// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_shared_vars.css.js';
import './profile_creation_shared.css.js';

import {CrButtonElement} from 'chrome://resources/cr_elements/cr_button/cr_button.js';
import {assert} from 'chrome://resources/js/assert.js';
import {focusWithoutInk} from 'chrome://resources/js/focus_without_ink.js';
import {I18nMixin} from 'chrome://resources/cr_elements/i18n_mixin.js';
import {loadTimeData} from 'chrome://resources/js/load_time_data.js';
import {WebUiListenerMixin} from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import {afterNextRender, PolymerElement} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

import {AutogeneratedThemeColorInfo, ManageProfilesBrowserProxy, ManageProfilesBrowserProxyImpl} from '../manage_profiles_browser_proxy.js';
import {navigateToPreviousRoute, navigateToStep, ProfileCreationSteps, recordPageVisited, Routes} from '../navigation_mixin.js';
// <if expr="chromeos_lacros">
import {AvailableAccount} from '../manage_profiles_browser_proxy.js';
import {navigateTo} from '../navigation_mixin.js';
// </if>

import {getTemplate} from './profile_type_choice.html.js';

export interface ProfileTypeChoiceElement {
  $: {
    backButton: CrButtonElement,
    notNowButton: CrButtonElement,
    signInButton: CrButtonElement,
  };
}

const ProfileTypeChoiceElementBase =
    WebUiListenerMixin(I18nMixin(PolymerElement));

export class ProfileTypeChoiceElement extends ProfileTypeChoiceElementBase {
  static get is() {
    return 'profile-type-choice';
  }

  static get template() {
    return getTemplate();
  }

  static get properties() {
    return {
      profileThemeInfo: Object,

      /**
       * True if a new profile (local or signed-in) is being created, all
       * buttons that create a new profile are then disabled (to avoid creating
       * two profiles).
       */
      profileCreationInProgress: {
        type: Boolean,
        notify: true,
      },

      /**
       * The disclaimer for managed devices.
       */
      managedDeviceDisclaimer_: {
        type: Boolean,
        value() {
          return loadTimeData.getString('managedDeviceDisclaimer').length > 0;
        },
      },

      // <if expr="chromeos_lacros">
      /**
       * If there are no available accounts, the account selection screen is
       * skipped.
       */
      hasAvailableAccounts_: {
        type: Boolean,
        value: false,
      },
      // </if>
    };
  }

  profileThemeInfo: AutogeneratedThemeColorInfo;
  profileCreationInProgress: boolean;
  private managedDeviceDisclaimer_: boolean;
  private manageProfilesBrowserProxy_: ManageProfilesBrowserProxy =
      ManageProfilesBrowserProxyImpl.getInstance();

  // <if expr="chromeos_lacros">
  private hasAvailableAccounts_: boolean;

  override connectedCallback() {
    super.connectedCallback();
    this.addWebUiListener(
        'available-accounts-changed',
        (accounts: AvailableAccount[]) =>
            this.handleAvailableAccountsChanged_(accounts));
    this.manageProfilesBrowserProxy_.getAvailableAccounts();
  }
  // </if>

  override ready() {
    super.ready();
    this.addWebUiListener(
        'load-signin-finished',
        (success: boolean) => this.handleLoadSigninFinished_(success));
    this.manageProfilesBrowserProxy_.recordSignInPromoImpression();

    this.addEventListener('view-enter-start', this.onViewEnterStart_);
  }

  private onViewEnterStart_() {
    afterNextRender(this, () => focusWithoutInk(this.$.backButton));
  }

  private onNotNowClick_() {
    navigateToStep(
        Routes.NEW_PROFILE, ProfileCreationSteps.LOCAL_PROFILE_CUSTOMIZATION);
  }

  private onSignInClick_() {
    // <if expr="chromeos_lacros">
    if (this.hasAvailableAccounts_) {
      navigateTo(Routes.ACCOUNT_SELECTION_LACROS);
      return;
    }
    // </if>

    assert(!this.profileCreationInProgress);
    this.profileCreationInProgress = true;

    // Explicitly record the page visit as this step is not pushed to the
    // history stack.
    recordPageVisited(ProfileCreationSteps.LOAD_SIGNIN);
    this.manageProfilesBrowserProxy_.selectNewAccount(
        this.profileThemeInfo.color);
  }

  private onBackClick_() {
    navigateToPreviousRoute();
  }

  private handleLoadSigninFinished_(_success: boolean) {
    // TODO(crbug.com/1444046): If failed, show some error message to inform the
    // user.
    this.profileCreationInProgress = false;
  }

  private getBackButtonAriaLabel_(): string {
    return this.i18n(
        'backButtonAriaLabel', this.i18n('profileTypeChoiceTitle'));
  }

  // <if expr="chromeos_lacros">
  private handleAvailableAccountsChanged_(availableAccounts:
                                              AvailableAccount[]) {
    this.hasAvailableAccounts_ = availableAccounts.length > 0;
  }
  // </if>
}

declare global {
  interface HTMLElementTagNameMap {
    'profile-type-choice': ProfileTypeChoiceElement;
  }
}

customElements.define(ProfileTypeChoiceElement.is, ProfileTypeChoiceElement);
