///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __COLOR_CODING_MODIFIER_H
#define __COLOR_CODING_MODIFIER_H

#include <core/Core.h>
#include <core/scene/animation/controller/Controller.h>
#include <atomviz/AtomViz.h>
#include "../AtomsObjectModifierBase.h"

namespace AtomViz {

/******************************************************************************
* Abstract base class for color gradients that can be used with the
* ColorCodingModifier.
* Its main purpose is to convert a scalar value in the range [0,1] to a color value.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ColorCodingGradient : public RefTarget
{
protected:

	/// Default constructor.
	ColorCodingGradient(bool isLoading = false) : RefTarget(isLoading) {}

public:

	/// \brief Converts a scalar value to a color value.
	/// \param t A value between 0 and 1.
	/// \return The color that visualizes the given scalar value.
	virtual Color valueToColor(FloatType t) = 0;

private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ColorCodingGradient)
};

/******************************************************************************
* Converts a scalar value to a color using the HSV color sytem.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ColorCodingHSVGradient : public ColorCodingGradient
{
public:
	/// Default constructor.
	ColorCodingHSVGradient(bool isLoading = false) : ColorCodingGradient(isLoading) {}

	/// \brief Converts a scalar value to a color value.
	/// \param t A value between 0 and 1.
	/// \return The color that visualizes the given scalar value.
	virtual Color valueToColor(FloatType t) { return Color::fromHSV((1.0 - t) * 0.7, 1, 1); }

private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ColorCodingHSVGradient)
};

/******************************************************************************
* Converts a scalar value to a color using a gray-scale ramp.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ColorCodingGrayscaleGradient : public ColorCodingGradient
{
public:
	/// Default constructor.
	ColorCodingGrayscaleGradient(bool isLoading = false) : ColorCodingGradient(isLoading) {}

	/// \brief Converts a scalar value to a color value.
	/// \param t A value between 0 and 1.
	/// \return The color that visualizes the given scalar value.
	virtual Color valueToColor(FloatType t) { return Color(t, t, t); }

private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ColorCodingGrayscaleGradient)
};

/******************************************************************************
* This modifier assigns a colors to the atoms based on the value of a
* selected data channel.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ColorCodingModifier : public AtomsObjectModifierBase
{
public:

	/// Default constructor.
	ColorCodingModifier(bool isLoading = false);

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// Selects the data channel that is the source for the coloring.
	void setSourceDataChannel(const QString& name) { _sourceDataChannel = name; }
	/// Returns the name of the data channel that is the source for the coloring.
	const QString& sourceDataChannel() const { return _sourceDataChannel; }

	/// Sets the vector component to be used from the source data channel if this channel contains multiple values per atom.
	void setSourceVectorComponent(int component) { _sourceVectorComponent = component; }
	/// Returns the vector component to be used from the source data channel if this channel contains multiple values per atom.
	int sourceVectorComponent() const { return _sourceVectorComponent; }

	/// Returns the range start value.
	FloatType startValue() const { return startValueCtrl ? startValueCtrl->getCurrentValue() : 0; }
	/// Sets the range start value.
	void setStartValue(FloatType value) { if(startValueCtrl) startValueCtrl->setCurrentValue(value); }
	/// Returns the controller for the range start value.
	FloatController* startValueController() const { return startValueCtrl; }
	/// Sets the controller for the range start value.
	void setStartValueController(const FloatController::SmartPtr& ctrl) { startValueCtrl = ctrl; }

	/// Returns the range end value.
	FloatType endValue() const { return endValueCtrl ? endValueCtrl->getCurrentValue() : 0; }
	/// Sets the range end value.
	void setEndValue(FloatType value) { if(endValueCtrl) endValueCtrl->setCurrentValue(value); }
	/// Returns the controller for the range end value.
	FloatController* endValueController() const { return endValueCtrl; }
	/// Sets the controller for the range end value.
	void setEndValueController(const FloatController::SmartPtr& ctrl) { endValueCtrl = ctrl; }

	/// Returns the color gradient used by the modifier to convert scalar atom properties to colors.
	ColorCodingGradient* colorGradient() const { return _colorGradient; }
	/// Sets the color gradient for the modifier to convert scalar atom properties to colors.
	void setColorGradient(const ColorCodingGradient::SmartPtr& gradient) { _colorGradient = gradient; }

	/// Sets the start and end value to the minimum and maximum value in the selected data channel.
	bool adjustRange();

public:

	Q_PROPERTY(QString sourceDataChannel READ sourceDataChannel WRITE setSourceDataChannel)
	Q_PROPERTY(int sourceVectorComponent READ sourceVectorComponent WRITE setSourceVectorComponent)
	Q_PROPERTY(FloatType startValue READ startValue WRITE setStartValue)
	Q_PROPERTY(FloatType endValue READ endValue WRITE setEndValue)

protected:

	/// Modifies the atoms object. The time interval passed
	/// to the function is reduced to the interval where the modified object is valid/constant.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// This controller stores the start value of the color scale.
	ReferenceField<FloatController> startValueCtrl;
	/// This controller stores the end value of the color scale.
	ReferenceField<FloatController> endValueCtrl;

	/// This object converts scalar atom properties to colors.
	ReferenceField<ColorCodingGradient> _colorGradient;

	/// The name of the data channel that is the source for the coloring.
	PropertyField<QString> _sourceDataChannel;
	/// The vector component to be used from the source data channel if this channel contains multiple values per atom.
	PropertyField<int> _sourceVectorComponent;

	friend class ColorCodingModifierEditor;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ColorCodingModifier)
	DECLARE_REFERENCE_FIELD(startValueCtrl)
	DECLARE_REFERENCE_FIELD(endValueCtrl)
	DECLARE_REFERENCE_FIELD(_colorGradient)
	DECLARE_PROPERTY_FIELD(_sourceDataChannel)
	DECLARE_PROPERTY_FIELD(_sourceVectorComponent)
};

/******************************************************************************
* A properties editor for the ColorCodingModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ColorCodingModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	/// The list of data channels.
	QComboBox* channelList;

	/// The list of available color gradients.
	QComboBox* colorGradientList;

	QLabel* colorLegendLabel;

protected Q_SLOTS:

	/// Updates the contents of the combo box.
	void updateChannelList();

	/// Updates the display for the color gradient.
	void updateColorGradient();

	/// Is called when the user selects a DataChannel in the list box.
	void onDataChannelSelected(int index);

	/// Is called when the user selects a color gradient in the list box.
	void onColorGradientSelected(int index);

	/// Is called when the user presses the "Adjust Range" button.
	void onAdjustRange();

	/// Is called when the user presses the "Reverse Range" button.
	void onReverseRange();

protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(ColorCodingModifierEditor)
};

};	// End of namespace AtomViz

#endif // __COLOR_CODING_MODIFIER_H
