#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QThread>
#include <QtWidgets/QTreeWidgetItem>
#include <jack/jack.h>
#include <jack/midiport.h>
#include "model.h"
#include "jackprocessor.h"

JackProcessor::JackProcessor(Model *p_model, QObject *parent) : QObject(parent) {

  int i1;

  model = p_model;
  sustain = false;
  for (i1 = 0 ; i1 < 2; i1++) {
    last_frame[i1] = 0;
  }  
}

JackProcessor::~JackProcessor() {

  jack_client_close (jackHandle);
}

int JackProcessor::initJack() {

  int i1;
  QString qs;
  
  if ((jackHandle = jack_client_open("Add64", JackNullOption, NULL)) == 0) {
    fprintf (stderr, "jack server not running ?\n");
    exit (1);
  }
  for (i1 = 0; i1 < 2; i1++) {
    qs.sprintf("add64_out_%d", i1);
    jackOut [i1] = jack_port_register (jackHandle, qs.toLatin1(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
  }
  jackMidiIn = jack_port_register (jackHandle, "add64_midi_in", JACK_DEFAULT_MIDI_TYPE, JackPortIsInput, 0);
  jackMidiOut = jack_port_register (jackHandle, "add64_midi_out", JACK_DEFAULT_MIDI_TYPE, JackPortIsOutput, 0);
  jack_set_process_callback (jackHandle, jack_static_callback, (void *)this);
  if (model->getInterpolate()) {
    model->setRate(jack_get_sample_rate(jackHandle) >> 1);
  } else {
    model->setRate(jack_get_sample_rate(jackHandle));
  }  
  if (jack_activate (jackHandle)) {
    fprintf(stderr, "Can't activate JACK");
    exit (1);
  }  
  return(0);
}

int JackProcessor::jack_static_callback(jack_nframes_t nframes, void *arg)
{
  return ((JackProcessor *) arg)->jack_callback (nframes);
}

int JackProcessor::jack_callback(jack_nframes_t nframes)
{
  jack_default_audio_sample_t *p[2];
  jack_midi_event_t ev;
  jack_nframes_t ev_count;
  int type, ch, index, val;
  jack_nframes_t cframes;
  int i1, i2, voiceIndexNoGate, voiceIndexGate, voiceIndex;
  unsigned int i3;
  long maxSamples, maxSamplesNoGate;
  voiceType *v;
  envelopeType *e[MAX_POLY];
  bool noteOn, noteOff;  
  modulationLfo modLfo, modSpacingLfo;
  int vStart, vEnd;
  MidiEventList outputEvents;
  unsigned char *buffer;
  long frame;
   
  if (nframes > MAX_FRAMES) { 
    fprintf(stderr, "nframes > %d\n", MAX_FRAMES);
    return 0;
  }
  cframes = (model->getInterpolate()) ? nframes>>1 : nframes;
  void* midiOutBuf = jack_port_get_buffer(jackMidiOut, nframes);
  jack_midi_clear_buffer(midiOutBuf);
  QMutexLocker jackMutexLocker(&model->jackMutex);
  outputEvents = model->fetchEvents(nframes);
  for (i1 = 0; i1 < outputEvents.count(); i1++) {
    frame = outputEvents.at(i1).frame;
    if (outputEvents.at(i1).type == 0xB0) {
      buffer = jack_midi_event_reserve(midiOutBuf, frame, 3);
      buffer[0] = outputEvents.at(i1).type | outputEvents.at(i1).ch;
      buffer[1] = outputEvents.at(i1).index;
      buffer[2] = outputEvents.at(i1).val;
    }
  }
  model->setBufSize(cframes);
  model->getEditRange(vStart, vEnd);
  v = model->getVoices();
  for (i1 = 0; i1 < model->getPoly(); i1++) {
    e[i1] = model->getVoiceData(i1)->getEnvelopes();
  }  
  void* midiBuf = jack_port_get_buffer(jackMidiIn, nframes);
  ev_count = jack_midi_get_event_count(midiBuf);
  while (ev_count) {
    ev_count--;
    jack_midi_event_get(&ev, midiBuf, ev_count);
    type = *(ev.buffer) & 0xf0;
    ch = *(ev.buffer) & 0x0f;
    index = *(ev.buffer + 1);
    val = *(ev.buffer + 2);
    noteOn = false;
    noteOff = false;
    if (type == 0x90) {
      if (val) {
        noteOn = true;
      } else {
        noteOff = true;
      }
    }
    if (type == 0x80) {
      noteOff = true;
    }
    if (noteOn) {
      voiceIndexNoGate = -1;
      voiceIndexGate = -1;
      voiceIndex = -1;
      maxSamples = 0;
      maxSamplesNoGate = 0;
      for (i1 = 0; i1 < model->getPoly(); i1++) {
        if ((model->getVoiceData(i1)->getChannel() == ch) && !v[i1].gate && (v[i1].samples >= maxSamplesNoGate)) {
          maxSamplesNoGate = v[i1].samples;
          voiceIndexNoGate = i1;
        }  
      }  
      for (i1 = 0; i1 < model->getPoly(); i1++) {
        if ((model->getVoiceData(i1)->getChannel() == ch) && v[i1].gate && (v[i1].samples >= maxSamples)) {
          maxSamples = v[i1].samples;
          voiceIndexGate = i1;
        }  
      }  
      voiceIndex = (maxSamplesNoGate) ? voiceIndexNoGate : voiceIndexGate; // use voice in release if available
      if (voiceIndex >= 0) {
        emit MIDI_noteOn(voiceIndex);
        v[voiceIndex].note = index;
        v[voiceIndex].velocity = val;
        v[voiceIndex].gate = true;
        v[voiceIndex].active = true;
        v[voiceIndex].sustain = false;
        v[voiceIndex].samples = 0;
        v[voiceIndex].cut = 20000.0;
        for (i1 = 0; i1 < MAX_SOURCES * MAX_SCENES; i1++) {
          for (i2 = model->getHarmonicSplit(i1); i2 < model->getHarmonicSplit(i1 + 1); i2++) {
            e[voiceIndex][i2].state = 1; 
            e[voiceIndex][i2].val = 0;
//            e[voiceIndex][i2].lfo_state = 1;
          }
          if (model->getVoiceData(voiceIndex)->getFilterMixLfo(i1 % 4)) {
            if (model->getVoiceData(voiceIndex)->getFilterMixRetrigger(i1 % 4)) {
              v[voiceIndex].filterMixEnv[i1 % 4] = 0;
              v[voiceIndex].filterMixEnvState[i1 % 4] = 1;
              v[voiceIndex].filterMixSampleStack[i1 % 4] = 0;
            } else {
              if (v[voiceIndex].filterMixEnvState[i1 % 4] < 1) {
                v[voiceIndex].filterMixEnvState[i1 % 4] = 1;
                v[voiceIndex].filterMixSampleStack[i1 % 4] = 0;
              }
            }  
          } else {
            v[voiceIndex].filterMixEnvState[i1 % 4] = 1;
            v[voiceIndex].filterMixEnv[i1 % 4] = 0;
            v[voiceIndex].filterMixSampleStack[i1 % 4] = 0;
          }   
        }
        for (i1 = 0; i1 < 2; i1++) {
          modLfo = model->getVoiceData(voiceIndex)->getFilterModLfo(i1);
          modSpacingLfo = model->getVoiceData(voiceIndex)->getFilterModSpacingLfo(i1);
          v[voiceIndex].filterModEnv[i1] = 0;
          v[voiceIndex].filterModEnvState[i1] = 1;
          if (modLfo.retrigger) {
            v[voiceIndex].filterModLfo[i1] = 0;
            v[voiceIndex].filterModLfoState[i1] = 1;
          } else {
            if (v[voiceIndex].filterModLfoState[i1] < 1) {
              v[voiceIndex].filterModLfo[i1] = 0;
              v[voiceIndex].filterModLfoState[i1] = 1;
            }  
          }
          if (modSpacingLfo.retrigger) {  
            v[voiceIndex].filterModSpacingLfo[i1] = 0;
            v[voiceIndex].filterModSpacingLfoState[i1] = 1;
          } else {
            if (v[voiceIndex].filterModSpacingLfoState[i1] < 1) {
              v[voiceIndex].filterModSpacingLfo[i1] = 0;
              v[voiceIndex].filterModSpacingLfoState[i1] = 1;
            }
          }  
        }    
      }
      emit MIDI_controlEvent(1, ch, 0, index, voiceIndex);
      emit MIDI_controlEvent(2, ch, 0, val, voiceIndex);
      srand(index * val);
    }
    if (noteOff) {
      for (i1 = 0; i1 < model->getPoly(); i1++) {
        if ((model->getVoiceData(i1)->getChannel() == ch) 
          && (v[i1].note == index) && v[i1].gate) {
          if (sustain) {
            v[i1].sustain = true;
          } else {   
            v[i1].gate = false;
            for (i2 = 0; i2 < model->getNumHarmonics(); i2++) {
              e[i1][i2].state = 3;
            }  
            for (i2 = 0; i2 < 2; i2++) {
              v[i1].filterModEnvState[i2] = 3;
            }
            break;
          }
        }
      }  
    }
    if (type == 0xE0) {
      for (int vl = vStart; vl <= vEnd; vl++) {
        model->getVoiceData(vl)->setPitchBend((double)((index + (val<<7)) - 8192) / 8192.0);
      }  
    }
    if (type == 0xB0) {
      if (index == 64) {
        if (val > 63) {
          sustain = true;
        } else {
          sustain = false;
          for (i1 = 0; i1 < model->getPoly(); i1++) {
            if ((model->getVoiceData(i1)->getChannel() == ch) 
              && (v[i1].sustain == true)) {
              v[i1].gate = false;
              v[i1].sustain = false;
              for (i2 = 0; i2 < model->getNumHarmonics(); i2++) {
                e[i1][i2].state = 3;
              }  
              for (i2 = 0; i2 < 2; i2++) {
                v[i1].filterModEnvState[i2] = 3;
              }
            }  
          }
        }
      }  
      emit MIDI_controlEvent(3, ch, index, val, -1);
    }
    if (type == 0xD0) {
      emit MIDI_controlEvent(4, ch, 0, index, -1);
    }
    if (type == 0xA0) {
      emit MIDI_controlEvent(5, ch, index, val, -1);
    }
  }
  if (model->getInterpolate()) {
    for (i1 = 0; i1 < 2; i1++) {
      p[i1] = (jack_default_audio_sample_t *)(jack_port_get_buffer (jackOut[i1], nframes));
      p[i1][0] = 0.5e-9 * (last_frame[i1] + model->buf[i1][0]);
      p[i1][1] = 1e-9 * model->buf[i1][0];
      for (i3 = 1; i3 < cframes; i3++) {
        p[i1][i3 * 2] = 0.5e-9 * (model->buf[i1][i3 - 1] + model->buf[i1][i3]);
        p[i1][i3 * 2 + 1] = 1e-9 * model->buf[i1][i3];
      }
      last_frame[i1] = model->buf[i1][cframes -1];
    }
  } else {
    for (i1 = 0; i1 < 2; i1++) {
      p[i1] = (jack_default_audio_sample_t *)(jack_port_get_buffer (jackOut[i1], nframes));
      for (i3 = 0; i3 < nframes; i3++) {
        p[i1][i3] = 1e-9 * model->buf[i1][i3];
      }
    }  
  }
  if (model->doSynthesis()) {
    model->jackWait.wakeAll();
  } else {
    model->jackWait.wakeAll();
    QThread::msleep(30);
    model->voiceWait.wakeAll();
    QThread::msleep(30);
    model->synthWait.wakeAll();
    model->jackMutex.unlock();  
    jack_client_close (jackHandle);
    model->quitMutex.lock();
    model->quitWait.wait(&model->quitMutex);
    model->quitMutex.unlock();
  }
  return 0;
}

QStringList JackProcessor::getConnections() {

  const char **connectedAudioPorts[2], **connectedMidiInPorts, **connectedMidiOutPorts;
  int i1, i2;
  QStringList connectionList;
  
  for (i1 = 0; i1 < 2; i1++) {
    connectedAudioPorts[i1] = jack_port_get_connections(jackOut[i1]);
  }  
  connectedMidiInPorts = jack_port_get_connections(jackMidiIn);
  connectedMidiOutPorts = jack_port_get_connections(jackMidiOut);
  for (i2 = 0; i2 < 2; i2++) {
    if (i2) connectionList.append("--");
    i1 = 0;
    while ((connectedAudioPorts[i2] != NULL) && connectedAudioPorts[i2][i1]) {
      connectionList.append(QString::fromLatin1(connectedAudioPorts[i2][i1]));
      i1++;
    }  
  }
  connectionList.append("--");
  i1 = 0;
  while ((connectedMidiInPorts != NULL) && connectedMidiInPorts[i1]) {
    connectionList.append(QString::fromLatin1(connectedMidiInPorts[i1]));
    i1++;
  }
  connectionList.append("--");
  i1 = 0;
  while ((connectedMidiOutPorts != NULL) && connectedMidiOutPorts[i1]) {
    connectionList.append(QString::fromLatin1(connectedMidiOutPorts[i1]));
    i1++;
  }
  connectionList.append("--");
  return(connectionList);
}

void JackProcessor::setConnections(QStringList connectionList) {

  int i1;
  
  i1 = 0;
  while (!connectionList.at(i1).contains("--")) {
    jack_connect(jackHandle, "Add64:add64_out_0", connectionList.at(i1).trimmed().toLatin1().constData());
    i1++;
  }
  i1++;
  while (!connectionList.at(i1).contains("--")) {
    jack_connect(jackHandle, "Add64:add64_out_1", connectionList.at(i1).trimmed().toLatin1().constData());
    i1++;
  }
  i1++;
  while (!connectionList.at(i1).contains("--")) {
    jack_connect(jackHandle, connectionList.at(i1).trimmed().toLatin1().constData(), "Add64:add64_midi_in");
    i1++;
  }
  i1++;
  while (!connectionList.at(i1).contains("--")) {
    jack_connect(jackHandle, "Add64:add64_midi_out", connectionList.at(i1).trimmed().toLatin1().constData());
    i1++;
  }
}
