/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "RenderScene.h"
#include "TimeFunction.h"
#include <glm/gtx/transform.hpp>
#include <glm/gtc/matrix_inverse.hpp>
#include <glm/gtc/type_ptr.hpp>
#include <glm/gtc/quaternion.hpp>
#include <glm/gtx/quaternion.hpp>
#include <cerrno>

using namespace std;

namespace libgltf
{

RenderWithFBO::RenderWithFBO()
    : mFboProId(0)
    , mFboId(0)
    , mRboId(0)
    , mTexId(0)
    , mShotTexId(0)
    , mRenderTexId(0)
    , mInverseFboId(0)
    , mInverseRboId(0)
    , mInverseTexId(0)
    , mMSAAFboId(0)
    , mMSAARboId(0)
    , mMSAATexId(0)
    , mVertexBuf(0)
    , mTexCoordBuf(0)
{
}

void RenderWithFBO::inverseBitMap(int width, int height)
{
    GLuint proId = loadFboShader(INVERSEVERT, INVERSEFRAG);
    GLuint texCoordBuf = 0;
    GLuint vertexBuf = 0;
    GLfloat coordVertices[] =
    {
        0.0f, 1.0f,
        1.0f, 1.0f,
        1.0f, 0.0f,
        0.0f, 0.0f,
    };
    GLfloat squareVertices[] =
    {
        -1.0f, -1.0f, -1.0,
         1.0f, -1.0f, -1.0,
         1.0f,  1.0f, -1.0,
        -1.0f,  1.0f, -1.0
    };
    setBufferForFbo(texCoordBuf, vertexBuf, coordVertices,
                    sizeof(coordVertices), squareVertices,
                    sizeof(squareVertices));
    createAndBindInverseFBO(width, height);
    glViewport(0, 0, width, height);
    inverseTexture(proId, texCoordBuf, vertexBuf);
}

void RenderWithFBO::releaseBitMapFBO()
{
    glDeleteFramebuffers(1, &mInverseFboId);
    glDeleteRenderbuffers(1, &mInverseRboId);
    glDeleteTextures(1, &mInverseTexId);

}

void RenderWithFBO::releaseBitmapTexture()
{
    glDeleteTextures(1, &mShotTexId);
}

void RenderWithFBO::releaseMSAAFBO()
{
    if( mMSAAFboId != 0 )
    {
        glDeleteFramebuffers(1, &mMSAAFboId);
        glDeleteRenderbuffers(1, &mMSAARboId);
        glDeleteTextures(1, &mMSAATexId);
    }
}
GLuint RenderWithFBO::loadFboShader(const char* vShader,
                                    const char* fShader)
{
    ShaderProgram shaderPro;
    GLuint progId = glCreateProgram();
    if(false == shaderPro.loadShader(progId, vShader,
                                     strlen(vShader), GL_VERTEX_SHADER))
    {
        return 0;
    }
    if(false == shaderPro.loadShader(progId, fShader,
                                     strlen(fShader), GL_FRAGMENT_SHADER))
    {
        return 0;
    }
    return progId;
}

void RenderWithFBO::setBufferForFbo(GLuint& texCoordBuf, GLuint& vertexBuf,
                                    GLfloat* pCoord, GLuint numCoord,
                                    GLfloat* pSquare, GLuint numSquare)
{
    glGenBuffers(1, &texCoordBuf);
    glBindBuffer(GL_ARRAY_BUFFER, texCoordBuf);
    glBufferData(GL_ARRAY_BUFFER, numCoord, pCoord, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);

    glGenBuffers(1, &vertexBuf);
    glBindBuffer(GL_ARRAY_BUFFER, vertexBuf);
    glBufferData(GL_ARRAY_BUFFER, numSquare, pSquare, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void RenderWithFBO::createAndBindInverseFBO(int width, int height)
{
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    createRenderObj(width, height, mInverseRboId);
    createTextureObj(width, height, mInverseTexId);
    createFrameBufferObj(mInverseFboId, mInverseTexId, mInverseRboId);
    glBindFramebuffer(GL_FRAMEBUFFER, mInverseFboId);
}

void RenderWithFBO::createBitmapTexture(int width, int height)
{
    unsigned char* buf = (unsigned char*)malloc(width*height*3);
    glBindFramebuffer(GL_FRAMEBUFFER, mFboId);
    glReadPixels(0, 0, width, height, GL_BGR, GL_UNSIGNED_BYTE, buf);

    glGenTextures(1, &mShotTexId);
    glBindTexture(GL_TEXTURE_2D, mShotTexId);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, width, height, 0, GL_BGR,
        GL_UNSIGNED_BYTE, buf);
    glBindTexture(GL_TEXTURE_2D, 0);
    free(buf);
}

void RenderWithFBO::inverseTexture(GLuint proId, GLuint texCoordBuf,
                                   GLuint vertexBuf)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    GLuint texCoordId = glGetAttribLocation(proId, "texCoord");
    GLuint vertexId   = glGetAttribLocation(proId, "vPosition");
    GLint textureId  = glGetUniformLocation(proId, "RenderTex");
    if( textureId == -1 )
        return;

    glUseProgram(proId);
    glEnableVertexAttribArray(vertexId);
    glBindBuffer(GL_ARRAY_BUFFER, vertexBuf);
    glVertexAttribPointer(vertexId, 3, GL_FLOAT, GL_FALSE, 0, (void*)0);
    glEnableVertexAttribArray(texCoordId);
    glBindBuffer(GL_ARRAY_BUFFER, texCoordBuf);
    glVertexAttribPointer(texCoordId, 2, GL_FLOAT, GL_FALSE, 0, (void*)0);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, mShotTexId);
    glUniform1i(textureId, 0);
    glDrawArrays(GL_QUADS, 0, 4);
    glDisableVertexAttribArray(vertexId);
    glDisableVertexAttribArray(texCoordId);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glUseProgram(0);
}

void RenderWithFBO::createRenderObj(int width, int height, GLuint& rboId)
{
    glGenRenderbuffers(1, &rboId);
    glBindRenderbuffer(GL_RENDERBUFFER, rboId);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT, width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
}

void RenderWithFBO::createTextureObj(int width, int height, GLuint& texId)
{
    glGenRenderbuffers(1, &texId);
    glBindRenderbuffer(GL_RENDERBUFFER, texId);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_RGB, width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);

    glGenTextures(1, &mRenderTexId);
    glBindTexture(GL_TEXTURE_2D, mRenderTexId);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
}

int RenderWithFBO::createFrameBufferObj(GLuint& fboId, GLuint texId,
                                        GLuint rboId)
{
    int status;
    glGenFramebuffers(1, &fboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindFramebuffer(GL_FRAMEBUFFER, fboId);

    glBindRenderbuffer(GL_RENDERBUFFER, texId);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,
        GL_RENDERBUFFER, texId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindRenderbuffer(GL_RENDERBUFFER, 0);

    glBindRenderbuffer(GL_RENDERBUFFER, rboId);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                              GL_RENDERBUFFER, rboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::createMultiSampleTextureFrameBufObj(GLuint& fboId,
                                                       GLuint& texId,
                                                       GLuint&  rboId,
                                                       int width,
                                                       int height)
{
    GLenum status;
    glGenFramebuffers(1, &fboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindFramebuffer(GL_FRAMEBUFFER, fboId);
    //create MSAA texture
    glGenRenderbuffers(1, &texId);
    glBindRenderbuffer(GL_RENDERBUFFER, texId);
    glRenderbufferStorageMultisample(GL_RENDERBUFFER, 4,  GL_RGB, width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,
                              GL_RENDERBUFFER, texId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glGenRenderbuffers(1, &rboId);
    glBindRenderbuffer(GL_RENDERBUFFER, rboId);
    glRenderbufferStorageMultisample(GL_RENDERBUFFER, 4, GL_DEPTH_COMPONENT24,
                                     width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                              GL_RENDERBUFFER, rboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::renderFbo(int srcWidth, int srcHeight)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glBindFramebuffer(GL_FRAMEBUFFER, mFboId);
    glBindTexture(GL_TEXTURE_2D, mRenderTexId);
    glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 0, 0, srcWidth, srcHeight, 0);
    glBindTexture(GL_TEXTURE_2D, 0);
    renderFboTexture();
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::renderFboTexture()
{
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glUseProgram(mFboProId);

    GLuint texCoordId = glGetAttribLocation(mFboProId, "texCoord");
    GLuint vertexId   = glGetAttribLocation(mFboProId, "vPosition");
    GLint textureId  = glGetUniformLocation(mFboProId, "RenderTex");
    if( textureId == -1 )
        return LIBGLTF_UNKNOWN_ERROR;

    glEnableVertexAttribArray(vertexId);
    glBindBuffer(GL_ARRAY_BUFFER, mVertexBuf);
    glVertexAttribPointer(vertexId, 2, GL_FLOAT, GL_FALSE, 0, (void*)0);

    glEnableVertexAttribArray(texCoordId);
    glBindBuffer(GL_ARRAY_BUFFER, mTexCoordBuf);
    glVertexAttribPointer(texCoordId, 2, GL_FLOAT, GL_FALSE, 0, (void*)0);

    glBindTexture(GL_TEXTURE_2D, mRenderTexId);
    glUniform1i(textureId, 0);
    glDrawArrays(GL_QUADS, 0, 4);
    glDisableVertexAttribArray(texCoordId);
    glDisableVertexAttribArray(vertexId);
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::createAndBindFbo(int width, int height)
{
    if (0 != mFboId)
    {
        return LIBGLTF_SUCCESS;
    }

    mFboProId = loadFboShader(FBOVERT, FBOFRAG);
    if(mFboProId == 0)
    {
        return LIBGLTF_SHADER_ERROR;
    }
    createRenderObj(width, height, mRboId);
    createTextureObj(width, height, mTexId);
    int status = createFrameBufferObj(mFboId, mTexId, mRboId);
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
#if USE_MSAA
    status = createMultiSampleTextureFrameBufObj(mMSAAFboId, mMSAATexId,
                                                 mMSAARboId, width, height);
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
#endif //USE_MSAA
    GLfloat coordVertices[] =
    {
        0.0f, 0.0f,
        1.0f, 0.0f,
        1.0f, 1.0f,
        0.0f, 1.0f,
    };
    GLfloat squareVertices[] =
    {
        -1.0f, -1.0f,
         1.0f, -1.0f,
         1.0f,  1.0f,
        -1.0f,  1.0f,
    };
    setBufferForFbo(mTexCoordBuf, mVertexBuf, coordVertices,
                    sizeof(coordVertices), squareVertices,
                    sizeof(squareVertices));
    return LIBGLTF_SUCCESS;
}

void RenderWithFBO::releaseFbo()
{
    if (0 != mFboId)
    {
        glDeleteFramebuffers(1, &mFboId);
        glDeleteRenderbuffers(1, &mRboId);
        glDeleteTextures(1, &mTexId);
        mFboId = 0;
    }
    if (0 != mShotTexId)
    {
        glDeleteTextures(1, &mShotTexId);
    }
    if (0 != mRenderTexId)
    {
        glDeleteTextures(1, &mRenderTexId);
    }
}

RenderPrimitive::RenderPrimitive()
    : mIndicesDataType(DataType_UNKNOW)
    , mpMaterial(0)
    , pNode(0)
    , mVerterCount(0)
    , mIndicesCount(0)
    , mVertexBuffer(0)
    , mNormalBuffer(0)
    , mTexCoordBuffer(0)
    , mJointBuffer(0)
    , mWeightBuffer(0)
    , mIndicesBuffer(0)
    , mVertexBufferData(0)
    , mIndiceBufferData(0)
    , mSortedIndiceBufferData(0)
    , mIndiceBufferLen(0)
    , mEyeSpaceVerts()
    , mPrimitiveZ()
    , mIsPolygonSorted(false)
{
}

RenderPrimitive::~RenderPrimitive()
{
    delete[] mVertexBufferData;
    delete[] mIndiceBufferData;
    delete[] mSortedIndiceBufferData;

    glDeleteBuffers(1, &mVertexBuffer);
    glDeleteBuffers(1, &mNormalBuffer);
    glDeleteBuffers(1, &mTexCoordBuffer);
    glDeleteBuffers(1, &mIndicesBuffer);
}

Node* RenderPrimitive::getNode()
{
    return pNode;
}

void RenderPrimitive::setNode(Node* node)
{
    pNode = node;
}

Material* RenderPrimitive::getMaterial()
{
    return mpMaterial;
}

void RenderPrimitive::setMaterial(Material* pMaterial)
{
    mpMaterial = pMaterial;
}

unsigned int RenderPrimitive::getVerterCount()
{
    return mVerterCount;
}

void RenderPrimitive::setVerterCount(unsigned int count)
{
    mVerterCount = count;
}

unsigned int RenderPrimitive::getIndicesCount()
{
    return mIndicesCount;
}

void RenderPrimitive::setIndicesCount(unsigned int count)
{
    mIndicesCount = count;
}

DataType RenderPrimitive::getIndicesDataType()
{
    return mIndicesDataType;
}

void RenderPrimitive::setIndicesDataType(DataType type)
{
    mIndicesDataType = type;
}

void RenderPrimitive::copyIndiceBufferData(const char* srcBuf,
                                           unsigned int bufSize)
{
    if (0 == mIndiceBufferData)
    {
        mIndiceBufferData = new char[bufSize];
        memcpy(mIndiceBufferData,srcBuf,bufSize);
        mIndiceBufferLen = bufSize;
    }
    if (0 == mSortedIndiceBufferData)
    {
        mSortedIndiceBufferData = new char[bufSize];
    }
}
void RenderPrimitive::polyonSorting(glm::mat4& viewMatrix)
{
    mEyeSpaceVerts.resize(mVerterCount);
    glm::vec3* pVertices = (glm::vec3*)mVertexBufferData;
    for (size_t i = 0; i < mVerterCount; ++i)
    {
        mEyeSpaceVerts[i] = pVertices[i].x * viewMatrix[2][0]
                          + pVertices[i].y * viewMatrix[2][1]
                          + pVertices[i].z * viewMatrix[2][2]
                          + viewMatrix[2][3];
    }
    sortIndices();
}

void RenderPrimitive::sortIndices()
{
    mPrimitiveZ.resize(mIndicesCount / 3);
    if (mPrimitiveZ.empty())
    {
        mIsPolygonSorted = false;
        return;
    }
    if (DataType_UNSIGNED_SHORT == mIndicesDataType)
    {
        unsigned short* indexBuf = (unsigned short*)mIndiceBufferData;
        for (unsigned iz = 0; iz < mIndicesCount / 3; ++iz)
        {
            mPrimitiveZ[iz].mZ = (float)(mEyeSpaceVerts[indexBuf[iz * 3]]
                               + mEyeSpaceVerts[indexBuf[iz * 3 + 1]]
                               + mEyeSpaceVerts[indexBuf[iz * 3 + 2]]);
            mPrimitiveZ[iz].mPrimitiveIndex = iz;
        }
        std::sort(mPrimitiveZ.begin(), mPrimitiveZ.end(),
                  SorterBackToFront());
        unsigned short* sortedIndexBuf =
            (unsigned short*)mSortedIndiceBufferData;
        for (unsigned int i = 0; i < mPrimitiveZ.size(); ++i)
        {
            sortedIndexBuf[i * 3] =
                indexBuf[mPrimitiveZ[i].mPrimitiveIndex * 3];
            sortedIndexBuf[i * 3 + 1] =
                indexBuf[mPrimitiveZ[i].mPrimitiveIndex * 3 + 1];
            sortedIndexBuf[i * 3 + 2] =
                indexBuf[mPrimitiveZ[i].mPrimitiveIndex * 3 + 2];
        }
        mIsPolygonSorted = true;
    }
    else if (DataType_UNSIGNED_INT == mIndicesDataType)
    {
        unsigned int* indexBuf = (unsigned int*)mIndiceBufferData;
        for (unsigned iz = 0; iz < mIndicesCount / 3; ++iz)
        {
            mPrimitiveZ[iz].mZ = (float)(mEyeSpaceVerts[indexBuf[iz * 3]]
                               + mEyeSpaceVerts[indexBuf[iz * 3 + 1]]
                               + mEyeSpaceVerts[indexBuf[iz * 3 + 2]]);
            mPrimitiveZ[iz].mPrimitiveIndex = iz;
        }
        std::sort(mPrimitiveZ.begin(), mPrimitiveZ.end(),
                  SorterBackToFront());
        unsigned int* sortedIndexBuf = (unsigned int *)mSortedIndiceBufferData;
        for (unsigned int i= 0; i < mPrimitiveZ.size(); ++i)
        {
            sortedIndexBuf[i * 3] =
                indexBuf[ mPrimitiveZ[i].mPrimitiveIndex * 3];
            sortedIndexBuf[i * 3 + 1] =
                indexBuf[ mPrimitiveZ[i].mPrimitiveIndex * 3 + 1];
            sortedIndexBuf[i * 3 + 2] =
                indexBuf[ mPrimitiveZ[i].mPrimitiveIndex * 3 + 2];
        }
        mIsPolygonSorted = true;
    }
    else
    {
        mIsPolygonSorted = false;
    }
}

void RenderPrimitive::bindSortedIndicesBuf()
{
    if (mIsPolygonSorted)
    {
        glDeleteBuffers(1, &mIndicesBuffer);
        glGenBuffers(1, &mIndicesBuffer);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, mIndicesBuffer);
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, mIndiceBufferLen,
                     mSortedIndiceBufferData, GL_STATIC_DRAW);
    }
}

void RenderPrimitive::copyVertexBufferData(const char* srcBuf,
                                           unsigned int bufSize)
{
    if (0 == mVertexBufferData)
    {
        mVertexBufferData = new char[bufSize];
        memcpy(mVertexBufferData, srcBuf, bufSize);
    }
}

void RenderPrimitive::getPrimitiveBoundary(glm::vec3* vertexMax,
    glm::vec3* vertexMin)
{
    const glm::mat4& modelMatrix = pNode->getGlobalMatrix();
    glm::vec3* pVertices = (glm::vec3*)mVertexBufferData;
    for (size_t i = 0; i < mVerterCount; ++i)
    {
        glm::vec4 tmpPos = glm::vec4(pVertices[i],1.0f);
        glm::vec4 globalPos = modelMatrix*tmpPos;
        vertexMax->x = max(globalPos.x,vertexMax->x);
        vertexMax->y = max(globalPos.y,vertexMax->y);
        vertexMax->z = max(globalPos.z,vertexMax->z);
        vertexMin->x = min(globalPos.x,vertexMin->x);
        vertexMin->y = min(globalPos.y,vertexMin->y);
        vertexMin->z = min(globalPos.z,vertexMin->z);
    }
}

unsigned int RenderPrimitive::getVertexBuffer()
{
    return mVertexBuffer;
}

void RenderPrimitive::setVertexBuffer(unsigned int bufferId)
{
    mVertexBuffer = bufferId;
}

unsigned int RenderPrimitive::getNormalBuffer()
{
    return mNormalBuffer;
}

void RenderPrimitive::setNormalBuffer(unsigned int bufferId)
{
    mNormalBuffer = bufferId;
}

unsigned int RenderPrimitive::getTexCoordBuffer()
{
    return mTexCoordBuffer;
}

void RenderPrimitive::setTexCoordBuffer(unsigned int bufferId)
{
    mTexCoordBuffer = bufferId;
}

unsigned int RenderPrimitive::getJointBuffer()
{
    return mJointBuffer;
}

void RenderPrimitive::setJointBuffer(unsigned int bufferId)
{
    mJointBuffer = bufferId;
}

unsigned int RenderPrimitive::getWeightBuffer()
{
    return mWeightBuffer;
}

void RenderPrimitive::setWeightBuffer(unsigned int bufferId)
{
    mWeightBuffer = bufferId;
}

unsigned int RenderPrimitive::getIndicesBuffer()
{
    return mIndicesBuffer;
}

void RenderPrimitive::setIndicesBuffer(unsigned int bufferId)
{
    mIndicesBuffer = bufferId;
}

RenderShader::RenderShader()
    : mPrimitiveVec()
    , mpTechnique(0)
{
}

RenderShader::~RenderShader()
{
    for (unsigned int i = 0, size = mPrimitiveVec.size();
         i < size; ++i)
    {
        delete mPrimitiveVec[i];
    }
    mPrimitiveVec.clear();
}

Technique* RenderShader::getTechnique()
{
    return mpTechnique;
}

void RenderShader::setTechnique(Technique* pTechnique)
{
    mpTechnique = pTechnique;
}

void RenderShader::pushRenderPrim(RenderPrimitive* p)
{
    (mPrimitiveVec).push_back(p);
}

RenderPrimitive* RenderShader::getRenderPrim(unsigned int index)
{
    if (mPrimitiveVec.size() <= index)
    {
        return 0;
    }
    return mPrimitiveVec[index];
}

unsigned int RenderShader::getRenderPrimSize()
{
    return (mPrimitiveVec).size();
}

ShaderProgram RenderScene::mShaderProgram = ShaderProgram();

RenderScene::RenderScene()
    : maCamera()
    , cCamera(0)
    , vCameraIndex()
    , mOrbitInitViewMatrix(0.0)
    , mWalkthroughInitViewMatrix(0.0)
    , IsAerialMode(false)
    , bAerialView(false)
    , oldTime(0.0)
    , flyInfo(1.0f)
    , flyTime(0.0)
    , bFlyCamera(false)
    , bAnimation(true)
    , pLight(0)
    , pTempMatrix(0)
    , mAnimationPlay(false)
    , mAnimationLoop(true)
    , mCurrentTime(0)
    , mLastPlaying(0)
    , mUpdateTimeOut(0)
    , mDuration(0)
    , mCameraMoveTime(0)
    , mPreCameraTime(0)
    , mShaderVec()
    , pScene(0)
    , mLoadJson()
    , mBindBufferMap()
    , mCurrentViewport()
    , fbo()
    , mEnableTransparency(false)
    , mEnableRotation(true)
    , mLastModelView(glm::mat4(0.0))
    , bIsTimeAvailable(false)
    , pFPSCounter(0)
#if ENABLE_TIMER
    , pTimer(new Timer())
#endif
#if LOAD_ONCE
    , mCurrentImage()
    , mCurrentTextNumber(0)
#endif
{
}

RenderScene::~RenderScene()
{
    delete pFPSCounter;

#if ENABLE_TIMER
    delete pTimer;
#endif

    delete pLight;

    for (unsigned int i = 0, size = mShaderVec.size();
         i < size; ++i)
    {
         delete mShaderVec[i];
    }
    mShaderVec.clear();
    mBindBufferMap.clear();
    delete pScene;
    delete[] pTempMatrix;
}

int RenderScene::loadScene(const std::vector<glTFFile>& inputFiles)
{
    try
    {
        int iStatus = mLoadJson.parseScene(inputFiles);
        return iStatus;
    }
    catch (boost::property_tree::ptree_error& e)
    {
        fprintf(stderr, "%s\n", e.what());
        return LIBGLTF_PARSE_JSON_ERROR;
    }
    catch (boost::exception const&)
    {
        fprintf(stderr, "%s\n", "unknown boost error");
        return LIBGLTF_UNKNOWN_BOOST_ERROR;
    }
}

glTFHandle* RenderScene::initScene(const std::string& jsonfile, std::vector<glTFFile>& o_glTFFiles)
{
    if (jsonfile.empty())
    {
        return 0;
    }
    if (!mLoadJson.parseJsonFile(jsonfile))
    {
        return 0;
    }
    mLoadJson.getFileNamesInJson(o_glTFFiles);
    pScene = new Scene();
    glTFHandle* gHandle = new glTFHandle;
    pScene->setGltfHandle(gHandle);
    mLoadJson.setScene(pScene);
    return gHandle;
}

unsigned int RenderScene::bindAttribute(const Attribute* pAttr)
{
    unsigned int bufferId;
    glGenBuffers(1, &bufferId);
    glBindBuffer(GL_ARRAY_BUFFER, bufferId);
    glBufferData(GL_ARRAY_BUFFER,
                 pAttr->getDataCount() * pAttr->getByteStride(),
                 pAttr->getAttributeData(), GL_STATIC_DRAW);
    return bufferId;
}

unsigned int RenderScene::bindIndices(const Attribute* pAttr)
{
    unsigned int bufferId;
    glGenBuffers(1, &bufferId);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferId);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER,
                 pAttr->getDataCount() * pAttr->getByteStride(),
                 pAttr->getAttributeData(), GL_STATIC_DRAW);
    return bufferId;
}

void RenderScene::constructShader()
{
    for (unsigned int i = 0, size = pScene->getTechSize();
         i < size; ++i)
    {
        RenderShader* pRenderShader = new RenderShader();
        pRenderShader->setTechnique(pScene->getTechnique(i));
        mShaderVec.push_back(pRenderShader);
    }
}

void RenderScene::constructMesh(const std::string& meshName, Node* pNode)
{
    const Mesh* pMesh = pScene->findMesh(meshName);
    for (unsigned int i = 0, size = pMesh->getPrimitiveVecSize();
         i < size; ++i)
    {
        constructPrimitive(pMesh->getPrimitiveVec(i), pNode);
    }
}

void RenderScene::constructPrimitive(const Primitives* pPrimitive,
                                     Node* pNode)
{
    const std::string& materialIndex = pPrimitive->getMaterialIndex();
    Material* pMaterial = pScene->findMaterial(materialIndex);
    if (0 != pMaterial)
    {
        for (unsigned int i = 0, size = mShaderVec.size();
             i < size; ++i)
        {
            RenderShader* pRenderShader = mShaderVec[i];
            Technique* pTechnique = pRenderShader->getTechnique();
            if (pMaterial->getTechniqueId() == pTechnique->getTechId())
            {
                RenderPrimitive* pRenderPrimitive = new RenderPrimitive();
                bindAttributeBuffer(pPrimitive, pRenderPrimitive);
                pRenderPrimitive->setMaterial(pMaterial);
                pRenderPrimitive->setNode(pNode);

                pRenderShader->pushRenderPrim(pRenderPrimitive);
                break;
            }
        }
    }
}

void RenderScene::bindAttributeBuffer(const Primitives* pPrimitive,
                                      RenderPrimitive* pRP)
{
    const Attribute* pAttr;
    BindBufferInfo bindBufferInfo;
    std::map<std::string, BindBufferInfo>::const_iterator it;

    const std::string& posId = pPrimitive->getAttributeIndex("POSITION");
    it = mBindBufferMap.find(posId);
    if (mBindBufferMap.end() == it)
    {
        pAttr = pScene->findAttribute(posId);
        if (0 != pAttr)
        {
            bindBufferInfo.mBufferId = bindAttribute(pAttr);
            bindBufferInfo.mDataCount = pAttr->getDataCount();
            bindBufferInfo.mBufferLen = pAttr->getDataCount()
                                        * pAttr->getByteStride();
            bindBufferInfo.mSrcData = pAttr->getAttributeData();
            pRP->setVertexBuffer(bindBufferInfo.mBufferId);
            pRP->setVerterCount(bindBufferInfo.mDataCount);
            pRP->copyVertexBufferData(bindBufferInfo.mSrcData,
                                      bindBufferInfo.mBufferLen);
            mBindBufferMap.insert(
                std::map<std::string, BindBufferInfo>::value_type
                (posId, bindBufferInfo));
        }
    }
    else
    {
        pRP->setVertexBuffer(it->second.mBufferId);
        pRP->setVerterCount(it->second.mDataCount);
        pRP->copyVertexBufferData(it->second.mSrcData,
                                  it->second.mBufferLen);
    }

    const std::string& NorId = pPrimitive->getAttributeIndex("NORMAL");
    it = mBindBufferMap.find(NorId);
    if (mBindBufferMap.end() == it)
    {
        pAttr = pScene->findAttribute(NorId);
        if (0 != pAttr)
        {
            bindBufferInfo.mBufferId = bindAttribute(pAttr);
            pRP->setNormalBuffer(bindBufferInfo.mBufferId);
            mBindBufferMap.insert(
                std::map<std::string, BindBufferInfo>::value_type
                (NorId, bindBufferInfo));
        }
    }
    else
    {
        pRP->setNormalBuffer(it->second.mBufferId);
    }

    const std::string& TexId = pPrimitive->getAttributeIndex("TEXCOORD_0");
    it = mBindBufferMap.find(TexId);
    if (mBindBufferMap.end() == it)
    {
        pAttr = pScene->findAttribute(TexId);
        if (0 != pAttr)
        {
            float* pData = (float*)pAttr->getAttributeData();
            for (unsigned int i = 0, size = pAttr->getDataCount();
                 i < size; i++)
            {
                pData[i * 2 + 1] = 1 - pData[i * 2 + 1];
            }
            bindBufferInfo.mBufferId = bindAttribute(pAttr);
            pRP->setTexCoordBuffer(bindBufferInfo.mBufferId);
            mBindBufferMap.insert(
                std::map<std::string, BindBufferInfo>::value_type
                (TexId, bindBufferInfo));
        }
    }
    else
    {
        pRP->setTexCoordBuffer(it->second.mBufferId);
    }

    pAttr = pScene->findAttribute(pPrimitive->getAttributeIndex("JOINT"));
    if (0 != pAttr)
    {
        pRP->setJointBuffer(bindAttribute(pAttr));
    }

    pAttr = pScene->findAttribute(pPrimitive->getAttributeIndex("WEIGHT"));
    if (0 != pAttr)
    {
        pRP->setWeightBuffer(bindAttribute(pAttr));
    }

    pAttr = pScene->findAttribute(pPrimitive->getIndicesIndex());
    if (0 != pAttr)
    {
        pRP->setIndicesBuffer(bindIndices(pAttr));
        pRP->setIndicesCount(pAttr->getDataCount());
        pRP->setIndicesDataType(pAttr->getDataType());
        pRP->copyIndiceBufferData(pAttr->getAttributeData(),
                                  pAttr->getDataCount() *
                                  pAttr->getByteStride());
    }
}

void RenderScene::getCameraIndex(Node* pNode)
{
    Node* pChild = 0;
    for (unsigned int i = 0, size = pNode->getChildNodeSize();
         i < size; ++i)
    {
        pChild = pNode->getChildNode(i);
        if (!pChild->getCameraIndex().empty())
        {
            vCameraIndex.push_back(pChild->getCameraIndex());
        }
        getCameraIndex(pChild);
    }
}
void RenderScene::initOpengl()
{
    glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
    glClear(GL_COLOR_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glDepthFunc(GL_LESS);
}

void RenderScene::createDefaultCamera()
{
    glm::vec3 vDelta = pScene->getVertexMax() - pScene->getVertexMin();
    glm::vec3 vHalf =  pScene->getVertexMin() + (vDelta / 2.0f);

    float fPos = sqrt(vDelta.x * vDelta.x
                    + vDelta.y * vDelta.y
                    + vDelta.z * vDelta.z);

    glm::vec3 vEye;
    vEye = glm::vec3(0, 0, 1.5 * fPos);
    maCamera.setViewMatrix(glm::lookAt(vEye, vHalf, glm::vec3(0.0f, 1.0f, 0.0f)));
    mOrbitInitViewMatrix = maCamera.getViewMatrix();
    maCamera.flength = fPos;
    maCamera.vModelCenterPos = vHalf;

    getCameraIndex(pScene->getRootNode());
    if (vCameraIndex.size() > 0)
    {
        std::vector<std::string>::iterator it;
        it = vCameraIndex.begin();
        cCamera = pScene->findCamera(*it);
    }

    float fovy = 37.8492f;
    float aspect = 1.5f;
    float zNear = 1.0f;
    float zFar = 500000.0f;

    if (pScene->getUseCameraInJson())
    {
        fovy = cCamera->getXFov();
        aspect = cCamera->getAspectRatio();
        zNear = cCamera->getNear();
        zFar = cCamera->getFar();
        Node* pNode = cCamera->getCameraNode();
        maCamera.setViewMatrix(glm::inverse(pNode->getGlobalMatrix()));
    }
    if (fPos * 6 > zFar)
    {
        zFar =fPos * 6;
    }
    maCamera.setPerspective(glm::perspective(fovy, aspect, zNear, zFar));
    mWalkthroughInitViewMatrix = maCamera.getViewMatrix();
}

void RenderScene::initNodeTree(Node* pNode, const glm::mat4& matrix,
                               bool parentJointFlag, bool updateFlag)
{
    glm::mat4 globalMatrix;
    bool currJointFlag = pNode->getJointFlag();
    if (!parentJointFlag && currJointFlag)
    {
        globalMatrix = pNode->getLocalMatrix();
    }
    else
    {
        globalMatrix = matrix * pNode->getLocalMatrix();
    }

    if (!pNode->getMatrixFlag())
    {
        updateFlag = true;
    }
    pNode->setUpdateFlag(updateFlag);

    //if (pNode->getSkinIndex().empty())
    {
        pNode->setGlobalMatrix(globalMatrix);
        if( cCamera && pNode == cCamera->getCameraNode() )
        {
            maCamera.setViewMatrix(glm::inverse(globalMatrix));
            mWalkthroughInitViewMatrix = maCamera.getViewMatrix();
        }
    }

    for (unsigned int i = 0, nSize = pNode->getChildNodeSize();
         i < nSize; ++i)
    {
        initNodeTree(pNode->getChildNode(i), globalMatrix,
                     currJointFlag, updateFlag);
    }
}

int RenderScene::initRender(const std::vector<glTFFile>& inputFiles)
{
    if( !glewIsSupported("GL_VERSION_3_0") )
    {
        return LIBGLTF_UNKNOWN_ERROR;
    }

    initOpengl();

    int iResult = loadScene(inputFiles);
    if (iResult != LIBGLTF_SUCCESS)
    {
        return iResult;
    }
    pTempMatrix = new glm::mat4[MAX_BONE_SIZE];

    Node* pRootNode = pScene->getRootNode();

    constructShader();
    initNodeTree(pRootNode, pRootNode->getGlobalMatrix(), false, false);

    for (unsigned int i = 0, nSize = pScene->getNodeSize();
         i < nSize; ++i)
    {
        Node* pNode = pScene->getNode(i);

        if (0 != pScene->getAnimationCount())
        {
            const std::string& nodeName = pNode->getNodeName();
            pNode->setAnimPoint(pScene->findAnimation(nodeName));
        }

        const std::string& skinIndex = pNode->getSkinIndex();
        if (!skinIndex.empty())
        {
            Node* pPareBone = 0;
            Node* pBone = 0;
            pPareBone = findNodeByName(pRootNode, pNode->getSkeleIndex());
            for (unsigned int j = 0, size = pScene->getSkinSize();
                 j < size; ++j)
            {
                Skin* pSkin = pScene->getSkin(j);
                if (pSkin->getSkinName() == skinIndex)
                {
                    pNode->setSkinPoint(pSkin);
                    for (unsigned int k = 0, jointCount = pSkin->getBoneIdSize();
                         k < jointCount; k++)
                    {
                        pBone = findNodeByJoint(pPareBone, pSkin->getBoneId(k));
                        pNode->pushBoneNode(pBone);
                    }
                    break;
                }
                continue;
            }
        }

        if (pNode->hasMesh())
        {
            for (unsigned int j = 0, size = pNode->getMeshIndexSize();
                 j < size; ++j)
            {
                constructMesh(pNode->getMeshIndex(j), pNode);
            }
        }
    }
    setModelBoundaryValue();
    createDefaultCamera();
    trackball(maCamera.curquatY, 0.0f, 0.0f, 0.0f, 0.0f);
    trackball(maCamera.curquatX, 0.0f, 0.0f, 0.0f, 0.0f);
    pScene->clearAttributeMap();

    mDuration = pScene->getDuration();
    return LIBGLTF_SUCCESS;
}

void RenderScene::initFPS()
{
    if (0 != pFPSCounter)
    {
        delete pFPSCounter;
    }
    pFPSCounter = new FPSCounter(&mShaderProgram);
}

void RenderScene::renderPrimitive(RenderPrimitive* pPrimitive,
                                  unsigned int progId)
{
    upLoadMatrixInfo(progId, pPrimitive);
    if (mEnableTransparency)
    {
        pPrimitive->bindSortedIndicesBuf();
    }

    {
        TRACE_TIME;
        upLoadUniform(progId, pPrimitive);
    }

    {
        TRACE_TIME;
        upLoadAttribute(progId, pPrimitive);
    }

    {
        TRACE_TIME;
        upLoadAnimation(progId, pPrimitive);
    }

    {
        TRACE_TIME;
        drawTriangle(pPrimitive);
    }
}

void RenderScene::upLoadMatrixInfo(unsigned int progId,
                                   RenderPrimitive* pPrimitive)
{
    Node* pNode = pPrimitive->getNode();
    glm::mat4& tempMatrix = pNode->getGlobalMatrix();
    const glm::mat4& lookMatrix = maCamera.getViewMatrix();
    {
        TRACE_TIME;
        mShaderProgram.setUniform(progId, "u_modelViewMatrix",
            maCamera.getModelViewMatrix(lookMatrix,
                                        tempMatrix,
                                        mCameraMoveTime,
                                        mPreCameraTime));
        mShaderProgram.setUniform(progId, "u_normalMatrix",
            glm::mat3(lookMatrix) *
            glm::transpose(glm::inverse(glm::mat3(tempMatrix))));
        mShaderProgram.setUniform(progId, "u_projectionMatrix",
                                  maCamera.getPerspective());
        mShaderProgram.setUniform(progId, "M", tempMatrix);
        mShaderProgram.setUniform(progId, "V", lookMatrix);
    }
}

bool RenderScene::upLoadTechProperty()
{
    glEnable(GL_BLEND);
    glBlendEquation(GL_FUNC_ADD);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glDepthMask(GL_TRUE);
    glEnable(GL_CULL_FACE);
    glEnable(GL_DEPTH_TEST);
    return true;
}

void RenderScene::upLoadTechPropertyOfJsonFile(Technique* pTech)
{
    if (1 == pTech->getTechState()->blendEnable )
    {
        glEnable(GL_BLEND);
    }
    else
    {
        glDisable(GL_BLEND);
    }

    if (0 != pTech->getTechState()->blendEquation)
    {
        glBlendEquation(pTech->getTechState()->blendEquation);
        glBlendFunc(pTech->getTechState()->blendFuncSfactor,
                    pTech->getTechState()->blendFuncDfactor);
    }

    if (1 == pTech->getTechState()->cullFaceEnable)
    {
        glEnable(GL_CULL_FACE);
    }
    else
    {
        glDisable(GL_CULL_FACE);
    }

    if (1 == pTech->getTechState()->depthMask)
    {
        glDepthMask(GL_TRUE);
    }
    else
    {
        glDepthMask(GL_FALSE);
    }

    if (1 == pTech->getTechState()->depthTestEnable)
    {
        glEnable(GL_DEPTH_TEST);
    }
    else
    {
        glDisable(GL_DEPTH_TEST);
    }
}

void RenderScene::upLoadTechInfo(unsigned int progId, Technique* pTech)
{
    if (mEnableTransparency)
    {
        upLoadTechProperty();
    }
    else
    {
        upLoadTechPropertyOfJsonFile(pTech);
    }

    std::vector<techLight*> vecLight = pTech->poptLight();
    std::vector<techLight*>::const_iterator it = vecLight.begin();
    std::vector<techLight*>::const_iterator itEnd = vecLight.end();
    for (; it != itEnd; ++it)
    {
        if ((*it)->mSource.empty())
        {
            switch ((*it)->type)
            {
            case DataType_FLOAT:
                mShaderProgram.setUniform(progId, (*it)->mName.c_str(),
                    (*it)->floatValue);
                break;
            case DataType_FLOAT_VEC3:
                mShaderProgram.setUniform(progId, (*it)->mName.c_str(),
                    (*it)->vecValue);
                break;
            default: break;
            }
        }
        else
        {
            Node* pNode = pScene->findLightNodeMap((*it)->mSource);
            mShaderProgram.setUniform(progId, (*it)->mName.c_str(),
                                      maCamera.getViewMatrix() *
                                      pNode->getGlobalMatrix());
        }
    }
}

void RenderScene::upLoadAnimation(unsigned int progId,
                                  RenderPrimitive* pPrimitive)
{
    Node* pNode = pPrimitive->getNode();
    Skin* pSkin = pNode->getSkinPoint();
    if (0 == pSkin)
    {
        return;
    }

    unsigned int count = pSkin->getBindMatrixCount();
    memcpy(pTempMatrix, pSkin->getBindMatrix(), sizeof(glm::mat4)*count);

    Node* pBone;
    for (unsigned int i = 0; i < count; ++i)
    {
        pBone = pNode->getBoneNode(i);
        if (0 != pBone)
            pTempMatrix[i] = pBone->getGlobalMatrix() * pTempMatrix[i];
    }

    glUniformMatrix4fv(glGetUniformLocation(progId, "u_jointMat"),
                       count, false, (GLfloat*)pTempMatrix);
}

void RenderScene::updateAnimInfo(Node* pNode)
{
    Animation* pAnimation = pNode->getAnimPoint();
    if (0 == pAnimation)
    {
        return;
    }

    double time = fmod(mCurrentTime, pAnimation->getDuration());
    glm::mat4 localMatrix = pAnimation->findTimeValue(time);
    if (pAnimation->getChannelBits() == ROTATE_CHANNEL)
    {
        localMatrix = pNode->getTranslate() * localMatrix * pNode->getScale();
    }
    pNode->setLocalMatrix(localMatrix);
}

void RenderScene::updateNodeMatrix(Node* pNode, const glm::mat4& matrix,
                                   bool jointFlag)
{
    glm::mat4 globalMatrix;
    bool flag = pNode->getJointFlag();

    if (pNode->getUpdateFlag())
    {
        updateAnimInfo(pNode);
        if (!jointFlag && flag)
        {
            globalMatrix = pNode->getLocalMatrix();
        }
        else
        {
            globalMatrix = matrix * pNode->getLocalMatrix();
        }

        if (pNode->getSkinIndex().empty())
        {
            if( cCamera && pNode == cCamera->getCameraNode() && !bAerialView )
            {
                for (unsigned int i = 0; i < 4; i++)
                {
                    for (unsigned int j = 0; j < 4; j++)
                    {
                        if (std::abs(pNode->getGlobalMatrix()[i][j] - globalMatrix[i][j]) > 0.01)
                        {
                            maCamera.setViewMatrix(glm::inverse(globalMatrix));
                            mWalkthroughInitViewMatrix = maCamera.getViewMatrix();
                            break;
                        }
                    }
                }
            }
            pNode->setGlobalMatrix(globalMatrix);
        }
    }
    else
    {
        globalMatrix = pNode->getGlobalMatrix();
    }

    for (unsigned int i = 0, nSize = pNode->getChildNodeSize();
         i < nSize; ++i)
    {
        updateNodeMatrix(pNode->getChildNode(i), globalMatrix, flag);
    }
}

void RenderScene::updateFlyCamera()
{
    static bool startFly = false;
    if (flyTime <= 0)
    {
        bFlyCamera = false;
        startFly = false;
        return;
    }
    glm::mat4 viewMatrix = maCamera.getViewMatrix();
    if (!startFly)
    {
        startFly = true;
        oldTime = libgltf::time::getCurrentTime();
        maCamera.setViewMatrix(viewMatrix);
        return;
    }
    double newTime = libgltf::time::getCurrentTime();
    double timeDifference = libgltf::time::diffTime(newTime,oldTime)/1000.0;
    flyTime -= timeDifference;
    viewMatrix += flyInfo * (float)timeDifference;
    maCamera.setViewMatrix(viewMatrix);
}
void RenderScene::upLoadUniform(unsigned int progId,
                                RenderPrimitive* pPrimitive)
{
    int textureCount = 0;
    Material* pMaterial = pPrimitive->getMaterial();
    for (unsigned int i = 0, size = pMaterial->getMaterialProperSize();
         i < size; ++i)
    {
        const MaterialProperty* pProper = pMaterial->getMaterialProper(i);
        switch (pProper->getDataType())
        {
        case DataType_FLOAT:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      (float*)(pProper->getPropertyData()));
            break;
        case DataType_FLOAT_VEC2:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      (glm::vec2*)(pProper->getPropertyData()));
            break;
        case DataType_FLOAT_VEC3:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      (glm::vec3*)(pProper->getPropertyData()));
            break;
        case DataType_FLOAT_VEC4:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      (glm::vec4*)(pProper->getPropertyData()));
            break;
        case DataType_FLOAT_MAT3:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      (glm::mat3*)(pProper->getPropertyData()));
            break;
        case DataType_FLOAT_MAT4:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      (glm::mat4*)(pProper->getPropertyData()));
            break;
        case DataType_SAMPLER_2D:
#if LOAD_ONCE
            if (mCurrentImage == pProper->getImagePath() &&
                textureCount == mCurrentTextNumber)
            {
                break;
            }
            mCurrentImage = pProper->getImagePath();
            mCurrentTextNumber = textureCount;
#endif
            mShaderProgram.setUniform(progId,
                                      pProper->getPropertyName().c_str(),
                                      textureCount);
            pScene->findTexture(
                pProper->getImagePath())->bindTexture(textureCount);
            textureCount++;
            break;
        default:
            break;
        }
    }
}

void RenderScene::upLoadAttribute(unsigned int progId,
                                  RenderPrimitive* pPrimitive)
{
    int tmpId;
    if ((tmpId = glGetAttribLocation(progId, "a_position")) != -1 &&
        0 != pPrimitive->getVertexBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getVertexBuffer());
        glVertexAttribPointer(tmpId, 3, GL_FLOAT, GL_FALSE, 0, (void*)0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_normal")) != -1 &&
        0 != pPrimitive->getNormalBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getNormalBuffer());
        glVertexAttribPointer(tmpId, 3, GL_FLOAT, GL_FALSE, 0, (void*)0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_texcoord0")) != -1 &&
        0 != pPrimitive->getTexCoordBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getTexCoordBuffer());
        glVertexAttribPointer(tmpId, 2, GL_FLOAT, GL_FALSE, 0, (void*)0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_weight")) != -1 &&
        0 != pPrimitive->getWeightBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getWeightBuffer());
        glVertexAttribPointer(tmpId, 4, GL_FLOAT, GL_FALSE, 0, (void*)0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_joint")) != -1 &&
        0 != pPrimitive->getJointBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getJointBuffer());
        glVertexAttribPointer(tmpId, 4, GL_FLOAT, GL_FALSE, 0, (void*)0);
    }
}

void RenderScene::drawTriangle(RenderPrimitive* pPrimitive)
{
    if (pPrimitive->getIndicesCount() > 0)
    {
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, pPrimitive->getIndicesBuffer());
#if ENABLE_TIMER
        glFinish();
#endif
        glDrawElements(GL_TRIANGLES, pPrimitive->getIndicesCount(),
                       pPrimitive->getIndicesDataType(), 0);
#if ENABLE_TIMER
        glFinish();
#endif
    }
    else
    {
        glDrawArrays(GL_TRIANGLES, 0, pPrimitive->getVerterCount());
    }
}

int RenderScene::prepareRender(glTFViewport* pViewport)
{
    int status;
    if (0 != pFPSCounter)
    {
        pFPSCounter->timeStamp();
    }
    {
        TRACE_TIME;
        status = initSSAAFrameBuf(pViewport);
        if (LIBGLTF_SUCCESS != status)
            return status;
    }
    return 0;
}

int RenderScene::initSSAAFrameBuf(glTFViewport* pViewport)
{
    if (0 == pViewport->width)
        return LIBGLTF_INVALID_SIZE;

    // When viewport changes we need to release fbo and create a new one
    if( mCurrentViewport.x != pViewport->x || mCurrentViewport.y != pViewport->y ||
    mCurrentViewport.width != pViewport->width || mCurrentViewport.height != pViewport->height )
    {
        fbo.releaseFbo();
        mCurrentViewport = *pViewport;
    }

#if DEFAULT_VIEW
    unsigned int width  = SSAA * DEFAULT_VIEW;
    unsigned int height = (SSAA * DEFAULT_VIEW *
                           pViewport->height / pViewport->width);
#else
    unsigned int width  = SSAA * pViewport->width;
    unsigned int height = SSAA * pViewport->height;
#endif //DEFAULT_VIEW
    int status = fbo.createAndBindFbo(width, height);
    if (LIBGLTF_SUCCESS != status)
        return status;
#if USE_MSAA
    glBindFramebuffer(GL_FRAMEBUFFER, fbo.getMSAAFboId());
#else
    glBindFramebuffer(GL_FRAMEBUFFER, fbo.getFboId());
#endif  //USE_MSAA
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glViewport(pViewport->x, pViewport->y, width, height);
    return LIBGLTF_SUCCESS;
}

void RenderScene::render()
{
    setTimeForAnim();
    realRender();
}

void RenderScene::realRender()
{
    if (mCurrentTime >= mUpdateTimeOut || !bIsTimeAvailable)
    {
        if (mAnimationPlay)
        {
            Node* pRootNode = pScene->getRootNode();
            updateNodeMatrix(pRootNode, pRootNode->getGlobalMatrix(), false);
        }
        mUpdateTimeOut = mCurrentTime;
    }
    if (bFlyCamera)
    {
        updateFlyCamera();
    }
    if (mEnableTransparency)
    {
        updatePolygonSorting();
    }

    for (unsigned int i = 0, size = mShaderVec.size();
         i < size; ++i)
    {
        renderShader(mShaderVec[i]);
    }
    {
        TRACE_TIME;
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, 0);
        mCurrentImage = "";
        mCurrentTextNumber = -1;
    }
}

void RenderScene::primitivePolygonSorting(RenderPrimitive* pPrimitive)
{
    Node* pNode = pPrimitive->getNode();
    glm::mat4 viewNatrix =
        maCamera.getModelViewMatrix(maCamera.getViewMatrix(),
                                    pNode->getGlobalMatrix(),
                                    mCameraMoveTime,
                                    mPreCameraTime);
    pPrimitive->polyonSorting(viewNatrix);
}

void RenderScene::updatePolygonSorting()
{
    bool viewChanged = false;
    const glm::mat4& currentView = maCamera.getViewMatrix();
    for (unsigned int i = 0; i < 4; i++)
    {
        for (unsigned int j = 0; j < 4; j++)
        {
            if (std::abs(currentView[i][j] - mLastModelView[i][j]) > 0.0001)
            {
                viewChanged = true;
                break;
            }
        }
    }
    if (!viewChanged)
    {
        return;
    }
    mLastModelView = currentView;
    unsigned int shaderSize = mShaderVec.size();
    for (unsigned int i = 0; i < shaderSize; ++i)
    {
        RenderShader* pRenderShader = mShaderVec[i];
        unsigned int primitiveSize = pRenderShader->getRenderPrimSize();
        for (unsigned int j = 0; j < primitiveSize; ++j)
        {
            primitivePolygonSorting(pRenderShader->getRenderPrim(j));
        }
    }
}

void RenderScene::renderShader(RenderShader* pRenderShader)
{
    Technique* pTechnique = pRenderShader->getTechnique();

    if (!pTechnique->useTechnique())
    {
        return;
    }

    unsigned int progId = pTechnique->getProgramId();

    {
        TRACE_TIME;
        upLoadTechInfo(progId, pTechnique);
    }

    for (unsigned int i = 0, size = pRenderShader->getRenderPrimSize();
         i < size; ++i)
    {
        renderPrimitive(pRenderShader->getRenderPrim(i), progId);
    }
}

int RenderScene::completeRender()
{
    if (0 != pFPSCounter)
    {
        pFPSCounter->printFPS(&mCurrentViewport);
    }

    {
        TRACE_TIME;
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
#if DEFAULT_VIEW
        unsigned int width  = SSAA * DEFAULT_VIEW;
        unsigned int height = (SSAA * DEFAULT_VIEW *
            mCurrentViewport.height / mCurrentViewport.width);
#else
        unsigned int width  = SSAA * mCurrentViewport.width;
        unsigned int height = SSAA * mCurrentViewport.height;
#endif //DEFAULT_VIEW

#if USE_MSAA
        glBindFramebuffer(GL_READ_FRAMEBUFFER, fbo.getMSAAFboId());
        GLenum status = glCheckFramebufferStatus(GL_READ_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            return LIBGLTF_BIND_FBO_ERROR;
        }
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo.getFboId());
        status = glCheckFramebufferStatus(GL_DRAW_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            return LIBGLTF_BIND_FBO_ERROR;
        }

        glBlitFramebuffer(0, 0 ,width, height, 0, 0,width ,height,
                          GL_COLOR_BUFFER_BIT, GL_LINEAR);
        glBindFramebuffer(GL_READ_FRAMEBUFFER,0);
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER,0);
#endif //USE_MSAA
        glViewport(mCurrentViewport.x, mCurrentViewport.y,
                   mCurrentViewport.width, mCurrentViewport.height);
        fbo.renderFbo(width, height);
    }
    return LIBGLTF_SUCCESS;
}

void RenderScene::releaseRender()
{
    delete pScene->getGltfHandle();
    fbo.releaseFbo();
#if USE_MSAA
    fbo.releaseMSAAFBO();
#endif  //USE_MSAA
    return;
}

CPhysicalCamera& RenderScene::getCamera()
{
    return maCamera;
}

Node* RenderScene::findNodeByName(Node* pPareNode,
                                  const std::string& nodeId)
{
    if (0 == pPareNode)
        return 0;
    if (pPareNode->getNodeName() == nodeId)
        return pPareNode;

    Node* pNode = 0;
    for (unsigned int i = 0, size = pPareNode->getChildNodeSize();
         i < size; ++i)
    {
        pNode = findNodeByName(pPareNode->getChildNode(i), nodeId);
        if (0 != pNode)
            break;
    }
    return pNode;
}

Node* RenderScene::findNodeByJoint(Node* pPareNode,
                                   const std::string& jointId)
{
    if (0 == pPareNode || !pPareNode->getJointFlag())
        return 0;
    if (pPareNode->getJointId() == jointId)
        return pPareNode;

    Node* pNode = 0;
    for (unsigned int i = 0, size = pPareNode->getChildNodeSize();
         i < size; ++i)
    {
        pNode = findNodeByJoint(pPareNode->getChildNode(i), jointId);
        if (0 != pNode)
            break;
    }
    return pNode;
}

int RenderScene::prepareRenderBitmap(glTFViewport* pViewport)
{
    int status = initSSAAFrameBuf(pViewport);
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
    startAnimation();
    glEnable(GL_DEPTH_TEST);
    return LIBGLTF_SUCCESS;
}

void RenderScene::renderBitmap(double time)
{
    mCurrentTime = time;
    realRender();
}

void gaussianFilter(unsigned char* corrupted, glTFViewport* pViewport)
{
    int templates[25] = { 1, 4,  7,  4,  1,
                          4, 16, 26, 16, 4,
                          7, 26, 41, 26, 7,
                          4, 16, 26, 16, 4,
                          1, 4,  7,  4,  1 };
    int desLineSize = (pViewport->width * 3 + 3) / 4 * 4;
    int size = desLineSize * pViewport->height;
    unsigned char* smooth = new unsigned char[size];
    memcpy(smooth, corrupted, size);
    for (int j = 2; j < pViewport->height - 2; j++)
    {
        for (int i = 2; i < pViewport->width - 2; i++)
        {
            int sum = 0;
            int index = 0;
            for ( int m = j - 2; m < j + 3; m++)
            {
                for (int n = i - 2; n < i + 3; n++)
                {
                    sum += corrupted[m * desLineSize + n * 3] *
                           templates[index++];
                }
            }
            sum /= 273;
            if (sum > 255)
                sum = 255;
            smooth[j * desLineSize + i * 3] = (unsigned char)sum;
        }
    }
    memcpy(corrupted, smooth, size);
    delete[] smooth;
    smooth = 0;
}

void RenderScene::setBitZoom(unsigned char* Dstbuffer,
                             unsigned char* Srcbuffer,
                             glTFViewport* pViewport,
                             int bufferDepth)
{
#if DEFAULT_VIEW
    unsigned int width  = SSAA * DEFAULT_VIEW;
    unsigned int height = (SSAA * DEFAULT_VIEW *
                           pViewport->height / pViewport->width);
#else
    unsigned int width  = SSAA * pViewport->width;
    unsigned int height = SSAA * pViewport->height;
#endif //DEFAULT_VIEW
    int dstLineByte = pViewport->width * bufferDepth;
    int srcLineByte = width * bufferDepth;
    for (int i = 0; i < pViewport->height; i++)
    {
        float fy = (float)((i + 0.5) * SSAA - 0.5);
        int sy =(int)floor(fy);
        fy -= sy;
        sy = min(sy, (int)(height - 2));
        sy = max(0, sy);
        short cbufy[2];
        cbufy[0] = (short)((1.0f - fy) * 2048);
        cbufy[1] = 2048 - cbufy[0];
        for (int j = 0; j < pViewport->width; j++)
        {
            float fx = (float)((j + 0.5) * SSAA - 0.5);
            unsigned int sx = (int)floor(fx);
            fx -= sx;
            if (sx >= width - 1)
            {
                fx = 0;
                sx = width - 2;
            }
            short cbufx[2];
            cbufx[0] = (short)((1.0f - fx) * 2048);
            cbufx[1] = 2048 - cbufx[0];
            for (int k = 0; k < bufferDepth; k++)
            {
                *(Dstbuffer + i * dstLineByte + bufferDepth * j + k) = (
                    *(Srcbuffer + sy * srcLineByte + bufferDepth * sx + k)
                    * cbufx[0] * cbufy[0] +
                    *(Srcbuffer + (sy + 1) * srcLineByte + bufferDepth * sx + k)
                    * cbufx[0] * cbufy[1] +
                    *(Srcbuffer + sy * srcLineByte + bufferDepth * (sx + 1) + k)
                    * cbufx[1] * cbufy[0] +
                    *(Srcbuffer + (sy + 1) * srcLineByte + bufferDepth
                    *(sx + 1) + k) * cbufx[1] * cbufy[1]) >> 22;
            }
        }
    }
}


int RenderScene::completeRenderBitmap(glTFViewport* pViewport,
                                       unsigned char* buffer, GLenum format)
{
#if DEFAULT_VIEW
    unsigned int width  = SSAA * DEFAULT_VIEW;
    unsigned int height = (SSAA * DEFAULT_VIEW *
                           pViewport->height / pViewport->width);
#else
    unsigned int width  = SSAA * pViewport->width;
    unsigned int height = SSAA * pViewport->height;
#endif //DEFAULT_VIEW
    glBindFramebuffer(GL_FRAMEBUFFER, 0);

#if USE_MSAA
    glBindFramebuffer(GL_READ_FRAMEBUFFER, fbo.getMSAAFboId());
    GLenum status = glCheckFramebufferStatus(GL_READ_FRAMEBUFFER);
    if (status != GL_FRAMEBUFFER_COMPLETE)
    {
        return LIBGLTF_BIND_FBO_ERROR;
    }
    glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo.getFboId());
    status = glCheckFramebufferStatus(GL_DRAW_FRAMEBUFFER);
    if (status != GL_FRAMEBUFFER_COMPLETE)
    {
        return LIBGLTF_BIND_FBO_ERROR;
    }
    glBlitFramebuffer(0, 0 ,width, height, 0, 0,width ,height,
        GL_COLOR_BUFFER_BIT, GL_LINEAR);
    glBindFramebuffer(GL_READ_FRAMEBUFFER,0);
    glBindFramebuffer(GL_DRAW_FRAMEBUFFER,0);
#endif //USE_MSAA

    glDisable(GL_DEPTH_TEST);
    glViewport(pViewport->x, pViewport->y, width, height);
    fbo.renderFbo(width, height);
    fbo.createBitmapTexture(width, height);
    fbo.inverseBitMap(width, height);
    if (GL_FRAMEBUFFER_COMPLETE != glCheckFramebufferStatus(GL_FRAMEBUFFER))
    {
        return LIBGLTF_BIND_FBO_ERROR;
    }
    int bufferDepth;
    if (format == GL_RGB || format == GL_BGR)
        bufferDepth = 3;
    else if (format == GL_RGBA || format == GL_BGRA)
        bufferDepth = 4;
    else
        return LIBGLTF_UNKNOWN_ERROR;
    int imageSize = width * height * bufferDepth;

    unsigned char *pbuffer = new unsigned char[imageSize];
    glReadPixels(0, 0, width, height, format,
                 GL_UNSIGNED_BYTE, pbuffer);
    setBitZoom(buffer, pbuffer, pViewport, bufferDepth);
    if (pbuffer)
    {
        delete[] pbuffer;
        pbuffer = 0;
    }
    fbo.releaseBitMapFBO();
    fbo.releaseBitmapTexture();
    return LIBGLTF_SUCCESS;
}

void RenderScene::getCameraPos(glm::vec3* Eye, glm::vec3* view,
                               glm::vec3* up)
{
    maCamera.getCameraPosVectors(Eye, view, up);
}

glm::vec3* RenderScene::getModelCenterPos()
{
    return &maCamera.vModelCenterPos;
}

void RenderScene::enableRotation()
{
    mEnableRotation = true;
}

void RenderScene::disableRotation()
{
    mEnableRotation = false;
}

bool RenderScene::isRotationEnabled() const
{
    return mEnableRotation;
}

void RenderScene::enableTransparency()
{
    mEnableTransparency = true;
}

void RenderScene::disableTransparency()
{
    mEnableTransparency = false;
}

double RenderScene::getModelSize()
{
    return maCamera.flength;
}

void RenderScene::startAnimation()
{
    mAnimationPlay = true;
    mCurrentTime = 0;
    mUpdateTimeOut = 0;
    bAnimation = true;
}

void RenderScene::stopAnimation()
{
    if (pScene->getSkinSize() != 0)
    {
        mAnimationPlay = false;
        bAnimation = false;
    }
}

void RenderScene::stopAerialView()
{
    bAerialView = false;
    if (IsAerialMode)
    {
        pScene->setUseCameraInJson(true);
    }
    maCamera.setViewMatrix(mWalkthroughInitViewMatrix);
    maCamera.setAerialView(false);
}

void RenderScene::startAerialView()
{
    bAerialView = true;
    maCamera.AerialViewY = 0.0;
    maCamera.mTrackBallY= glm::mat4(1.0);
    maCamera.mTrackBallX= glm::mat4(1.0);
    trackball(maCamera.curquatY, 0.0f, 0.0f, 0.0f, 0.0f);
    trackball(maCamera.curquatX, 0.0f, 0.0f, 0.0f, 0.0f);
    if (pScene->getUseCameraInJson())
    {
        IsAerialMode = true;
        pScene->setUseCameraInJson(false);
    }
    maCamera.setViewMatrix(mOrbitInitViewMatrix);
    maCamera.setAerialView(true);
}

void RenderScene::setAnimTime(double time)
{
    mCurrentTime = time;
    mUpdateTimeOut = time;
}

double RenderScene::getAnimTime()
{
    errno = 0;
    double time = fmod(mCurrentTime, mDuration);
    return errno == EDOM ? 0.0 : time;
}

void RenderScene::setAnimLoop(bool loop)
{
    mAnimationLoop = loop;
}

bool RenderScene::getAnimLoop()
{
    return mAnimationLoop;
}

double RenderScene::getAnimDuration()
{
    return mDuration;
}

bool RenderScene::isAnimPlay()
{
    return mAnimationPlay;
}

void RenderScene::resumeAnim()
{
    bAnimation = true;
    mAnimationPlay = true;
}

void RenderScene::setTimeForAnim()
{
    double timeNow = libgltf::time::getCurrentTime();
    if (mAnimationPlay)
    {
        if (bIsTimeAvailable)
        {
            mCurrentTime += libgltf::time::diffTime(timeNow,mLastPlaying);
        }
        else
        {
            bIsTimeAvailable = true;
        }
        mLastPlaying = timeNow;
    }
    else
    {
        bIsTimeAvailable = false;
    }
    if (!mAnimationLoop && mCurrentTime > mDuration)
    {
        stopAnimation();
        setAnimTime(0.0);
    }
}

void RenderScene::renderFlyCamera(glm::vec3 glPosInfo, double time)
{
    glm::mat4 newViewMatrix = glm::lookAt(glPosInfo, glm::vec3(0, 0, 0),
                                          glm::vec3(0, 1, 0));
    if( std::abs(time) > 0.0001 )
    {
        glm::mat4 oldViewMatrix = maCamera.getViewMatrix();
        flyInfo = newViewMatrix - oldViewMatrix;
        bFlyCamera = true;
        flyTime = time * 1000 * 1000;
        flyInfo = flyInfo / (float)(time * 1000 * 1000);
    }
    else
    {
        maCamera.setViewMatrix(newViewMatrix);
    }
}

void RenderScene::setModelBoundaryValue()
{
    unsigned int shaderSize = mShaderVec.size();
    glm::vec3 vertexMax = glm::vec3(-FLT_MAX,-FLT_MAX,-FLT_MAX);
    glm::vec3 vertexMin = glm::vec3(FLT_MAX,FLT_MAX,FLT_MAX);
    for (unsigned int i = 0;i < shaderSize; ++i)
    {
        RenderShader* pRenderShader = mShaderVec[i];
        unsigned int primitiveSize = pRenderShader->getRenderPrimSize();
        for (unsigned int j = 0;j < primitiveSize; ++j)
        {
            RenderPrimitive* pPrimitive = pRenderShader->getRenderPrim(j);
            if (NULL != pPrimitive)
            {
                pPrimitive->getPrimitiveBoundary(&vertexMax,&vertexMin);
            }
        }
    }
    pScene->setVertexMax(vertexMax[0],vertexMax[1],vertexMax[2]);
    pScene->setVertexMin(vertexMin[0],vertexMin[1],vertexMin[2]);
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
