/*
 * Copyright (C) 2020 Open Source Robotics Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/
#ifndef GZ_GAZEBO_GUI_GUIEVENTS_HH_
#define GZ_GAZEBO_GUI_GUIEVENTS_HH_

#include <QEvent>
#include <string>
#include <utility>
#include <vector>
#include <gz/math/Vector3.hh>
#include "gz/sim/Entity.hh"
#include "gz/sim/config.hh"

namespace ignition
{
namespace gazebo
{
namespace gui {
// Inline bracket to help doxygen filtering.
inline namespace IGNITION_GAZEBO_VERSION_NAMESPACE {
/// \brief Namespace for all events. Refer to the EventManager class for
/// more information about events.
namespace events
{
  /// \brief The class for sending and receiving custom snap value events.
  class SnapIntervals : public QEvent
  {
    /// \brief Constructor
    /// \param[in] _xyz XYZ snapping values.
    /// \param[in] _rpy RPY snapping values.
    /// \param[in] _scale Scale snapping values.
    public: SnapIntervals(
                const math::Vector3d &_xyz,
                const math::Vector3d &_rpy,
                const math::Vector3d &_scale)
    : QEvent(kType), xyz(_xyz), rpy(_rpy), scale(_scale)
    {
    }

    /// \brief Get the XYZ snapping values.
    /// \return The XYZ snapping values.
    public: math::Vector3d XYZ() const
    {
      return this->xyz;
    }

    /// \brief Get the RPY snapping values.
    /// \return The RPY snapping values.
    public: math::Vector3d RPY() const
    {
      return this->rpy;
    }

    /// \brief Get the scale snapping values.
    /// \return The scale snapping values.
    public: math::Vector3d Scale() const
    {
      return this->scale;
    }

    /// \brief The QEvent representing a snap event occurrence.
    static const QEvent::Type kType = QEvent::Type(QEvent::User);

    /// \brief XYZ snapping values in meters, these values must be positive.
    private: math::Vector3d xyz;

    /// \brief RPY snapping values in degrees, these values must be positive.
    private: math::Vector3d rpy;

    /// \brief Scale snapping values - a multiplier of the current size,
    /// these values must be positive.
    private: math::Vector3d scale;
  };

  /// \brief Event that notifies when new entities have been selected.
  class EntitiesSelected : public QEvent
  {
    /// \brief Constructor
    /// \param[in] _entities All the selected entities
    /// \param[in] _fromUser True if the event was directly generated by the
    /// user, false in case it's been propagated through a different mechanism.
    public: explicit EntitiesSelected(
        const std::vector<Entity> &_entities,  // NOLINT
        bool _fromUser = false)
        : QEvent(kType), entities(_entities), fromUser(_fromUser)
    {
    }

    /// \brief Get the data sent with the event.
    /// \return The entities being selected.
    public: std::vector<Entity> Data() const
    {
      return this->entities;
    }

    /// \brief Get whether the event was generated by the user.
    /// \return True for the user.
    public: bool FromUser() const
    {
      return this->fromUser;
    }

    /// \brief Unique type for this event.
    static const QEvent::Type kType = QEvent::Type(QEvent::User + 1);

    /// \brief The selected entities.
    private: std::vector<Entity> entities;

    /// \brief Whether the event was generated by the user,
    private: bool fromUser{false};
  };

  /// \brief Event that notifies when all entities have been deselected.
  class DeselectAllEntities : public QEvent
  {
    /// \brief Constructor
    /// \param[in] _fromUser True if the event was directly generated by the
    /// user, false in case it's been propagated through a different mechanism.
    public: explicit DeselectAllEntities(bool _fromUser = false)
        : QEvent(kType), fromUser(_fromUser)
    {
    }

    /// \brief Get whether the event was generated by the user.
    /// \return True for the user.
    public: bool FromUser() const
    {
      return this->fromUser;
    }

    /// \brief Unique type for this event.
    static const QEvent::Type kType = QEvent::Type(QEvent::User + 2);

    /// \brief Whether the event was generated by the user,
    private: bool fromUser{false};
  };

  /// \brief Event called in the render thread of a 3D scene.
  /// It's safe to make rendering calls in this event's callback.
  class Render : public QEvent
  {
    public: Render()
        : QEvent(kType)
    {
    }
    /// \brief Unique type for this event.
    static const QEvent::Type kType = QEvent::Type(QEvent::User + 3);
  };

  /// \brief Event called to spawn a preview model.
  /// Used by plugins that spawn models.
  class SpawnPreviewModel : public QEvent
  {
    /// \brief Constructor
    /// \param[in] _modelSdfString The model's SDF file as a string.
    public: explicit SpawnPreviewModel(std::string &_modelSdfString)
        : QEvent(kType), modelSdfString(_modelSdfString)
    {
    }

    /// \brief Unique type for this event.
    static const QEvent::Type kType = QEvent::Type(QEvent::User + 4);

    /// \brief Get the sdf string of the model.
    /// \return The model sdf string
    public: std::string ModelSdfString() const
    {
      return this->modelSdfString;
    }

    /// \brief The sdf string of the model to be previewed.
    std::string modelSdfString;
  };

  /// \brief Event called to spawn a preview resource, which takes the path
  /// to the SDF file. Used by plugins that spawn resources.
  class SpawnPreviewPath : public QEvent
  {
    /// \brief Constructor
    /// \param[in] _filePath The path to an SDF file.
    public: explicit SpawnPreviewPath(const std::string &_filePath)
        : QEvent(kType), filePath(_filePath)
    {
    }

    /// \brief Unique type for this event.
    static const QEvent::Type kType = QEvent::Type(QEvent::User + 5);

    /// \brief Get the path of the SDF file.
    /// \return The file path.
    public: std::string FilePath() const
    {
      return this->filePath;
    }

    /// \brief The path of SDF file to be previewed.
    std::string filePath;
  };
}  // namespace events
}
}  // namespace gui
}  // namespace gazebo
}  // namespace ignition

#endif  // GZ_GAZEBO_GUI_GUIEVENTS_HH_
