/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *               2003 Dirk Ziegelmeier <dziegel@gmx.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <klocale.h>
#include <kdebug.h>
#include <kconfig.h>
#include <assert.h>

#include <qwidget.h>
#include <qlayout.h>
#include <qframe.h>
#include <qcheckbox.h>

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>

extern "C" {
#include <libzvbi.h>
}

#include "simon.h"

#include <kdetv.h>
#include <osdmanager.h>
#include <statusmanager.h>
#include <vbimanager.h>


KdetvSimon::KdetvSimon(Kdetv *ktv, QWidget *parent, const char* name)
    : KdetvMiscPlugin(ktv, "simon-misc", parent, name),
      _vbimgr(ktv->vbiManager())
{
    kdDebug() << "Simon: Loading plugin." << endl;

    _cfg->setGroup("Displays");
    _cfg_displayText  = _cfg->readBoolEntry("Display Text", true);
    _cfg_displayTitle = _cfg->readBoolEntry("Display Title", true);

    _vbimgr->addClient();

    connect(_vbimgr, SIGNAL( caption(int) ),
            this, SLOT( captionData(int) ));
    connect(_vbimgr, SIGNAL( progTitle(const QString&) ),
            this, SLOT( programTitle(const QString&) ));
    connect(driver()->statusManager(), SIGNAL( channelChanged() ),
            this, SLOT( channelChanged() ));
}

KdetvSimon::~KdetvSimon()
{
    _vbimgr->removeClient();
}

void KdetvSimon::channelChanged()
{
    if (_cfg_displayTitle)
        driver()->statusManager()->setProgram("");
    if (_cfg_displayText)
        driver()->osdManager()->displayCC("");
}

void KdetvSimon::captionData(int pgno)
{
    vbi_page page;

    if (_cfg_displayText &&
        vbi_fetch_cc_page(static_cast<vbi_decoder*>(_vbimgr->internalData()),
                          &page,
                          pgno,
                          1)) {
        char buf[1024];
        int rc = vbi_print_page(&page, buf, sizeof(buf), "ISO-8859-1", true, true);
        if (rc > 0) {
            buf[rc] = 0;
            driver()->osdManager()->displayCC(QString::fromLatin1(buf).stripWhiteSpace());
        }
        vbi_unref_page(&page);
    }
}

void KdetvSimon::programTitle(const QString& title)
{
    if (_cfg_displayTitle)
        driver()->statusManager()->setProgram(title);
}

QWidget *KdetvSimon::configWidget(QWidget *parent, const char *name)
{
    QFrame *w = new QFrame(parent, name);
    QGridLayout *g = new QGridLayout(w, 2, 1);
    
    w->setMargin(4);
    
    _cb_dispText = new QCheckBox(i18n("Display closed caption text"), w);
    _cb_dispTitle = new QCheckBox(i18n("Display the program title in the statusbar"), w);
    _cb_dispText->setChecked(_cfg_displayText);
    _cb_dispTitle->setChecked(_cfg_displayTitle);
    
    g->addWidget(_cb_dispText,  0, 0);
    g->addWidget(_cb_dispTitle, 1, 0);
    
    return w;
}

void KdetvSimon::saveConfig()
{
    _cfg->setGroup("Displays");
    _cfg->writeEntry("Display Text",  (_cfg_displayText  = _cb_dispText->isChecked()));
    _cfg->writeEntry("Display Title", (_cfg_displayTitle = _cb_dispTitle->isChecked()));
    _cfg->sync();

    if (!_cfg_displayText)
        driver()->osdManager()->displayCC("");
    if (!_cfg_displayTitle)
        driver()->statusManager()->setProgram("");
}


extern "C" {
    KdetvSimon* create_simon(Kdetv *ktv, QWidget *parent) 
    {
        return new KdetvSimon(ktv, parent);
    }
}

#include "simon.moc"
