// -*- c++ -*-

/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *               2003 Dirk Ziegelmeier <dziegel@gmx.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __LIBKDETV_H
#define __LIBKDETV_H

#include <qmap.h>
#include <qcolor.h>
#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qguardedptr.h>

#include "kdetviface.h"

class KConfig;
class KdetvView;
class ViewManager;
class AudioManager;
class SourceManager;
class OSDManager;
class ChannelStore;
class StatusManager;
class MiscManager;
class VbiManager;
class PluginFactory;
class KActionCollection;
class KXMLGUIFactory;
class ConfigData;
class Channel;

/**
 * A reusable TV object.
 * A client should only use the functions of this class and signals of the managers.
 * If manager functions are used, it is likely to get things out of sync!
 */
class Kdetv : public QObject, virtual public KdetvIface
{
    Q_OBJECT

public:
    Kdetv( QObject *parent = 0, const char* name = 0);
    virtual ~Kdetv();

    /**
     * Returns the current screen widget.
     */
    KdetvView *screen() { return _view; }

    /**
     * Returns a DCOP reference to the ChannelStore.
     */
    DCOPRef channelStoreIface();

    /**
     * Returns a DCOP reference to the current Channel.
     */
    DCOPRef channelIface();

    /**
     * Returns the name of the current Channel.
     */
    QString channelName() const;

    /**
     * Returns the number of the current Channel.
     */
    int channelNumber() const;

    /**
     * Returns the manager controlling the views.
     */
    ViewManager *views() const { return _viewmng; }

    /**
     * Returns the audio manager.
     */
    AudioManager *audioManager() const { return _am; }

    /**
     * Returns the source manager.
     */
    SourceManager *sourceManager() const { return _srcm; }

    /**
     * Returns the OSD manager.
     */
    OSDManager *osdManager() const { return _osd; }

    /**
     * Returns the miscellaneous plugin manager.
     */
    MiscManager *miscManager() const { return _mm; }

    /**
     * Returns the status plugin manager.
     */
    StatusManager *statusManager() const { return _sm; }

    /**
     * Returns the VBI plugin manager.
     */
    VbiManager *vbiManager() const { return _vbim; }

    /**
     * Returns the plugin factory.
     */
    PluginFactory *pluginFactory() const { return _pf; }

    /**
     * Returns current ChannelStore.
     */
    ChannelStore *channels() { return _cs; }

    /**
     * Returns the current configuration.
     */
    ConfigData *config() const { return _cfg; }
    
    /**
     * Returns list of supported channel formats with read support.
     * For use with importChannelFile()
     */
    const QStringList& fileFormatsRead() const;

    /**
     * Returns list of supported channel formats with write support.
     * For use with exportChannelFile()
     */
    const QStringList& fileFormatsWrite() const;

    /**
     * Factory method for creating the screen the picture will be displayed
     * on. This widget returned will depend on the facilities supported by
     * the TV card and X server (eg. there will be special screen widgets
     * for Xv support). Note that this widget is automatically made the
     * screen for this kdetv object.
     */
    virtual KdetvView *createScreen( QWidget *parent=0, const char *name=0 );

    /**
     * Return a copy of the client's GUI factory, if available, so that plugins
     * or other components can modify the UI dynamically.
     * May return 0L!
     * Note: After the guiFactory is set all the plugins are notified of the (new) gui
     * factory so they usually dont need to call this.
     * This method is only needed for the plugin factory when creating new plugins.
     */
    virtual KXMLGUIFactory *guiFactory() const;

    /**
     * Return a copy of the client's action collection, if available, so that
     * plugins or other components can modify the UI dynamically.
     * May return 0L!
     * Note: After the actionCollection is set all the plugins are notified of the (new) action
     * collection so they usually dont need to call this.
     * This method is only needed for the plugin factory when creating new plugins.
     */
    virtual KActionCollection *actionCollection() const;

    /**
     * To get various values.
     */
    QColor colourKey();

    /**
     * Get current channel
     */
    Channel *channel() { return _chan; }

    /**
     * FOR CLIENTS ONLY!  Set the guiFactory and actionCollection if available.
     */
    virtual void setGuiFactory(KXMLGUIFactory *guiFactory, KActionCollection *actionCollection);


public slots:
    /**
     * Start video playback. If configured, try to start with last device used.
     * Launch device selection and channel wizard if necessary.
     * Normally only called internally at startup, but may be useful for clients
     */
    virtual bool start();

    /**
     * Start playing desired device. Restore previous channel,
     * launch source selection/wizard dialogs if necessary. If 
     * no channel file exists for device, apply src and encoding supplied
     * as arguments. Restore picture settings.
     * You most probably want to call this function from the client
     * or when you have manually altered SourceManager settings to get
     * things in sync again.
     */
    virtual bool playDevice(const QString& dev,
			    const QString& src = QString::null,
			    const QString& enc = QString::null);

    /**
     * Stop video playback.
     */
    virtual void stop();

    /**
     * Sets the current current channel.
     */
    virtual void setChannel( Channel *channel );

    /**
     * Overloaded version of the above method, sets channel if not already set.
     */
    virtual void setChannel(int channel);

    /**
     * Sets channel, but does not check if it is already set.
     */
    void doSetChannel( int channel );

    /**
     * Changes the channel to the one preceding the current, wrapping
     * round if we are currently showing the first channel.
     */
    virtual void channelDown();

    /**
     * Changes the channel to the one following the current, wrapping
     * round if we are currently showing the last channel.
     */
    virtual void channelUp();
    
    /**
     * Recalls the previous channel.
     */
    virtual void previousChannel();

    /**
     * Mutes and unmutes the sound.
     */
    virtual void toggleMute();

    /**
     * Increases the volume.
     */
    virtual void volumeUp();

    /**
     * Decreases the volume.
     */
    virtual void volumeDown();

    /**
     * Sets the volume.
     */
    virtual void setVolume(int left, int right);
    virtual void setVolume(int vol);

    /**
     * Audio mode selection
     */
    virtual void setAudioMode(const QString& mode);

    /**
     * Restore picture settings for current state
     */
    void applyPictureSettings();

    /**
     * Save current picture settings
     */
    void savePictureSettings();

    /**
     * Save snapshot to disk.
     */
    virtual void snapshot();

    /**
     *  Imports the old KWinTV channels if they exist.
     */
    virtual bool importLegacyChannels(bool warn = true);

    /**
     *  Open a file dialog to allow the user to add the channels from a file
     *  to the channel list.
     */
    virtual void importChannelFile(const QString& fmt=QString::null);

    /**
     *  Open a file dialog to allow the user export channels in selected format
     */
    virtual void exportChannelFile(const QString& fmt=QString::null);

    /**
     * Process the press of a number key (0-9) from keyboard, DCOP or lirc.
     */
    virtual void processNumberKeyEvent(int);

    /**
     * Reload channel file
     */
    virtual void reloadChannels();

    /**
     * Save channel file
     */
    virtual void saveChannels();

    /**
     * Launch settings dialog.
     */
    void settings();

    /**
     * Launch channel wizard.
     */
    void launchWizard();

    /**
     * Launch import frequency list dialog.
     */
    void importDefault();

    /**
     * Launch picture settings dialog
     */
    void pictureSettings();

    /**
     * Launch channel editor
     */
    void editChannels();


signals:
    /**
     * Emitted when the volume changes
     */
    void volumeChanged( int left, int right );
    
    /**
     * Emitted whenever the volume is muted.
     */
    void volumeMuted(bool);

    /**
     * Emitted when the current channel changes.
     */
    void channelChanged( int num );

    /**
     * Emitted when the current channel changes.
     */
    void channelChanged( const QString &name );

    /**
     * Emitted when the current channel changes.
     */
    void channelChanged( Channel *channel );

    /**
     * Emitted when the channel text changes during setting channel with numeric keys.
     */
    void channelText(const QString &);


protected:
    void timerEvent (QTimerEvent *ev);

    
private slots:
    /** Mute/Unmute volume... */
    void setMuteVolume (bool);

    void slotKeyPressTimeout();

    void mouseWheelUp();
    void mouseWheelDown();


private:
    /**
     *  Do migration from kwintv to kdetv
     */
    bool doMigration();

    /**
     * Set current channel to last used channel
     */
    void setLastChannel();

    ChannelStore *_cs;

    QGuardedPtr<Channel> _chan;

    OSDManager *_osd;
    AudioManager *_am;
    SourceManager *_srcm;
    StatusManager *_sm;
    MiscManager *_mm;
    VbiManager *_vbim;
    KdetvView *_view;
    PluginFactory *_pf;
    ViewManager *_viewmng;
    ConfigData* _cfg;
    KConfig* _cfgHandle;
    KXMLGUIFactory *_guiFactory;
    KActionCollection *_actionCollection;

    QStringList _fileFormatsRead;
    QStringList _fileFormatsWrite;

    QTimer *_keypresstimer;
    QString _number;

    int _prevChannel;

    bool _muted;    // We have to keep track since some devices are broken    

    struct {
        int left; // Left volume
        int right; // Right volume
        int changeEventId; // Id 
    } ChannelVolState;

    unsigned long _grabNumber;
};

#endif

