/***************************************************************************
                          kaspasql.h  -  description
                             -------------------                                         
    begin                : Tue Sep 7 1999                                           
    copyright            : (C) 1999 by Jan Mueller                         
    email                : janmueller7@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef SQL_H
#define SQL_H

#ifdef HAVE_CONFIG_H
	#include <config.h>
	#ifdef DEBUG
	#undef DEBUG
	#endif
#endif
// #include <postgres.h>
#include <libpq/libpq-fs.h>
#include <libpq-fe.h>
#include "kaspaerr.h"
#include "str.h"

//////////////////////////////
// Global:

/** Expands "'" to "\'" */
char *esc4sql(const char *s);

/** Converts the ID's of the tuples or large objects (Oids). */
char *oid2str(Oid o);

/** Same as above. Str added to S. */
void oid2str(Oid o, Str *s);

/** Converts a oid stored as string to a numeric oid. */
Oid str2oid(const char *c);


/**
 * @short All exception classes indicating server operation problems are derived from SqlErr.
 */
class SqlErr: public KaspaErr {
 public:
  SqlErr(const char *s=0);
};

/**
 * @short Thrown by large-object-IO operations, e.g. lRead(), lWrite().
 */
class LoIOErr: public SqlErr {
 public:
  LoIOErr(const char *s=0);
};

/**
 * @short LO-handle errors, e.g. lOpen(), lClose().
 */
class LoHandleErr: public SqlErr {
 public:
  LoHandleErr(const char *s=0);
};

/**
 * @short If the server dies...
 */
class SqlConnectionErr: public SqlErr {
 public:
  SqlConnectionErr(const char *s=0);
};

/**
 * @short Handles errors occuring on the organization of the LOs, e.g. lImport(),
 * lExport, lCreate(), lUnlink().
 */
class LoOidErr: public SqlErr {
 public:
  LoOidErr(const char *s=0);
};

/**
 * @short Range errors in querys, e.g. getValue().
 */
class SqlOutOfRange: public SqlErr {
 public:
  SqlOutOfRange(const char *s=0);
};

/**
 * @short Query execution errors, e.g. exec().
 */
class SqlExecErr: public SqlErr {
 public:
  SqlExecErr(const char *s=0);
};

/**
 * @short Query canceled, e.g. exec().
 */
class SqlExecCanceled: public SqlErr {
 public:
  SqlExecCanceled(const char *s=0);
};

/**
 * @author Jan Mller
 * @version 0.2
 * @short Implements interface to the postgres server.
 * Methods throw @ref SqlErr derived execptions. Sql.error()
 * is set up with the server error messages.
 */
class Sql {
	private:
	Sql& operator=(const Sql&);

	public:	

  /** Copy constructor. Takes the server connection. Both objects
   *  use the same connection afterwards!
   * Throws @ref SqlConnectionErr.
   */
  Sql(const Sql& s) throw (SqlConnectionErr);

  /** Opens new connection.
   * Throws @ref SqlConnectionErr.
   * @param name name of the database.
   */
  Sql(const char *name) throw (SqlConnectionErr);

  /** Destructor...*/
  virtual ~Sql() throw();

  /** Returns connection. */
  PGconn *getConn() const throw() ;

  /** Returns TRUE, if a query is processed.
   * See @ref idle.
   */
  bool isWorking() throw();

  /** Dump client-server dialog in a file.*/
  void traceOn(FILE *f) throw();

  /** Opens LO.
   * Throws @ref LoHandleErr, @ref SqlConnectionErr.
   * @return handle
   * @param id Oid
   * @param mode INV_READ|INV_WRITE|INV_ARCHIVE
   */
  virtual int lOpen(Oid id, int mode) throw (LoHandleErr, SqlConnectionErr);

  /** Closes LO.
   * Throws @ref LoHandleErr, @ref SqlConnectionErr.
   * @param fd Oid
   */
  virtual void lClose(int fd) throw (LoHandleErr, SqlConnectionErr);

  /** Creating new LO.
   * Throws @ref LoHandleErr, @ref SqlConnectionErr.
   * @return Oid
   * @param mode INV_READ|INV_WRITE|INV_ARCHIVE
   */
  virtual Oid lCreate(int mode) throw (LoOidErr, SqlConnectionErr);

  /** Changes cursor position of LOs.
   * Throws @ref LoIOErr, @ref SqlConnectionErr.
   * @return new position
   * @param fd handle
   * @param offs offset
   * @param mode SEEK_SET or SEEK_CUR or SEEK_END, only SEEK_SET implemented.
   */
  virtual int lSeek(int fd, int offs, int mode) throw (LoIOErr, SqlConnectionErr);

  /** Returns cursor position of LOs.
   * Throws @ref LoIOErr, @ref SqlConnectionErr.
   * @return position
   * @param fd handle
   */
  virtual int lTell(int fd) throw (LoIOErr, SqlConnectionErr);

  /** Import file. This is done by the server. That means that the
	 * permissions may differ from the client's premissions.
   * Throws @ref LoOidErr, @ref SqlConnectionErr.
   * @return Oid.
   * @param name filename
   */
  virtual Oid lImport(const char *name) throw (LoOidErr, SqlConnectionErr);

  /** Writes the LO into a file. Done by the backend. See lImport().
   * Throws @ref LoOidErr, @ref SqlConnectionErr.
   * @param id Oid
   * @param name filename
   */
  virtual void lExport(Oid id, const char *name) throw (LoOidErr, SqlConnectionErr);

  /** Writes to LO.
   * Throws @ref LoIOErr, @ref SqlConnectionErr.
   * @return written bytes.
   * @param fd handle
   * @param *buf data
   * @param len length of data
   */
  virtual int lWrite(int fd, const char *buf, int len) throw (LoIOErr, SqlConnectionErr);

  /** Reads from LO.
   * Throws @ref LoIOErr, @ref SqlConnectionErr.
   * @return read bytes.
   * @param fd handle
   * @param *buf buffer
   * @param len length of buffer.
   */
  virtual int lRead(int fd, char *buf, int len) throw (LoIOErr, SqlConnectionErr);

  /** Deletes LO from database.
   * Throws @ref LoOidErr, @ref SqlConnectionErr.
   * @param id Oid
   */
  virtual void lUnlink(Oid id) throw (LoOidErr, SqlConnectionErr);

  /** Processes querys. Calls @ref idle while processing the query (if not <block>).
   * If @ref idle returns false, the query is canceled and the
   * method throws @ref SqlExecCanceled.
	 * The function is not reentrent.
	 * It will throw @ref SqlExecErr if a query is already in progress.
   * Throws @ref SqlExecCanceled, @ref SqlExecErr, @ref SqlConnectionErr.
   * @param q query
	 * @param block no idle processing
   */
  virtual void exec(const char *q, bool block=false)
											throw (SqlExecCanceled, SqlExecErr, SqlConnectionErr);

	/*	virtual void cancel(); */

  /** Called by @ref exec to avoid blocking.
   *  @return Returns TRUE by default. Returning FALSE would result
   * in cancellation of the query.
   */		
  virtual bool idle() throw();

  /** Returns the number of tuples resulting from a query. */
  int tuples() throw ();

  /** Returns the number of rows resulting from a query. */
  int fields() throw();

  /** Returns a field resulting from a query.
   * Throws @ref SqlOutOfRange.
   * @return fields as a string. The string is valid until the
   * next query by @ref exec or until the destruction of the object.
   * @param t Index of tuple.
   * @param f Index of row
   */
  const char *getValue(int t, int f) throw(SqlOutOfRange);

  /** As above, name of the row instead of index.
      Throws @ref SqlOutOfRange.
  */
  const char *getValue(int t, const char *f) throw(SqlOutOfRange);

  /** Returns the name of a row.
   * Throws @ref SqlOutOfRange
   * @return name is valid until next query or destruction of object.
   * @param num index of the row.
   */
  const char *fieldName(int num) throw(SqlOutOfRange);
	const bool fieldIsText(int num);
	const Oid type(int num);

  /** Returns the ID of the last insert operation.
   * @return Oid of the inserted object or InvalidOid if failed.
   */
  Oid oidStatus() throw() {
    if(pgResult) return str2oid(PQoidStatus(pgResult));
    else return InvalidOid;
  }

  /** Returns the last error message. */
  const char* errorMessage() const throw();

  /** Returns true if the connection to the server failed. */
  int connectionBad() const throw() { return PQstatus(pgConn)==CONNECTION_BAD; }

	/** returns true if the table exists */
	bool table(Str name);

	/** returns true if the index exists */
	bool index(Str name);

	/** */
	Oid type();

  void reset() {
    PQreset(pgConn);
    }

 private:
  /** Checks the connection, throws @ref SqlConnectionErr if bad. */
  inline void checkErr()  throw(SqlConnectionErr);
	bool check4rel(const Str rel, const Str field, const Str name);
  bool working;

  PGconn* pgConn;				
  PGresult* pgResult;		
  int pgCloseConnection; // TRUE if connection should be closed by destructor
};

#endif
