/**
 * entrytoken.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "entrytoken.h"
#include "undomanager.h"
#include "memento.h"

#include <kdebug.h>

namespace KCfgCreator
{

EntryToken::EntryToken( GroupToken* parent, const QString& key,
                        const QString& name, bool hidden )
    : Token( name ), m_parent( parent ), m_key( key ),
      m_hidden( hidden ), m_fromEnv( false )
{
}

QString
EntryToken::label() const
{
    return m_label;
}

void
EntryToken::setLabel( const QString& lab )
{
    addEntryMemento( m_label, &EntryToken::setLabel );
    m_label = lab;
}

QString
EntryToken::whatsThis() const
{
    return m_whatsThis;
}

void
EntryToken::setWhatsThis( const QString& whatsThis )
{
    addEntryMemento( m_whatsThis, &EntryToken::setWhatsThis );
    m_whatsThis = whatsThis;
}

QValueList<EntryToken::Choice>
EntryToken::choices() const
{
    return m_choices;
}

void
EntryToken::setChoices( const QValueList<EntryToken::Choice>& ch )
{
    //addEntryMemento( m_choices, &EntryToken::setChoices );
    m_choices = ch;
}

EntryToken::DefaultValue
EntryToken::defaultValue() const
{
    return m_default;
}

void
EntryToken::setDefaultValue( const EntryToken::DefaultValue& val )
{
    //addEntryMemento( m_default, &EntryToken::setDefaultValue );
    m_default = val;
}

QString
EntryToken::name() const
{
    if ( Token::name().isEmpty() ) {
        QString key = m_key;
        return key.remove( ' ' );
    }
    return Token::name();
}

QString
EntryToken::key() const
{
    return m_key;
}

void
EntryToken::setKey( const QString& key )
{
    addEntryMemento( m_key, &EntryToken::setKey );
    m_key = key;
}

bool
EntryToken::contains( const QString& str ) const
{
    return ( Token::name().contains( str ) || valueAsString().contains( str ) );
}

GroupToken*
EntryToken::parent() const
{
    return m_parent;
}

void
EntryToken::addEntryMemento( const QString& data,
                             void (EntryToken::*func)( const QString& ) )
{
    if ( undoEnabled() )
        UndoManager::Instance()->addMemento( new EntryMemento( this, data, func ) );
}

void
EntryToken::addEntryMemento( const QStringList& data,
                             void (EntryToken::*func)( const QStringList& ) )
{
    if ( undoEnabled() )
        UndoManager::Instance()->addMemento( new EntryMemento( this, data, func ) );
}

void
EntryToken::setHidden( bool h )
{
    m_hidden = h;
}

bool
EntryToken::hidden() const
{
    return m_hidden;
}

void
EntryToken::setFromEnvironment( bool en )
{
    m_fromEnv = en;
}

bool
EntryToken::fromEnvironment() const
{
    return m_fromEnv;
}

void
EntryToken::copyFromToken( EntryToken* token )
{
    m_parent = token->m_parent;
    m_label  = token->m_label;
    m_whatsThis = token->m_whatsThis;
    m_choices = token->m_choices;
    m_default = token->m_default;
    m_key = token->m_key;
    m_hidden = token->m_hidden;
    // token->m_immutable;
    m_fromEnv = token->m_fromEnv;
    setValueFromString( token->valueAsString() );
}

EntryToken::Value
EntryToken::min() const
{
    return m_min;
}

void
EntryToken::setMin( const EntryToken::Value& m )
{
    m_min = m;
}

EntryToken::Value
EntryToken::max() const
{
    return m_max;
}

void
EntryToken::setMax( const EntryToken::Value& m )
{
    m_max = m;
}

EntryToken::Parameter
EntryToken::parameter() const
{
    return m_param;
}

void
EntryToken::setParameter( const EntryToken::Parameter& p )
{
    m_param = p;
}

QString
EntryToken::code() const
{
    return m_code;
}

void
EntryToken::setCode( const QString& c )
{
    m_code = c;
}

QStringList
EntryToken::values() const
{
    return m_values;
}

void
EntryToken::setValues( const QStringList& v )
{
    m_values = v;
}

namespace Utils
{
    static QMap<QString, int> s_typeMap;

    void
    initTypeMap()
    {
        if ( s_typeMap.isEmpty() ) {
            s_typeMap.insert( "String",     EntryToken::String );
            s_typeMap.insert( "Password",   EntryToken::Password );
            s_typeMap.insert( "StringList", EntryToken::StringList );
            s_typeMap.insert( "Font",       EntryToken::Font );
            s_typeMap.insert( "Rect",       EntryToken::Rect );
            s_typeMap.insert( "Size",       EntryToken::Size );
            s_typeMap.insert( "Color",      EntryToken::Color );
            s_typeMap.insert( "Point",      EntryToken::Point );
            s_typeMap.insert( "Int",        EntryToken::Int );
            s_typeMap.insert( "UInt",       EntryToken::UInt );
            s_typeMap.insert( "Bool",       EntryToken::Bool );
            s_typeMap.insert( "Double",     EntryToken::Double );
            s_typeMap.insert( "DateTime",   EntryToken::DateTime );
            s_typeMap.insert( "Int64",      EntryToken::Int64 );
            s_typeMap.insert( "UInt64",     EntryToken::UInt64 );
            s_typeMap.insert( "IntList",    EntryToken::IntList );
            s_typeMap.insert( "Enum",       EntryToken::Enum );
            s_typeMap.insert( "Path",       EntryToken::Path );
            s_typeMap.insert( "Invalid",    EntryToken::Invalid );
        }
    }

    EntryToken::Type stringToType( const QString& typeName )
    {
        if ( s_typeMap.isEmpty() ) initTypeMap();
        if ( s_typeMap.contains( typeName ) )
            return static_cast<EntryToken::Type>(  s_typeMap[ typeName ] );
        else return EntryToken::Invalid;
    }
} //end namespace Utils

} //end namespace KCfgCreator
