/**
 * configmanager.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "configmanager.h"
#include "kcfgparser.h"
#include "kconfigparser.h"
#include "hosttoken.h"
#include "applicationtoken.h"
#include "hostdata.h"
#include "kernel.h"

#include <kurl.h>
#include <klocale.h>
#include <kdebug.h>

#include <qtimer.h>

namespace KCfgCreator
{

ConfigManager::ConfigManager( QObject *parent )
    : QObject( parent, "configmanager" ),
      m_currentHost( 0 ), m_data( 0 )
{
    m_tokens.setAutoDelete( true );

    m_kcfgParser = new KCfgParser( this );
    m_kconfigParser = new KConfigParser( this );

    connect( m_kcfgParser, SIGNAL(parsed(HostToken*)),
             SLOT(slotParsedHost(HostToken*)) );
    connect( m_kconfigParser, SIGNAL(parsed(HostToken*)),
             SLOT(slotParsedHost(HostToken*)) );
}

void
ConfigManager::slotParseData( HostData *data )
{
    m_data = data;
    QTimer::singleShot( 0, this, SLOT(timerParse()) );
}

void
ConfigManager::slotParsedHost( HostToken *host )
{
    if ( m_tokens.find( host->url().prettyURL() ) ) {
        emit hostReparsed( host );
    } else {
        m_tokens.insert( host->url().prettyURL(), host );
        emit newHostParsed( host );
    }
}

void
ConfigManager::timerParse()
{
    if ( !m_data )
        return;

    if ( !m_currentHost )
        m_currentHost = new HostToken( m_data->url(), m_data->url().host() );

    ApplicationToken* atoken = 0;

    FileData *kcfg = m_data->popKCfg();
    if ( kcfg ) {
        atoken = m_kcfgParser->parseAndAddFile( m_currentHost, kcfg );
        if ( !atoken ) {
            kernel->emitLogMsg( "KCFGParser",
                                i18n( "%1: file has wrong KCFG format." ).arg( kcfg->url().prettyURL() )
                );
            delete kcfg;
            QTimer::singleShot( 0, this, SLOT(timerParse()) );
            return;
        }

        QPtrListIterator<FileData> mergeFile( m_data->kconfigFiles() );
        while ( mergeFile.current() ) {
            QString file = atoken->name();
            if ( mergeFile.current()->url().fileName() == file ) {
                m_kconfigParser->parseDataMerging( atoken, mergeFile.current() );
                m_data->kconfigFiles().remove( mergeFile.current() );
                QTimer::singleShot( 0, this, SLOT(timerParse()) );
                return;
            }
            ++mergeFile;
        }
    } else {
        FileData *kconfig = m_data->popKConfig();
        if ( kconfig ) {
            m_kconfigParser->parseAndAddFile( m_currentHost, kconfig );
        } else {
            //finished
            m_data = 0;
            emit newHostParsed( m_currentHost );
            m_currentHost = 0;
            return;
        }
    }
    QTimer::singleShot( 0, this, SLOT(timerParse()) );
}

}

#include "configmanager.moc"
