/***************************************************************************
    Description          : KPuzzle - A KDE Jigsaw Puzzle Game
    Version              : 0.2
    Copyright            : (C) 2000-2001 by Michael Wand
    EMail                : mwand@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "main.h"
#include "gamewidget.h"

#include "kpuzzle.h"
#include "kpuzzleapp.h"
#include "kpuzzleview.h"

#include <qpaintdevice.h>
#include <qcursor.h>

#include <kconfig.h>
#include <kcursor.h>

KGameWidget::KGameWidget(QWidget *parent) : QWidget(parent)
{
	_app = NULL;
	setMouseTracking(true);
        setCursor(KCursor::blankCursor());
	_piecePos = QPoint(-1,-1);
	_moveTimer = new QTimer(this);
	connect(_moveTimer,SIGNAL(timeout()),this,SLOT(slotMoveTimer()));
	_scrollPos = QPoint(0,0);
        _moveKeyPressed = 0;
        _rightPressed = false;
	setBackgroundMode(NoBackground);
	connect(this,SIGNAL(sigShowAt(QPoint)),this,SLOT(slotShowAt(QPoint)));
        _backBuffer = NULL;

}

KGameWidget::~KGameWidget()
{
       delete _moveTimer;
       delete _backBuffer;
}

void KGameWidget::initialize(QSize picSize)
{
	// This is necessary to avoid drawing problems
	if (picSize.width() < this->width()) setFixedWidth(picSize.width());
	if (picSize.height() < this->height()) setFixedHeight(picSize.height());
        _backBuffer = new QPixmap(size());
}

QSize KGameWidget::pieceSize()
{
        return _app->pieceSize();
}

QSize KGameWidget::pieceSizeDisp()
{
        return _app->pieceSizeDisp();
}

QRect KGameWidget::pieceRect()
{
	return QRect(pieceTopLeft(),pieceSizeDisp());
}

QPoint KGameWidget::pieceTopLeft()
{
        QPoint sizePt(pieceSize().width() / 2,pieceSize().height() / 2);
        return piecePos() - sizePt - QPoint(_app->displace(),_app->displace());
}

void KGameWidget::paintEvent(QPaintEvent* event)
{
        if (((KPuzzleView*) parent())->status() != GS_RUNNING) return;
	QRect logRect(DPtoLP(event->rect()));
        ASSERT(_backBuffer->size() == this->size());
	bitBlt(_backBuffer,event->rect().topLeft(),_app->gamePixmap(),logRect,CopyROP,false);
	if (!_rightPressed && piecePos().x() != -1) {
                QPixmap* pieceTurned = _app->currentPieceTurned(); // CAUTION: implicit new
                bitBlt(_backBuffer,pieceTopLeft(),
                       pieceTurned,QRect(QPoint(0,0),_app->pieceSizeDisp()),CopyROP,false);
                delete pieceTurned;
        }
        bitBlt(this,event->rect().topLeft(),_backBuffer,event->rect());
}

void KGameWidget::mouseMoveEvent(QMouseEvent* event)
{
	if (((KPuzzleView*) parent())->status() != GS_RUNNING) return;
        if (_rightPressed) { // panning around
                QPoint diffPos = _movePos - event->pos();
                QPoint newPos = scrollPos() + diffPos;
                if (newPos.x() < 0) newPos.setX(0);
                if (newPos.x() > _app->pixmapSize().width() - this->width()) newPos.setX( _app->pixmapSize().width() - this->width());
                if (newPos.y() < 0) newPos.setY(0);
                if (newPos.y() > _app->pixmapSize().height() - this->height()) newPos.setY( _app->pixmapSize().height() - this->height());
                emit sigShowAt(newPos);
                _movePos = event->pos();
        }

        // These are all absolute coordinates
	QPoint prevPos(piecePos());
	QRect oldRect(pieceRect());
	_piecePos = event->pos();
	QRect newRect(pieceRect());
	if (prevPos.x() != -1) newRect = newRect.unite(oldRect);
	update(newRect);
}

void KGameWidget::mousePressEvent(QMouseEvent* event)
{
	QPoint p; QPoint t;
        _rightPressed = false;
        setCursor(KCursor::blankCursor());
        switch (event->button()) {
	case LeftButton:
		if (((KPuzzleView*) parent())->status() != GS_RUNNING) return;
		p = DPtoLP(event->pos());
		t = calculatePiecePos(p);
		if (!_app->setPiece(t)) return;
		break;
	case RightButton:
		if (kapp->sessionConfig()->readBoolEntry("RightKeyPansAround",SV_RIGHT_KEY_PANS_AROUND)) {
                        _rightPressed = true;
                        _movePos = event->pos();
                        setCursor(KCursor::sizeAllCursor());
                } else {
                        if (kapp->sessionConfig()->readBoolEntry("TurnPiecesAround",SV_TURN_PIECES_AROUND))
                                _app->turnCW();
                        else
                                _app->changeLRTurn();

                        _app->updateAll();
                }
                break;
	case MidButton:
		if (kapp->sessionConfig()->readBoolEntry("TurnPiecesAround",SV_TURN_PIECES_AROUND))
			_app->turnCCW();
		else
			_app->changeUDTurn();

		_app->updateAll();
		break;
            default: ;
        }
}

void KGameWidget::mouseReleaseEvent(QMouseEvent* event)
{
        switch(event->button()) {
            case RightButton:
                    if (!_rightPressed) debug("mouseReleaseEvent without right key pressed before");
                    _rightPressed = false;
                    setCursor(KCursor::blankCursor());
                    break;
            default:
                    break; // To avoid compiler warnings
        }
}

void KGameWidget::leaveEvent(QEvent*)
{
	if (((KPuzzleView*) parent())->status() != GS_RUNNING) return;
	QRect rect(pieceRect());
	_piecePos = QPoint(-1,-1);
        update(rect);
}

QPoint KGameWidget::calculatePiecePos(QPoint p)
{
	// p is in widget coordinates
	// return value is the logical position of the piece
	QSize ts(_app->pieceSize());
	QPoint szpt(ts.width(),ts.height());
	QPoint r(p.x() / szpt.x(),p.y() / szpt.y());
	return r;
}

void KGameWidget::slotLeftPress()
{
	_moveTimer->start(MOVE_TIMER_INTERVAL);
	_moveKeyPressed = 1;
	slotMoveTimer();
}

void KGameWidget::slotLeftRelease()
{
	_moveTimer->stop();
	_moveKeyPressed = 0;
}

void KGameWidget::slotRightPress()
{
	_moveTimer->start(MOVE_TIMER_INTERVAL);
	_moveKeyPressed = 3;
	slotMoveTimer();
}

void KGameWidget::slotRightRelease()
{
	_moveTimer->stop();
	_moveKeyPressed = 0;
}

void KGameWidget::slotUpPress()
{
	_moveTimer->start(MOVE_TIMER_INTERVAL);
	_moveKeyPressed = 2;
	slotMoveTimer();
}

void KGameWidget::slotUpRelease()
{
	_moveTimer->stop();
	_moveKeyPressed = 0;
}

void KGameWidget::slotDownPress()
{
	slotMoveTimer();
	_moveTimer->start(MOVE_TIMER_INTERVAL);
	_moveKeyPressed = 4;
	slotMoveTimer();
}

void KGameWidget::slotDownRelease()
{
	_moveTimer->stop();
	_moveKeyPressed = 0;
}

void KGameWidget::slotMoveTimer()
{
	QPoint newPt;
	switch(_moveKeyPressed) {
	case 1: if (scrollPos().x() > 0) {
		newPt = QPoint(scrollPos().x() - 20,scrollPos().y());
		if (newPt.x() < 0) newPt.setX(0);
		emit sigShowAt(newPt);
	}
	//		scrollPos().setX(scrollPos().x() - 20);
		break;
	case 2: if (scrollPos().y() > 0) {
		newPt = QPoint(scrollPos().x(),scrollPos().y() - 20);
		if (newPt.y() < 0) newPt.setY(0);
		emit sigShowAt(newPt);
	}
		break;
	case 3: if (scrollPos().x() < _app->pixmapSize().width() -
		    this->width()) {
		newPt = QPoint(scrollPos().x() + 20,scrollPos().y());
		if (newPt.x() > _app->pixmapSize().width() - this->width())
			newPt.setX(_app->pixmapSize().width() - this->width());
		emit sigShowAt(newPt);
	}
		break;
	case 4: if (scrollPos().y() < _app->pixmapSize().height() -
		    this->height()) {
		newPt = QPoint(scrollPos().x(),scrollPos().y() + 20);
		if (newPt.y() > _app->pixmapSize().height() -  this->height())
			newPt.setY(_app->pixmapSize().height() - this->height());
		emit sigShowAt(newPt);
	}
		break;


	}
}

void KGameWidget::slotShowAt(QPoint pos)
{
	// To avoid rounding problems
	pos.setX(min(pos.x(),_app->pixmapSize().width() - this->width()));
	pos.setY(min(pos.y(),_app->pixmapSize().height() - this->height()));
	scrollPos() = pos;
	((KPuzzleView*) parent())->updateButtons(!pos.x() == 0,!pos.y() == 0,
		   pos.x() < _app->pixmapSize().width() - this->width(),
		   pos.y() < _app->pixmapSize().height() - this->height());

	update();
}
