/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A table widget with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtablewidget.h"

#include <QKeyEvent>
#include <qapplication.h>
#include <QClipboard>
#include <QScrollBar>

#include "skgtraces.h"

SKGTableWidget::SKGTableWidget(QWidget* iParent)
    : QTableWidget(iParent), stickH(false), stickV(false)
{
    this->installEventFilter(this);
    connect(horizontalScrollBar(), &QScrollBar::valueChanged, this, &SKGTableWidget::onActionTriggered);
    connect(verticalScrollBar(), &QScrollBar::valueChanged, this, &SKGTableWidget::onActionTriggered);
    connect(horizontalScrollBar(), &QScrollBar::rangeChanged, this, &SKGTableWidget::onRangeChanged);
    connect(verticalScrollBar(), &QScrollBar::rangeChanged, this, &SKGTableWidget::onRangeChanged);
}

SKGTableWidget::~SKGTableWidget()
{
}

void SKGTableWidget::onRangeChanged()
{
    QScrollBar* scrollb = qobject_cast<QScrollBar*>(sender());
    if ((stickH && scrollb == horizontalScrollBar()) || (stickV && scrollb == verticalScrollBar())) {
        scrollb->setValue(scrollb->maximum());
    }
}

void SKGTableWidget::onActionTriggered()
{
    QScrollBar* scrollb = qobject_cast<QScrollBar*>(sender());
    if (scrollb) {
        if (scrollb == horizontalScrollBar()) {
            stickH = (scrollb->value() == scrollb->maximum());
        }
        if (scrollb == verticalScrollBar()) {
            stickV = (scrollb->value() == scrollb->maximum());
        }
    }
}

void SKGTableWidget::setStickHorizontal(bool iStick)
{
    stickH = iStick;
}

bool SKGTableWidget::stickHorizontal() const
{
    return stickH;
}

void SKGTableWidget::setStickVertical(bool iStick)
{
    stickV = iStick;
}
bool SKGTableWidget::stickVertical() const
{
    return stickV;
}

bool SKGTableWidget::eventFilter(QObject* iObject, QEvent* iEvent)
{
    if (iObject == this && iEvent != NULL && iEvent->type() == QEvent::KeyPress) {
        QKeyEvent* kevent = static_cast<QKeyEvent*>(iEvent);
        if (kevent) {
            if (kevent->key() == Qt::Key_Delete && state() != QAbstractItemView::EditingState) {
                QList<QTableWidgetItem*> listOfSelectedItems = this->selectedItems();
                int nb = listOfSelectedItems.count();
                if (nb > 0) {
                    // Build list of indexes
                    QList<int> listIndex;
                    for (int i = 0; i < nb; ++i) {
                        QModelIndex mIndex = this->indexFromItem(listOfSelectedItems[i]);
                        if (!listIndex.contains(mIndex.row())) {
                            listIndex.append(mIndex.row());
                        }
                    }

                    // Sort reverse
                    qSort(listIndex.begin(), listIndex.end(), qGreater<int>());

                    // Emit events
                    nb = listIndex.count();
                    for (int i = 0; i < nb; ++i) {
                        Q_EMIT removeLine(listIndex[i]);
                    }

                    if (iEvent) {
                        iEvent->accept();
                    }
                    return true;  // stop the process
                }
            } else if (kevent->matches(QKeySequence::Copy) && this->state() != QAbstractItemView::EditingState) {
                copy();
                if (iEvent) {
                    iEvent->accept();
                }
                return true;  // stop the process
            }
        }
    }

    return false;
}

void SKGTableWidget::copy()
{
    QItemSelectionModel* selection = selectionModel();
    if (selection) {
        QModelIndexList indexes = selection->selectedIndexes();

        if (indexes.size() < 1) {
            return;
        }

        qSort(indexes.begin(), indexes.end());

        // You need a pair of indexes to find the row changes
        QModelIndex previous = indexes.first();
        indexes.removeFirst();
        QString header_text;
        bool header_done = false;
        QString selected_text;
        foreach(const QModelIndex & current, indexes) {
            selected_text.append(model()->data(previous).toString());
            if (!header_done) {
                header_text.append(model()->headerData(previous.column(), Qt::Horizontal).toString());
            }
            if (current.row() != previous.row()) {
                selected_text.append(QLatin1Char('\n'));
                header_done = true;
            } else {
                selected_text.append(QLatin1Char(';'));
                if (!header_done) {
                    header_text.append(QLatin1Char(';'));
                }
            }
            previous = current;
        }

        // add last element
        selected_text.append(model()->data(previous).toString());
        selected_text.append(QLatin1Char('\n'));
        QApplication::clipboard()->setText(header_text + '\n' + selected_text);
    }
}


