/***************************************************************************
 *
 * knetworkmanager-openvpn.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <klocale.h>
#include <qmessagebox.h>
#include <qbutton.h>
#include <kcombobox.h>
#include <klineedit.h>
#include <kurlrequester.h>
#include <qobjectlist.h>
#include <qobject.h>
#include <qcheckbox.h>
#include <kpassdlg.h>
#include <kgenericfactory.h>
#include <qwidgetstack.h>
#include <qfileinfo.h>
#include <qhostaddress.h>

#include "knetworkmanager-openvpn.h"

typedef KGenericFactory<OpenVPNPlugin> OpenVPNPluginFactory;
K_EXPORT_COMPONENT_FACTORY( knetworkmanager_openvpn, OpenVPNPluginFactory("knetworkmanager_openvpn"));

/************************************
* OpenVPNPlugin
************************************/
OpenVPNPlugin::OpenVPNPlugin(QObject* parent, const char* name, const QStringList& args)
	: VPNPlugin(parent, name, args)
{

}

OpenVPNPlugin::~OpenVPNPlugin()
{

}

VPNConfigWidget* OpenVPNPlugin::CreateConfigWidget(QWidget* parent)
{
	return new OpenVPNConfig(parent);
}

VPNAuthenticationWidget* OpenVPNPlugin::CreateAuthenticationWidget(QWidget* parent)
{
	return new OpenVPNAuthentication(parent);
}

/************************************
* OpenVPNConnectionType
************************************/
OpenVPNConnectionType::CONNECTIONTYPE OpenVPNConnectionType::mapString2ConnectionType(QString prop)
{
	if (prop == "x509")
		return X509;
	else if (prop == "shared-key")
		return SHARED_KEY;
	else if (prop == "password")
		return PASSWORD;
	else if (prop == "x509userpass")
		return X509USERPASS;
	return UNKNOWN;
}

QString OpenVPNConnectionType::mapConnectionType2String(CONNECTIONTYPE connType)
{
	switch(connType)
	{
		case X509:
			return QString("x509");
		case SHARED_KEY:
			return QString("shared-key");
		case PASSWORD:
			return QString("password");
		case X509USERPASS:
			return QString("x509userpass");
		default:
			return QString::null;
	}
	return QString::null;
}

/************************************
* OpenVPNConfig
************************************/

OpenVPNConfig::OpenVPNConfig(QWidget* parent)
	: VPNConfigWidget(parent)
{
	QVBoxLayout* layout = new QVBoxLayout(this, 1, 1);
	_openvpnWidget = new OpenVPNConfigWidget(this);
	layout->addWidget(_openvpnWidget);

	connect(_openvpnWidget->chkUseCipher, SIGNAL(toggled(bool)), _openvpnWidget->cboCipher, SLOT(setEnabled(bool)));
	connect(_openvpnWidget->chkUseTLS, SIGNAL(toggled(bool)), _openvpnWidget->cboDirection, SLOT(setEnabled(bool)));
	connect(_openvpnWidget->chkUseTLS, SIGNAL(toggled(bool)), _openvpnWidget->editTLSAuth, SLOT(setEnabled(bool)));
	connect(_openvpnWidget->chkIPAdresses, SIGNAL(toggled(bool)), _openvpnWidget->routes, SLOT(setEnabled(bool)));
		
	// add all Cipher modes to the Combobox
	getCipherModes();

	// switch to the right configuration interface when selecting the connection type
	connect(_openvpnWidget->cboConnectionType, SIGNAL( activated(int)), _openvpnWidget->widgetStack, SLOT(raiseWidget(int)));
	this->languageChange();
}

OpenVPNConfig::~OpenVPNConfig()
{

}

void OpenVPNConfig::languageChange()
{
	_openvpnWidget->cboConnectionType->insertItem(i18n("X.509 Certificates"), OpenVPNConnectionType::X509 );
	_openvpnWidget->cboConnectionType->insertItem(i18n("Pre-shared key") , OpenVPNConnectionType::SHARED_KEY );
	_openvpnWidget->cboConnectionType->insertItem(i18n("Password Authentication") , OpenVPNConnectionType::PASSWORD );
	_openvpnWidget->cboConnectionType->insertItem(i18n("X.509 with Password Authentication") , OpenVPNConnectionType::X509USERPASS );

	_openvpnWidget->cboDirection->insertItem(i18n("none"));
	_openvpnWidget->cboDirection->insertItem(i18n("0"));
	_openvpnWidget->cboDirection->insertItem(i18n("1"));
}

QString OpenVPNConfig::findOpenVPNBinary()
{
	static const char *openvpn_binary_paths[] =
	{
	  "/usr/sbin/openvpn",
	  "/sbin/openvpn",
	  NULL
	};

	const char  **openvpn_binary = openvpn_binary_paths;

	while (*openvpn_binary != NULL) {
		if ( QFileInfo(*openvpn_binary).exists())
			break;
		openvpn_binary++;
	}
	
	return *openvpn_binary;
}


void OpenVPNConfig::receiveCipherData(KProcess*, char* buffer, int len)
{
	// add possible cipher modes to the combobox
	QStringList cipherModes = QStringList::split("\n", QString::fromLatin1(buffer, len), false );
	for (QStringList::ConstIterator it = cipherModes.begin(); it != cipherModes.end(); ++it)
	{
		_openvpnWidget->cboCipher->insertItem((*it));
	}
}

void OpenVPNConfig::getCipherModes()
{
	// get all possible cipher modes
	QString openvpn = findOpenVPNBinary();
	if (!openvpn.isNull())
	{
		KProcess* cipherHelper = new KProcess();
		cipherHelper->setUseShell(true, "/bin/sh");
		*cipherHelper << QString::fromLatin1("%1 --show-ciphers | /bin/awk '/^[A-Z][A-Z0-9]+-/ { print $1 }'").arg(openvpn); 
		connect (cipherHelper, SIGNAL(receivedStdout(KProcess*, char*, int)), this, SLOT(receiveCipherData(KProcess*, char*, int)));
		kdDebug() << "starting openvpn to get cipher modes" << endl;
		if (!cipherHelper->start(KProcess::Block, KProcess::Stdout))
			kdDebug() << "error starting openvpn" << endl;
	}

}

void OpenVPNConfig::setVPNData(const QStringList& routes, const QStringList& properties)
{

	// fill up our inputfields 
	for(QStringList::ConstIterator it = properties.begin(); it != properties.end(); ++it)
	{
		QString entry = (*it);
		if (entry == "connection-type")
		{
			OpenVPNConnectionType::CONNECTIONTYPE type = OpenVPNConnectionType::mapString2ConnectionType((*(++it)));
			_openvpnWidget->cboConnectionType->setCurrentItem(type);
			_openvpnWidget->widgetStack->raiseWidget(type);
		}
		else if (entry == "remote")
		{
			_openvpnWidget->gateway->setText(*(++it));
		}
		else if (entry == "port")
		{
			if (! (*(++it)).isEmpty())
			{
				_openvpnWidget->port->setText(*it);
				_openvpnWidget->chkDefaultPort->setChecked(false);
			}
			else
			{
				_openvpnWidget->chkDefaultPort->setChecked(true);
			}
		}
		else if (entry == "proto")
		{
			_openvpnWidget->chkUseTCP->setChecked( ((*(++it)) == "tcp") );
		}
		else if (entry == "ca")
		{
			_openvpnWidget->editCA->setURL(*(++it));
		}
		else if (entry == "cert")
		{
			_openvpnWidget->editCert->setURL(*(++it));
		}
		else if (entry == "key")
		{
			_openvpnWidget->editKey->setURL(*(++it));
		}
		else if (entry == "cipher")
		{
			_openvpnWidget->chkUseCipher->setChecked(true);
			_openvpnWidget->cboCipher->setCurrentItem(*(++it));
		}
		else if (entry == "comp-lzo")
		{
			if (*(++it) == "yes")
				_openvpnWidget->chkUseLZO->setChecked(true);
		}
		else if (entry == "shared-key")
		{
			_openvpnWidget->editSharedKey->setURL(*(++it));
		}
		else if (entry == "username")
		{
			_openvpnWidget->editUsername->setText(*(++it));
		}
		else if (entry == "local-ip")
		{
			_openvpnWidget->editLocalIP->setText(*(++it));
		}
		else if (entry == "remote-ip")
		{
			_openvpnWidget->editRemoteIP->setText(*(++it));
		}
		else if (entry == "dev")
			_openvpnWidget->chkUseTAP->setChecked( (*(++it))=="tap" );
		else if (entry == "ta")
		{
			_openvpnWidget->chkUseTLS->setChecked(true);
			_openvpnWidget->editTLSAuth->setURL(*(++it));
		}
		else if (entry == "ta-dir")
		{
			_openvpnWidget->cboDirection->setCurrentItem(*(++it));
		}
		else
		{
			kdDebug() << QString("OpenVPN: Property '%1' not handled").arg(entry) << endl;
		}
	}

	// set routes
	if (!routes.empty())
	{
		_openvpnWidget->chkIPAdresses->setChecked(true);
		_openvpnWidget->routes->setText(routes.join(" "));
	}
}

QStringList OpenVPNConfig::getVPNProperties()
{
	// build a StingList of properties
	QStringList strlist;

	strlist.append("connection-type");
	strlist.append(OpenVPNConnectionType::mapConnectionType2String((OpenVPNConnectionType::CONNECTIONTYPE)_openvpnWidget->cboConnectionType->currentItem()));	

	strlist.append("remote");
	strlist.append(_openvpnWidget->gateway->text());

	// port is not necessary
	if (!_openvpnWidget->port->text().isEmpty() && !_openvpnWidget->chkDefaultPort->isChecked())
	{
		strlist.append("port");
		strlist.append(_openvpnWidget->port->text());
	}

	strlist.append("ca");
	strlist.append(_openvpnWidget->editCA->url());

	strlist.append("cert");
	strlist.append(_openvpnWidget->editCert->url());

	strlist.append("key");
	strlist.append(_openvpnWidget->editKey->url());

	if (_openvpnWidget->chkUseCipher->isChecked())
	{
		strlist.append("cipher");
		strlist.append(_openvpnWidget->cboCipher->currentText());
	}

	if (_openvpnWidget->chkUseLZO->isChecked())
	{
		strlist.append("comp-lzo");
		strlist.append("yes");
	}

	strlist.append("shared-key");
	strlist.append(_openvpnWidget->editSharedKey->url());

	strlist.append("username");
	strlist.append(_openvpnWidget->editUsername->text());

	strlist.append("local-ip");
	strlist.append(_openvpnWidget->editLocalIP->text());

	strlist.append("remote-ip");
	strlist.append(_openvpnWidget->editRemoteIP->text());

	strlist.append("dev");
	if (_openvpnWidget->chkUseTAP->isChecked())	
		strlist.append("tap");
	else
		strlist.append("tun");

	strlist.append("proto");
	if (_openvpnWidget->chkUseTCP->isChecked())
		strlist.append("tcp");
	else
		strlist.append("udp");

	if (_openvpnWidget->chkUseTLS->isChecked())
	{
		strlist.append("ta");
		strlist.append(_openvpnWidget->editTLSAuth->url());
	}

	strlist.append("ta-dir");
	strlist.append(_openvpnWidget->cboDirection->currentText());

	return strlist;
}

QStringList OpenVPNConfig::getVPNRoutes()
{
	QStringList strlist;
	if(_openvpnWidget->chkIPAdresses->isChecked())
	{
		strlist = QStringList::split(" ", _openvpnWidget->routes->text());
	}
	return strlist;
}

bool OpenVPNConfig::hasChanged()
{
	return true;
}

bool OpenVPNConfig::isValid(QStringList& err_msg)
{
	bool retval = true;

	// check gateway
	if (_openvpnWidget->gateway->text().isEmpty())
	{
		err_msg.append(i18n("You have to specify a gateway"));
		retval = false;
	}

	bool ok = false;
	_openvpnWidget->port->text().toULong(&ok);
	if (!ok && !_openvpnWidget->port->text().isEmpty() )
	{
		err_msg.append(i18n("The port number has to be numeric"));
		retval = false;
	}

	switch(_openvpnWidget->cboConnectionType->currentItem())
	{
		case OpenVPNConnectionType::X509:
			// check if ca file is correct
			if (_openvpnWidget->editCA->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no CA file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editCA->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("CA file not valid"));
			}
			
			// check if cert file is correct
			if (_openvpnWidget->editCert->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no CERT file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editCert->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("CERT file not valid"));
			}
			
			// check if key file is correct
			if (_openvpnWidget->editKey->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no Key file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editKey->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("Key file not valid"));
			}

			break;


		case OpenVPNConnectionType::SHARED_KEY:
			// check if a shared key is selected
			if (_openvpnWidget->editSharedKey->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("Please provide a valid shared key"));
			}
			// check if the shared key file exists
			else if (!QFileInfo(_openvpnWidget->editSharedKey->url()).exists())
			{
				retval = false;
				err_msg.append(i18n("Please provide a valid shared key"));
			}
			
			// check if local ip is valid
			if (!QHostAddress().setAddress(_openvpnWidget->editLocalIP->text()))
			{
				retval = false;
				err_msg.append(i18n("local IP is invalid"));
			}
			// check if remote ip is valid
			if (!QHostAddress().setAddress(_openvpnWidget->editRemoteIP->text()))
			{
				retval = false;	
				err_msg.append(i18n("remote IP is invalid"));
			}

			break;
		case OpenVPNConnectionType::PASSWORD:
			// check if username is suplied
			if (_openvpnWidget->editUsername->text().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no username provided"));
			}

			// check if ca file is correct
			if (_openvpnWidget->editCA->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no CA file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editCA->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("CA file not valid"));
			}

			break;


		case OpenVPNConnectionType::X509USERPASS:
			// check if username is suplied
			if (_openvpnWidget->editUsername->text().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no username provided"));
			}

			// check if ca file is correct
			if (_openvpnWidget->editCA->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no CA file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editCA->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("CA file not valid"));
			}
			
			// check if cert file is correct
			if (_openvpnWidget->editCert->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no CERT file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editCert->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("CERT file not valid"));
			}
			
			// check if key file is correct
			if (_openvpnWidget->editKey->url().isEmpty())
			{
				retval = false;
				err_msg.append(i18n("no Key file provided"));
			}
			else if (!QFileInfo(_openvpnWidget->editKey->url()).isFile())
			{
				retval = false;
				err_msg.append(i18n("Key file not valid"));
			}
			break;
	}


	return retval;
}

/************************************
* OpenVPNAuthentication
************************************/

OpenVPNAuthentication::OpenVPNAuthentication(QWidget* parent, char* name)
	: VPNAuthenticationWidget(parent, name)
{
	QVBoxLayout* layout = new QVBoxLayout(this, 1, 1);
	_openvpnAuth = new OpenVPNAuthenticationWidget(this);
	layout->addWidget(_openvpnAuth);
}

OpenVPNAuthentication::~OpenVPNAuthentication()
{

}

void OpenVPNAuthentication::setVPNData(const QStringList& /*routes*/, const QStringList& properties)
{
	// find the connection type property
	for(QStringList::ConstIterator it = properties.begin(); it != properties.end(); ++it)
	{
		if ((*it) == "connection-type")
		{
			_connectionType = OpenVPNConnectionType::mapString2ConnectionType((*(++it)));
			break;
		}
	}
}

QStringList OpenVPNAuthentication::getPasswords()
{
	QStringList pwds;
	if (_connectionType == OpenVPNConnectionType::PASSWORD | _connectionType == OpenVPNConnectionType::X509USERPASS) 
		pwds.push_back(_openvpnAuth->editUserPassword->password());
	return pwds;
}

bool OpenVPNAuthentication::needsUserInteraction()
{
	if (_connectionType == OpenVPNConnectionType::PASSWORD | _connectionType == OpenVPNConnectionType::X509USERPASS) 
		return true;
	return false;
}

