/***************************************************************************
                          utils.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef UTILS_H
#define UTILS_H

#include <qdialog.h>
#include <qstring.h>
#include "kplnamespace.h"

class QLineEdit;
class QWidget;
class QString;
class QListBox;
class QFile;
class KURL;
class KplDoc;

/**
  * Choose function dialog class. Provides a dialog for the selection
  * of a function from list of functions in a shared library.
  * @author Werner Stille
  */
class ChooseFuncDlg : public QDialog {
  Q_OBJECT

public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param f pointer to QFile opened for input from a file containing
   * a list of functions obtained using nm.
   * @param func pointer to QLineEdit widget for the function name.
   */
  ChooseFuncDlg(QWidget* parent, QFile* f, QLineEdit* func);
  /** Destructor. */
  ~ChooseFuncDlg();

protected:
  QLineEdit* func;
  QListBox* funcList;

protected slots:
  /**
   * Accepts selection and leaves the dialog.
   * @param i index of selected function
   */
  void slotList(int i);
  /** Accepts selection and leaves the dialog. */
  void slotOK();
};

/**
  * Edit parameter dialog class. Provides a dialog for the editing
  * the parameters of a function.
  * @author Werner Stille
  */
class EditParameterDlg : public QDialog {
  Q_OBJECT

public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param m pointer to KplDoc model object.
   * @param p parameter array.
   * @param file absolute path of the function library.
   * @param func function name.
   */
  EditParameterDlg(QWidget* parent, KplDoc* m, double* p, const QString& file,
                   const QString& func);
  /** Destructor. */
  ~EditParameterDlg();

protected:
  double* p;
  QLineEdit* par[KPL_NPMAX];

protected slots:
  /** Accepts changes and leaves the dialog. */
  void slotOK();
};

/**
  * Utility class. Provides static utility functions in a namespace
  * given by the class.
  * @author Werner Stille
  */
class Utils {
public:
  /**
   * Retrieves URL for saving.
   * @param parent parent widget.
   * @param url KURL receiving the chosen file URL.
   * @param filter file name filter for file dialog.
   * @param m KplDoc model object.
   * @return true when file can be saved.
   */
  static bool getWriteURL(QWidget* parent, KURL& url, const QString& filter,
                          KplDoc* m);
  /**
   * Opens a file selection dialog.
   * @param e QLineEdit widget to display the filename.
   * @param m KplDoc model object.
   */
  static void getFile(QLineEdit* e, KplDoc* m);
  /**
   * Opens a function selection dialog.
   * @param parent parent widget.
   * @param name QLineEdit widget displaying the library filename.
   * @param func QLineEdit widget to display the function name.
   */
  static void getFunc(QWidget* parent, QLineEdit* name, QLineEdit* func);
  /**
   * Loads a function parameter file.
   * @param parent parent widget.
   * @param p double array for parameters.
   * @param m KplDoc model object.
   */
  static bool loadPar(const KURL& url, QWidget* parent, double* p, KplDoc* m);
  /**
   * Opens a dialog to select a function parameter file.
   * @param url file URL.
   * @param parent parent widget.
   * @param p double array for parameters.
   * @param m KplDoc model object.
   */
  static void getPar(QWidget* parent, double* p, KplDoc* m);
  /**
   * Saves parameters in a file.
   * @param parent parent widget.
   * @param url file URL.
   * @param p double array with parameters.
   * @param err double array with parameters (0 for no error output).
   * @param m KplDoc model object.
   * @return true for success.
   */
  static bool saveParameter(QWidget* parent, const KURL& url, double* p,
                            double* err, KplDoc* m);
  /**
   * Opens a dialog to save function parameters in a file.
   * @param parent parent widget.
   * @param p double array with parameters.
   * @param m KplDoc model object.
   */
  static void saveFunPar(QWidget* parent, double* p, KplDoc* m);
  /**
   * Gets module handle and address of a shared library function.
   * @param path path of the library.
   * @param parent parent widget.
   * @param name function name.
   * @param hmod pointer to module handle.
   * @param fkt pointer to function address.
   */
  static bool getFuncAddr(const QString& path, const QString& name,
                          void** hmod, double (**fkt)(double, const double*));
  /**
   * Calculates minimum and maximum values of an array.
   * @param xmin pointer to minimum value.
   * @param xmax pointer to maximum value.
   * @param x double array.
   * @param n size of array.
   * @return true for error.
   */
  static bool minMaxFile(double* xmin, double* xmax, const double* x, int n);
  /**
   * Calculates scaling parameters.
   * @param xmi pointer to double for rounded minimum value.
   * @param xma pointer to double for rounded maximum value.
   * @param tic pointer to double for major tic distance.
   * @param mtic pointer to int for intervals between major tics.
   * @param ndig pointer to int for number of places after decimal point
   * for lettering.
   * @param xmin minimum value.
   * @param xmax maximum value.
   * @param fx normalization factor.
   * @param c scaling constant, desired value for tic / (xma - xmi).
   * @param log true for logarithmic axis.
   * @return true when xmin and xmax are equal.
   */
  static bool autoSc(double* xmi, double* xma, double* tic, int* mtic,
                     int* ndig, double xmin, double xmax, double fx, double c,
                     bool log);

  /**
   * Calculates normalization factor.
   * @param a absolute double value.
   * @param ie pointer to int for logarithm of normalization factor.
   * @param f pointer to double for normalization factor.
   */
  static void expo(double a, int* ie, double* f);
  /**
   * Converts double to integer.
   * @param a double value.
   * @return integer result.
   */
  static int int9(double a);
  /**
   * Converts Qt1 QRgb to QRgb.
   * @param u Qt1 rgb value.
   * @return QRgb value for actual Qt version.
   */
  static unsigned rgbQt(unsigned u);
  /**
   * Converts QRgb to Qt1 QRgb.
   * @param rgb QRgb value of actual Qt version.
   * @return QRgb value for actual Qt1 version.
   */
  static unsigned rgbQt1(unsigned rgb);
  /**
   * Constructs relative file path (if possible).
   * @param url URL of plot file.
   * @param path file URL.
   * @param abs return absolute path if true.
   * @return relative path (if possible).
   */
  static QString relPath(const KURL& url, const KURL& path, bool abs);
};

#endif
