/***************************************************************************
                          fitdlg.cpp  -  description
                             -------------------
    begin                : Tue May 25 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <dlfcn.h>
#include <qgroupbox.h>
#include <qfileinfo.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qdir.h>
#include <qlayout.h>
#include <qtable.h>
#include <kapp.h>
#include <klocale.h>
#include <kurl.h>
#include <kcolorbtn.h>
#include <kseparator.h>
#include <kconfig.h>
#include <sourcewidget.h>
#include <kmessagebox.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <kglobal.h>
#include "fitdlg.h"
#include "lmfit.h"
#include "kpldoc.h"
#include "arrayitem.h"
#include "funitem.h"
#include "utils.h"
#include "kpldoubleedit.h"
#include "kplspinbox.h"

CorrDlg::CorrDlg(QWidget* parent, int np, const int* ip, const double* corr) :
  QDialog(parent, 0, true)
{
  setCaption(i18n("Parameter correlation coefficients"));
  QTable* table = new QTable(np, np, this);
  table->setMinimumSize(100, 40);
  QString s;
  for (int iy = 0; iy < np; iy++)
    table->verticalHeader()->setLabel(iy, s.sprintf("p%i", ip[iy]));
  for (int ix = 0; ix < np; ix++) {
    table->horizontalHeader()->setLabel(ix, s.sprintf("p%i", ip[ix]));
    for (int iy = 0; iy < np; iy++) {
      double temp = corr[np * iy + ix];
      if ((temp >= -1.0004) && (temp  <= 1.0004))
        s.setNum(temp, 'f', 3);
      else
        s = "NaN";
      table->setItem(iy, ix, new QTableItem(table, QTableItem::Never, s));
    }
    table->adjustColumn(ix);
  }
  table->adjustSize();
  (new QVBoxLayout(this, 0, 0))->addWidget(table);
}

CorrDlg::~CorrDlg()
{
}

ErrModDlg::ErrModDlg(QWidget* parent, KplDoc* model,
                     Kpl::DataErrorStruct* err0) : QDialog(parent, 0, true),
 m(model), err(err0)
{
  errt = new Kpl::DataErrorStruct(*err);
  setCaption(i18n("Error model Function"));
  QVBoxLayout* vbox = new QVBoxLayout(this, 11, 6);
  QGridLayout* grid = new QGridLayout(vbox, 3, 2);
  grid->addWidget(new QLabel(i18n("Library"), this), 0, 0);
  QHBoxLayout* hbox = new QHBoxLayout();
  grid->addLayout(hbox, 0, 1);
  hbox->addWidget(fileName = new QLineEdit(err->errModPath, this));
  QPushButton* b = new QPushButton("...", this);
  b->setFixedWidth(28);
  hbox->addWidget(b);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  grid->addWidget(new QLabel(i18n("Function"), this), 1, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 1, 1);
  hbox->addWidget(func = new QLineEdit(err->errModName, this));
  hbox->addWidget(b = new QPushButton("...", this));
  b->setFixedWidth(28);
  connect(b, SIGNAL(clicked()), SLOT(slotFunc()));
  grid->addWidget(new QLabel(i18n("Parameter"), this), 2, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 2, 1);
  hbox->addWidget(b = new QPushButton(i18n("Edit"), this));
  connect(b, SIGNAL(clicked()), SLOT(slotEditParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Load"), this));
  connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Save"), this));
  connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
  KSeparator* hl = new KSeparator(KSeparator::HLine, this);
  vbox->addWidget(hl);
  hbox = new QHBoxLayout(vbox);
  hbox->addWidget(b = new QPushButton(i18n("Help"), this));
  connect(b, SIGNAL(clicked()), SLOT(slotHelp()));
  hbox->addItem(new QSpacerItem(20, 10));
  hbox->addWidget(b = new QPushButton(i18n("OK"), this));
  b->setDefault(true);
  connect(b, SIGNAL(clicked()), SLOT(slotOK()));
  hbox->addWidget(b = new QPushButton(i18n("Cancel"), this));
  connect(b, SIGNAL(clicked()), SLOT(reject()));
}

ErrModDlg::~ErrModDlg()
{
  delete errt;
}

void ErrModDlg::slotFile()
{
  Utils::getFile(fileName, m);
}

void ErrModDlg::slotFunc()
{
  QFileInfo fi(fileName->text());
  QString path = fi.dirPath(true);
  QFile f(path + "/" + fi.baseName() + ".def");
  if (f.open(IO_ReadOnly)) {
    ChooseFuncDlg dlg(this, &f, func);
    dlg.exec();
  } else
    KMessageBox::error(this,
                       i18n("while trying to open module definition file"));
}

void ErrModDlg::slotGetParameter()
{
  Utils::getPar(this, errt->pErrMod, m);
}

void ErrModDlg::slotEditParameter()
{
  EditParameterDlg dlg(this, m, errt->pErrMod, fileName->text(), func->text());
  dlg.exec();
}

void ErrModDlg::slotSaveParameter()
{
  Utils::saveFunPar(this, errt->pErrMod, m);
}

void ErrModDlg::slotOK()
{
  errt->errModPath = fileName->text();
  errt->errModName = func->text();
  void *hErrMod = 0;
  double (*fkt)(double, const double *);
  if (Utils::getFuncAddr(errt->errModPath, errt->errModName, &hErrMod, &fkt))
    return;
  dlclose(hErrMod);
  *err = *errt;
  accept();
}

void ErrModDlg::slotHelp()
{
  kapp->invokeHelp("ERRORMODEL");
}

FitDlg::FitDlg(QWidget* parent, KplDoc* model, QList<ArrayItem>* ad0,
               QList<FunItem>* fd0, int mode) : QDialog(parent, 0, true),
 dlgMode(mode), m(model), ad(ad0), fd(fd0), lm(0), sw(0)
{
  aut = new Kpl::AutoStruct(m->aut);
  config = KGlobal::config();
  err = new Kpl::DataErrorStruct[ad->count()];
  int i;
  QString s;
  ArrayItem* a;
  for (i = 0; i < (int) ad->count(); i++) {
    err[i] = m->aut.err;
    s.sprintf("DataErrors%i", i);
    config->setGroup(s);
    err[i].errModPath = config->readEntry("ErrorModelPath",
                                          m->aut.err.errModPath);
    err[i].errModName = config->readEntry("ErrorModelName",
                                          m->aut.err.errModName);
    QStringList list = config->readListEntry("ErrorModelParam", ' ');
    int cnt = list.count();
    for (int j = 0; j < KPL_NPMAX; j++)
      err[i].pErrMod[j] = (j < cnt) ? list[j].toDouble() : 0.0;
    a = ad->at(i);
    if (a->ie < a->ncols)
      err[i].fitErrCol = config->readBoolEntry("FitErrorColumn",
                                               m->aut.err.fitErrCol);
    else
      err[i].fitErrCol = false;
  }
  FunItem* ff = fd->first();
  for (i = 0; i < KPL_NPMAX; i++) {
    m->pLast[i] = ff->py[i];
    m->pErr[i] = 0.0;
  }
  if (dlgMode & ShowDlg) {
    config->setGroup("FitDialog");
    resize(config->readNumEntry("Width", 530),
           config->readNumEntry("Height", 560));
    setCaption(i18n("Parameter fit"));
    QHBoxLayout* hbox = 0;
    QVBoxLayout* vbox;
    if (m->aut.showSource) {
      hbox = new QHBoxLayout(this, 11, 6);
      vbox = new QVBoxLayout(hbox, 6);
    } else
      vbox = new QVBoxLayout(this, 11, 6);
    QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Parameter"), this);
    vbox->addWidget(g);
    QVBoxLayout* vbox2 = new QVBoxLayout(g->layout(), 6);
    int nGr = 1 + (KPL_NPMAX - 1) / 10;
    QGridLayout* grid = new QGridLayout(vbox2, QMIN(KPL_NPMAX, 10),
                                        5 * nGr - 1);
    for (i = 1; i < nGr; i++)
      grid->addItem(new QSpacerItem(20, 10), 0, 4 * i);
    for (i = 0; i < KPL_NPMAX; i++) {
      int ix = 5 * (i / 10);
      int iy = i % 10;
      grid->addWidget(enFit[i] = new QCheckBox(s.sprintf("p%i", i), g, "f"),
                      iy, ix);
      enFit[i]->setChecked(aut->fitPar[i]);
      grid->addWidget(par[i] = new QLineEdit(QString::number(m->pLast[i],
        m->aut.format, m->aut.prec), g), iy, ix + 1);
      par[i]->setMaximumWidth(80);
      grid->addWidget(new QLabel("\261", g), iy, ix + 2);
      grid->addWidget(epar[i] = new QLabel("0", g), iy, ix + 3);
      epar[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
      epar[i]->setMaximumWidth(100);
      epar[i]->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding,
                                         QSizePolicy::Fixed));
    }
    QHBoxLayout* hbox2 = new QHBoxLayout(vbox2, 6);
    hbox2->addWidget(nonLin = new QCheckBox(i18n("Nonlinear fit"), g));
    nonLin->setChecked(aut->fitNonLin);
    widgetList.append(nonLin);
    QPushButton* b = new QPushButton(i18n("Load"), g);
    hbox2->addWidget(b);
    connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
    widgetList.append(b);
    hbox2->addWidget(b = new QPushButton(i18n("Save"), g));
    connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
    widgetList.append(b);
    hbox2->addWidget(showCorr = new QPushButton(i18n("Correlations"), g));
    showCorr->setEnabled(false);
    connect(showCorr, SIGNAL(clicked()), SLOT(slotCorr()));
    g = new QGroupBox(0, Qt::Horizontal, i18n("Data errors"), this);
    hbox2 = new QHBoxLayout(g->layout(), 6);
    hbox2->addWidget(new QLabel(i18n("Array"), g));
    a = ad->at(0);
    QHBoxLayout* hbox3 = new QHBoxLayout(hbox2, 6);
    hbox3->addWidget(sArr = new KplSpinBox(0, ad->count() - 1, 1, g));
    connect(sArr, SIGNAL(valueChanged(int)), SLOT(updArray(int)));
    sArr->setValue(0);
    hbox3->addWidget(lArr = new QLabel(g));
    lArr->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox2->addWidget(errCol = new QCheckBox(i18n("Error column"), g));
    connect(errCol, SIGNAL(toggled(bool)), SLOT(errColToggled(bool)));
    hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
    hbox2->addWidget(errMod = new QPushButton(i18n("Model"), g));
    connect(errMod, SIGNAL(clicked()), SLOT(slotErrMod()));
    vbox->addWidget(g);
    hbox2 = new QHBoxLayout(vbox, 6);
    hbox2->addWidget(new QLabel(i18n("Maximum iterations"), this));
    hbox2->addWidget(sItmax = new KplSpinBox(1, 200, 1, this));
    sItmax->setValue(aut->fitMaxIt);
    widgetList.append(sItmax);
    hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
    hbox2->addWidget(new QLabel(i18n("Tolerance"), this));
    hbox2->addWidget(eTol = new KplDoubleEdit(aut->fitTol, 0.0, 1.0, this,
                                              "%.6lg"));
    connect(nonLin, SIGNAL(toggled(bool)), SLOT(enableNonLin(bool)));
    enableNonLin(aut->fitNonLin);
    widgetList.append(eTol);
    vbox->addWidget(results = new QListBox(this));
    results->setMinimumHeight(40);
    hbox2 = new QHBoxLayout(vbox, 6);
    hbox2->addWidget(b = new QPushButton(i18n("Help"), this));
    connect(b, SIGNAL(clicked()), SLOT(slotHelp()));
    hbox2->addWidget(b = new QPushButton(i18n("Start"), this));
    connect(b, SIGNAL(clicked()), SLOT(slotFit()));
    hbox2->addWidget(b = new QPushButton(i18n("OK"), this));
    b->setDefault(true);
    connect(b, SIGNAL(clicked()), SLOT(slotOK()));
    widgetList.append(b);
    hbox2->addWidget(b = new QPushButton(i18n("Apply"), this));
    connect(b, SIGNAL(clicked()), SLOT(slotApply()));
    widgetList.append(b);
    hbox2->addWidget(b = new QPushButton(i18n("Cancel"), this));
    connect(b, SIGNAL(clicked()), SLOT(slotCancel()));
    if (m->aut.showSource) {
      hbox->addWidget(sw = new SourceWidget(this, 0), 1);
      sw->setMinimumWidth(200);
    }
    updArray(0);
    if (m->aut.showSource)
      QTimer::singleShot(0, sw, SLOT(slotHighlight()));
  }
}

FitDlg::~FitDlg()
{
  if (dlgMode & ShowDlg) {
    config->setGroup("FitDialog");
    config->writeEntry("Width", width());
    config->writeEntry("Height", height());
  }
  delete [] err;
  delete aut;
}

void FitDlg::getValues(bool ok)
{
  int i, j;
  if (dlgMode & ShowDlg) {
    aut->fitNonLin = nonLin->isChecked();
    for (i = 0; i < KPL_NPMAX; i++)
      aut->fitPar[i] = enFit[i]->isChecked();
    if (aut->fitNonLin) {
      aut->fitMaxIt = sItmax->interpretedValue();
      aut->fitTol = eTol->value();
    }
    m->aut = *aut;
    QString s;
    for (j = 0; j < (int) fd->count(); j++) {
      s.sprintf("DataErrors%i", j);
      config->setGroup(s);
      config->writeEntry("FitErrorColumn", err[j].fitErrCol);
      config->writeEntry("ErrorModelPath", err[j].errModPath);
      config->writeEntry("ErrorModelName", err[j].errModName);
      QStrList s1;
      for (i = 0; i < KPL_NPMAX; i++)
        s1.insert(i, QString::number(err[j].pErrMod[i], aut->format,
                                     aut->prec));
      config->writeEntry("ErrorModelParam", s1, ' ');
    }
  }
  for (j = 0; j < (int) fd->count(); j++)
    for (i = 0; i < KPL_NPMAX; i++)
      fd->at(j)->py[i] = m->pLast[i];
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
}

void FitDlg::slotMessage(const QString& msg)
{
  results->insertItem(msg);
  results->setTopItem(results->count() - 1);
  kapp->processEvents();
}

void FitDlg::slotGetParameter()
{
  Utils::getPar(this, m->pLast, m);
  for (int i = 0; i < KPL_NPMAX; i++) {
    par[i]->setText(QString::number(m->pLast[i], m->aut.format, m->aut.prec));
    m->pErr[i] = 0.0;
    epar[i]->setText("0");
  }
}

void FitDlg::slotSaveParameter()
{
  KURL url;
  if (Utils::getWriteURL(this, url, "*.par\n*", m))
    Utils::saveParameter(this, url, m->pLast, m->pErr, m);
}

void FitDlg::errColToggled(bool state)
{
  errMod->setEnabled(!state);
  err[sArr->interpretedValue()].fitErrCol = state;
}

void FitDlg::enableNonLin(bool on)
{
  eTol->setEnabled(on);
  sItmax->setEnabled(on);
}

void FitDlg::updArray(int i)
{
  ArrayItem* a = ad->at(i);
  lArr->setText(a->url.fileName() + " " + QString::number(a->ix) + " " +
                QString::number(a->iy) + " " + QString::number(a->ie));
  errCol->setChecked(err[i].fitErrCol);
  errCol->setEnabled(a->ie < a->ncols);
  if (m->aut.showSource) {
    FunItem* ff = fd->at(i);
    QFileInfo fi(ff->pathy.path());
    int pp = ff->namey.find("__");
    QString s = fi.dirPath(true) + "/" + fi.baseName() +
      ((pp != -1) ? ".cpp" : ".c");
    QFile f(s);
    if (f.open(IO_ReadOnly)) {
      sw->fill(&f);
      if (pp != -1) {
        s = ff->namey;
        sw->highlight(s.left(pp));
      } else
        sw->highlight(ff->namey);
    }
  }
}

void FitDlg::slotOK()
{
  getValues(true);
}

void FitDlg::slotCancel()
{
  if (lm)
    lm->userBreak = true;
  else
    reject();
}

void FitDlg::slotApply()
{
  getValues(false);
}

void FitDlg::slotHelp()
{
  kapp->invokeHelp("SEC-FIT");
}

void FitDlg::slotFit()
{
  for (FunItem *f = fd->first(); f; f = fd->next())
    if (!f->fkty) {
      KMessageBox::error(this, i18n("no function address!"));
      return;
    }
  bool nLin;
  if (dlgMode & ShowDlg) {
    nLin = nonLin->isChecked();
    if (nLin) {
      aut->fitTol = eTol->value();
      aut->fitMaxIt = sItmax->interpretedValue();
    }
    showCorr->setEnabled(false);
    results->clear();
  } else
    nLin = aut->fitNonLin;
  np = 0;
  int i;
  bool valid;
  QString s;
  for (i = 0; i < KPL_NPMAX; i++) {
    if (dlgMode & ShowDlg) {
      m->pLast[i] = par[i]->text().toDouble(&valid);
      if (!valid) {
        KMessageBox::sorry(this, s.sprintf(i18n("invalid parameter %i"), i));
        return;
      }
      aut->fitPar[i] = enFit[i]->isChecked();
    }
    if (aut->fitPar[i])
      np++;
  }
  if (!np) {
    KMessageBox::sorry(this, i18n("No parameter to fit!"));
    return;
  }
  int n = 0;
  ArrayItem* a;
  for (a = ad->first(); a; a = ad->next())
    n += a->n;
  lm = new LMFit(this);
  lm->sig = new double[n];
  int j = 0;
  int k;
  for (k = 0; k < (int) ad->count(); k++) {
    a = ad->at(k);
    if (err[k].fitErrCol) {
      for (i = 0; i < a->n; i++)
        lm->sig[j++] = a->x[a->ie][a->istart + i];
    } else {
      double (*fktErrMod)(double, const double*);
      void* hErrMod;
      if (Utils::getFuncAddr(err[k].errModPath, err[k].errModName, &hErrMod,
                             &fktErrMod)) {
        delete [] lm->sig;
        delete lm;
        lm = 0;
        return;
      }
      for (i = 0; i < a->n; i++)
        lm->sig[j++] = fktErrMod(a->x[a->iy][a->istart + i], err[k].pErrMod);
      dlclose(hErrMod);
    }
  }
  lm->ad = ad;
  lm->fd = fd;
  lm->bFit = aut->fitPar;
  lm->p = m->pLast;
  double* wa = (double *) calloc((6 + np + n) * np + 2 * n, sizeof(double));
  double* fvec = wa + 3 * np + n;
  double* guess = fvec + n;
  double* diag = guess + np;
  double* fjac = diag + np;
  double* qtf = fjac + n * np;
  double* alpha = qtf + np;
  j = 0;
  for (i = 0; i < KPL_NPMAX; i++)
    if (aut->fitPar[i])
      guess[j++] = m->pLast[i];
  if (dlgMode & ShowDlg) {
    for (QWidget* w = widgetList.first(); w; w = widgetList.next())
      w->setEnabled(false);
    for (i = 0; i < KPL_NPMAX; i++) {
      par[i]->setEnabled(false);
      enFit[i]->setEnabled(false);
    }
    connect(lm, SIGNAL(updateMessage(const QString &)),
            SLOT(slotMessage(const QString &)));
  }
  kapp->processEvents();
  bool aborted = false;
  if (nLin) {
    int info = 0;
    int nfev;
    lm->lmdif(n, np, guess, fvec, aut->fitTol, aut->fitTol, 0.0,
              (np + 1) * aut->fitMaxIt, 0.0, diag, 1, 100.0, 1, &info, &nfev,
              fjac, n, ip, qtf, wa, wa + np, wa + 2 * np, wa + 3 * np);
    if (info >= 8)
      info = 4;
    if (dlgMode & ShowDlg) {
      switch (info) {
        case 0:
          s = i18n("Improper input parameters");
          break;
        case 1:
          s = i18n("Relative error in chi-square is at most tolerance");
          break;
        case 2:
          s = i18n("Relative error in parameters is at most tolerance");
          break;
        case 3:
          s = i18n("Relative error in chi-square and in parameters is at most "
                   "tolerance");
          break;
        case 4:
          s = i18n("Function vector is orthogonal to the columns of "
                   "the jacobian");
          break;
        case 5:
          s = i18n("Number of iterations has reached or exceeded maximum");
          break;
        case 6:
          s = i18n("Tolerance too small. No further reduction of "
                   "chi-square possible");
          break;
        case 7:
          s = i18n("Tolerance too small. No further improvement of "
                   "parameters possible");
          break;
        default:
          s = i18n("Terminated by user");
      }
      slotMessage(s);
    }
  } else {
    aborted = lm->linFit(n, np, guess, fvec, fjac, ip, qtf, wa, wa + np,
                         wa + 2 * np, wa + 3 * np);
  }
  if (!aborted) {
    j = 0;
    for (i = 0; i < KPL_NPMAX; i++)
      if (aut->fitPar[i])
        m->pLast[i] = guess[j++];
    m->chisq = enorm(n, fvec);
    m->chisq *= m->chisq;
    for (i = 0; i < np; i++) {
      for (j = 0; j < np; j++) {
        alpha[ip[i] + ip[j] * np] = 0.0;
        for (k = 0; k <= j; k++)
          if (k <= i)
            alpha[ip[i] + ip[j] * np] += fjac[i * n + k] * fjac[j * n + k];
      }
    }
    minv(alpha, corr, np, wa, ip);
    int ij = 0;
    for (i = 0; i < np; i++) {
      diag[i] = sqrt(corr[ij]);
      ij += np + 1;
    }
    k = 0;
    for (i = 0; i < np; i++)
      for (j = 0; j < np; j++) {
        if (double temp = diag[i] * diag[j])
          corr[k] /= temp;
        k++;
      }
    int ny = n - np;
    double q = 0.0;
    if (ny) {
      q = igamc(0.5 * ny, 0.5 * m->chisq);
      double temp = sqrt(m->chisq / ny);
      for (i = 0; i < np; i++)
        diag[i] *= temp;
    }
    k = 0;
    for (i = 0; i < KPL_NPMAX; i++) {
      if (aut->fitPar[i]) {
        ip[k] = i;
        m->pErr[i] = diag[k++];
      } else
        m->pErr[i] = 0.0;
      if (dlgMode & ShowDlg) {
        par[i]->setText(QString::number(m->pLast[i], m->aut.format,
                        m->aut.prec));
        epar[i]->setText(QString::number(m->pErr[i], m->aut.format,
                         m->aut.prec));
      }
      if (dlgMode & Follow)
        aut->pFit[i] = m->pLast[i];
    }
    double avgErr = 0.0;
    for (i = 0; i < n; i++) {
      double temp = fvec[i] * lm->sig[i];
      avgErr += temp * temp;
    }
    avgErr = sqrt(avgErr / n);
    if (dlgMode & ShowDlg)
      slotMessage(s.sprintf(i18n("Average error = %.3g    ny = %i    "
                                 "Significance Q = %.3g"),
                            avgErr, ny, q));
    if (dlgMode & SavePar) {
      KURL url = m->URL();
      QFileInfo fi(url.path());
      url.setPath(fi.dirPath(true) + "/" + fi.baseName() + ".par");
      Utils::saveParameter(this, url, m->pLast, m->pErr, m);
    }
  }
  if (dlgMode & ShowDlg) {
    showCorr->setEnabled(true);
    for (QWidget* w = widgetList.first(); w; w = widgetList.next())
      w->setEnabled(true);
    enableNonLin(nonLin->isChecked());
    for (i = 0; i < KPL_NPMAX; i++) {
      par[i]->setEnabled(true);
      enFit[i]->setEnabled(true);
    }
  }
  delete [] lm->sig;
  free(wa);
  delete lm;
  lm = 0;
}

void FitDlg::slotErrMod()
{
  ErrModDlg dlg(this, m, &err[sArr->interpretedValue()]);
  dlg.exec();
}

void FitDlg::slotCorr()
{
  CorrDlg dlg(this, np, ip, corr);
  dlg.exec();
}
