/* dlltool.c -- tool to generate stuff for PE style DLLs
   Copyright (C) 1995-2026 Free Software Foundation, Inc.

   This file is part of GNU Binutils.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street - Fifth Floor, Boston, MA
   02110-1301, USA.  */


/* This program allows you to build the files necessary to create
   DLLs to run on a system which understands PE format image files.
   (eg, Windows NT)

   See "Peering Inside the PE: A Tour of the Win32 Portable Executable
   File Format", MSJ 1994, Volume 9 for more information.
   Also see "Microsoft Portable Executable and Common Object File Format,
   Specification 4.1" for more information.

   A DLL contains an export table which contains the information
   which the runtime loader needs to tie up references from a
   referencing program.

   The export table is generated by this program by reading
   in a .DEF file or scanning the .a and .o files which will be in the
   DLL.  A .o file can contain information in special  ".drectve" sections
   with export information.

   A DEF file contains any number of the following commands:


   NAME <name> [ , <base> ]
   The result is going to be <name>.EXE

   LIBRARY <name> [ , <base> ]
   The result is going to be <name>.DLL

   EXPORTS  ( (  ( <name1> [ = <name2> ] )
               | ( <name1> = <module-name> . <external-name>))
            [ @ <integer> ] [ NONAME ] [CONSTANT] [DATA] [PRIVATE] ) *
   Declares name1 as an exported symbol from the
   DLL, with optional ordinal number <integer>.
   Or declares name1 as an alias (forward) of the function <external-name>
   in the DLL <module-name>.

   IMPORTS  (  (   <internal-name> =   <module-name> . <integer> )
             | ( [ <internal-name> = ] <module-name> . <external-name> )) *
   Declares that <external-name> or the exported function whose ordinal number
   is <integer> is to be imported from the file <module-name>.  If
   <internal-name> is specified then this is the name that the imported
   function will be refereed to in the body of the DLL.

   DESCRIPTION <string>
   Puts <string> into output .exp file in the .rdata section

   [STACKSIZE|HEAPSIZE] <number-reserve> [ , <number-commit> ]
   Generates --stack|--heap <number-reserve>,<number-commit>
   in the output .drectve section.  The linker will
   see this and act upon it.

   [CODE|DATA] <attr>+
   SECTIONS ( <sectionname> <attr>+ )*
   <attr> = READ | WRITE | EXECUTE | SHARED
   Generates --attr <sectionname> <attr> in the output
   .drectve section.  The linker will see this and act
   upon it.


   A -export:<name> in a .drectve section in an input .o or .a
   file to this program is equivalent to a EXPORTS <name>
   in a .DEF file.



   The program generates output files with the prefix supplied
   on the command line, or in the def file, or taken from the first
   supplied argument.

   The .exp.s file contains the information necessary to export
   the routines in the DLL.  The .lib.s file contains the information
   necessary to use the DLL's routines from a referencing program.



   Example:

 file1.c:
   asm (".section .drectve");
   asm (".ascii \"-export:adef\"");

   void adef (char * s)
   {
     printf ("hello from the dll %s\n", s);
   }

   void bdef (char * s)
   {
     printf ("hello from the dll and the other entry point %s\n", s);
   }

 file2.c:
   asm (".section .drectve");
   asm (".ascii \"-export:cdef\"");
   asm (".ascii \"-export:ddef\"");

   void cdef (char * s)
   {
     printf ("hello from the dll %s\n", s);
   }

   void ddef (char * s)
   {
     printf ("hello from the dll and the other entry point %s\n", s);
   }

   int printf (void)
   {
     return 9;
   }

 themain.c:
   int main (void)
   {
     cdef ();
     return 0;
   }

 thedll.def

   LIBRARY thedll
   HEAPSIZE 0x40000, 0x2000
   EXPORTS bdef @ 20
           cdef @ 30 NONAME

   SECTIONS donkey READ WRITE
   aardvark EXECUTE

 # Compile up the parts of the dll and the program

   gcc -c file1.c file2.c themain.c

 # Optional: put the dll objects into a library
 # (you don't have to, you could name all the object
 # files on the dlltool line)

   ar  qcv thedll.in file1.o file2.o
   ranlib thedll.in

 # Run this tool over the DLL's .def file and generate an exports
 # file (thedll.o) and an imports file (thedll.a).
 # (You may have to use -S to tell dlltool where to find the assembler).

   dlltool --def thedll.def --output-exp thedll.o --output-lib thedll.a

 # Build the dll with the library and the export table

   ld -o thedll.dll thedll.o thedll.in

 # Link the executable with the import library

   gcc -o themain.exe themain.o thedll.a

 This example can be extended if relocations are needed in the DLL:

 # Compile up the parts of the dll and the program

   gcc -c file1.c file2.c themain.c

 # Run this tool over the DLL's .def file and generate an imports file.

   dlltool --def thedll.def --output-lib thedll.lib

 # Link the executable with the import library and generate a base file
 # at the same time

   gcc -o themain.exe themain.o thedll.lib -Wl,--base-file -Wl,themain.base

 # Run this tool over the DLL's .def file and generate an exports file
 # which includes the relocations from the base file.

   dlltool --def thedll.def --base-file themain.base --output-exp thedll.exp

 # Build the dll with file1.o, file2.o and the export table

   ld -o thedll.dll thedll.exp file1.o file2.o  */

/* .idata section description

   The .idata section is the import table.  It is a collection of several
   subsections used to keep the pieces for each dll together: .idata$[234567].
   IE: Each dll's .idata$2's are catenated together, each .idata$3's, etc.

   .idata$2 = Import Directory Table
   = array of IMAGE_IMPORT_DESCRIPTOR's.

	DWORD   Import Lookup Table;  - pointer to .idata$4
	DWORD   TimeDateStamp;        - currently always 0
	DWORD   ForwarderChain;       - currently always 0
	DWORD   Name;                 - pointer to dll's name
	PIMAGE_THUNK_DATA FirstThunk; - pointer to .idata$5

   .idata$3 = null terminating entry for .idata$2.

   .idata$4 = Import Lookup Table
   = array of array of numbers, which has meaning based on its highest bit:
     - when cleared - pointer to entry in Hint Name Table
     - when set - 16-bit function's ordinal number (rest of the bits are zeros)
   Function ordinal number subtracted by Export Directory Table's
   Ordinal Base is an index entry into the Export Address Table.
   There is one for each dll being imported from, and each dll's set is
   terminated by a trailing NULL.

   .idata$5 = Import Address Table
   There is one for each dll being imported from, and each dll's set is
   terminated by a trailing NULL.
   Initially, this table is identical to the Import Lookup Table.  However,
   at load time, the loader overwrites the entries with the address of the
   function.

   .idata$6 = Hint Name Table
   = Array of { short, asciz } entries, one for each imported function.
   The `short' is the name hint - index into Export Name Pointer Table.
   The `asciz` is the name string - value in Export Name Table referenced
   by some entry in Export Name Pointer Table.  Name hint should be the
   index of that entry in Export Name Pointer Table.  It has no connection
   with the function's ordinal number.

   .idata$7 = dll name (eg: "kernel32.dll").  */

#include "sysdep.h"
#include "bfd.h"
#include "libiberty.h"
#include "getopt.h"
#include "demangle.h"
#include "dyn-string.h"
#include "bucomm.h"
#include "dlltool.h"
#include "safe-ctype.h"
#include "coff-bfd.h"

#include <time.h>
#include <assert.h>

#ifdef DLLTOOL_ARM
#include "coff/arm.h"
#include "coff/internal.h"
#endif
#ifdef DLLTOOL_DEFAULT_MX86_64
#include "coff/x86_64.h"
#endif
#ifdef DLLTOOL_DEFAULT_I386
#include "coff/i386.h"
#endif

#ifndef COFF_PAGE_SIZE
#define COFF_PAGE_SIZE ((bfd_vma) 4096)
#endif

#ifndef PAGE_MASK
#define PAGE_MASK ((bfd_vma) (- COFF_PAGE_SIZE))
#endif

#ifndef NAME_MAX
#define NAME_MAX	255
#endif

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#else /* ! HAVE_SYS_WAIT_H */
#if ! defined (_WIN32) || defined (__CYGWIN32__)
#ifndef WIFEXITED
#define WIFEXITED(w)	(((w) & 0377) == 0)
#endif
#ifndef WIFSIGNALED
#define WIFSIGNALED(w)	(((w) & 0377) != 0177 && ((w) & ~0377) == 0)
#endif
#ifndef WTERMSIG
#define WTERMSIG(w)	((w) & 0177)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(w)	(((w) >> 8) & 0377)
#endif
#else /* defined (_WIN32) && ! defined (__CYGWIN32__) */
#ifndef WIFEXITED
#define WIFEXITED(w)	(((w) & 0xff) == 0)
#endif
#ifndef WIFSIGNALED
#define WIFSIGNALED(w)	(((w) & 0xff) != 0 && ((w) & 0xff) != 0x7f)
#endif
#ifndef WTERMSIG
#define WTERMSIG(w)	((w) & 0x7f)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(w)	(((w) & 0xff00) >> 8)
#endif
#endif /* defined (_WIN32) && ! defined (__CYGWIN32__) */
#endif /* ! HAVE_SYS_WAIT_H */

#define show_allnames 0

/* ifunc and ihead data structures: ttk@cygnus.com 1997

   When IMPORT declarations are encountered in a .def file the
   function import information is stored in a structure referenced by
   the global variable IMPORT_LIST.  The structure is a linked list
   containing the names of the dll files each function is imported
   from and a linked list of functions being imported from that dll
   file.  This roughly parallels the structure of the .idata section
   in the PE object file.

   The contents of .def file are interpreted from within the
   process_def_file function.  Every time an IMPORT declaration is
   encountered, it is broken up into its component parts and passed to
   def_import.  IMPORT_LIST is initialized to NULL in function main.  */

typedef struct ifunct
{
  const char *	 name;	/* Name of function being imported.  */
  const char *its_name;	/* Optional import table symbol name.  */
  int		 ord;	/* Two-byte ordinal value associated with function.  */
  struct ifunct *next;
} ifunctype;

typedef struct iheadt
{
  const char *	 dllname;  /* Name of dll file imported from.  */
  long		 nfuncs;   /* Number of functions in list.  */
  struct ifunct *funchead; /* First function in list.  */
  struct ifunct *functail; /* Last  function in list.  */
  struct iheadt *next;     /* Next dll file in list.  */
} iheadtype;

/* Structure containing all import information as defined in .def file
   (qv "ihead structure").  */

static iheadtype *import_list = NULL;
static char *as_name = NULL;
static char * as_flags = "";
static char *tmp_prefix = NULL;
static int no_idata4;
static int no_idata5;
static char *exp_name;
static char *imp_name;
static char *delayimp_name;
static char *identify_imp_name;
static bool identify_strict;
static bool deterministic = DEFAULT_AR_DETERMINISTIC;

/* Types used to implement a linked list of dllnames associated
   with the specified import lib. Used by the identify_* code.
   The head entry is acts as a sentinal node and is always empty
   (head->dllname is NULL).  */
typedef struct dll_name_list_node_t
{
  char *			dllname;
  struct dll_name_list_node_t * next;
} dll_name_list_node_type;

typedef struct dll_name_list_t
{
  dll_name_list_node_type * head;
  dll_name_list_node_type * tail;
} dll_name_list_type;

/* Types used to pass data to iterator functions.  */
typedef struct symname_search_data_t
{
  const char *symname;
  bool found;
} symname_search_data_type;

typedef struct identify_data_t
{
   dll_name_list_type *list;
   bool ms_style_implib;
} identify_data_type;


static char *head_label;
static char *imp_name_lab;
static char *dll_name;
static int dll_name_set_by_exp_name;
static int add_indirect = 0;
static int add_underscore = 0;
static int add_stdcall_underscore = 0;
static char *leading_underscore = NULL;
static int dontdeltemps = 0;

/* TRUE if we should export all symbols.  Otherwise, we only export
   symbols listed in .drectve sections or in the def file.  */
static bool export_all_symbols;

/* TRUE if we should exclude the symbols in DEFAULT_EXCLUDES when
   exporting all symbols.  */
static bool do_default_excludes = true;

static bool use_nul_prefixed_import_tables = false;

/* Default symbols to exclude when exporting all the symbols.  */
static const char *default_excludes = "DllMain@12,DllEntryPoint@0,impure_ptr";

/* TRUE if we should add __imp_<SYMBOL> to import libraries for backward
   compatibility to old Cygwin releases.  */
static bool create_compat_implib;

/* TRUE if we have to write PE+ import libraries.  */
static bool create_for_pep;

static char *def_file;

extern char * program_name;

static int machine;
static int killat;
static int add_stdcall_alias;
static const char *ext_prefix_alias;
static int verbose;
static FILE *output_def;
static FILE *base_file;

#ifdef DLLTOOL_DEFAULT_ARM
static const char *mname = "arm";
#endif

#ifdef DLLTOOL_DEFAULT_ARM_WINCE
static const char *mname = "arm-wince";
#endif

#ifdef DLLTOOL_DEFAULT_AARCH64
/* arm64 rather than aarch64 to match llvm-dlltool */
static const char *mname = "arm64";
#endif

#ifdef DLLTOOL_DEFAULT_I386
static const char *mname = "i386";
#endif

#ifdef DLLTOOL_DEFAULT_MX86_64
static const char *mname = "i386:x86-64";
#endif

#ifdef DLLTOOL_DEFAULT_SH
static const char *mname = "sh";
#endif

#ifdef DLLTOOL_DEFAULT_MIPS
static const char *mname = "mips";
#endif

#ifdef DLLTOOL_DEFAULT_MCORE
static const char * mname = "mcore-le";
#endif

#ifdef DLLTOOL_DEFAULT_MCORE_ELF
static const char * mname = "mcore-elf";
static char * mcore_elf_out_file = NULL;
static char * mcore_elf_linker   = NULL;
static char * mcore_elf_linker_flags = NULL;

#define DRECTVE_SECTION_NAME ((machine == MMCORE_ELF || machine == MMCORE_ELF_LE) ? ".exports" : ".drectve")
#endif

#ifndef DRECTVE_SECTION_NAME
#define DRECTVE_SECTION_NAME ".drectve"
#endif

/* External name alias numbering starts here.  */
#define PREFIX_ALIAS_BASE	20000

static char *tmp_asm_buf;
static char *tmp_head_s_buf;
static char *tmp_head_o_buf;
static char *tmp_tail_s_buf;
static char *tmp_tail_o_buf;
static char *tmp_stub_buf;

#define TMP_ASM		dlltmp (&tmp_asm_buf, "%sc.s")
#define TMP_HEAD_S	dlltmp (&tmp_head_s_buf, "%sh.s")
#define TMP_HEAD_O	dlltmp (&tmp_head_o_buf, "%sh.o")
#define TMP_TAIL_S	dlltmp (&tmp_tail_s_buf, "%st.s")
#define TMP_TAIL_O	dlltmp (&tmp_tail_o_buf, "%st.o")
#define TMP_STUB	dlltmp (&tmp_stub_buf, "%ssnnnnn.o")

/* This bit of assembly does jmp * ....  */
static const unsigned char i386_jtab[] =
{
  0xff, 0x25, 0x00, 0x00, 0x00, 0x00, 0x90, 0x90
};

static const unsigned char i386_dljtab[] =
{
  0xFF, 0x25, 0x00, 0x00, 0x00, 0x00, /* jmp __imp__function		 */
  0xB8, 0x00, 0x00, 0x00, 0x00,	      /* mov eax, offset __imp__function */
  0xE9, 0x00, 0x00, 0x00, 0x00	      /* jmp __tailMerge__dllname	 */
};

static const unsigned char i386_x64_dljtab[] =
{
  0xFF, 0x25, 0x00, 0x00, 0x00, 0x00, /* jmp __imp__function		 */
  0x48, 0x8d, 0x05,		      /* leaq rax, (__imp__function)	 */
	0x00, 0x00, 0x00, 0x00,
  0xE9, 0x00, 0x00, 0x00, 0x00	      /* jmp __tailMerge__dllname	 */
};

static const unsigned char arm_jtab[] =
{
  0x00, 0xc0, 0x9f, 0xe5,	/* ldr  ip, [pc] */
  0x00, 0xf0, 0x9c, 0xe5,	/* ldr  pc, [ip] */
  0,	0,    0,    0
};

static const unsigned char arm_interwork_jtab[] =
{
  0x04, 0xc0, 0x9f, 0xe5,	/* ldr	ip, [pc] */
  0x00, 0xc0, 0x9c, 0xe5,	/* ldr	ip, [ip] */
  0x1c, 0xff, 0x2f, 0xe1,	/* bx	ip	 */
  0,	0,    0,    0
};

static const unsigned char thumb_jtab[] =
{
  0x40, 0xb4,		/* push {r6}	     */
  0x02, 0x4e,		/* ldr	r6, [pc, #8] */
  0x36, 0x68,		/* ldr	r6, [r6]     */
  0xb4, 0x46,		/* mov	ip, r6	     */
  0x40, 0xbc,		/* pop	{r6}	     */
  0x60, 0x47,		/* bx	ip	     */
  0,	0,    0,    0
};

static const unsigned char mcore_be_jtab[] =
{
  0x71, 0x02,		 /* lrw r1,2	   */
  0x81, 0x01,		 /* ld.w r1,(r1,0) */
  0x00, 0xC1,		 /* jmp r1	   */
  0x12, 0x00,		 /* nop		   */
  0x00, 0x00, 0x00, 0x00 /* <address>	   */
};

static const unsigned char mcore_le_jtab[] =
{
  0x02, 0x71,		 /* lrw r1,2	   */
  0x01, 0x81,		 /* ld.w r1,(r1,0) */
  0xC1, 0x00,		 /* jmp r1	   */
  0x00, 0x12,		 /* nop		   */
  0x00, 0x00, 0x00, 0x00 /* <address>	   */
};

static const unsigned char aarch64_jtab[] =
{
  0x10, 0x00, 0x00, 0x90, /* adrp x16, 0	*/
  0x10, 0x02, 0x00, 0x91, /* add x16, x16, #0x0 */
  0x10, 0x02, 0x40, 0xf9, /* ldr x16, [x16]	*/
  0x00, 0x02, 0x1f, 0xd6  /* br x16		*/
};

static const char i386_trampoline[] =
  "\tpushl %%ecx\n"
  "\tpushl %%edx\n"
  "\tpushl %%eax\n"
  "\tpushl $__DELAY_IMPORT_DESCRIPTOR_%s\n"
  "\tcall ___delayLoadHelper2@8\n"
  "\tpopl %%edx\n"
  "\tpopl %%ecx\n"
  "\tjmp *%%eax\n";

/* Save integer arg regs in parameter space reserved by our caller
   above the return address.  Allocate space for six fp arg regs plus
   parameter space possibly used by __delayLoadHelper2 plus alignment.
   We enter with the stack offset from 16-byte alignment by the return
   address, so allocate 96 + 32 + 8 = 136 bytes.  Note that only the
   first four xmm regs are used to pass fp args, but the first six
   vector ymm (zmm too?) are used to pass vector args.  We are
   assuming that volatile vector regs are not modified inside
   __delayLoadHelper2.  However, it is known that at least xmm0 and
   xmm1 are trashed in some versions of Microsoft dlls, and if xmm4 or
   xmm5 are also used then that would trash the lower bits of ymm4 and
   ymm5.  If it turns out that vector insns with a vex prefix are used
   then we'll need to save ymm0-5 here but that can't be done without
   first testing cpuid and xcr0.  */
static const char i386_x64_trampoline[] =
  "\tsubq $136, %%rsp\n"
  "\t.seh_stackalloc 136\n"
  "\t.seh_endprologue\n"
  "\tmovq %%rcx, 136+8(%%rsp)\n"
  "\tmovq %%rdx, 136+16(%%rsp)\n"
  "\tmovq %%r8, 136+24(%%rsp)\n"
  "\tmovq %%r9, 136+32(%%rsp)\n"
  "\tmovaps %%xmm0, 32(%%rsp)\n"
  "\tmovaps %%xmm1, 48(%%rsp)\n"
  "\tmovaps %%xmm2, 64(%%rsp)\n"
  "\tmovaps %%xmm3, 80(%%rsp)\n"
  "\tmovaps %%xmm4, 96(%%rsp)\n"
  "\tmovaps %%xmm5, 112(%%rsp)\n"
  "\tmovq %%rax, %%rdx\n"
  "\tleaq __DELAY_IMPORT_DESCRIPTOR_%s(%%rip), %%rcx\n"
  "\tcall __delayLoadHelper2\n"
  "\tmovq 136+8(%%rsp), %%rcx\n"
  "\tmovq 136+16(%%rsp), %%rdx\n"
  "\tmovq 136+24(%%rsp), %%r8\n"
  "\tmovq 136+32(%%rsp), %%r9\n"
  "\tmovaps 32(%%rsp), %%xmm0\n"
  "\tmovaps 48(%%rsp), %%xmm1\n"
  "\tmovaps 64(%%rsp), %%xmm2\n"
  "\tmovaps 80(%%rsp), %%xmm3\n"
  "\tmovaps 96(%%rsp), %%xmm4\n"
  "\tmovaps 112(%%rsp), %%xmm5\n"
  "\taddq $136, %%rsp\n"
  "\tjmp *%%rax\n";

struct mac
{
  const char *type;
  const char *how_byte;
  const char *how_short;
  const char *how_long;
  const char *how_asciz;
  const char *how_comment;
  const char *how_jump;
  const char *how_global;
  const char *how_space;
  const char *how_align_short;
  const char *how_align_long;
  const char *how_default_as_switches;
  const char *how_bfd_target;
  enum bfd_architecture how_bfd_arch;
  const unsigned char *how_jtab;
  int how_jtab_size; /* Size of the jtab entry.  */
  int how_jtab_roff; /* Offset into it for the ind 32 reloc into idata 5.  */
  const unsigned char *how_dljtab;
  int how_dljtab_size; /* Size of the dljtab entry.  */
  int how_dljtab_roff1; /* Offset for the ind 32 reloc into idata 5.  */
  int how_dljtab_roff2; /* Offset for the ind 32 reloc into idata 5.  */
  int how_dljtab_roff3; /* Offset for the ind 32 reloc into idata 5.  */
  bool how_seh;
  const char *trampoline;
};

static const struct mac
mtable[] =
{
  {
#define MARM 0
    "arm", ".byte", ".short", ".long", ".asciz", "@",
    "ldr\tip,[pc]\n\tldr\tpc,[ip]\n\t.long",
    ".global", ".space", ".align\t2", ".align\t4", "-mapcs-32",
    "pe-arm-little", bfd_arch_arm,
    arm_jtab, sizeof (arm_jtab), 8,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define M386 1
    "i386", ".byte", ".short", ".long", ".asciz", "#",
    "jmp *", ".global", ".space", ".align\t2", ".align\t4", "",
    "pe-i386",bfd_arch_i386,
    i386_jtab, sizeof (i386_jtab), 2,
    i386_dljtab, sizeof (i386_dljtab), 2, 7, 12, false, i386_trampoline
  }
  ,
  {
#define MTHUMB 2
    "thumb", ".byte", ".short", ".long", ".asciz", "@",
    "push\t{r6}\n\tldr\tr6, [pc, #8]\n\tldr\tr6, [r6]\n\tmov\tip, r6\n\tpop\t{r6}\n\tbx\tip",
    ".global", ".space", ".align\t2", ".align\t4", "-mthumb-interwork",
    "pe-arm-little", bfd_arch_arm,
    thumb_jtab, sizeof (thumb_jtab), 12,
    0, 0, 0, 0, 0, false, 0
  }
  ,
#define MARM_INTERWORK 3
  {
    "arm_interwork", ".byte", ".short", ".long", ".asciz", "@",
    "ldr\tip,[pc]\n\tldr\tip,[ip]\n\tbx\tip\n\t.long",
    ".global", ".space", ".align\t2", ".align\t4", "-mthumb-interwork",
    "pe-arm-little", bfd_arch_arm,
    arm_interwork_jtab, sizeof (arm_interwork_jtab), 12,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define MMCORE_BE 4
    "mcore-be", ".byte", ".short", ".long", ".asciz", "//",
    "lrw r1,[1f]\n\tld.w r1,(r1,0)\n\tjmp r1\n\tnop\n1:.long",
    ".global", ".space", ".align\t2", ".align\t4", "",
    "pe-mcore-big", bfd_arch_mcore,
    mcore_be_jtab, sizeof (mcore_be_jtab), 8,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define MMCORE_LE 5
    "mcore-le", ".byte", ".short", ".long", ".asciz", "//",
    "lrw r1,[1f]\n\tld.w r1,(r1,0)\n\tjmp r1\n\tnop\n1:.long",
    ".global", ".space", ".align\t2", ".align\t4", "-EL",
    "pe-mcore-little", bfd_arch_mcore,
    mcore_le_jtab, sizeof (mcore_le_jtab), 8,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define MMCORE_ELF 6
    "mcore-elf-be", ".byte", ".short", ".long", ".asciz", "//",
    "lrw r1,[1f]\n\tld.w r1,(r1,0)\n\tjmp r1\n\tnop\n1:.long",
    ".global", ".space", ".align\t2", ".align\t4", "",
    "elf32-mcore-big", bfd_arch_mcore,
    mcore_be_jtab, sizeof (mcore_be_jtab), 8,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define MMCORE_ELF_LE 7
    "mcore-elf-le", ".byte", ".short", ".long", ".asciz", "//",
    "lrw r1,[1f]\n\tld.w r1,(r1,0)\n\tjmp r1\n\tnop\n1:.long",
    ".global", ".space", ".align\t2", ".align\t4", "-EL",
    "elf32-mcore-little", bfd_arch_mcore,
    mcore_le_jtab, sizeof (mcore_le_jtab), 8,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define MARM_WINCE 8
    "arm-wince", ".byte", ".short", ".long", ".asciz", "@",
    "ldr\tip,[pc]\n\tldr\tpc,[ip]\n\t.long",
    ".global", ".space", ".align\t2", ".align\t4", "-mapcs-32",
    "pe-arm-wince-little", bfd_arch_arm,
    arm_jtab, sizeof (arm_jtab), 8,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  {
#define MX86 9
    "i386:x86-64", ".byte", ".short", ".long", ".asciz", "#",
    "jmp *", ".global", ".space", ".align\t2", ".align\t4", "",
    "pe-x86-64",bfd_arch_i386,
    i386_jtab, sizeof (i386_jtab), 2,
    i386_x64_dljtab, sizeof (i386_x64_dljtab), 2, 9, 14, true, i386_x64_trampoline
  }
  ,
  {
#define MAARCH64 10
    "arm64", ".byte", ".short", ".long", ".asciz", "//",
    "bl ", ".global", ".space", ".balign\t2", ".balign\t4", "",
    "pe-aarch64-little", bfd_arch_aarch64,
    aarch64_jtab, sizeof (aarch64_jtab), 0,
    0, 0, 0, 0, 0, false, 0
  }
  ,
  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
};

typedef struct dlist
{
  const char *text;
  struct dlist *next;
}
dlist_type;

typedef struct export
{
  const char *name;
  const char *internal_name;
  const char *import_name;
  const char *its_name;
  int ordinal;
  int constant;
  int noname;		/* Don't put name in image file.  */
  int private;	/* Don't put reference in import lib.  */
  int data;
  int hint;
  int forward;	/* Number of forward label, 0 means no forward.  */
  struct export *next;
}
export_type;

/* A list of symbols which we should not export.  */

struct string_list
{
  struct string_list *next;
  char *string;
};

static struct string_list *excludes;

/* Forward references.  */
static char *deduce_name (const char *);
static char *xlate (const char *);
static void dll_name_list_free_contents (dll_name_list_node_type *);
static void identify_search_archive
  (bfd *, void (*) (bfd *, bfd *, void *),  void *);
static void identify_search_member (bfd *, bfd *, void *);
static void identify_search_section (bfd *, asection *, void *);
static void inform (const char *, ...) ATTRIBUTE_PRINTF_1;

#ifdef DLLTOOL_MCORE_ELF
static void mcore_elf_cache_filename (const char *);
static void mcore_elf_gen_out_file (void);
#endif

/* Get current BFD error message.  */
static inline const char *
bfd_get_errmsg (void)
{
  return bfd_errmsg (bfd_get_error ());
}

static char *
prefix_encode (char *start, unsigned code)
{
  static char buf[32];
  char *p = stpcpy (buf, start);
  do
    *p++ = "abcdefghijklmnopqrstuvwxyz"[code % 26];
  while ((code /= 26) != 0);
  *p = '\0';
  return buf;
}

static char *
dlltmp (char **buf, const char *fmt)
{
  if (!*buf)
    {
      *buf = xmalloc (strlen (tmp_prefix) + 64);
      sprintf (*buf, fmt, tmp_prefix);
    }
  return *buf;
}

static void
inform (const char * message, ...)
{
  va_list args;

  va_start (args, message);

  if (!verbose)
    return;

  report (message, args);

  va_end (args);
}

static const char *
rvaafter (int mach)
{
  switch (mach)
    {
    case MARM:
    case M386:
    case MX86:
    case MTHUMB:
    case MARM_INTERWORK:
    case MMCORE_BE:
    case MMCORE_LE:
    case MMCORE_ELF:
    case MMCORE_ELF_LE:
    case MARM_WINCE:
    case MAARCH64:
      break;
    default:
      /* xgettext:c-format */
      fatal (_("Internal error: Unknown machine type: %d"), mach);
      break;
    }
  return "";
}

static const char *
rvabefore (int mach)
{
  switch (mach)
    {
    case MARM:
    case M386:
    case MX86:
    case MTHUMB:
    case MARM_INTERWORK:
    case MMCORE_BE:
    case MMCORE_LE:
    case MMCORE_ELF:
    case MMCORE_ELF_LE:
    case MARM_WINCE:
    case MAARCH64:
      return ".rva\t";
    default:
      /* xgettext:c-format */
      fatal (_("Internal error: Unknown machine type: %d"), mach);
      break;
    }
  return "";
}

static const char *
asm_prefix (const char *name)
{
  /* Symbol names starting with ? do not have a leading underscore.  */
  if (name && *name == '?')
    return "";
  return leading_underscore;
}

#define ASM_BYTE		mtable[machine].how_byte
#define ASM_SHORT		mtable[machine].how_short
#define ASM_LONG		mtable[machine].how_long
#define ASM_TEXT		mtable[machine].how_asciz
#define ASM_C			mtable[machine].how_comment
#define ASM_JUMP		mtable[machine].how_jump
#define ASM_GLOBAL		mtable[machine].how_global
#define ASM_SPACE		mtable[machine].how_space
#define ASM_ALIGN_SHORT		mtable[machine].how_align_short
#define ASM_RVA_BEFORE		rvabefore (machine)
#define ASM_RVA_AFTER		rvaafter (machine)
#define ASM_PREFIX(NAME)	asm_prefix (NAME)
#define ASM_ALIGN_LONG  	mtable[machine].how_align_long
#define HOW_BFD_READ_TARGET	0  /* Always default.  */
#define HOW_BFD_WRITE_TARGET	mtable[machine].how_bfd_target
#define HOW_BFD_ARCH		mtable[machine].how_bfd_arch
#define HOW_JTAB		(delay ? mtable[machine].how_dljtab \
					: mtable[machine].how_jtab)
#define HOW_JTAB_SIZE		(delay ? mtable[machine].how_dljtab_size \
					: mtable[machine].how_jtab_size)
#define HOW_JTAB_ROFF		(delay ? mtable[machine].how_dljtab_roff1 \
					: mtable[machine].how_jtab_roff)
#define HOW_JTAB_ROFF2		(delay ? mtable[machine].how_dljtab_roff2 : 0)
#define HOW_JTAB_ROFF3		(delay ? mtable[machine].how_dljtab_roff3 : 0)
#define ASM_SWITCHES		mtable[machine].how_default_as_switches
#define HOW_SEH			mtable[machine].how_seh

static char **oav;

static void
process_def_file (const char *name)
{
  FILE *f = fopen (name, FOPEN_RT);

  if (!f)
    /* xgettext:c-format */
    fatal (_("Can't open def file: %s"), name);

  yyin = f;

  /* xgettext:c-format */
  inform (_("Processing def file: %s"), name);

  yyparse ();

  inform (_("Processed def file"));
}

/**********************************************************************/

/* Communications with the parser.  */

static int d_nfuncs;		/* Number of functions exported.  */
static int d_named_nfuncs;	/* Number of named functions exported.  */
static int d_low_ord;		/* Lowest ordinal index.  */
static int d_high_ord;		/* Highest ordinal index.  */
static export_type *d_exports;	/* List of exported functions.  */
static export_type **d_exports_lexically;  /* Vector of exported functions in alpha order.  */
static dlist_type *d_list;	/* Descriptions.  */
static dlist_type *a_list;	/* Stuff to go in directives.  */
static int d_nforwards = 0;	/* Number of forwarded exports.  */

static int d_is_dll;
static int d_is_exe;

void
yyerror (const char * err ATTRIBUTE_UNUSED)
{
  /* xgettext:c-format */
  non_fatal (_("Syntax error in def file %s:%d"), def_file, linenumber);
}

void
def_exports (const char *name, const char *internal_name, int ordinal,
	     int noname, int constant, int data, int private,
	     const char *its_name)
{
  struct export *p = (struct export *) xmalloc (sizeof (*p));

  p->name = name;
  p->internal_name = internal_name ? internal_name : name;
  p->its_name = its_name;
  p->import_name = name;
  p->ordinal = ordinal;
  p->constant = constant;
  p->noname = noname;
  p->private = private;
  p->data = data;
  p->next = d_exports;
  d_exports = p;
  d_nfuncs++;

  if (internal_name != NULL
      && strchr (internal_name, '.') != NULL)
    p->forward = ++d_nforwards;
  else
    p->forward = 0; /* no forward */
}

static void
set_dll_name_from_def (const char *name, char is_dll)
{
  const char *image_basename = lbasename (name);
  if (image_basename != name)
    non_fatal (_("%s: Path components stripped from image name, '%s'."),
	       def_file, name);
  /* Append the default suffix, if none specified.  */
  if (strchr (image_basename, '.') == 0)
    dll_name = xasprintf ("%s%s", image_basename, is_dll ? ".dll" : ".exe");
  else
    dll_name = xstrdup (image_basename);
}

void
def_name (const char *name, int base)
{
  /* xgettext:c-format */
  inform (_("NAME: %s base: %x"), name, base);

  if (d_is_dll)
    non_fatal (_("Can't have LIBRARY and NAME"));

  if (dll_name_set_by_exp_name && name && *name != 0)
    {
      free (dll_name);
      dll_name = NULL;
      dll_name_set_by_exp_name = 0;
    }
  /* If --dllname not provided, use the one in the DEF file.
     FIXME: Is this appropriate for executables?  */
  if (!dll_name)
    set_dll_name_from_def (name, 0);
  free ((char *) name);
  d_is_exe = 1;
}

void
def_library (const char *name, int base)
{
  /* xgettext:c-format */
  inform (_("LIBRARY: %s base: %x"), name, base);

  if (d_is_exe)
    non_fatal (_("Can't have LIBRARY and NAME"));

  if (dll_name_set_by_exp_name && name && *name != 0)
    {
      free (dll_name);
      dll_name = NULL;
      dll_name_set_by_exp_name = 0;
    }

  /* If --dllname not provided, use the one in the DEF file.  */
  if (!dll_name)
    set_dll_name_from_def (name, 1);
  free ((char *) name);
  d_is_dll = 1;
}

void
def_description (const char *desc)
{
  dlist_type *d = (dlist_type *) xmalloc (sizeof (dlist_type));
  d->text = desc;
  d->next = d_list;
  d_list = d;
}

static void
new_directive (char *dir)
{
  dlist_type *d = (dlist_type *) xmalloc (sizeof (dlist_type));
  d->text = dir;
  d->next = a_list;
  a_list = d;
}

void
def_heapsize (int reserve, int commit)
{
  char *s;
  if (commit > 0)
    s = xasprintf ("-heap 0x%x,0x%x ", reserve, commit);
  else
    s = xasprintf ("-heap 0x%x ", reserve);
  new_directive (s);
}

void
def_stacksize (int reserve, int commit)
{
  char *s;
  if (commit > 0)
    s = xasprintf ("-stack 0x%x,0x%x ", reserve, commit);
  else
    s = xasprintf ( "-stack 0x%x ", reserve);
  new_directive (s);
}

/* append_import simply adds the given import definition to the global
   import_list.  It is used by def_import.  */

static void
append_import (const char *symbol_name, const char *dllname, int func_ordinal,
	       const char *its_name)
{
  iheadtype **pq;
  iheadtype *q;

  for (pq = &import_list; *pq != NULL; pq = &(*pq)->next)
    {
      if (strcmp ((*pq)->dllname, dllname) == 0)
	{
	  q = *pq;
	  q->functail->next = xmalloc (sizeof (ifunctype));
	  q->functail = q->functail->next;
	  q->functail->ord  = func_ordinal;
	  q->functail->name = symbol_name;
	  q->functail->its_name = its_name;
	  q->functail->next = NULL;
	  q->nfuncs++;
	  return;
	}
    }

  q = xmalloc (sizeof (iheadtype));
  q->dllname = dllname;
  q->nfuncs = 1;
  q->funchead = xmalloc (sizeof (ifunctype));
  q->functail = q->funchead;
  q->next = NULL;
  q->functail->name = symbol_name;
  q->functail->its_name = its_name;
  q->functail->ord  = func_ordinal;
  q->functail->next = NULL;

  *pq = q;
}

/* def_import is called from within defparse.y when an IMPORT
   declaration is encountered.  Depending on the form of the
   declaration, the module name may or may not need ".dll" to be
   appended to it, the name of the function may be stored in internal
   or entry, and there may or may not be an ordinal value associated
   with it.  */

/* A note regarding the parse modes:
   In defparse.y we have to accept import declarations which follow
   any one of the following forms:
     <func_name_in_app> = <dll_name>.<func_name_in_dll>
     <func_name_in_app> = <dll_name>.<number>
     <dll_name>.<func_name_in_dll>
     <dll_name>.<number>
   Furthermore, the dll's name may or may not end with ".dll", which
   complicates the parsing a little.  Normally the dll's name is
   passed to def_import() in the "module" parameter, but when it ends
   with ".dll" it gets passed in "module" sans ".dll" and that needs
   to be reappended.

  def_import gets five parameters:
  APP_NAME - the name of the function in the application, if
             present, or NULL if not present.
  MODULE   - the name of the dll, possibly sans extension (ie, '.dll').
  DLLEXT   - the extension of the dll, if present, NULL if not present.
  ENTRY    - the name of the function in the dll, if present, or NULL.
  ORD_VAL  - the numerical tag of the function in the dll, if present,
             or NULL.  Exactly one of <entry> or <ord_val> must be
             present (i.e., not NULL).  */

void
def_import (const char *app_name, const char *module, const char *dllext,
	    const char *entry, int ord_val, const char *its_name)
{
  const char *application_name;

  if (entry != NULL)
    application_name = entry;
  else
    {
      if (app_name != NULL)
	application_name = app_name;
      else
	application_name = "";
    }

  const char *mod_name = module;
  if (dllext != NULL)
    {
      mod_name = concat (module, ".", dllext, NULL);
      free ((char *) module);
    }

  append_import (application_name, mod_name, ord_val, its_name);
}

void
def_version (int major, int minor)
{
  printf (_("VERSION %d.%d\n"), major, minor);
}

void
def_section (const char *name, int attr)
{
  char atts[5];
  char *d = atts;
  if (attr & 1)
    *d++ = 'R';
  if (attr & 2)
    *d++ = 'W';
  if (attr & 4)
    *d++ = 'X';
  if (attr & 8)
    *d++ = 'S';
  *d++ = 0;
  char *s = xasprintf ("-attr %s %s", name, atts);
  new_directive (s);
}

void
def_code (int attr)
{
  def_section ("CODE", attr);
}

void
def_data (int attr)
{
  def_section ("DATA", attr);
}

/**********************************************************************/

static void
run (const char *what, char *args)
{
  char *s;
  int pid, wait_status;
  int i;
  const char **argv;
  char *errmsg_fmt = NULL, *errmsg_arg = NULL;
  char *temp_base = make_temp_file ("");

  inform (_("run: %s %s"), what, args);

  /* Count the args */
  i = 0;
  for (s = args; *s; s++)
    if (*s == ' ')
      i++;
  i++;
  argv = xmalloc (sizeof (char *) * (i + 3));
  i = 0;
  argv[i++] = what;
  s = args;
  while (1)
    {
      while (*s == ' ')
	++s;
      argv[i++] = s;
      while (*s != ' ' && *s != 0)
	s++;
      if (*s == 0)
	break;
      *s++ = 0;
    }
  argv[i++] = NULL;

  pid = pexecute (argv[0], (char * const *) argv, program_name, temp_base,
		  &errmsg_fmt, &errmsg_arg, PEXECUTE_ONE | PEXECUTE_SEARCH);
  free (argv);
  free (temp_base);

  if (pid == -1)
    {
      inform ("%s", strerror (errno));

      fatal (errmsg_fmt, errmsg_arg);
    }

  pid = pwait (pid, & wait_status, 0);

  if (pid == -1)
    {
      /* xgettext:c-format */
      fatal (_("wait: %s"), strerror (errno));
    }
  else if (WIFSIGNALED (wait_status))
    {
      /* xgettext:c-format */
      fatal (_("subprocess got fatal signal %d"), WTERMSIG (wait_status));
    }
  else if (WIFEXITED (wait_status))
    {
      if (WEXITSTATUS (wait_status) != 0)
	/* xgettext:c-format */
	non_fatal (_("%s exited with status %d"),
		   what, WEXITSTATUS (wait_status));
    }
  else
    abort ();
}

/* Look for a list of symbols to export in the .drectve section of
   ABFD.  Pass each one to def_exports.  */

static void
scan_drectve_symbols (bfd *abfd)
{
  /* Look for .drectve's */
  asection *s = bfd_get_section_by_name (abfd, DRECTVE_SECTION_NAME);
  if (s == NULL)
    return;

  bfd_byte *buf;
  if (!bfd_malloc_and_get_section (abfd, s, &buf))
    return;

  /* xgettext:c-format */
  inform (_("Sucking in info from %s section in %s"),
	  DRECTVE_SECTION_NAME, bfd_get_filename (abfd));

  /* Search for -export: strings. The exported symbols can optionally
     have type tags (eg., -export:foo,data), so handle those as well.
     Currently only data tag is supported.  */
  const char *p = (const char *) buf;
  const char *e = (const char *) buf + bfd_section_size (s);
  while (p < e)
    {
      if (p[0] == '-'
	  && startswith (p, "-export:"))
	{
	  const char *name;
	  char *c;
	  flagword flags = BSF_FUNCTION;

	  p += 8;
	  /* Do we have a quoted export?  */
	  if (*p == '"')
	    {
	      p++;
	      name = p;
	      while (p < e && *p != '"')
		++p;
	    }
	  else
	    {
	      name = p;
	      while (p < e && *p != ',' && *p != ' ' && *p != '-')
		p++;
	    }
	  c = xmemdup (name, p - name, p - name + 1);
	  /* Advance over trailing quote.  */
	  if (p < e && *p == '"')
	    ++p;
	  if (p < e && *p == ',')       /* found type tag.  */
	    {
	      const char *tag_start = ++p;
	      while (p < e && *p != ' ' && *p != '-')
		p++;
	      if (startswith (tag_start, "data"))
		flags &= ~BSF_FUNCTION;
	    }

	  /* FIXME: The 5th arg is for the `constant' field.
	     What should it be?  Not that it matters since it's not
	     currently useful.  */
	  def_exports (c, 0, -1, 0, 0, ! (flags & BSF_FUNCTION), 0, NULL);

	  if (add_stdcall_alias && strchr (c, '@'))
	    {
	      int lead_at = (*c == '@') ;
	      char *exported_name = xstrdup (c + lead_at);
	      char *atsym = strchr (exported_name, '@');
	      if (atsym)
		*atsym = '\0';
	      /* Note: stdcall alias symbols can never be data.  */
	      def_exports (exported_name, c, -1, 0, 0, 0, 0, NULL);
	    }
	}
      else
	p++;
    }
  free (buf);
}

/* Look through the symbols in MINISYMS, and add each one to list of
   symbols to export.  */

static void
scan_filtered_symbols (bfd *abfd, void *minisyms, long symcount,
		       unsigned int size)
{
  asymbol *store;
  bfd_byte *from, *fromend;

  store = bfd_make_empty_symbol (abfd);
  if (store == NULL)
    bfd_fatal (bfd_get_filename (abfd));

  from = (bfd_byte *) minisyms;
  fromend = from + symcount * size;
  for (; from < fromend; from += size)
    {
      asymbol *sym;
      const char *symbol_name;

      sym = bfd_minisymbol_to_symbol (abfd, false, from, store);
      if (sym == NULL)
	bfd_fatal (bfd_get_filename (abfd));

      symbol_name = bfd_asymbol_name (sym);
      if (*symbol_name
	  && *symbol_name == bfd_get_symbol_leading_char (abfd))
	++symbol_name;
      symbol_name = xstrdup (symbol_name);

      def_exports (symbol_name , 0, -1, 0, 0,
		   ! (sym->flags & BSF_FUNCTION), 0, NULL);

      if (add_stdcall_alias && strchr (symbol_name, '@'))
	{
	  int lead_at = (*symbol_name == '@');
	  char *exported_name = xstrdup (symbol_name + lead_at);
	  char *atsym = strchr (exported_name, '@');
	  if (atsym)
	    *atsym = '\0';
	  /* Note: stdcall alias symbols can never be data.  */
	  def_exports (exported_name, symbol_name,
		       -1, 0, 0, 0, 0, NULL);
	}
    }
}

/* Add a list of symbols to exclude.  */

static void
add_excludes (const char *new_excludes)
{
  char *local_copy;
  char *exclude_string;

  local_copy = xstrdup (new_excludes);

  exclude_string = strtok (local_copy, ",:");
  for (; exclude_string; exclude_string = strtok (NULL, ",:"))
    {
      struct string_list *new_exclude = xmalloc (sizeof (*new_exclude));
      /* Don't add a leading underscore for fastcall symbols.  */
      if (*exclude_string == '@')
	new_exclude->string = xstrdup (exclude_string);
      else
	new_exclude->string = xasprintf ("%s%s", leading_underscore,
					 exclude_string);
      new_exclude->next = excludes;
      excludes = new_exclude;

      /* xgettext:c-format */
      inform (_("Excluding symbol: %s"), exclude_string);
    }

  free (local_copy);
}

/* See if STRING is on the list of symbols to exclude.  */

static bool
match_exclude (const char *string)
{
  struct string_list *excl_item;

  for (excl_item = excludes; excl_item; excl_item = excl_item->next)
    if (strcmp (string, excl_item->string) == 0)
      return true;
  return false;
}

/* Add the default list of symbols to exclude.  */

static void
set_default_excludes (void)
{
  add_excludes (default_excludes);
}

/* Choose which symbols to export.  */

static long
filter_symbols (bfd *abfd, void *minisyms, long symcount, unsigned int size)
{
  bfd_byte *from, *fromend, *to;
  asymbol *store;

  store = bfd_make_empty_symbol (abfd);
  if (store == NULL)
    bfd_fatal (bfd_get_filename (abfd));

  from = (bfd_byte *) minisyms;
  fromend = from + symcount * size;
  to = (bfd_byte *) minisyms;

  for (; from < fromend; from += size)
    {
      int keep = 0;
      asymbol *sym;

      sym = bfd_minisymbol_to_symbol (abfd, false, (const void *) from, store);
      if (sym == NULL)
	bfd_fatal (bfd_get_filename (abfd));

      /* Check for external and defined only symbols.  */
      keep = (((sym->flags & BSF_GLOBAL) != 0
	       || (sym->flags & BSF_WEAK) != 0
	       || bfd_is_com_section (sym->section))
	      && ! bfd_is_und_section (sym->section));

      keep = keep && ! match_exclude (sym->name);

      if (keep)
	{
	  memcpy (to, from, size);
	  to += size;
	}
    }

  return (to - (bfd_byte *) minisyms) / size;
}

/* Export all symbols in ABFD, except for ones we were told not to
   export.  */

static void
scan_all_symbols (bfd *abfd)
{
  long symcount;
  void *minisyms;
  unsigned int size;

  /* Ignore bfds with an import descriptor table.  We assume that any
     such BFD contains symbols which are exported from another DLL,
     and we don't want to reexport them from here.  */
  if (bfd_get_section_by_name (abfd, ".idata$4"))
    return;

  if (! (bfd_get_file_flags (abfd) & HAS_SYMS))
    {
      /* xgettext:c-format */
      non_fatal (_("%s: no symbols"), bfd_get_filename (abfd));
      return;
    }

  symcount = bfd_read_minisymbols (abfd, false, &minisyms, &size);
  if (symcount < 0)
    bfd_fatal (bfd_get_filename (abfd));

  if (symcount == 0)
    {
      /* xgettext:c-format */
      non_fatal (_("%s: no symbols"), bfd_get_filename (abfd));
      return;
    }

  /* Discard the symbols we don't want to export.  It's OK to do this
     in place; we'll free the storage anyway.  */

  symcount = filter_symbols (abfd, minisyms, symcount, size);
  scan_filtered_symbols (abfd, minisyms, symcount, size);

  free (minisyms);
}

/* Look at the object file to decide which symbols to export.  */

static void
scan_open_obj_file (bfd *abfd)
{
  if (export_all_symbols)
    scan_all_symbols (abfd);
  else
    scan_drectve_symbols (abfd);

  /* FIXME: we ought to read in and block out the base relocations.  */

  /* xgettext:c-format */
  inform (_("Done reading %s"), bfd_get_filename (abfd));
}

static void
scan_obj_file (const char *filename)
{
  bfd * f = bfd_openr (filename, 0);

  if (!f)
    /* xgettext:c-format */
    fatal (_("Unable to open object file: %s: %s"), filename, bfd_get_errmsg ());

  /* xgettext:c-format */
  inform (_("Scanning object file %s"), filename);

  if (bfd_check_format (f, bfd_archive))
    {
      bfd *arfile = bfd_openr_next_archived_file (f, 0);
      while (arfile)
	{
	  bfd *next;
	  if (bfd_check_format (arfile, bfd_object))
	    scan_open_obj_file (arfile);
	  next = bfd_openr_next_archived_file (f, arfile);
	  bfd_close (arfile);
	  /* PR 17512: file: 58715298.  */
	  if (next == arfile)
	    break;
	  arfile = next;
	}

#ifdef DLLTOOL_MCORE_ELF
      if (mcore_elf_out_file)
	inform (_("Cannot produce mcore-elf dll from archive file: %s"),
		filename);
#endif
    }
  else if (bfd_check_format (f, bfd_object))
    {
      scan_open_obj_file (f);

#ifdef DLLTOOL_MCORE_ELF
      if (mcore_elf_out_file)
	mcore_elf_cache_filename (filename);
#endif
    }

  bfd_close (f);
}


static void
dump_def_info (FILE *f)
{
  int i;
  export_type *exp;
  fprintf (f, "%s ", ASM_C);
  for (i = 0; oav[i]; i++)
    fprintf (f, "%s ", oav[i]);
  fprintf (f, "\n");
  for (i = 0, exp = d_exports; exp; i++, exp = exp->next)
    {
      fprintf (f, "%s  %d = %s %s @ %d %s%s%s%s%s%s\n",
	       ASM_C,
	       i,
	       exp->name,
	       exp->internal_name,
	       exp->ordinal,
	       exp->noname ? "NONAME " : "",
	       exp->private ? "PRIVATE " : "",
	       exp->constant ? "CONSTANT" : "",
	       exp->data ? "DATA" : "",
	       exp->its_name ? " ==" : "",
	       exp->its_name ? exp->its_name : "");
    }
}

/* Generate the .exp file.  */

static int
sfunc (const void *a, const void *b)
{
  if (*(const bfd_vma *) a == *(const bfd_vma *) b)
    return 0;

  return ((*(const bfd_vma *) a > *(const bfd_vma *) b) ? 1 : -1);
}

static void
flush_page (FILE *f, bfd_vma *need, bfd_vma page_addr, int on_page)
{
  int i;

  /* Flush this page.  */
  fprintf (f, "\t%s\t0x%08x\t%s Starting RVA for chunk\n",
	   ASM_LONG,
	   (int) page_addr,
	   ASM_C);
  fprintf (f, "\t%s\t0x%x\t%s Size of block\n",
	   ASM_LONG,
	   (on_page * 2) + (on_page & 1) * 2 + 8,
	   ASM_C);

  for (i = 0; i < on_page; i++)
    {
      bfd_vma needed = need[i];

      if (needed)
	{
	  if (!create_for_pep)
	    {
	      /* Relocation via HIGHLOW.  */
	      needed = ((needed - page_addr) | 0x3000) & 0xffff;
	    }
	  else
	    {
	      /* Relocation via DIR64.  */
	      needed = ((needed - page_addr) | 0xa000) & 0xffff;
	    }
	}

      fprintf (f, "\t%s\t0x%lx\n", ASM_SHORT, (long) needed);
    }

  /* And padding */
  if (on_page & 1)
    fprintf (f, "\t%s\t0x%x\n", ASM_SHORT, 0 | 0x0000);
}

static void
gen_def_file (void)
{
  int i;
  export_type *exp;

  inform (_("Adding exports to output file"));

  fprintf (output_def, ";");
  for (i = 0; oav[i]; i++)
    fprintf (output_def, " %s", oav[i]);

  fprintf (output_def, "\nEXPORTS\n");

  for (i = 0, exp = d_exports; exp; i++, exp = exp->next)
    {
      char *quote = strchr (exp->name, '.') ? "\"" : "";
      char *res = cplus_demangle (exp->internal_name, DMGL_ANSI | DMGL_PARAMS);

      if (res)
	{
	  fprintf (output_def, ";\t%s\n", res);
	  free (res);
	}

      if (strcmp (exp->name, exp->internal_name) == 0)
	{
	  fprintf (output_def, "\t%s%s%s @ %d%s%s%s%s%s\n",
		   quote,
		   exp->name,
		   quote,
		   exp->ordinal,
		   exp->noname ? " NONAME" : "",
		   exp->private ? "PRIVATE " : "",
		   exp->data ? " DATA" : "",
		   exp->its_name ? " ==" : "",
		   exp->its_name ? exp->its_name : "");
	}
      else
	{
	  char * quote1 = strchr (exp->internal_name, '.') ? "\"" : "";
	  /* char *alias =  */
	  fprintf (output_def, "\t%s%s%s = %s%s%s @ %d%s%s%s%s%s\n",
		   quote,
		   exp->name,
		   quote,
		   quote1,
		   exp->internal_name,
		   quote1,
		   exp->ordinal,
		   exp->noname ? " NONAME" : "",
		   exp->private ? "PRIVATE " : "",
		   exp->data ? " DATA" : "",
		   exp->its_name ? " ==" : "",
		   exp->its_name ? exp->its_name : "");
	}
    }

  inform (_("Added exports to output file"));
}

/* generate_idata_ofile generates the portable assembly source code
   for the idata sections.  It appends the source code to the end of
   the file.  */

static void
generate_idata_ofile (FILE *filvar)
{
  iheadtype *headptr;
  ifunctype *funcptr;
  int        headindex;
  int        funcindex;
  int	     nheads;

  if (import_list == NULL)
    return;

  fprintf (filvar, "%s Import data sections\n", ASM_C);
  fprintf (filvar, "\n\t.section\t.idata$2\n");
  fprintf (filvar, "\t%s\tdoi_idata\n", ASM_GLOBAL);
  fprintf (filvar, "doi_idata:\n");

  nheads = 0;
  for (headptr = import_list; headptr != NULL; headptr = headptr->next)
    {
      fprintf (filvar, "\t%slistone%d%s\t%s %s\n",
	       ASM_RVA_BEFORE, nheads, ASM_RVA_AFTER,
	       ASM_C, headptr->dllname);
      fprintf (filvar, "\t%s\t0\n", ASM_LONG);
      fprintf (filvar, "\t%s\t0\n", ASM_LONG);
      fprintf (filvar, "\t%sdllname%d%s\n",
	       ASM_RVA_BEFORE, nheads, ASM_RVA_AFTER);
      fprintf (filvar, "\t%slisttwo%d%s\n\n",
	       ASM_RVA_BEFORE, nheads, ASM_RVA_AFTER);
      nheads++;
    }

  fprintf (filvar, "\t%s\t0\n", ASM_LONG); /* NULL record at */
  fprintf (filvar, "\t%s\t0\n", ASM_LONG); /* end of idata$2 */
  fprintf (filvar, "\t%s\t0\n", ASM_LONG); /* section        */
  fprintf (filvar, "\t%s\t0\n", ASM_LONG);
  fprintf (filvar, "\t%s\t0\n", ASM_LONG);

  fprintf (filvar, "\n\t.section\t.idata$4\n");
  headindex = 0;
  for (headptr = import_list; headptr != NULL; headptr = headptr->next)
    {
      fprintf (filvar, "listone%d:\n", headindex);
      for (funcindex = 0; funcindex < headptr->nfuncs; funcindex++)
	{
	  if (create_for_pep)
	    fprintf (filvar, "\t%sfuncptr%d_%d%s\n%s\t0\n",
		     ASM_RVA_BEFORE, headindex, funcindex, ASM_RVA_AFTER,
		     ASM_LONG);
	  else
	    fprintf (filvar, "\t%sfuncptr%d_%d%s\n",
		     ASM_RVA_BEFORE, headindex, funcindex, ASM_RVA_AFTER);
	}
      if (create_for_pep)
	fprintf (filvar, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
      else
	fprintf (filvar, "\t%s\t0\n", ASM_LONG); /* NULL terminating list.  */
      headindex++;
    }

  fprintf (filvar, "\n\t.section\t.idata$5\n");
  headindex = 0;
  for (headptr = import_list; headptr != NULL; headptr = headptr->next)
    {
      fprintf (filvar, "listtwo%d:\n", headindex);
      for (funcindex = 0; funcindex < headptr->nfuncs; funcindex++)
	{
	  if (create_for_pep)
	    fprintf (filvar, "\t%sfuncptr%d_%d%s\n%s\t0\n",
		     ASM_RVA_BEFORE, headindex, funcindex, ASM_RVA_AFTER,
		     ASM_LONG);
	  else
	    fprintf (filvar, "\t%sfuncptr%d_%d%s\n",
		     ASM_RVA_BEFORE, headindex, funcindex, ASM_RVA_AFTER);
	}
      if (create_for_pep)
	fprintf (filvar, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
      else
	fprintf (filvar, "\t%s\t0\n", ASM_LONG); /* NULL terminating list.  */
      headindex++;
    }

  fprintf (filvar, "\n\t.section\t.idata$6\n");
  headindex = 0;
  for (headptr = import_list; headptr != NULL; headptr = headptr->next)
    {
      funcindex = 0;
      for (funcptr = headptr->funchead; funcptr != NULL;
	   funcptr = funcptr->next)
	{
	  fprintf (filvar, "funcptr%d_%d:\n", headindex, funcindex);
	  fprintf (filvar, "\t%s\t%d\n", ASM_SHORT, funcptr->ord & 0xFFFF);
	  fprintf (filvar, "\t%s\t\"%s\"\n", ASM_TEXT,
		   funcptr->its_name ? funcptr->its_name : funcptr->name);
	  fprintf (filvar, "\t%s\t0\n", ASM_BYTE);
	  funcindex++;
	}
      headindex++;
    }

  fprintf (filvar, "\n\t.section\t.idata$7\n");
  headindex = 0;
  for (headptr = import_list; headptr != NULL; headptr = headptr->next)
    {
      fprintf (filvar, "dllname%d:\n", headindex);
      fprintf (filvar, "\t%s\t\"%s\"\n", ASM_TEXT, headptr->dllname);
      fprintf (filvar, "\t%s\t0\n", ASM_BYTE);
      headindex++;
    }
}

/* Assemble the specified file.  */
static void
assemble_file (const char * source, const char * dest)
{
  char *cmd = xasprintf ("%s %s -o %s %s",
			 ASM_SWITCHES, as_flags, dest, source);
  run (as_name, cmd);
  free (cmd);
}

static const char * temp_file_to_remove[5];
#define TEMP_EXPORT_FILE 0
#define TEMP_HEAD_FILE   1
#define TEMP_TAIL_FILE   2
#define TEMP_HEAD_O_FILE 3
#define TEMP_TAIL_O_FILE 4

static void
unlink_temp_files (void)
{
  unsigned i;

  if (dontdeltemps > 0)
    return;

  for (i = 0; i < ARRAY_SIZE (temp_file_to_remove); i++)
    {
      if (temp_file_to_remove[i])
	{
	  unlink (temp_file_to_remove[i]);
	  temp_file_to_remove[i] = NULL;
	}
    }
}

static void
gen_exp_file (void)
{
  FILE *f;
  int i;
  export_type *exp;
  dlist_type *dl;

  /* xgettext:c-format */
  inform (_("Generating export file: %s"), exp_name);

  f = fopen (TMP_ASM, FOPEN_WT);
  if (!f)
    /* xgettext:c-format */
    fatal (_("Unable to open temporary assembler file: %s"), TMP_ASM);

  temp_file_to_remove[TEMP_EXPORT_FILE] = TMP_ASM;

  /* xgettext:c-format */
  inform (_("Opened temporary file: %s"), TMP_ASM);

  dump_def_info (f);

  if (d_exports)
    {
      fprintf (f, "\t.section	.edata\n\n");
      fprintf (f, "\t%s	0	%s Allways 0\n", ASM_LONG, ASM_C);
      fprintf (f, "\t%s	0x%lx	%s Time and date\n", ASM_LONG,
	       (unsigned long) time(0), ASM_C);
      fprintf (f, "\t%s	0	%s Major and Minor version\n", ASM_LONG, ASM_C);
      fprintf (f, "\t%sname%s	%s Ptr to name of dll\n",
	       ASM_RVA_BEFORE, ASM_RVA_AFTER, ASM_C);
      fprintf (f, "\t%s	%d	%s Starting ordinal of exports\n",
	       ASM_LONG, d_low_ord, ASM_C);


      fprintf (f, "\t%s	%d	%s Number of functions\n",
	       ASM_LONG, d_high_ord - d_low_ord + 1, ASM_C);
      fprintf (f, "\t%s named funcs %d, low ord %d, high ord %d\n",
	       ASM_C, d_named_nfuncs, d_low_ord, d_high_ord);
      fprintf (f, "\t%s	%d	%s Number of names\n", ASM_LONG,
	       show_allnames ? d_high_ord - d_low_ord + 1 : d_named_nfuncs,
	       ASM_C);
      fprintf (f, "\t%safuncs%s  %s Address of functions\n",
	       ASM_RVA_BEFORE, ASM_RVA_AFTER, ASM_C);

      fprintf (f, "\t%sanames%s	%s Address of Name Pointer Table\n",
	       ASM_RVA_BEFORE, ASM_RVA_AFTER, ASM_C);

      fprintf (f, "\t%sanords%s	%s Address of ordinals\n",
	       ASM_RVA_BEFORE, ASM_RVA_AFTER, ASM_C);

      fprintf (f, "name:	%s	\"%s\"\n", ASM_TEXT, dll_name);


      fprintf (f, "%s Export address Table\n", ASM_C);
      fprintf (f, "\t%s\n", ASM_ALIGN_LONG);
      fprintf (f, "afuncs:\n");
      i = d_low_ord;

      for (exp = d_exports; exp; exp = exp->next)
	{
	  if (exp->ordinal != i)
	    {
	      while (i < exp->ordinal)
		{
		  fprintf (f, "\t%s\t0\n", ASM_LONG);
		  i++;
		}
	    }

	  if (exp->forward == 0)
	    {
	      if (exp->internal_name[0] == '@')
		fprintf (f, "\t%s%s%s\t%s %d\n", ASM_RVA_BEFORE,
			 exp->internal_name, ASM_RVA_AFTER, ASM_C, exp->ordinal);
	      else
		fprintf (f, "\t%s%s%s%s\t%s %d\n", ASM_RVA_BEFORE,
			 ASM_PREFIX (exp->internal_name),
			 exp->internal_name, ASM_RVA_AFTER, ASM_C, exp->ordinal);
	    }
	  else
	    fprintf (f, "\t%sf%d%s\t%s %d\n", ASM_RVA_BEFORE,
		     exp->forward, ASM_RVA_AFTER, ASM_C, exp->ordinal);
	  i++;
	}

      fprintf (f, "%s Export Name Pointer Table\n", ASM_C);
      fprintf (f, "anames:\n");

      for (i = 0; (exp = d_exports_lexically[i]); i++)
	{
	  if (!exp->noname || show_allnames)
	    fprintf (f, "\t%sn%d%s\n",
		     ASM_RVA_BEFORE, exp->ordinal, ASM_RVA_AFTER);
	}

      fprintf (f, "%s Export Ordinal Table\n", ASM_C);
      fprintf (f, "anords:\n");
      for (i = 0; (exp = d_exports_lexically[i]); i++)
	{
	  if (!exp->noname || show_allnames)
	    fprintf (f, "\t%s	%d\n", ASM_SHORT, exp->ordinal - d_low_ord);
	}

      fprintf (f, "%s Export Name Table\n", ASM_C);
      for (i = 0; (exp = d_exports_lexically[i]); i++)
	{
	  if (!exp->noname || show_allnames)
	    {
	      const char *xname = (exp->its_name ? exp->its_name
				   : xlate (exp->name));
	      fprintf (f, "n%d:	%s	\"%s\"\n",
		       exp->ordinal, ASM_TEXT, xname);
	      if (!exp->its_name)
		free ((char *) xname);
	    }
	  if (exp->forward != 0)
	    fprintf (f, "f%d:	%s	\"%s\"\n",
		     exp->forward, ASM_TEXT, exp->internal_name);
	}

      if (a_list)
	{
	  fprintf (f, "\t.section %s\n", DRECTVE_SECTION_NAME);
	  for (dl = a_list; dl; dl = dl->next)
	    {
	      fprintf (f, "\t%s\t\"%s\"\n", ASM_TEXT, dl->text);
	    }
	}

      if (d_list)
	{
	  fprintf (f, "\t.section .rdata\n");
	  for (dl = d_list; dl; dl = dl->next)
	    {
	      const char *p;
	      int l;

	      /* We don't output as ascii because there can
		 be quote characters in the string.  */
	      l = 0;
	      for (p = dl->text; *p; p++)
		{
		  if (l == 0)
		    fprintf (f, "\t%s\t", ASM_BYTE);
		  else
		    fprintf (f, ",");
		  fprintf (f, "%d", *p);
		  if (p[1] == 0)
		    {
		      fprintf (f, ",0\n");
		      break;
		    }
		  if (++l == 10)
		    {
		      fprintf (f, "\n");
		      l = 0;
		    }
		}
	    }
	}
    }

  /* Add to the output file a way of getting to the exported names
     without using the import library.  */
  if (add_indirect)
    {
      fprintf (f, "\t.section\t.rdata\n");
      for (i = 0, exp = d_exports; exp; i++, exp = exp->next)
	if (!exp->noname || show_allnames)
	  {
	    /* We use a single underscore for MS compatibility, and a
	       double underscore for backward compatibility with old
	       cygwin releases.  */
	    if (create_compat_implib)
	      fprintf (f, "\t%s\t__imp_%s\n", ASM_GLOBAL, exp->name);
	    fprintf (f, "\t%s\t_imp_%s%s\n", ASM_GLOBAL,
		     leading_underscore, exp->name);
	    if (create_compat_implib)
	      fprintf (f, "__imp_%s:\n", exp->name);
	    fprintf (f, "_imp_%s%s:\n", leading_underscore, exp->name);
	    fprintf (f, "\t%s\t%s\n", ASM_LONG, exp->name);
	  }
    }

  /* Dump the reloc section if a base file is provided.  */
  if (base_file)
    {
      bfd_vma addr;
      bfd_vma need[COFF_PAGE_SIZE];
      bfd_vma page_addr;
      bfd_size_type numbytes;
      int num_entries;
      bfd_vma *copy;
      int j;
      int on_page;
      fprintf (f, "\t.section\t.init\n");
      fprintf (f, "lab:\n");

      fseek (base_file, 0, SEEK_END);
      numbytes = ftell (base_file);
      fseek (base_file, 0, SEEK_SET);
      copy = xmalloc (numbytes);
      if (fread (copy, 1, numbytes, base_file) < numbytes)
	fatal (_("failed to read the number of entries from base file"));
      num_entries = numbytes / sizeof (bfd_vma);

      fprintf (f, "\t.section\t.reloc\n");
      if (num_entries)
	{
	  int src;
	  int dst = 0;
	  bfd_vma last = (bfd_vma) -1;
	  qsort (copy, num_entries, sizeof (bfd_vma), sfunc);
	  /* Delete duplicates */
	  for (src = 0; src < num_entries; src++)
	    {
	      if (last != copy[src])
		last = copy[dst++] = copy[src];
	    }
	  num_entries = dst;
	  addr = copy[0];
	  page_addr = addr & PAGE_MASK;		/* work out the page addr */
	  on_page = 0;
	  for (j = 0; j < num_entries; j++)
	    {
	      addr = copy[j];
	      if ((addr & PAGE_MASK) != page_addr)
		{
		  flush_page (f, need, page_addr, on_page);
		  on_page = 0;
		  page_addr = addr & PAGE_MASK;
		}
	      need[on_page++] = addr;
	    }
	  flush_page (f, need, page_addr, on_page);
#if 0
	  fprintf (f, "\t%s\t0,0\t%s End\n", ASM_LONG, ASM_C);
#endif
	}
      free (copy);
    }

  generate_idata_ofile (f);

  fclose (f);

  /* Assemble the file.  */
  assemble_file (TMP_ASM, exp_name);

  if (dontdeltemps == 0)
    {
      temp_file_to_remove[TEMP_EXPORT_FILE] = NULL;
      unlink (TMP_ASM);
    }

  inform (_("Generated exports file"));
}

static char *
xlate (const char *name)
{
  int lead_at = *name == '@';
  int is_stdcall = !lead_at && strchr (name, '@') != NULL;
  char *copy;

  if (!lead_at && (add_underscore
		   || (add_stdcall_underscore && is_stdcall)))
    copy = xasprintf ("_%s", name);
  else
    copy = xstrdup (name + (killat ? lead_at : 0));

  if (killat)
    {
      /* PR 9766: Look for the last @ sign in the name.  */
      char *p = strrchr (copy, '@');
      if (p && ISDIGIT (p[1]))
	*p = 0;
    }
  return copy;
}

typedef struct
{
  int id;
  const char *name;
  int flags;
  int align;
  asection *sec;
  asymbol *sym;
  asymbol **sympp;
  int size;
  unsigned char *data;
} sinfo;

#define INIT_SEC_DATA(id, name, flags, align) \
  { id, name, flags, align, NULL, NULL, NULL, 0, NULL }

#define TEXT 0
#define DATA 1
#define BSS 2
#define IDATA7 3
#define IDATA5 4
#define IDATA4 5
#define IDATA6 6

#define NSECS 7

#define TEXT_SEC_FLAGS   \
  (SEC_ALLOC | SEC_LOAD | SEC_CODE | SEC_READONLY | SEC_HAS_CONTENTS)
#define DATA_SEC_FLAGS   (SEC_ALLOC | SEC_LOAD | SEC_DATA)
#define BSS_SEC_FLAGS     SEC_ALLOC

static sinfo secdata_plain[NSECS] =
{
  INIT_SEC_DATA (TEXT,   ".text",    TEXT_SEC_FLAGS,   2),
  INIT_SEC_DATA (DATA,   ".data",    DATA_SEC_FLAGS,   2),
  INIT_SEC_DATA (BSS,    ".bss",     BSS_SEC_FLAGS,    2),
  INIT_SEC_DATA (IDATA7, ".idata$7", SEC_HAS_CONTENTS, 2),
  INIT_SEC_DATA (IDATA5, ".idata$5", SEC_HAS_CONTENTS, 2),
  INIT_SEC_DATA (IDATA4, ".idata$4", SEC_HAS_CONTENTS, 2),
  INIT_SEC_DATA (IDATA6, ".idata$6", SEC_HAS_CONTENTS, 1)
};

static sinfo secdata_delay[NSECS] =
{
  INIT_SEC_DATA (TEXT,   ".text",    TEXT_SEC_FLAGS,   2),
  INIT_SEC_DATA (DATA,   ".data",    DATA_SEC_FLAGS,   2),
  INIT_SEC_DATA (BSS,    ".bss",     BSS_SEC_FLAGS,    2),
  INIT_SEC_DATA (IDATA7, ".didat$7", SEC_HAS_CONTENTS, 2),
  INIT_SEC_DATA (IDATA5, ".didat$5", SEC_HAS_CONTENTS, 2),
  INIT_SEC_DATA (IDATA4, ".didat$4", SEC_HAS_CONTENTS, 2),
  INIT_SEC_DATA (IDATA6, ".didat$6", SEC_HAS_CONTENTS, 1)
};

/* This is what we're trying to make.  We generate the imp symbols with
   both single and double underscores, for compatibility.

	.text
	.global	_GetFileVersionInfoSizeW@8
	.global	__imp_GetFileVersionInfoSizeW@8
_GetFileVersionInfoSizeW@8:
	jmp *	__imp_GetFileVersionInfoSizeW@8
	.section	.idata$7	# To force loading of head
	.long	__version_a_head
# Import Address Table
	.section	.idata$5
__imp_GetFileVersionInfoSizeW@8:
	.rva	ID2

# Import Lookup Table
	.section	.idata$4
	.rva	ID2
# Hint/Name table
	.section	.idata$6
ID2:	.short	2
	.asciz	"GetFileVersionInfoSizeW"  */

static char *
make_label (const char *prefix, const char *name)
{
  int len = strlen (ASM_PREFIX (name)) + strlen (prefix) + strlen (name);
  char *copy = xmalloc (len + 1);

  strcpy (copy, ASM_PREFIX (name));
  strcat (copy, prefix);
  strcat (copy, name);
  return copy;
}

static char *
make_imp_label (bfd *abfd, const char *prefix, const char *name)
{
  int len;
  char *copy;

  if (name[0] == '@')
    {
      len = strlen (prefix) + strlen (name);
      copy = bfd_xalloc (abfd, len + 1);
      strcpy (copy, prefix);
      strcat (copy, name);
    }
  else
    {
      len = strlen (ASM_PREFIX (name)) + strlen (prefix) + strlen (name);
      copy = bfd_xalloc (abfd, len + 1);
      strcpy (copy, prefix);
      strcat (copy, ASM_PREFIX (name));
      strcat (copy, name);
    }
  return copy;
}

static bfd *
make_one_lib_file (export_type *exp, int i, int delay)
{
  sinfo *const secdata = delay ? secdata_delay : secdata_plain;
  char *outname = TMP_STUB;
  size_t name_len = strlen (outname);
  sprintf (outname + name_len - 7, "%05d.o", i);

  bfd *abfd = bfd_openw (outname, HOW_BFD_WRITE_TARGET);
  if (!abfd)
    /* xgettext:c-format */
    fatal (_("bfd_open failed open stub file: %s: %s"),
	   outname, bfd_get_errmsg ());

  /* xgettext:c-format */
  inform (_("Creating stub file: %s"), outname);

  bfd_set_format (abfd, bfd_object);
  bfd_set_arch_mach (abfd, HOW_BFD_ARCH, 0);

#ifdef DLLTOOL_ARM
  if (machine == MARM_INTERWORK || machine == MTHUMB)
    bfd_set_private_flags (abfd, F_INTERWORK);
#endif

  /* First make symbols for the sections.  */
  flagword applicable = bfd_applicable_section_flags (abfd);
#ifndef EXTRA
#define EXTRA    0
#endif
  asymbol *ptrs[NSECS + 4 + EXTRA + 1];
  int oidx = 0;
  for (i = 0; i < NSECS; i++)
    {
      sinfo *si = secdata + i;

      if (si->id != i)
	abort ();
      si->sec = bfd_make_section_old_way (abfd, si->name);
      bfd_set_section_flags (si->sec, si->flags & applicable);

      bfd_set_section_alignment (si->sec, si->align);
      si->sec->output_section = si->sec;
      si->sym = bfd_make_empty_symbol(abfd);
      si->sym->name = si->sec->name;
      si->sym->section = si->sec;
      si->sym->flags = BSF_LOCAL;
      si->sym->value = 0;
      ptrs[oidx] = si->sym;
      si->sympp = ptrs + oidx;
      si->size = 0;
      si->data = NULL;

      oidx++;
    }

  if (! exp->data)
    {
      asymbol *exp_label = bfd_make_empty_symbol (abfd);
      exp_label->name = make_imp_label (abfd, "", exp->name);
      exp_label->section = secdata[TEXT].sec;
      exp_label->flags = BSF_GLOBAL;
      exp_label->value = 0;

#ifdef DLLTOOL_ARM
      if (machine == MTHUMB)
	bfd_coff_set_symbol_class (abfd, exp_label, C_THUMBEXTFUNC);
#endif
      ptrs[oidx++] = exp_label;
    }

  /* Generate imp symbols with one underscore for Microsoft
     compatibility, and with two underscores for backward
     compatibility with old versions of cygwin.  */
  asymbol *iname = NULL;
  if (create_compat_implib)
    {
      iname = bfd_make_empty_symbol (abfd);
      iname->name = make_imp_label (abfd, "___imp", exp->name);
      iname->section = secdata[IDATA5].sec;
      iname->flags = BSF_GLOBAL;
      iname->value = 0;
    }

  asymbol *iname2 = bfd_make_empty_symbol (abfd);
  iname2->name = make_imp_label (abfd, "__imp_", exp->name);
  iname2->section = secdata[IDATA5].sec;
  iname2->flags = BSF_GLOBAL;
  iname2->value = 0;

  asymbol *iname_lab = bfd_make_empty_symbol (abfd);
  iname_lab->name = head_label;
  iname_lab->section = bfd_und_section_ptr;
  iname_lab->flags = 0;
  iname_lab->value = 0;

  asymbol **iname_pp = ptrs + oidx;
  if (create_compat_implib)
    ptrs[oidx++] = iname;
  ptrs[oidx++] = iname2;

  asymbol **iname_lab_pp = ptrs + oidx;
  ptrs[oidx++] = iname_lab;

  ptrs[oidx] = 0;

  for (i = 0; i < NSECS; i++)
    {
      sinfo *si = secdata + i;
      asection *sec = si->sec;
      arelent *rel, *rel2 = 0, *rel3 = 0;
      arelent **rpp;

      switch (i)
	{
	case TEXT:
	  if (! exp->data)
	    {
	      unsigned int rpp_len;

	      si->size = HOW_JTAB_SIZE;
	      si->data = bfd_xalloc (abfd, HOW_JTAB_SIZE);
	      memcpy (si->data, HOW_JTAB, HOW_JTAB_SIZE);

	      /* Add the reloc into idata$5.  */
	      rel = bfd_xalloc (abfd, sizeof (arelent));

	      rpp_len = delay ? 4 : 2;

	      if (machine == MAARCH64)
		rpp_len++;

	      rpp = bfd_xalloc (abfd, sizeof (arelent *) * rpp_len);
	      rpp[0] = rel;
	      rpp[1] = 0;

	      rel->address = HOW_JTAB_ROFF;
	      rel->addend = 0;

	      if (delay)
		{
		  rel2 = bfd_xalloc (abfd, sizeof (arelent));
		  rpp[1] = rel2;
		  rel2->address = HOW_JTAB_ROFF2;
		  rel2->addend = 0;
		  rel3 = bfd_xalloc (abfd, sizeof (arelent));
		  rpp[2] = rel3;
		  rel3->address = HOW_JTAB_ROFF3;
		  rel3->addend = 0;
		  rpp[3] = 0;
		}

	      if (machine == MX86)
		{
		  rel->howto = bfd_reloc_type_lookup (abfd,
						      BFD_RELOC_32_PCREL);
		  rel->sym_ptr_ptr = iname_pp;
		}
	      else if (machine == MAARCH64)
		{
		  arelent *rel_add;

		  rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_AARCH64_ADR_HI21_NC_PCREL);
		  rel->sym_ptr_ptr = secdata[IDATA5].sympp;

		  rel_add = bfd_xalloc (abfd, sizeof (arelent));
		  rel_add->address = 4;
		  rel_add->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_AARCH64_ADD_LO12);
		  rel_add->sym_ptr_ptr = secdata[IDATA5].sympp;
		  rel_add->addend = 0;

		  rpp[rpp_len - 2] = rel_add;
		  rpp[rpp_len - 1] = 0;
		}
	      else
		{
		  rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_32);
		  rel->sym_ptr_ptr = secdata[IDATA5].sympp;
		}

	      if (delay)
		{
		  if (machine == MX86)
		    rel2->howto = bfd_reloc_type_lookup (abfd,
							 BFD_RELOC_32_PCREL);
		  else
		    rel2->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_32);
		  rel2->sym_ptr_ptr = rel->sym_ptr_ptr;
		  rel3->howto = bfd_reloc_type_lookup (abfd,
						       BFD_RELOC_32_PCREL);
		  rel3->sym_ptr_ptr = iname_lab_pp;
		}

	      sec->orelocation = rpp;
	      sec->reloc_count = rpp_len - 1;
	    }
	  break;

	case IDATA5:
	  if (delay)
	    {
	      si->size = create_for_pep ? 8 : 4;
	      si->data = bfd_xalloc (abfd, si->size);
	      sec->reloc_count = 1;
	      memset (si->data, 0, si->size);
	      /* Point after jmp [__imp_...] instruction.  */
	      si->data[0] = 6;
	      rel = bfd_xalloc (abfd, sizeof (arelent));
	      rpp = bfd_xalloc (abfd, sizeof (arelent *) * 2);
	      rpp[0] = rel;
	      rpp[1] = 0;
	      rel->address = 0;
	      rel->addend = 0;
	      if (create_for_pep)
		rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_64);
	      else
		rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_32);
	      rel->sym_ptr_ptr = secdata[TEXT].sympp;
	      sec->orelocation = rpp;
	      break;
	    }
	  /* Fall through.  */

	case IDATA4:
	  /* An idata$4 or idata$5 is one word long, and has an
	     rva to idata$6.  */

	  if (create_for_pep)
	    {
	      si->data = bfd_xalloc (abfd, 8);
	      si->size = 8;
	      if (exp->noname)
		{
		  si->data[0] = exp->ordinal ;
		  si->data[1] = exp->ordinal >> 8;
		  si->data[2] = exp->ordinal >> 16;
		  si->data[3] = exp->ordinal >> 24;
		  si->data[4] = 0;
		  si->data[5] = 0;
		  si->data[6] = 0;
		  si->data[7] = 0x80;
		}
	      else
		{
		  sec->reloc_count = 1;
		  memset (si->data, 0, si->size);
		  rel = bfd_xalloc (abfd, sizeof (arelent));
		  rpp = bfd_xalloc (abfd, sizeof (arelent *) * 2);
		  rpp[0] = rel;
		  rpp[1] = 0;
		  rel->address = 0;
		  rel->addend = 0;
		  rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_RVA);
		  rel->sym_ptr_ptr = secdata[IDATA6].sympp;
		  sec->orelocation = rpp;
		}
	    }
	  else
	    {
	      si->data = bfd_xalloc (abfd, 4);
	      si->size = 4;

	      if (exp->noname)
		{
		  si->data[0] = exp->ordinal ;
		  si->data[1] = exp->ordinal >> 8;
		  si->data[2] = exp->ordinal >> 16;
		  si->data[3] = 0x80;
		}
	      else
		{
		  sec->reloc_count = 1;
		  memset (si->data, 0, si->size);
		  rel = bfd_xalloc (abfd, sizeof (arelent));
		  rpp = bfd_xalloc (abfd, sizeof (arelent *) * 2);
		  rpp[0] = rel;
		  rpp[1] = 0;
		  rel->address = 0;
		  rel->addend = 0;
		  rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_RVA);
		  rel->sym_ptr_ptr = secdata[IDATA6].sympp;
		  sec->orelocation = rpp;
		}
	    }
	  break;

	case IDATA6:
	  if (!exp->noname)
	    {
	      /* This used to add 1 to exp->hint.  I don't know
		 why it did that, and it does not match what I see
		 in programs compiled with the MS tools.  */
	      int idx = exp->hint;
	      const char *xname = (exp->its_name ? exp->its_name
				   : xlate (exp->import_name));
	      si->size = strlen (xname) + 3;
	      si->data = bfd_xalloc (abfd, si->size);
	      si->data[0] = idx & 0xff;
	      si->data[1] = idx >> 8;
	      memcpy (si->data + 2, xname, si->size - 2);
	      if (!exp->its_name)
		free ((char *) xname);
	    }
	  break;
	case IDATA7:
	  if (delay)
	    break;
	  si->size = 4;
	  si->data = bfd_xalloc (abfd, 4);
	  memset (si->data, 0, si->size);
	  rel = bfd_xalloc (abfd, sizeof (arelent));
	  rpp = bfd_xalloc (abfd, sizeof (arelent *) * 2);
	  rpp[0] = rel;
	  rel->address = 0;
	  rel->addend = 0;
	  rel->howto = bfd_reloc_type_lookup (abfd, BFD_RELOC_RVA);
	  rel->sym_ptr_ptr = iname_lab_pp;
	  sec->orelocation = rpp;
	  sec->reloc_count = 1;
	  break;
	}
    }

  {
    bfd_vma vma = 0;
    /* Size up all the sections.  */
    for (i = 0; i < NSECS; i++)
      {
	sinfo *si = secdata + i;

	bfd_set_section_size (si->sec, si->size);
	bfd_set_section_vma (si->sec, vma);
      }
  }
  /* Write them out.  */
  for (i = 0; i < NSECS; i++)
    {
      sinfo *si = secdata + i;

      if (i == IDATA5 && no_idata5)
	continue;

      if (i == IDATA4 && no_idata4)
	continue;

      bfd_set_section_contents (abfd, si->sec,
				si->data, 0,
				si->size);
    }

  bfd_set_symtab (abfd, ptrs, oidx);
  bfd_close (abfd);
  abfd = bfd_openr (outname, HOW_BFD_READ_TARGET);
  if (!abfd)
    /* xgettext:c-format */
    fatal (_("bfd_open failed reopen stub file: %s: %s"),
	   outname, bfd_get_errmsg ());

  return abfd;
}

static bfd *
make_head (void)
{
  FILE *f = fopen (TMP_HEAD_S, FOPEN_WT);
  bfd *abfd;

  if (f == NULL)
    {
      fatal (_("failed to open temporary head file: %s"), TMP_HEAD_S);
      return NULL;
    }

  temp_file_to_remove[TEMP_HEAD_FILE] = TMP_HEAD_S;

  fprintf (f, "%s IMAGE_IMPORT_DESCRIPTOR\n", ASM_C);
  fprintf (f, "\t.section\t.idata$2\n");

  fprintf (f, "\t%s\t%s\n", ASM_GLOBAL, head_label);

  fprintf (f, "%s:\n", head_label);

  fprintf (f, "\t%shname%s\t%sPtr to image import by name list\n",
	   ASM_RVA_BEFORE, ASM_RVA_AFTER, ASM_C);

  fprintf (f, "\t%sthis should be the timestamp, but NT sometimes\n", ASM_C);
  fprintf (f, "\t%sdoesn't load DLLs when this is set.\n", ASM_C);
  fprintf (f, "\t%s\t0\t%s loaded time\n", ASM_LONG, ASM_C);
  fprintf (f, "\t%s\t0\t%s Forwarder chain\n", ASM_LONG, ASM_C);
  fprintf (f, "\t%s__%s_iname%s\t%s imported dll's name\n",
	   ASM_RVA_BEFORE,
	   imp_name_lab,
	   ASM_RVA_AFTER,
	   ASM_C);
  fprintf (f, "\t%sfthunk%s\t%s pointer to firstthunk\n",
	   ASM_RVA_BEFORE,
	   ASM_RVA_AFTER, ASM_C);

  fprintf (f, "%sStuff for compatibility\n", ASM_C);

  if (!no_idata5)
    {
      fprintf (f, "\t.section\t.idata$5\n");
      if (use_nul_prefixed_import_tables)
	{
	  if (create_for_pep)
	    fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
	  else
	    fprintf (f, "\t%s\t0\n", ASM_LONG);
	}
      fprintf (f, "fthunk:\n");
    }

  if (!no_idata4)
    {
      fprintf (f, "\t.section\t.idata$4\n");
      if (use_nul_prefixed_import_tables)
	{
	  if (create_for_pep)
	    fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
	  else
	    fprintf (f, "\t%s\t0\n", ASM_LONG);
	}
      fprintf (f, "hname:\n");
    }

  fclose (f);

  assemble_file (TMP_HEAD_S, TMP_HEAD_O);

  abfd = bfd_openr (TMP_HEAD_O, HOW_BFD_READ_TARGET);
  if (abfd == NULL)
    /* xgettext:c-format */
    fatal (_("failed to open temporary head file: %s: %s"),
	   TMP_HEAD_O, bfd_get_errmsg ());

  temp_file_to_remove[TEMP_HEAD_O_FILE] = TMP_HEAD_O;
  return abfd;
}

static bfd *
make_delay_head (void)
{
  FILE *f = fopen (TMP_HEAD_S, FOPEN_WT);
  bfd *abfd;

  if (f == NULL)
    {
      fatal (_("failed to open temporary head file: %s"), TMP_HEAD_S);
      return NULL;
    }

  temp_file_to_remove[TEMP_HEAD_FILE] = TMP_HEAD_S;

  /* Output the __tailMerge__xxx function */
  fprintf (f, "%s Import trampoline\n", ASM_C);
  fprintf (f, "\t.section\t.text\n");
  fprintf (f, "\t%s\t%s\n", ASM_GLOBAL, head_label);
  if (HOW_SEH)
    fprintf (f, "\t.seh_proc\t%s\n", head_label);
  fprintf (f, "%s:\n", head_label);
  fprintf (f, mtable[machine].trampoline, imp_name_lab);
  if (HOW_SEH)
    fprintf (f, "\t.seh_endproc\n");

  /* Output the delay import descriptor */
  fprintf (f, "\n%s DELAY_IMPORT_DESCRIPTOR\n", ASM_C);
  fprintf (f, ".section\t.didat$2\n");
  fprintf (f, "%s __DELAY_IMPORT_DESCRIPTOR_%s\n", ASM_GLOBAL,imp_name_lab);
  fprintf (f, "__DELAY_IMPORT_DESCRIPTOR_%s:\n", imp_name_lab);
  fprintf (f, "\t%s 1\t%s grAttrs\n", ASM_LONG, ASM_C);
  fprintf (f, "\t%s__%s_iname%s\t%s rvaDLLName\n",
	   ASM_RVA_BEFORE, imp_name_lab, ASM_RVA_AFTER, ASM_C);
  fprintf (f, "\t%s__DLL_HANDLE_%s%s\t%s rvaHmod\n",
	   ASM_RVA_BEFORE, imp_name_lab, ASM_RVA_AFTER, ASM_C);
  fprintf (f, "\t%s__IAT_%s%s\t%s rvaIAT\n",
	   ASM_RVA_BEFORE, imp_name_lab, ASM_RVA_AFTER, ASM_C);
  fprintf (f, "\t%s__INT_%s%s\t%s rvaINT\n",
	   ASM_RVA_BEFORE, imp_name_lab, ASM_RVA_AFTER, ASM_C);
  fprintf (f, "\t%s\t0\t%s rvaBoundIAT\n", ASM_LONG, ASM_C);
  fprintf (f, "\t%s\t0\t%s rvaUnloadIAT\n", ASM_LONG, ASM_C);
  fprintf (f, "\t%s\t0\t%s dwTimeStamp\n", ASM_LONG, ASM_C);

  /* Output the dll_handle */
  fprintf (f, "\n.section .data\n");
  fprintf (f, "__DLL_HANDLE_%s:\n", imp_name_lab);
  fprintf (f, "\t%s\t0\t%s Handle\n", ASM_LONG, ASM_C);
  if (create_for_pep)
    fprintf (f, "\t%s\t0\n", ASM_LONG);
  fprintf (f, "\n");

  fprintf (f, "%sStuff for compatibility\n", ASM_C);

  if (!no_idata5)
    {
      fprintf (f, "\t.section\t.didat$5\n");
      if (use_nul_prefixed_import_tables)
	{
	  if (create_for_pep)
	    fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
	  else
	    fprintf (f, "\t%s\t0\n", ASM_LONG);
	}
      fprintf (f, "__IAT_%s:\n", imp_name_lab);
    }

  if (!no_idata4)
    {
      fprintf (f, "\t.section\t.didat$4\n");
      if (use_nul_prefixed_import_tables)
	{
	  fprintf (f, "\t%s\t0\n", ASM_LONG);
	  if (create_for_pep)
	    fprintf (f, "\t%s\t0\n", ASM_LONG);
	}
      fprintf (f, "__INT_%s:\n", imp_name_lab);
    }

  fclose (f);

  assemble_file (TMP_HEAD_S, TMP_HEAD_O);

  abfd = bfd_openr (TMP_HEAD_O, HOW_BFD_READ_TARGET);
  if (abfd == NULL)
    /* xgettext:c-format */
    fatal (_("failed to open temporary head file: %s: %s"),
	   TMP_HEAD_O, bfd_get_errmsg ());

  temp_file_to_remove[TEMP_HEAD_O_FILE] = TMP_HEAD_O;
  return abfd;
}

static bfd *
make_tail (void)
{
  FILE *f = fopen (TMP_TAIL_S, FOPEN_WT);
  bfd *abfd;

  if (f == NULL)
    {
      fatal (_("failed to open temporary tail file: %s"), TMP_TAIL_S);
      return NULL;
    }

  temp_file_to_remove[TEMP_TAIL_FILE] = TMP_TAIL_S;

  if (!no_idata4)
    {
      fprintf (f, "\t.section\t.idata$4\n");
      if (create_for_pep)
	fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
      else
	fprintf (f, "\t%s\t0\n", ASM_LONG); /* NULL terminating list.  */
    }

  if (!no_idata5)
    {
      fprintf (f, "\t.section\t.idata$5\n");
      if (create_for_pep)
	fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
      else
	fprintf (f, "\t%s\t0\n", ASM_LONG); /* NULL terminating list.  */
    }

  fprintf (f, "\t.section\t.idata$7\n");
  fprintf (f, "\t%s\t__%s_iname\n", ASM_GLOBAL, imp_name_lab);
  fprintf (f, "__%s_iname:\t%s\t\"%s\"\n",
	   imp_name_lab, ASM_TEXT, dll_name);

  fclose (f);

  assemble_file (TMP_TAIL_S, TMP_TAIL_O);

  abfd = bfd_openr (TMP_TAIL_O, HOW_BFD_READ_TARGET);
  if (abfd == NULL)
    /* xgettext:c-format */
    fatal (_("failed to open temporary tail file: %s: %s"),
	   TMP_TAIL_O, bfd_get_errmsg ());

  temp_file_to_remove[TEMP_TAIL_O_FILE] = TMP_TAIL_O;
  return abfd;
}

static bfd *
make_delay_tail (void)
{
  FILE *f = fopen (TMP_TAIL_S, FOPEN_WT);
  bfd *abfd;

  if (f == NULL)
    {
      fatal (_("failed to open temporary tail file: %s"), TMP_TAIL_S);
      return NULL;
    }

  temp_file_to_remove[TEMP_TAIL_FILE] = TMP_TAIL_S;

  if (!no_idata4)
    {
      fprintf (f, "\t.section\t.didat$4\n");
      if (create_for_pep)
	fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
      else
	fprintf (f, "\t%s\t0\n", ASM_LONG); /* NULL terminating list.  */
    }

  if (!no_idata5)
    {
      fprintf (f, "\t.section\t.didat$5\n");
      if (create_for_pep)
	fprintf (f, "\t%s\t0\n\t%s\t0\n", ASM_LONG, ASM_LONG);
      else
	fprintf (f, "\t%s\t0\n", ASM_LONG); /* NULL terminating list.  */
    }

  fprintf (f, "\t.section\t.didat$7\n");
  fprintf (f, "\t%s\t__%s_iname\n", ASM_GLOBAL, imp_name_lab);
  fprintf (f, "__%s_iname:\t%s\t\"%s\"\n",
	   imp_name_lab, ASM_TEXT, dll_name);

  fclose (f);

  assemble_file (TMP_TAIL_S, TMP_TAIL_O);

  abfd = bfd_openr (TMP_TAIL_O, HOW_BFD_READ_TARGET);
  if (abfd == NULL)
    /* xgettext:c-format */
    fatal (_("failed to open temporary tail file: %s: %s"),
	   TMP_TAIL_O, bfd_get_errmsg ());

  temp_file_to_remove[TEMP_TAIL_O_FILE] = TMP_TAIL_O;
  return abfd;
}

static void
gen_lib_file (int delay)
{
  int i;
  export_type *exp;
  bfd *ar_head;
  bfd *ar_tail;
  bfd *outarch;
  bfd * head  = 0;

  unlink (imp_name);

  outarch = bfd_openw (imp_name, HOW_BFD_WRITE_TARGET);

  if (!outarch)
    /* xgettext:c-format */
    fatal (_("Can't create .lib file: %s: %s"),
	   imp_name, bfd_get_errmsg ());

  /* xgettext:c-format */
  inform (_("Creating library file: %s"), imp_name);

  xatexit (unlink_temp_files);

  bfd_set_format (outarch, bfd_archive);
  outarch->has_armap = 1;
  outarch->is_thin_archive = 0;

  if (deterministic)
    outarch->flags |= BFD_DETERMINISTIC_OUTPUT;

  /* Work out a reasonable size of things to put onto one line.  */
  if (delay)
    {
      ar_head = make_delay_head ();
      ar_tail = make_delay_tail();
    }
  else
    {
      ar_head = make_head ();
      ar_tail = make_tail();
    }

  if (ar_head == NULL || ar_tail == NULL)
    return;

  for (i = 0; (exp = d_exports_lexically[i]); i++)
    {
      bfd *n;
      /* Don't add PRIVATE entries to import lib.  */
      if (exp->private)
	continue;
      n = make_one_lib_file (exp, i, delay);
      n->archive_next = head;
      head = n;
      if (ext_prefix_alias)
	{
	  export_type alias_exp;

	  assert (i < PREFIX_ALIAS_BASE);
	  alias_exp.name = make_imp_label (outarch, ext_prefix_alias, exp->name);
	  alias_exp.internal_name = exp->internal_name;
	  alias_exp.its_name = exp->its_name;
	  alias_exp.import_name = exp->name;
	  alias_exp.ordinal = exp->ordinal;
	  alias_exp.constant = exp->constant;
	  alias_exp.noname = exp->noname;
	  alias_exp.private = exp->private;
	  alias_exp.data = exp->data;
	  alias_exp.hint = exp->hint;
	  alias_exp.forward = exp->forward;
	  alias_exp.next = exp->next;
	  n = make_one_lib_file (&alias_exp, i + PREFIX_ALIAS_BASE, delay);
	  n->archive_next = head;
	  head = n;
	}
    }

  /* Now stick them all into the archive.  */
  ar_head->archive_next = head;
  ar_tail->archive_next = ar_head;
  head = ar_tail;

  if (! bfd_set_archive_head (outarch, head))
    bfd_fatal ("bfd_set_archive_head");

  if (! bfd_close (outarch))
    bfd_fatal (imp_name);

  /* Delete all the temp files.  */
  unlink_temp_files ();

  if (dontdeltemps < 2)
    {
      char *name = TMP_STUB;
      size_t name_len = strlen (name);

      for (i = 0; (exp = d_exports_lexically[i]); i++)
	{
	  /* Don't delete non-existent stubs for PRIVATE entries.  */
	  if (exp->private)
	    continue;
	  sprintf (name + name_len - 7, "%05d.o", i);
	  if (unlink (name) < 0)
	    /* xgettext:c-format */
	    non_fatal (_("cannot delete %s: %s"), name, strerror (errno));
	  if (ext_prefix_alias)
	    {
	      sprintf (name + name_len - 7, "%05d.o", i + PREFIX_ALIAS_BASE);
	      if (unlink (name) < 0)
		/* xgettext:c-format */
		non_fatal (_("cannot delete %s: %s"), name, strerror (errno));
	    }
	}
    }

  inform (_("Created lib file"));
}

/* Append a copy of data (cast to char *) to list.  */

static void
dll_name_list_append (dll_name_list_type * list, bfd_byte * data)
{
  dll_name_list_node_type * entry;

  /* Error checking.  */
  if (! list || ! list->tail)
    return;

  /* Allocate new node.  */
  entry = ((dll_name_list_node_type *)
	   xmalloc (sizeof (dll_name_list_node_type)));

  /* Initialize its values.  */
  entry->dllname = xstrdup ((char *) data);
  entry->next = NULL;

  /* Add to tail, and move tail.  */
  list->tail->next = entry;
  list->tail = entry;
}

/* Count the number of entries in list.  */

static int
dll_name_list_count (dll_name_list_type * list)
{
  dll_name_list_node_type * p;
  int count = 0;

  /* Error checking.  */
  if (! list || ! list->head)
    return 0;

  p = list->head;

  while (p && p->next)
    {
      count++;
      p = p->next;
    }
  return count;
}

/* Print each entry in list to stdout.  */

static void
dll_name_list_print (dll_name_list_type * list)
{
  dll_name_list_node_type * p;

  /* Error checking.  */
  if (! list || ! list->head)
    return;

  p = list->head;

  while (p && p->next && p->next->dllname && *p->next->dllname)
    {
      printf ("%s\n", p->next->dllname);
      p = p->next;
    }
}

/* Free all entries in list, and list itself.  */

static void
dll_name_list_free (dll_name_list_type * list)
{
  if (list)
    {
      dll_name_list_free_contents (list->head);
      list->head = NULL;
      list->tail = NULL;
      free (list);
    }
}

/* Recursive function to free all nodes entry->next->next...
   as well as entry itself.  */

static void
dll_name_list_free_contents (dll_name_list_node_type * entry)
{
  while (entry)
    {
      dll_name_list_node_type *next = entry->next;
      free (entry->dllname);
      free (entry);
      entry = next;
    }
}

/* Allocate and initialize a dll_name_list_type object,
   including its sentinel node.  Caller is responsible
   for calling dll_name_list_free when finished with
   the list.  */

static dll_name_list_type *
dll_name_list_create (void)
{
  /* Allocate list.  */
  dll_name_list_type * list = xmalloc (sizeof (dll_name_list_type));

  /* Allocate and initialize sentinel node.  */
  list->head = xmalloc (sizeof (dll_name_list_node_type));
  list->head->dllname = NULL;
  list->head->next = NULL;

  /* Bookkeeping for empty list.  */
  list->tail = list->head;

  return list;
}

/* Search the symbol table of the suppled BFD for a symbol whose name matches
   OBJ (where obj is cast to const char *).  If found, set global variable
   identify_member_contains_symname_result TRUE.  It is the caller's
   responsibility to set the result variable FALSE before iterating with
   this function.  */

static void
identify_member_contains_symname (bfd  * abfd,
				  bfd  * archive_bfd ATTRIBUTE_UNUSED,
				  void * obj)
{
  long storage_needed;
  asymbol ** symbol_table;
  long number_of_symbols;
  long i;
  symname_search_data_type * search_data = (symname_search_data_type *) obj;

  /* If we already found the symbol in a different member,
     short circuit.  */
  if (search_data->found)
    return;

  storage_needed = bfd_get_symtab_upper_bound (abfd);
  if (storage_needed <= 0)
    return;

  symbol_table = xmalloc (storage_needed);
  number_of_symbols = bfd_canonicalize_symtab (abfd, symbol_table);
  if (number_of_symbols < 0)
    {
      free (symbol_table);
      return;
    }

  for (i = 0; i < number_of_symbols; i++)
    {
      if (strncmp (symbol_table[i]->name,
		   search_data->symname,
		   strlen (search_data->symname)) == 0)
	{
	  search_data->found = true;
	  break;
	}
    }
  free (symbol_table);
}

/* This is the main implementation for the --identify option.
   Given the name of an import library in identify_imp_name, first
   determine if the import library is a GNU binutils-style one (where
   the DLL name is stored in an .idata$7 section), or if it is a
   MS-style one (where the DLL name, along with much other data, is
   stored in the .idata$6 section).  We determine the style of import
   library by searching for the DLL-structure symbol inserted by MS
   tools: __NULL_IMPORT_DESCRIPTOR.

   Once we know which section to search, evaluate each section for the
   appropriate properties that indicate it may contain the name of the
   associated DLL (this differs depending on the style).  Add the contents
   of all sections which meet the criteria to a linked list of dll names.

   Finally, print them all to stdout. (If --identify-strict, an error is
   reported if more than one match was found).  */

static void
identify_dll_for_implib (void)
{
  bfd * abfd = NULL;
  int count = 0;
  identify_data_type identify_data;
  symname_search_data_type search_data;

  /* Initialize identify_data.  */
  identify_data.list = dll_name_list_create ();
  identify_data.ms_style_implib = false;

  /* Initialize search_data.  */
  search_data.symname = "__NULL_IMPORT_DESCRIPTOR";
  search_data.found = false;

  if (bfd_init () != BFD_INIT_MAGIC)
    fatal (_("fatal error: libbfd ABI mismatch"));

  abfd = bfd_openr (identify_imp_name, 0);
  if (abfd == NULL)
    /* xgettext:c-format */
    fatal (_("Can't open .lib file: %s: %s"),
	   identify_imp_name, bfd_get_errmsg ());

  if (! bfd_check_format (abfd, bfd_archive))
    {
      if (! bfd_close (abfd))
	bfd_fatal (identify_imp_name);

      fatal (_("%s is not a library"), identify_imp_name);
    }

  /* Detect if this a Microsoft import library.  */
  identify_search_archive (abfd,
			   identify_member_contains_symname,
			   (void *) &search_data);
  if (search_data.found)
    identify_data.ms_style_implib = true;

  /* Rewind the bfd.  */
  if (! bfd_close (abfd))
    bfd_fatal (identify_imp_name);
  abfd = bfd_openr (identify_imp_name, 0);
  if (abfd == NULL)
    bfd_fatal (identify_imp_name);

  if (!bfd_check_format (abfd, bfd_archive))
    {
      if (!bfd_close (abfd))
	bfd_fatal (identify_imp_name);

      fatal (_("%s is not a library"), identify_imp_name);
    }

  /* Now search for the dll name.  */
  identify_search_archive (abfd,
			   identify_search_member,
			   (void *) &identify_data);

  if (! bfd_close (abfd))
    bfd_fatal (identify_imp_name);

  count = dll_name_list_count (identify_data.list);
  if (count > 0)
    {
      if (identify_strict && count > 1)
	{
	  dll_name_list_free (identify_data.list);
	  identify_data.list = NULL;
	  fatal (_("Import library `%s' specifies two or more dlls"),
		 identify_imp_name);
	}
      dll_name_list_print (identify_data.list);
      dll_name_list_free (identify_data.list);
      identify_data.list = NULL;
    }
  else
    {
      dll_name_list_free (identify_data.list);
      identify_data.list = NULL;
      fatal (_("Unable to determine dll name for `%s' (not an import library?)"),
	     identify_imp_name);
    }
}

/* Loop over all members of the archive, applying the supplied function to
   each member that is a bfd_object.  The function will be called as if:
      func (member_bfd, abfd, user_storage)  */

static void
identify_search_archive (bfd * abfd,
			 void (* operation) (bfd *, bfd *, void *),
			 void * user_storage)
{
  bfd *last_arfile = NULL;

  while (1)
    {
      bfd *arfile = bfd_openr_next_archived_file (abfd, last_arfile);
      if (arfile == NULL
	  || arfile == last_arfile)
	{
	  if (arfile != NULL)
	    bfd_set_error (bfd_error_malformed_archive);
	  if (bfd_get_error () != bfd_error_no_more_archived_files)
	    bfd_fatal (bfd_get_filename (abfd));
	  break;
	}

      if (last_arfile != NULL)
	bfd_close (last_arfile);

      char **matching;
      if (bfd_check_format_matches (arfile, bfd_object, &matching))
	(*operation) (arfile, abfd, user_storage);
      else
	{
	  bfd_nonfatal (bfd_get_filename (arfile));
	  free (matching);
	}

      last_arfile = arfile;
    }

  if (last_arfile != NULL)
    bfd_close (last_arfile);
}

/* Call the identify_search_section() function for each section of this
   archive member.  */

static void
identify_search_member (bfd  *abfd,
			bfd  *archive_bfd ATTRIBUTE_UNUSED,
			void *obj)
{
  bfd_map_over_sections (abfd, identify_search_section, obj);
}

/* This predicate returns true if section->name matches the desired value.
   By default, this is .idata$7 (.idata$6 if the import library is
   ms-style).  */

static bool
identify_process_section_p (asection * section, bool ms_style_implib)
{
  static const char * SECTION_NAME = ".idata$7";
  static const char * MS_SECTION_NAME = ".idata$6";

  const char * section_name =
    (ms_style_implib ? MS_SECTION_NAME : SECTION_NAME);

  if (strcmp (section_name, section->name) == 0)
    return true;
  return false;
}

/* If *section has contents and its name is .idata$7 (.idata$6 if
   import lib ms-generated) -- and it satisfies several other constraints
   -- then add the contents of the section to obj->list.  */

static void
identify_search_section (bfd * abfd, asection * section, void * obj)
{
  bfd_byte *data = 0;
  bfd_size_type datasize;
  identify_data_type * identify_data = (identify_data_type *)obj;
  bool ms_style = identify_data->ms_style_implib;

  if ((section->flags & SEC_HAS_CONTENTS) == 0)
    return;

  if (! identify_process_section_p (section, ms_style))
    return;

  /* Binutils import libs seem distinguish the .idata$7 section that contains
     the DLL name from other .idata$7 sections by the absence of the
     SEC_RELOC flag.  */
  if (!ms_style && ((section->flags & SEC_RELOC) == SEC_RELOC))
    return;

  /* MS import libs seem to distinguish the .idata$6 section
     that contains the DLL name from other .idata$6 sections
     by the presence of the SEC_DATA flag.  */
  if (ms_style && ((section->flags & SEC_DATA) == 0))
    return;

  if ((datasize = bfd_section_size (section)) == 0)
    return;

  data = (bfd_byte *) xmalloc (datasize + 1);
  data[0] = '\0';

  bfd_get_section_contents (abfd, section, data, 0, datasize);
  data[datasize] = '\0';

  /* Use a heuristic to determine if data is a dll name.
     Possible to defeat this if (a) the library has MANY
     (more than 0x302f) imports, (b) it is an ms-style
     import library, but (c) it is buggy, in that the SEC_DATA
     flag is set on the "wrong" sections.  This heuristic might
     also fail to record a valid dll name if the dllname uses
     a multibyte or unicode character set (is that valid?).

     This heuristic is based on the fact that symbols names in
     the chosen section -- as opposed to the dll name -- begin
     at offset 2 in the data. The first two bytes are a 16bit
     little-endian count, and start at 0x0000. However, the dll
     name begins at offset 0 in the data. We assume that the
     dll name does not contain unprintable characters.   */
  if (data[0] != '\0' && ISPRINT (data[0])
      && (datasize < 2 || ISPRINT (data[1])))
    dll_name_list_append (identify_data->list, data);

  free (data);
}

/* Run through the information gathered from the .o files and the
   .def file and work out the best stuff.  */

static int
pfunc (const void *a, const void *b)
{
  export_type *ap = *(export_type **) a;
  export_type *bp = *(export_type **) b;

  if (ap->ordinal == bp->ordinal)
    return 0;

  /* Unset ordinals go to the bottom.  */
  if (ap->ordinal == -1)
    return 1;
  if (bp->ordinal == -1)
    return -1;
  return (ap->ordinal - bp->ordinal);
}

static int
nfunc (const void *a, const void *b)
{
  export_type *ap = *(export_type **) a;
  export_type *bp = *(export_type **) b;
  const char *an = ap->name;
  const char *bn = bp->name;
  if (ap->its_name)
    an = ap->its_name;
  if (bp->its_name)
    an = bp->its_name;
  if (killat)
    {
      an = (an[0] == '@') ? an + 1 : an;
      bn = (bn[0] == '@') ? bn + 1 : bn;
    }

  return (strcmp (an, bn));
}

static void
remove_null_names (export_type **ptr)
{
  int src;
  int dst;

  for (dst = src = 0; src < d_nfuncs; src++)
    {
      if (ptr[src])
	{
	  ptr[dst] = ptr[src];
	  dst++;
	}
    }
  d_nfuncs = dst;
}

static void
process_duplicates (export_type **d_export_vec)
{
  int more = 1;
  int i;

  while (more)
    {
      more = 0;
      /* Remove duplicates.  */
      qsort (d_export_vec, d_nfuncs, sizeof (export_type *), nfunc);

      for (i = 0; i < d_nfuncs - 1; i++)
	{
	  if (strcmp (d_export_vec[i]->name,
		      d_export_vec[i + 1]->name) == 0)
	    {
	      export_type *a = d_export_vec[i];
	      export_type *b = d_export_vec[i + 1];

	      more = 1;

	      /* xgettext:c-format */
	      inform (_("Warning, ignoring duplicate EXPORT %s %d,%d"),
		      a->name, a->ordinal, b->ordinal);

	      if (a->ordinal != -1
		  && b->ordinal != -1)
		/* xgettext:c-format */
		fatal (_("Error, duplicate EXPORT with ordinals: %s"),
		       a->name);

	      /* Merge attributes.  */
	      b->ordinal = a->ordinal > 0 ? a->ordinal : b->ordinal;
	      b->constant |= a->constant;
	      b->noname |= a->noname;
	      b->data |= a->data;
	      d_export_vec[i] = 0;
	    }

	  remove_null_names (d_export_vec);
	}
    }

  /* Count the names.  */
  for (i = 0; i < d_nfuncs; i++)
    if (!d_export_vec[i]->noname)
      d_named_nfuncs++;
}

static void
fill_ordinals (export_type **d_export_vec)
{
  int lowest = -1;
  int i;
  char *ptr;
  int size = 65536;

  qsort (d_export_vec, d_nfuncs, sizeof (export_type *), pfunc);

  /* Fill in the unset ordinals with ones from our range.  */
  ptr = (char *) xmalloc (size);

  memset (ptr, 0, size);

  /* Mark in our large vector all the numbers that are taken.  */
  for (i = 0; i < d_nfuncs; i++)
    {
      if (d_export_vec[i]->ordinal != -1)
	{
	  ptr[d_export_vec[i]->ordinal] = 1;

	  if (lowest == -1 || d_export_vec[i]->ordinal < lowest)
	    lowest = d_export_vec[i]->ordinal;
	}
    }

  /* Start at 1 for compatibility with MS toolchain.  */
  if (lowest == -1)
    lowest = 1;

  /* Now fill in ordinals where the user wants us to choose.  */
  for (i = 0; i < d_nfuncs; i++)
    {
      if (d_export_vec[i]->ordinal == -1)
	{
	  int j;

	  /* First try within or after any user supplied range.  */
	  for (j = lowest; j < size; j++)
	    if (ptr[j] == 0)
	      {
		ptr[j] = 1;
		d_export_vec[i]->ordinal = j;
		goto done;
	      }

	  /* Then try before the range.  */
	  for (j = lowest; j >0; j--)
	    if (ptr[j] == 0)
	      {
		ptr[j] = 1;
		d_export_vec[i]->ordinal = j;
		goto done;
	      }
	done:;
	}
    }

  free (ptr);

  /* And resort.  */
  qsort (d_export_vec, d_nfuncs, sizeof (export_type *), pfunc);

  /* Work out the lowest and highest ordinal numbers.  */
  if (d_nfuncs)
    {
      if (d_export_vec[0])
	d_low_ord = d_export_vec[0]->ordinal;
      if (d_export_vec[d_nfuncs-1])
	d_high_ord = d_export_vec[d_nfuncs-1]->ordinal;
    }
}

static void
mangle_defs (void)
{
  /* First work out the minimum ordinal chosen.  */
  export_type *exp;

  int i;
  int hint = 0;
  export_type **d_export_vec = xmalloc (sizeof (export_type *) * d_nfuncs);

  inform (_("Processing definitions"));

  for (i = 0, exp = d_exports; exp; i++, exp = exp->next)
    d_export_vec[i] = exp;

  process_duplicates (d_export_vec);
  fill_ordinals (d_export_vec);

  /* Put back the list in the new order.  */
  d_exports = 0;
  for (i = d_nfuncs - 1; i >= 0; i--)
    {
      d_export_vec[i]->next = d_exports;
      d_exports = d_export_vec[i];
    }
  free (d_export_vec);

  /* Build list in alpha order.  */
  d_exports_lexically = (export_type **)
    xmalloc (sizeof (export_type *) * (d_nfuncs + 1));

  for (i = 0, exp = d_exports; exp; i++, exp = exp->next)
    d_exports_lexically[i] = exp;

  d_exports_lexically[i] = 0;

  qsort (d_exports_lexically, i, sizeof (export_type *), nfunc);

  /* Fill exp entries with their hint values.  */
  for (i = 0; i < d_nfuncs; i++)
    if (!d_exports_lexically[i]->noname || show_allnames)
      d_exports_lexically[i]->hint = hint++;

  inform (_("Processed definitions"));
}

static void
usage (FILE *file, int status)
{
  /* xgetext:c-format */
  fprintf (file, _("Usage %s <option(s)> <object-file(s)>\n"), program_name);
  /* xgetext:c-format */
  fprintf (file, _("   -m --machine <machine>    Create as DLL for <machine>.  [default: %s]\n"), mname);
  fprintf (file, _("        possible <machine>: arm[_interwork], arm64, i386, mcore[-elf]{-le|-be}, thumb\n"));
  fprintf (file, _("   -e --output-exp <outname> Generate an export file.\n"));
  fprintf (file, _("   -l --output-lib <outname> Generate an interface library.\n"));
  fprintf (file, _("   -y --output-delaylib <outname> Create a delay-import library.\n"));
  fprintf (file, _("      --deterministic-libraries\n"));
  if (DEFAULT_AR_DETERMINISTIC)
    fprintf (file, _("                             Use zero for timestamps and uids/gids in output libraries (default)\n"));
  else
    fprintf (file, _("                             Use zero for timestamps and uids/gids in output libraries\n"));
  fprintf (file, _("      --non-deterministic-libraries\n"));
  if (DEFAULT_AR_DETERMINISTIC)
    fprintf (file, _("                             Use actual timestamps and uids/gids in output libraries\n"));
  else
    fprintf (file, _("                             Use actual timestamps and uids/gids in output libraries (default)\n"));
  fprintf (file, _("   -a --add-indirect         Add dll indirects to export file.\n"));
  fprintf (file, _("   -D --dllname <name>       Name of input dll to put into interface lib.\n"));
  fprintf (file, _("   -d --input-def <deffile>  Name of .def file to be read in.\n"));
  fprintf (file, _("   -z --output-def <deffile> Name of .def file to be created.\n"));
  fprintf (file, _("      --export-all-symbols   Export all symbols to .def\n"));
  fprintf (file, _("      --no-export-all-symbols  Only export listed symbols\n"));
  fprintf (file, _("      --exclude-symbols <list> Don't export <list>\n"));
  fprintf (file, _("      --no-default-excludes  Clear default exclude symbols\n"));
  fprintf (file, _("   -b --base-file <basefile> Read linker generated base file.\n"));
  fprintf (file, _("   -x --no-idata4            Don't generate idata$4 section.\n"));
  fprintf (file, _("   -c --no-idata5            Don't generate idata$5 section.\n"));
  fprintf (file, _("      --use-nul-prefixed-import-tables Use zero prefixed idata$4 and idata$5.\n"));
  fprintf (file, _("   -U --add-underscore       Add underscores to all symbols in interface library.\n"));
  fprintf (file, _("      --add-stdcall-underscore Add underscores to stdcall symbols in interface library.\n"));
  fprintf (file, _("      --no-leading-underscore All symbols shouldn't be prefixed by an underscore.\n"));
  fprintf (file, _("      --leading-underscore   All symbols should be prefixed by an underscore.\n"));
  fprintf (file, _("   -k --kill-at              Kill @<n> from exported names.\n"));
  fprintf (file, _("   -A --add-stdcall-alias    Add aliases without @<n>.\n"));
  fprintf (file, _("   -p --ext-prefix-alias <prefix> Add aliases with <prefix>.\n"));
  fprintf (file, _("   -S --as <name>            Use <name> for assembler.\n"));
  fprintf (file, _("   -f --as-flags <flags>     Pass <flags> to the assembler.\n"));
  fprintf (file, _("   -C --compat-implib        Create backward compatible import library.\n"));
  fprintf (file, _("   -n --no-delete            Keep temp files (repeat for extra preservation).\n"));
  fprintf (file, _("   -t --temp-prefix <prefix> Use <prefix> to construct temp file names.\n"));
  fprintf (file, _("   -I --identify <implib>    Report the name of the DLL associated with <implib>.\n"));
  fprintf (file, _("      --identify-strict      Causes --identify to report error when multiple DLLs.\n"));
  fprintf (file, _("   -v --verbose              Be verbose.\n"));
  fprintf (file, _("   -V --version              Display the program version.\n"));
  fprintf (file, _("   -h --help                 Display this information.\n"));
  fprintf (file, _("   @<file>                   Read options from <file>.\n"));
#ifdef DLLTOOL_MCORE_ELF
  fprintf (file, _("   -M --mcore-elf <outname>  Process mcore-elf object files into <outname>.\n"));
  fprintf (file, _("   -L --linker <name>        Use <name> as the linker.\n"));
  fprintf (file, _("   -F --linker-flags <flags> Pass <flags> to the linker.\n"));
#endif
  if (REPORT_BUGS_TO[0] && status == 0)
    fprintf (file, _("Report bugs to %s\n"), REPORT_BUGS_TO);
  exit (status);
}

/* 150 isn't special; it's just an arbitrary non-ASCII char value.  */
enum command_line_switch
{
  OPTION_EXPORT_ALL_SYMS = 150,
  OPTION_NO_EXPORT_ALL_SYMS,
  OPTION_EXCLUDE_SYMS,
  OPTION_NO_DEFAULT_EXCLUDES,
  OPTION_ADD_STDCALL_UNDERSCORE,
  OPTION_USE_NUL_PREFIXED_IMPORT_TABLES,
  OPTION_IDENTIFY_STRICT,
  OPTION_NO_LEADING_UNDERSCORE,
  OPTION_LEADING_UNDERSCORE,
  OPTION_DETERMINISTIC_LIBRARIES,
  OPTION_NON_DETERMINISTIC_LIBRARIES
};

static const struct option long_options[] =
{
  {"add-indirect", no_argument, NULL, 'a'},
  {"add-stdcall-alias", no_argument, NULL, 'A'},
  {"add-stdcall-underscore", no_argument, NULL, OPTION_ADD_STDCALL_UNDERSCORE},
  {"add-underscore", no_argument, NULL, 'U'},
  {"as", required_argument, NULL, 'S'},
  {"as-flags", required_argument, NULL, 'f'},
  {"base-file", required_argument, NULL, 'b'},
  {"compat-implib", no_argument, NULL, 'C'},
  {"def", required_argument, NULL, 'd'},     /* For compatibility with older versions.  */
  {"deterministic-libraries", no_argument, NULL, OPTION_DETERMINISTIC_LIBRARIES},
  {"dllname", required_argument, NULL, 'D'},
  {"exclude-symbols", required_argument, NULL, OPTION_EXCLUDE_SYMS},
  {"export-all-symbols", no_argument, NULL, OPTION_EXPORT_ALL_SYMS},
  {"ext-prefix-alias", required_argument, NULL, 'p'},
  {"help", no_argument, NULL, 'h'},
  {"identify", required_argument, NULL, 'I'},
  {"identify-strict", no_argument, NULL, OPTION_IDENTIFY_STRICT},
  {"input-def", required_argument, NULL, 'd'},
  {"kill-at", no_argument, NULL, 'k'},
  {"leading-underscore", no_argument, NULL, OPTION_LEADING_UNDERSCORE},
  {"machine", required_argument, NULL, 'm'},
  {"mcore-elf", required_argument, NULL, 'M'},
  {"no-default-excludes", no_argument, NULL, OPTION_NO_DEFAULT_EXCLUDES},
  {"no-delete", no_argument, NULL, 'n'},
  {"no-export-all-symbols", no_argument, NULL, OPTION_NO_EXPORT_ALL_SYMS},
  {"no-idata4", no_argument, NULL, 'x'},
  {"no-idata5", no_argument, NULL, 'c'},
  {"no-leading-underscore", no_argument, NULL, OPTION_NO_LEADING_UNDERSCORE},
  {"non-deterministic-libraries", no_argument, NULL, OPTION_NON_DETERMINISTIC_LIBRARIES},
  {"output-def", required_argument, NULL, 'z'},
  {"output-delaylib", required_argument, NULL, 'y'},
  {"output-exp", required_argument, NULL, 'e'},
  {"output-lib", required_argument, NULL, 'l'},
  {"temp-prefix", required_argument, NULL, 't'},
  {"use-nul-prefixed-import-tables", no_argument, NULL, OPTION_USE_NUL_PREFIXED_IMPORT_TABLES},
  {"verbose", no_argument, NULL, 'v'},
  {"version", no_argument, NULL, 'V'},
  {NULL,0,NULL,0}
};

int main (int, char **);

int
main (int ac, char **av)
{
  int c;
  int i;
  char *firstarg = 0;
  program_name = av[0];
  oav = av;

#ifdef HAVE_LC_MESSAGES
  setlocale (LC_MESSAGES, "");
#endif
  setlocale (LC_CTYPE, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  bfd_set_error_program_name (program_name);
  expandargv (&ac, &av);

  while ((c = getopt_long (ac, av,
#ifdef DLLTOOL_MCORE_ELF
			   "m:e:l:aD:d:z:b:xp:cCuUkAS:t:f:nI:vVHhM:L:F:",
#else
			   "m:e:l:y:aD:d:z:b:xp:cCuUkAS:t:f:nI:vVHh",
#endif
			   long_options, 0))
	 != EOF)
    {
      switch (c)
	{
	case OPTION_EXPORT_ALL_SYMS:
	  export_all_symbols = true;
	  break;
	case OPTION_NO_EXPORT_ALL_SYMS:
	  export_all_symbols = false;
	  break;
	case OPTION_EXCLUDE_SYMS:
	  add_excludes (optarg);
	  break;
	case OPTION_NO_DEFAULT_EXCLUDES:
	  do_default_excludes = false;
	  break;
	case OPTION_USE_NUL_PREFIXED_IMPORT_TABLES:
	  use_nul_prefixed_import_tables = true;
	  break;
	case OPTION_ADD_STDCALL_UNDERSCORE:
	  add_stdcall_underscore = 1;
	  break;
	case OPTION_NO_LEADING_UNDERSCORE:
	  leading_underscore = "";
	  break;
	case OPTION_LEADING_UNDERSCORE:
	  leading_underscore = "_";
	  break;
	case OPTION_IDENTIFY_STRICT:
	  identify_strict = 1;
	  break;
	case 'x':
	  no_idata4 = 1;
	  break;
	case 'c':
	  no_idata5 = 1;
	  break;
	case 'S':
	  as_name = optarg;
	  break;
	case 't':
	  tmp_prefix = optarg;
	  break;
	case 'f':
	  as_flags = optarg;
	  break;

	  /* Ignored for compatibility.  */
	case 'u':
	  break;
	case 'a':
	  add_indirect = 1;
	  break;
	case 'z':
	  output_def = fopen (optarg, FOPEN_WT);
	  if (!output_def)
	    /* xgettext:c-format */
	    fatal (_("Unable to open def-file: %s"), optarg);
	  break;
	case 'D':
	  dll_name = (char*) lbasename (optarg);
	  if (dll_name != optarg)
	    non_fatal (_("Path components stripped from dllname, '%s'."),
		       optarg);
	  break;
	case 'l':
	  imp_name = optarg;
	  break;
	case 'e':
	  exp_name = optarg;
	  break;
	case 'H':
	case 'h':
	  usage (stdout, 0);
	  break;
	case 'm':
	  mname = optarg;
	  break;
	case 'I':
	  identify_imp_name = optarg;
	  break;
	case 'v':
	  verbose = 1;
	  break;
	case 'V':
	  print_version (program_name);
	  break;
	case 'U':
	  add_underscore = 1;
	  break;
	case 'k':
	  killat = 1;
	  break;
	case 'A':
	  add_stdcall_alias = 1;
	  break;
	case 'p':
	  ext_prefix_alias = optarg;
	  break;
	case 'd':
	  def_file = optarg;
	  break;
	case 'n':
	  dontdeltemps++;
	  break;
	case 'b':
	  base_file = fopen (optarg, FOPEN_RB);

	  if (!base_file)
	    /* xgettext:c-format */
	    fatal (_("Unable to open base-file: %s"), optarg);

	  break;
#ifdef DLLTOOL_MCORE_ELF
	case 'M':
	  mcore_elf_out_file = optarg;
	  break;
	case 'L':
	  mcore_elf_linker = optarg;
	  break;
	case 'F':
	  mcore_elf_linker_flags = optarg;
	  break;
#endif
	case 'C':
	  create_compat_implib = 1;
	  break;
	case 'y':
	  delayimp_name = optarg;
	  break;
	case OPTION_DETERMINISTIC_LIBRARIES:
	  deterministic = true;
	  break;
	case OPTION_NON_DETERMINISTIC_LIBRARIES:
	  deterministic = false;
	  break;
	default:
	  usage (stderr, 1);
	  break;
	}
    }

  for (i = 0; mtable[i].type; i++)
    if (strcmp (mtable[i].type, mname) == 0)
      break;

  if (!mtable[i].type)
    /* xgettext:c-format */
    fatal (_("Machine '%s' not supported"), mname);

  machine = i;

  /* Check if we generated PE+.  */
  create_for_pep = (strcmp (mname, "i386:x86-64") == 0
		    || strcmp (mname, "arm64") == 0);

  /* Check the default underscore */
  if (leading_underscore == NULL)
    {
      int u;
      static char underscore[2];
      bfd_get_target_info (mtable[machine].how_bfd_target, NULL,
			   NULL, &u, NULL);
      if (u == -1)
	u = 0;
      underscore[0] = u;
      underscore[1] = 0;
      leading_underscore = underscore;
    }

  if (!dll_name && exp_name)
    {
      /* If we are inferring dll_name from exp_name,
	 strip off any path components, without emitting
	 a warning.  */
      const char* exp_basename = lbasename (exp_name);
      const int len = strlen (exp_basename) + 5;
      dll_name = xmalloc (len);
      strcpy (dll_name, exp_basename);
      strcat (dll_name, ".dll");
      dll_name_set_by_exp_name = 1;
    }

  if (as_name == NULL)
    as_name = deduce_name ("as");

  /* Don't use the default exclude list if we're reading only the
     symbols in the .drectve section.  The default excludes are meant
     to avoid exporting DLL entry point and Cygwin32 impure_ptr.  */
  if (! export_all_symbols)
    do_default_excludes = false;

  if (do_default_excludes)
    set_default_excludes ();

  if (def_file)
    process_def_file (def_file);

  while (optind < ac)
    {
      if (!firstarg)
	firstarg = av[optind];
      scan_obj_file (av[optind]);
      optind++;
    }

  if (tmp_prefix == NULL)
    {
      /* If possible use a deterministic prefix.  */
      const char *input = imp_name ? imp_name : delayimp_name;
      if (input && strlen (input) + sizeof ("_snnnnn.o") - 1 <= NAME_MAX)
	{
	  tmp_prefix = xasprintf ("%s_", input);
	  for (i = 0; tmp_prefix[i]; i++)
	    if (!ISALNUM (tmp_prefix[i]))
	      tmp_prefix[i] = '_';
	}
      else
	tmp_prefix = prefix_encode ("d", getpid ());
    }

  mangle_defs ();

  if (exp_name)
    gen_exp_file ();

  if (imp_name)
    {
      /* Make imp_name safe for use as a label.  */
      char *p;

      imp_name_lab = xstrdup (imp_name);
      for (p = imp_name_lab; *p; p++)
	{
	  if (!ISALNUM (*p))
	    *p = '_';
	}
      head_label = make_label ("_head_", imp_name_lab);
      gen_lib_file (0);
    }

  if (delayimp_name)
    {
      /* Make delayimp_name safe for use as a label.  */
      char *p;

      if (mtable[machine].how_dljtab == 0)
	{
	  inform (_("Warning, machine type (%d) not supported for "
		    "delayimport."), machine);
	}
      else
	{
	  killat = 1;
	  imp_name = delayimp_name;
	  imp_name_lab = xstrdup (imp_name);
	  for (p = imp_name_lab; *p; p++)
	    {
	      if (!ISALNUM (*p))
		*p = '_';
	    }
	  head_label = make_label ("__tailMerge_", imp_name_lab);
	  gen_lib_file (1);
	}
    }

  if (output_def)
    gen_def_file ();

  if (identify_imp_name)
    {
      identify_dll_for_implib ();
    }

#ifdef DLLTOOL_MCORE_ELF
  if (mcore_elf_out_file)
    mcore_elf_gen_out_file ();
#endif

  return 0;
}

/* Look for the program formed by concatenating PROG_NAME and the
   string running from PREFIX to END_PREFIX.  If the concatenated
   string contains a '/', try appending EXECUTABLE_SUFFIX if it is
   appropriate.  */

static char *
look_for_prog (const char *prog_name, const char *prefix, int end_prefix)
{
  struct stat s;
  char *cmd;

  cmd = xmalloc (strlen (prefix)
		 + strlen (prog_name)
#ifdef HAVE_EXECUTABLE_SUFFIX
		 + strlen (EXECUTABLE_SUFFIX)
#endif
		 + 10);
  memcpy (cmd, prefix, end_prefix);

  strcpy (cmd + end_prefix, prog_name);

  if (strchr (cmd, '/') != NULL)
    {
      int found;

      found = (stat (cmd, &s) == 0
#ifdef HAVE_EXECUTABLE_SUFFIX
	       || stat (strcat (cmd, EXECUTABLE_SUFFIX), &s) == 0
#endif
	       );

      if (! found)
	{
	  /* xgettext:c-format */
	  inform (_("Tried file: %s"), cmd);
	  free (cmd);
	  return NULL;
	}
    }

  /* xgettext:c-format */
  inform (_("Using file: %s"), cmd);

  return cmd;
}

/* Deduce the name of the program we are want to invoke.
   PROG_NAME is the basic name of the program we want to run,
   eg "as" or "ld".  The catch is that we might want actually
   run "i386-pe-as".

   If argv[0] contains the full path, then try to find the program
   in the same place, with and then without a target-like prefix.

   Given, argv[0] = /usr/local/bin/i586-cygwin32-dlltool,
   deduce_name("as") uses the following search order:

     /usr/local/bin/i586-cygwin32-as
     /usr/local/bin/as
     as

   If there's an EXECUTABLE_SUFFIX, it'll use that as well; for each
   name, it'll try without and then with EXECUTABLE_SUFFIX.

   Given, argv[0] = i586-cygwin32-dlltool, it will not even try "as"
   as the fallback, but rather return i586-cygwin32-as.

   Oh, and given, argv[0] = dlltool, it'll return "as".

   Returns a dynamically allocated string.  */

static char *
deduce_name (const char *prog_name)
{
  char *cmd;
  char *dash, *slash, *cp;

  dash = NULL;
  slash = NULL;
  for (cp = program_name; *cp != '\0'; ++cp)
    {
      if (*cp == '-')
	dash = cp;
      if (
#if defined(__DJGPP__) || defined (__CYGWIN__) || defined(__WIN32__)
	  *cp == ':' || *cp == '\\' ||
#endif
	  *cp == '/')
	{
	  slash = cp;
	  dash = NULL;
	}
    }

  cmd = NULL;

  if (dash != NULL)
    {
      /* First, try looking for a prefixed PROG_NAME in the
	 PROGRAM_NAME directory, with the same prefix as PROGRAM_NAME.  */
      cmd = look_for_prog (prog_name, program_name, dash - program_name + 1);
    }

  if (slash != NULL && cmd == NULL)
    {
      /* Next, try looking for a PROG_NAME in the same directory as
	 that of this program.  */
      cmd = look_for_prog (prog_name, program_name, slash - program_name + 1);
    }

  if (cmd == NULL)
    {
      /* Just return PROG_NAME as is.  */
      cmd = xstrdup (prog_name);
    }

  return cmd;
}

#ifdef DLLTOOL_MCORE_ELF
typedef struct fname_cache
{
  const char *         filename;
  struct fname_cache * next;
}
fname_cache;

static fname_cache fnames;

static void
mcore_elf_cache_filename (const char * filename)
{
  fname_cache * ptr;

  ptr = & fnames;

  while (ptr->next != NULL)
    ptr = ptr->next;

  ptr->filename = filename;
  ptr->next     = (fname_cache *) malloc (sizeof (fname_cache));
  if (ptr->next != NULL)
    ptr->next->next = NULL;
}

#define MCORE_ELF_TMP_OBJ "mcoreelf.o"
#define MCORE_ELF_TMP_EXP "mcoreelf.exp"
#define MCORE_ELF_TMP_LIB "mcoreelf.lib"

static void
mcore_elf_gen_out_file (void)
{
  fname_cache * ptr;
  dyn_string_t ds;

  /* Step one.  Run 'ld -r' on the input object files in order to resolve
     any internal references and to generate a single .exports section.  */
  ptr = & fnames;

  ds = dyn_string_new (100);
  dyn_string_append_cstr (ds, "-r ");

  if (mcore_elf_linker_flags != NULL)
    dyn_string_append_cstr (ds, mcore_elf_linker_flags);

  while (ptr->next != NULL)
    {
      dyn_string_append_cstr (ds, ptr->filename);
      dyn_string_append_cstr (ds, " ");

      ptr = ptr->next;
    }

  dyn_string_append_cstr (ds, "-o ");
  dyn_string_append_cstr (ds, MCORE_ELF_TMP_OBJ);

  if (mcore_elf_linker == NULL)
    mcore_elf_linker = deduce_name ("ld");

  run (mcore_elf_linker, ds->s);

  dyn_string_delete (ds);

  /* Step two. Create a .exp file and a .lib file from the temporary file.
     Do this by recursively invoking dlltool...  */
  ds = dyn_string_new (100);

  dyn_string_append_cstr (ds, "-S ");
  dyn_string_append_cstr (ds, as_name);

  dyn_string_append_cstr (ds, " -e ");
  dyn_string_append_cstr (ds, MCORE_ELF_TMP_EXP);
  dyn_string_append_cstr (ds, " -l ");
  dyn_string_append_cstr (ds, MCORE_ELF_TMP_LIB);
  dyn_string_append_cstr (ds, " " );
  dyn_string_append_cstr (ds, MCORE_ELF_TMP_OBJ);

  if (verbose)
    dyn_string_append_cstr (ds, " -v");

  if (dontdeltemps)
    {
      dyn_string_append_cstr (ds, " -n");

      if (dontdeltemps > 1)
	dyn_string_append_cstr (ds, " -n");
    }

  /* XXX - FIME: ought to check/copy other command line options as well.  */
  run (program_name, ds->s);

  dyn_string_delete (ds);

  /* Step four. Feed the .exp and object files to ld -shared to create the dll.  */
  ds = dyn_string_new (100);

  dyn_string_append_cstr (ds, "-shared ");

  if (mcore_elf_linker_flags)
    dyn_string_append_cstr (ds, mcore_elf_linker_flags);

  dyn_string_append_cstr (ds, " ");
  dyn_string_append_cstr (ds, MCORE_ELF_TMP_EXP);
  dyn_string_append_cstr (ds, " ");
  dyn_string_append_cstr (ds, MCORE_ELF_TMP_OBJ);
  dyn_string_append_cstr (ds, " -o ");
  dyn_string_append_cstr (ds, mcore_elf_out_file);

  run (mcore_elf_linker, ds->s);

  dyn_string_delete (ds);

  if (dontdeltemps == 0)
    unlink (MCORE_ELF_TMP_EXP);

  if (dontdeltemps < 2)
    unlink (MCORE_ELF_TMP_OBJ);
}
#endif /* DLLTOOL_MCORE_ELF */
