# -*- coding: utf-8; frozen_string_literal: true -*-
#
#--
# Copyright (C) 2009-2019 Thomas Leitner <t_leitner@gmx.at>
#
# This file is part of kramdown which is licensed under the MIT.
#++
#

module Kramdown

  module Utils

    # Provides convenience methods for handling named and numeric entities.
    module Entities

      # Represents an entity that has a +code_point+ and +name+.
      Entity = Struct.new(:code_point, :name) do
        # Return the UTF8 representation of the entity.
        def char
          [code_point].pack('U*') rescue nil
        end
      end

      # Array of arrays. Each sub-array specifies a code point and the associated name.
      #
      # This table is not used directly -- Entity objects are automatically created from it and put
      # into a Hash map when this file is loaded.
      ENTITY_TABLE = [

        # html4
        [913, 'Alpha'],
        [914, 'Beta'],
        [915, 'Gamma'],
        [916, 'Delta'],
        [917, 'Epsilon'],
        [918, 'Zeta'],
        [919, 'Eta'],
        [920, 'Theta'],
        [921, 'Iota'],
        [922, 'Kappa'],
        [923, 'Lambda'],
        [924, 'Mu'],
        [925, 'Nu'],
        [926, 'Xi'],
        [927, 'Omicron'],
        [928, 'Pi'],
        [929, 'Rho'],
        [931, 'Sigma'],
        [932, 'Tau'],
        [933, 'Upsilon'],
        [934, 'Phi'],
        [935, 'Chi'],
        [936, 'Psi'],
        [937, 'Omega'],
        [945, 'alpha'],
        [946, 'beta'],
        [947, 'gamma'],
        [948, 'delta'],
        [949, 'epsilon'],
        [950, 'zeta'],
        [951, 'eta'],
        [952, 'theta'],
        [953, 'iota'],
        [954, 'kappa'],
        [955, 'lambda'],
        [956, 'mu'],
        [957, 'nu'],
        [958, 'xi'],
        [959, 'omicron'],
        [960, 'pi'],
        [961, 'rho'],
        [963, 'sigma'],
        [964, 'tau'],
        [965, 'upsilon'],
        [966, 'phi'],
        [967, 'chi'],
        [968, 'psi'],
        [969, 'omega'],
        [962, 'sigmaf'],
        [977, 'thetasym'],
        [978, 'upsih'],
        [982, 'piv'],
        [988, 'Gammad'],
        [989, 'gammad'],
        [8204, 'zwnj'],
        [8205, 'zwj'],
        [8206, 'lrm'],
        [8207, 'rlm'],
        [8230, 'hellip'],
        [8242, 'prime'],
        [8243, 'Prime'],
        [8254, 'oline'],
        [8260, 'frasl'],
        [8472, 'weierp'],
        [8465, 'image'],
        [8476, 'real'],
        [8501, 'alefsym'],
        [8226, 'bull'],
        [8482, 'trade'],
        [8592, 'larr'],
        [8594, 'rarr'],
        [8593, 'uarr'],
        [8595, 'darr'],
        [8596, 'harr'],
        [8629, 'crarr'],
        [8657, 'uArr'],
        [8659, 'dArr'],
        [8656, 'lArr'],
        [8658, 'rArr'],
        [8660, 'hArr'],
        [8704, 'forall'],
        [8706, 'part'],
        [8707, 'exist'],
        [8709, 'empty'],
        [8711, 'nabla'],
        [8712, 'isin'],
        [8715, 'ni'],
        [8713, 'notin'],
        [8721, 'sum'],
        [8719, 'prod'],
        [8722, 'minus'],
        [8727, 'lowast'],
        [8730, 'radic'],
        [8733, 'prop'],
        [8734, 'infin'],
        [8736, 'ang'],
        [8743, 'and'],
        [8744, 'or'],
        [8745, 'cap'],
        [8746, 'cup'],
        [8747, 'int'],
        [8756, 'there4'],
        [8764, 'sim'],
        [8776, 'asymp'],
        [8773, 'cong'],
        [8800, 'ne'],
        [8801, 'equiv'],
        [8804, 'le'],
        [8805, 'ge'],
        [8834, 'sub'],
        [8835, 'sup'],
        [8838, 'sube'],
        [8839, 'supe'],
        [8836, 'nsub'],
        [8853, 'oplus'],
        [8855, 'otimes'],
        [8869, 'perp'],
        [8901, 'sdot'],
        [8942, 'vellip'],
        [8968, 'rceil'],
        [8969, 'lceil'],
        [8970, 'lfloor'],
        [8971, 'rfloor'],
        [9001, 'rang'],
        [9002, 'lang'],
        [9674, 'loz'],
        [9824, 'spades'],
        [9827, 'clubs'],
        [9829, 'hearts'],
        [9830, 'diams'],
        [38, 'amp'],
        [34, 'quot'],
        [39, 'apos'],
        [169, 'copy'],
        [60, 'lt'],
        [62, 'gt'],
        [338, 'OElig'],
        [339, 'oelig'],
        [352, 'Scaron'],
        [353, 'scaron'],
        [376, 'Yuml'],
        [710, 'circ'],
        [732, 'tilde'],
        [8211, 'ndash'],
        [8212, 'mdash'],
        [8216, 'lsquo'],
        [8217, 'rsquo'],
        [8220, 'ldquo'],
        [8221, 'rdquo'],
        [8224, 'dagger'],
        [8225, 'Dagger'],
        [8240, 'permil'],
        [8364, 'euro'],
        [8249, 'lsaquo'],
        [8250, 'rsaquo'],
        [160, 'nbsp'],
        [161, 'iexcl'],
        [163, 'pound'],
        [164, 'curren'],
        [165, 'yen'],
        [166, 'brvbar'],
        [167, 'sect'],
        [168, 'uml'],
        [171, 'laquo'],
        [187, 'raquo'],
        [174, 'reg'],
        [170, 'ordf'],
        [172, 'not'],
        [173, 'shy'],
        [175, 'macr'],
        [176, 'deg'],
        [177, 'plusmn'],
        [180, 'acute'],
        [181, 'micro'],
        [182, 'para'],
        [183, 'middot'],
        [184, 'cedil'],
        [186, 'ordm'],
        [162, 'cent'],
        [185, 'sup1'],
        [178, 'sup2'],
        [179, 'sup3'],
        [189, 'frac12'],
        [188, 'frac14'],
        [190, 'frac34'],
        [8531, 'frac13'],
        [8532, 'frac23'],
        [8533, 'frac15'],
        [8534, 'frac25'],
        [8535, 'frac35'],
        [8536, 'frac45'],
        [8537, 'frac16'],
        [8538, 'frac56'],
        [8539, 'frac18'],
        [8540, 'frac38'],
        [8541, 'frac58'],
        [8542, 'frac78'],
        [191, 'iquest'],
        [192, 'Agrave'],
        [193, 'Aacute'],
        [194, 'Acirc'],
        [195, 'Atilde'],
        [196, 'Auml'],
        [197, 'Aring'],
        [198, 'AElig'],
        [199, 'Ccedil'],
        [200, 'Egrave'],
        [201, 'Eacute'],
        [202, 'Ecirc'],
        [203, 'Euml'],
        [204, 'Igrave'],
        [205, 'Iacute'],
        [206, 'Icirc'],
        [207, 'Iuml'],
        [208, 'ETH'],
        [209, 'Ntilde'],
        [210, 'Ograve'],
        [211, 'Oacute'],
        [212, 'Ocirc'],
        [213, 'Otilde'],
        [214, 'Ouml'],
        [215, 'times'],
        [216, 'Oslash'],
        [217, 'Ugrave'],
        [218, 'Uacute'],
        [219, 'Ucirc'],
        [220, 'Uuml'],
        [221, 'Yacute'],
        [222, 'THORN'],
        [223, 'szlig'],
        [224, 'agrave'],
        [225, 'aacute'],
        [226, 'acirc'],
        [227, 'atilde'],
        [228, 'auml'],
        [229, 'aring'],
        [230, 'aelig'],
        [231, 'ccedil'],
        [232, 'egrave'],
        [233, 'eacute'],
        [234, 'ecirc'],
        [235, 'euml'],
        [236, 'igrave'],
        [237, 'iacute'],
        [238, 'icirc'],
        [239, 'iuml'],
        [240, 'eth'],
        [241, 'ntilde'],
        [242, 'ograve'],
        [243, 'oacute'],
        [244, 'ocirc'],
        [245, 'otilde'],
        [246, 'ouml'],
        [247, 'divide'],
        [248, 'oslash'],
        [249, 'ugrave'],
        [250, 'uacute'],
        [251, 'ucirc'],
        [252, 'uuml'],
        [253, 'yacute'],
        [254, 'thorn'],
        [255, 'yuml'],

        [8218, 'sbquo'],
        [402, 'fnof'],
        [8222, 'bdquo'],

        [128, 8364],
        [130, 8218],
        [131, 402],
        [132, 8222],
        [133, 8230],
        [134, 8224],
        [135, 8225],
        [136, 710],
        [137, 8240],
        [138, 352],
        [139, 8249],
        [140, 338],
        [142, 381],
        [145, 8216],
        [146, 8217],
        [147, 8220],
        [148, 8221],
        [149, 8226],
        [150, 8211],
        [151, 8212],
        [152, 732],
        [153, 8482],
        [154, 353],
        [155, 8250],
        [156, 339],
        [158, 382],
        [159, 376],

        [8194, 'ensp'],
        [8195, 'emsp'],
        [8201, 'thinsp'],

        # html5
        [10218, 'Lang'],
        [10219, 'Rang'],
        [10220, 'loang'],
        [10221, 'roang'],
        [10229, 'xlarr'],
        [10229, 'longleftarrow'],
        [10229, 'LongLeftArrow'],
        [10230, 'xrarr'],
        [10230, 'longrightarrow'],
        [10230, 'LongRightArrow'],
        [10231, 'xharr'],
        [10231, 'longleftrightarrow'],
        [10231, 'LongLeftRightArrow'],
        [10232, 'xlArr'],
        [10232, 'Longleftarrow'],
        [10232, 'DoubleLongLeftArrow'],
        [10233, 'xrArr'],
        [10233, 'Longrightarrow'],
        [10233, 'DoubleLongRightArrow'],
        [10234, 'xhArr'],
        [10234, 'Longleftrightarrow'],
        [10234, 'DoubleLongLeftRightArrow'],
        [10236, 'xmap'],
        [10236, 'longmapsto'],
        [10239, 'dzigrarr'],
        [10498, 'nvlArr'],
        [10499, 'nvrArr'],
        [10500, 'nvHarr'],
        [10501, 'Map'],
        [10508, 'lbarr'],
        [10509, 'rbarr'],
        [10509, 'bkarow'],
        [10510, 'lBarr'],
        [10511, 'rBarr'],
        [10511, 'dbkarow'],
        [10512, 'RBarr'],
        [10512, 'drbkarow'],
        [10513, 'DDotrahd'],
        [10514, 'UpArrowBar'],
        [10515, 'DownArrowBar'],
        [10518, 'Rarrtl'],
        [10521, 'latail'],
        [10522, 'ratail'],
        [10523, 'lAtail'],
        [10524, 'rAtail'],
        [10525, 'larrfs'],
        [10526, 'rarrfs'],
        [10527, 'larrbfs'],
        [10528, 'rarrbfs'],
        [10531, 'nwarhk'],
        [10532, 'nearhk'],
        [10533, 'searhk'],
        [10533, 'hksearow'],
        [10534, 'swarhk'],
        [10534, 'hkswarow'],
        [10535, 'nwnear'],
        [10536, 'nesear'],
        [10536, 'toea'],
        [10537, 'seswar'],
        [10537, 'tosa'],
        [10538, 'swnwar'],
        [10547, 'rarrc'],
        [10549, 'cudarrr'],
        [10550, 'ldca'],
        [10551, 'rdca'],
        [10552, 'cudarrl'],
        [10553, 'larrpl'],
        [10556, 'curarrm'],
        [10557, 'cularrp'],
        [10565, 'rarrpl'],
        [10568, 'harrcir'],
        [10569, 'Uarrocir'],
        [10570, 'lurdshar'],
        [10571, 'ldrushar'],
        [10574, 'LeftRightVector'],
        [10575, 'RightUpDownVector'],
        [10576, 'DownLeftRightVector'],
        [10577, 'LeftUpDownVector'],
        [10578, 'LeftVectorBar'],
        [10579, 'RightVectorBar'],
        [10580, 'RightUpVectorBar'],
        [10581, 'RightDownVectorBar'],
        [10582, 'DownLeftVectorBar'],
        [10583, 'DownRightVectorBar'],
        [10584, 'LeftUpVectorBar'],
        [10585, 'LeftDownVectorBar'],
        [10586, 'LeftTeeVector'],
        [10587, 'RightTeeVector'],
        [10588, 'RightUpTeeVector'],
        [10589, 'RightDownTeeVector'],
        [10590, 'DownLeftTeeVector'],
        [10591, 'DownRightTeeVector'],
        [10592, 'LeftUpTeeVector'],
        [10593, 'LeftDownTeeVector'],
        [10594, 'lHar'],
        [10595, 'uHar'],
        [10596, 'rHar'],
        [10597, 'dHar'],
        [10598, 'luruhar'],
        [10599, 'ldrdhar'],
        [10600, 'ruluhar'],
        [10601, 'rdldhar'],
        [10602, 'lharul'],
        [10603, 'llhard'],
        [10604, 'rharul'],
        [10605, 'lrhard'],
        [10606, 'udhar'],
        [10606, 'UpEquilibrium'],
        [10607, 'duhar'],
        [10607, 'ReverseUpEquilibrium'],
        [10608, 'RoundImplies'],
        [10609, 'erarr'],
        [10610, 'simrarr'],
        [10611, 'larrsim'],
        [10612, 'rarrsim'],
        [10613, 'rarrap'],
        [10614, 'ltlarr'],
        [10616, 'gtrarr'],
        [10617, 'subrarr'],
        [10619, 'suplarr'],
        [10620, 'lfisht'],
        [10621, 'rfisht'],
        [10622, 'ufisht'],
        [10623, 'dfisht'],
        [10629, 'lopar'],
        [10630, 'ropar'],
        [10635, 'lbrke'],
        [10636, 'rbrke'],
        [10637, 'lbrkslu'],
        [10638, 'rbrksld'],
        [10639, 'lbrksld'],
        [10640, 'rbrkslu'],
        [10641, 'langd'],
        [10642, 'rangd'],
        [10643, 'lparlt'],
        [10644, 'rpargt'],
        [10645, 'gtlPar'],
        [10646, 'ltrPar'],
        [10650, 'vzigzag'],
        [10652, 'vangrt'],
        [10653, 'angrtvbd'],
        [10660, 'ange'],
        [10661, 'range'],
        [10662, 'dwangle'],
        [10663, 'uwangle'],
        [10664, 'angmsdaa'],
        [10665, 'angmsdab'],
        [10666, 'angmsdac'],
        [10667, 'angmsdad'],
        [10668, 'angmsdae'],
        [10669, 'angmsdaf'],
        [10670, 'angmsdag'],
        [10671, 'angmsdah'],
        [10672, 'bemptyv'],
        [10673, 'demptyv'],
        [10674, 'cemptyv'],
        [10675, 'raemptyv'],
        [10676, 'laemptyv'],
        [10677, 'ohbar'],
        [10678, 'omid'],
        [10679, 'opar'],
        [10681, 'operp'],
        [10683, 'olcross'],
        [10684, 'odsold'],
        [10686, 'olcir'],
        [10687, 'ofcir'],
        [10688, 'olt'],
        [10689, 'ogt'],
        [10690, 'cirscir'],
        [10691, 'cirE'],
        [10692, 'solb'],
        [10693, 'bsolb'],
        [10697, 'boxbox'],
        [10701, 'trisb'],
        [10702, 'rtriltri'],
        [10703, 'LeftTriangleBar'],
        [10704, 'RightTriangleBar'],
        [10716, 'iinfin'],
        [10717, 'infintie'],
        [10718, 'nvinfin'],
        [10723, 'eparsl'],
        [10724, 'smeparsl'],
        [10725, 'eqvparsl'],
        [10731, 'lozf'],
        [10731, 'blacklozenge'],
        [10740, 'RuleDelayed'],
        [10742, 'dsol'],
        [10752, 'xodot'],
        [10752, 'bigodot'],
        [10753, 'xoplus'],
        [10753, 'bigoplus'],
        [10754, 'xotime'],
        [10754, 'bigotimes'],
        [10756, 'xuplus'],
        [10756, 'biguplus'],
        [10758, 'xsqcup'],
        [10758, 'bigsqcup'],
        [10764, 'qint'],
        [10764, 'iiiint'],
        [10765, 'fpartint'],
        [10768, 'cirfnint'],
        [10769, 'awint'],
        [10770, 'rppolint'],
        [10771, 'scpolint'],
        [10772, 'npolint'],
        [10773, 'pointint'],
        [10774, 'quatint'],
        [10775, 'intlarhk'],
        [10786, 'pluscir'],
        [10787, 'plusacir'],
        [10788, 'simplus'],
        [10789, 'plusdu'],
        [10790, 'plussim'],
        [10791, 'plustwo'],
        [10793, 'mcomma'],
        [10794, 'minusdu'],
        [10797, 'loplus'],
        [10798, 'roplus'],
        [10799, 'Cross'],
        [10800, 'timesd'],
        [10801, 'timesbar'],
        [10803, 'smashp'],
        [10804, 'lotimes'],
        [10805, 'rotimes'],
        [10806, 'otimesas'],
        [10807, 'Otimes'],
        [10808, 'odiv'],
        [10809, 'triplus'],
        [10810, 'triminus'],
        [10811, 'tritime'],
        [10812, 'iprod'],
        [10812, 'intprod'],
        [10815, 'amalg'],
        [10816, 'capdot'],
        [10818, 'ncup'],
        [10819, 'ncap'],
        [10820, 'capand'],
        [10821, 'cupor'],
        [10822, 'cupcap'],
        [10823, 'capcup'],
        [10824, 'cupbrcap'],
        [10825, 'capbrcup'],
        [10826, 'cupcup'],
        [10827, 'capcap'],
        [10828, 'ccups'],
        [10829, 'ccaps'],
        [10832, 'ccupssm'],
        [10835, 'And'],
        [10836, 'Or'],
        [10837, 'andand'],
        [10838, 'oror'],
        [10839, 'orslope'],
        [10840, 'andslope'],
        [10842, 'andv'],
        [10843, 'orv'],
        [10844, 'andd'],
        [10845, 'ord'],
        [10847, 'wedbar'],
        [10854, 'sdote'],
        [10858, 'simdot'],
        [10861, 'congdot'],
        [10862, 'easter'],
        [10863, 'apacir'],
        [10864, 'apE'],
        [10865, 'eplus'],
        [10866, 'pluse'],
        [10867, 'Esim'],
        [10868, 'Colone'],
        [10869, 'Equal'],
        [10871, 'eDDot'],
        [10871, 'ddotseq'],
        [10872, 'equivDD'],
        [10873, 'ltcir'],
        [10874, 'gtcir'],
        [10875, 'ltquest'],
        [10876, 'gtquest'],
        [10877, 'les'],
        [10877, 'LessSlantEqual'],
        [10877, 'leqslant'],
        [10878, 'ges'],
        [10878, 'GreaterSlantEqual'],
        [10878, 'geqslant'],
        [10879, 'lesdot'],
        [10880, 'gesdot'],
        [10881, 'lesdoto'],
        [10882, 'gesdoto'],
        [10883, 'lesdotor'],
        [10884, 'gesdotol'],
        [10885, 'lap'],
        [10885, 'lessapprox'],
        [10886, 'gap'],
        [10886, 'gtrapprox'],
        [10887, 'lne'],
        [10887, 'lneq'],
        [10888, 'gne'],
        [10888, 'gneq'],
        [10889, 'lnap'],
        [10889, 'lnapprox'],
        [10890, 'gnap'],
        [10890, 'gnapprox'],
        [10891, 'lEg'],
        [10891, 'lesseqqgtr'],
        [10892, 'gEl'],
        [10892, 'gtreqqless'],
        [10893, 'lsime'],
        [10894, 'gsime'],
        [10895, 'lsimg'],
        [10896, 'gsiml'],
        [10897, 'lgE'],
        [10898, 'glE'],
        [10899, 'lesges'],
        [10900, 'gesles'],
        [10901, 'els'],
        [10901, 'eqslantless'],
        [10902, 'egs'],
        [10902, 'eqslantgtr'],
        [10903, 'elsdot'],
        [10904, 'egsdot'],
        [10905, 'el'],
        [10906, 'eg'],
        [10909, 'siml'],
        [10910, 'simg'],
        [10911, 'simlE'],
        [10912, 'simgE'],
        [10913, 'LessLess'],
        [10914, 'GreaterGreater'],
        [10916, 'glj'],
        [10917, 'gla'],
        [10918, 'ltcc'],
        [10919, 'gtcc'],
        [10920, 'lescc'],
        [10921, 'gescc'],
        [10922, 'smt'],
        [10923, 'lat'],
        [10924, 'smte'],
        [10925, 'late'],
        [10926, 'bumpE'],
        [10927, 'pre'],
        [10927, 'preceq'],
        [10927, 'PrecedesEqual'],
        [10928, 'sce'],
        [10928, 'succeq'],
        [10928, 'SucceedsEqual'],
        [10931, 'prE'],
        [10932, 'scE'],
        [10933, 'prnE'],
        [10933, 'precneqq'],
        [10934, 'scnE'],
        [10934, 'succneqq'],
        [10935, 'prap'],
        [10935, 'precapprox'],
        [10936, 'scap'],
        [10936, 'succapprox'],
        [10937, 'prnap'],
        [10937, 'precnapprox'],
        [10938, 'scnap'],
        [10938, 'succnapprox'],
        [10939, 'Pr'],
        [10940, 'Sc'],
        [10941, 'subdot'],
        [10942, 'supdot'],
        [10943, 'subplus'],
        [10944, 'supplus'],
        [10945, 'submult'],
        [10946, 'supmult'],
        [10947, 'subedot'],
        [10948, 'supedot'],
        [10949, 'subE'],
        [10949, 'subseteqq'],
        [10950, 'supE'],
        [10950, 'supseteqq'],
        [10951, 'subsim'],
        [10952, 'supsim'],
        [10955, 'subnE'],
        [10955, 'subsetneqq'],
        [10956, 'supnE'],
        [10956, 'supsetneqq'],
        [10959, 'csub'],
        [10960, 'csup'],
        [10961, 'csube'],
        [10962, 'csupe'],
        [10963, 'subsup'],
        [10964, 'supsub'],
        [10965, 'subsub'],
        [10966, 'supsup'],
        [10967, 'suphsub'],
        [10968, 'supdsub'],
        [10969, 'forkv'],
        [10970, 'topfork'],
        [10971, 'mlcp'],
        [10980, 'Dashv'],
        [10980, 'DoubleLeftTee'],
        [10982, 'Vdashl'],
        [10983, 'Barv'],
        [10984, 'vBar'],
        [10985, 'vBarv'],
        [10987, 'Vbar'],
        [10988, 'Not'],
        [10989, 'bNot'],
        [10990, 'rnmid'],
        [10991, 'cirmid'],
        [10992, 'midcir'],
        [10993, 'topcir'],
        [10994, 'nhpar'],
        [10995, 'parsim'],
        [11005, 'parsl'],
        [64256, 'fflig'],
        [64257, 'filig'],
        [64258, 'fllig'],
        [64259, 'ffilig'],
        [64260, 'ffllig'],
        [119964, 'Ascr'],
        [119966, 'Cscr'],
        [119967, 'Dscr'],
        [119970, 'Gscr'],
        [119973, 'Jscr'],
        [119974, 'Kscr'],
        [119977, 'Nscr'],
        [119978, 'Oscr'],
        [119979, 'Pscr'],
        [119980, 'Qscr'],
        [119982, 'Sscr'],
        [119983, 'Tscr'],
        [119984, 'Uscr'],
        [119985, 'Vscr'],
        [119986, 'Wscr'],
        [119987, 'Xscr'],
        [119988, 'Yscr'],
        [119989, 'Zscr'],
        [119990, 'ascr'],
        [119991, 'bscr'],
        [119992, 'cscr'],
        [119993, 'dscr'],
        [119995, 'fscr'],
        [119997, 'hscr'],
        [119998, 'iscr'],
        [119999, 'jscr'],
        [120000, 'kscr'],
        [120001, 'lscr'],
        [120002, 'mscr'],
        [120003, 'nscr'],
        [120005, 'pscr'],
        [120006, 'qscr'],
        [120007, 'rscr'],
        [120008, 'sscr'],
        [120009, 'tscr'],
        [120010, 'uscr'],
        [120011, 'vscr'],
        [120012, 'wscr'],
        [120013, 'xscr'],
        [120014, 'yscr'],
        [120015, 'zscr'],
        [120068, 'Afr'],
        [120069, 'Bfr'],
        [120071, 'Dfr'],
        [120072, 'Efr'],
        [120073, 'Ffr'],
        [120074, 'Gfr'],
        [120077, 'Jfr'],
        [120078, 'Kfr'],
        [120079, 'Lfr'],
        [120080, 'Mfr'],
        [120081, 'Nfr'],
        [120082, 'Ofr'],
        [120083, 'Pfr'],
        [120084, 'Qfr'],
        [120086, 'Sfr'],
        [120087, 'Tfr'],
        [120088, 'Ufr'],
        [120089, 'Vfr'],
        [120090, 'Wfr'],
        [120091, 'Xfr'],
        [120092, 'Yfr'],
        [120094, 'afr'],
        [120095, 'bfr'],
        [120096, 'cfr'],
        [120097, 'dfr'],
        [120098, 'efr'],
        [120099, 'ffr'],
        [120100, 'gfr'],
        [120101, 'hfr'],
        [120102, 'ifr'],
        [120103, 'jfr'],
        [120104, 'kfr'],
        [120105, 'lfr'],
        [120106, 'mfr'],
        [120107, 'nfr'],
        [120108, 'ofr'],
        [120109, 'pfr'],
        [120110, 'qfr'],
        [120111, 'rfr'],
        [120112, 'sfr'],
        [120113, 'tfr'],
        [120114, 'ufr'],
        [120115, 'vfr'],
        [120116, 'wfr'],
        [120117, 'xfr'],
        [120118, 'yfr'],
        [120119, 'zfr'],
        [120120, 'Aopf'],
        [120121, 'Bopf'],
        [120123, 'Dopf'],
        [120124, 'Eopf'],
        [120125, 'Fopf'],
        [120126, 'Gopf'],
        [120128, 'Iopf'],
        [120129, 'Jopf'],
        [120130, 'Kopf'],
        [120131, 'Lopf'],
        [120132, 'Mopf'],
        [120134, 'Oopf'],
        [120138, 'Sopf'],
        [120139, 'Topf'],
        [120140, 'Uopf'],
        [120141, 'Vopf'],
        [120142, 'Wopf'],
        [120143, 'Xopf'],
        [120144, 'Yopf'],
        [120146, 'aopf'],
        [120147, 'bopf'],
        [120148, 'copf'],
        [120149, 'dopf'],
        [120150, 'eopf'],
        [120151, 'fopf'],
        [120152, 'gopf'],
        [120153, 'hopf'],
        [120154, 'iopf'],
        [120155, 'jopf'],
        [120156, 'kopf'],
        [120157, 'lopf'],
        [120158, 'mopf'],
        [120159, 'nopf'],
        [120160, 'oopf'],
        [120161, 'popf'],
        [120162, 'qopf'],
        [120163, 'ropf'],
        [120164, 'sopf'],
        [120165, 'topf'],
        [120166, 'uopf'],
        [120167, 'vopf'],
        [120168, 'wopf'],
        [120169, 'xopf'],
        [120170, 'yopf'],
        [120171, 'zopf'],
        [60, 'nvlt'],
        [61, 'bne'],
        [62, 'nvgt'],
        [102, 'fjlig'],
        [8287, 'ThickSpace'],
        [8605, 'nrarrw'],
        [8706, 'npart'],
        [8736, 'nang'],
        [8745, 'caps'],
        [8746, 'cups'],
        [8764, 'nvsim'],
        [8765, 'race'],
        [8766, 'acE'],
        [8770, 'nesim'],
        [8770, 'NotEqualTilde'],
        [8779, 'napid'],
        [8781, 'nvap'],
        [8782, 'nbump'],
        [8782, 'NotHumpDownHump'],
        [8783, 'nbumpe'],
        [8783, 'NotHumpEqual'],
        [8784, 'nedot'],
        [8801, 'bnequiv'],
        [8804, 'nvle'],
        [8805, 'nvge'],
        [8806, 'nlE'],
        [8806, 'nleqq'],
        [8807, 'ngE'],
        [8807, 'ngeqq'],
        [8807, 'NotGreaterFullEqual'],
        [8808, 'lvertneqq'],
        [8808, 'lvnE'],
        [8809, 'gvertneqq'],
        [8809, 'gvnE'],
        [8810, 'nLtv'],
        [8810, 'NotLessLess'],
        [8810, 'nLt'],
        [8811, 'nGtv'],
        [8811, 'NotGreaterGreater'],
        [8811, 'nGt'],
        [8831, 'NotSucceedsTilde'],
        [8834, 'NotSubset'],
        [8834, 'nsubset'],
        [8834, 'vnsub'],
        [8835, 'NotSuperset'],
        [8835, 'nsupset'],
        [8835, 'vnsup'],
        [8842, 'varsubsetneq'],
        [8842, 'vsubne'],
        [8843, 'varsupsetneq'],
        [8843, 'vsupne'],
        [8847, 'NotSquareSubset'],
        [8848, 'NotSquareSuperset'],
        [8851, 'sqcaps'],
        [8852, 'sqcups'],
        [8884, 'nvltrie'],
        [8885, 'nvrtrie'],
        [8920, 'nLl'],
        [8921, 'nGg'],
        [8922, 'lesg'],
        [8923, 'gesl'],
        [8949, 'notindot'],
        [8953, 'notinE'],
        [10547, 'nrarrc'],
        [10703, 'NotLeftTriangleBar'],
        [10704, 'NotRightTriangleBar'],
        [10861, 'ncongdot'],
        [10864, 'napE'],
        [10877, 'nleqslant'],
        [10877, 'nles'],
        [10877, 'NotLessSlantEqual'],
        [10878, 'ngeqslant'],
        [10878, 'nges'],
        [10878, 'NotGreaterSlantEqual'],
        [10913, 'NotNestedLessLess'],
        [10914, 'NotNestedGreaterGreater'],
        [10924, 'smtes'],
        [10925, 'lates'],
        [10927, 'NotPrecedesEqual'],
        [10927, 'npre'],
        [10927, 'npreceq'],
        [10928, 'NotSucceedsEqual'],
        [10928, 'nsce'],
        [10928, 'nsucceq'],
        [10949, 'nsubE'],
        [10949, 'nsubseteqq'],
        [10950, 'nsupE'],
        [10950, 'nsupseteqq'],
        [10955, 'varsubsetneqq'],
        [10955, 'vsubnE'],
        [10956, 'varsupsetneqq'],
        [10956, 'vsupnE'],
        [11005, 'nparsl'],

      ]

      # Contains the mapping of code point (or name) to the actual Entity object.
      ENTITY_MAP = Hash.new do |h, k|
        if k.kind_of?(Integer)
          h[k] = Entity.new(k, nil)
        else
          raise Kramdown::Error, "Can't handle generic non-integer character reference '#{k}'"
        end
      end

      ENTITY_TABLE.each do |code_point, data|
        ENTITY_MAP[code_point] = if data.kind_of?(String)
                                   ENTITY_MAP[data] = Entity.new(code_point, data)
                                 else
                                   ENTITY_MAP[data]
                                 end
      end

      # Return the entity for the given code point or name +point_or_name+.
      def entity(point_or_name)
        ENTITY_MAP[point_or_name]
      end

      module_function :entity

    end

  end

end
