package midterm

import "time"

type Screen struct {
	// Height and Width are the dimensions of the terminal.
	Height, Width int

	// Content is the text in the terminal.
	Content [][]rune

	// Format is the display properties of each cell.
	Format [][]Format

	// Cursor is the current state of the cursor.
	Cursor Cursor

	// ScrollRegion is the region of the terminal that is scrollable. If it is
	// nil, the entire terminal is scrollable.
	//
	// This value is set by the CSI ; Ps ; Ps r command.
	ScrollRegion *ScrollRegion

	// CursorVisible indicates whether the cursor is visible.
	//
	// This value is set by CSI ? 25 h and unset by CSI ? 25 l.
	CursorVisible bool

	// CursorBlinking indicates whether the cursor is blinking, and the start of
	// the blinking interval.
	CursorBlinkEpoch *time.Time

	// SavedCursor is the state of the cursor last time save() was called.
	SavedCursor Cursor

	// MaxY is the maximum vertical offset that a character has been printed.
	MaxY int
}

func newScreen(h, w int) *Screen {
	s := &Screen{
		Height: h,
		Width:  w,

		// start at -1 so there's no "used" height until first write
		MaxY: -1,
	}
	s.reset()
	return s
}

func (s *Screen) reset() {
	s.Content = make([][]rune, s.Height)
	s.Format = make([][]Format, s.Height)
	for row := 0; row < s.Height; row++ {
		s.Content[row] = make([]rune, s.Width)
		s.Format[row] = make([]Format, s.Width)
		for col := 0; col < s.Width; col++ {
			s.Content[row][col] = ' '
		}
	}
	s.Cursor.X = 0
	s.Cursor.Y = 0
}

func (v *Screen) resize(h, w int) {
	if h < v.MaxY {
		v.MaxY = h - 1
	}

	if h > v.Height {
		n := h - v.Height
		for row := 0; row < n; row++ {
			v.Content = append(v.Content, make([]rune, v.Width))
			v.Format = append(v.Format, make([]Format, v.Width))
			for col := 0; col < v.Width; col++ {
				v.clear(v.Height+row, col, Format{})
			}
		}
	} else if h < v.Height {
		v.Content = v.Content[:h]
		v.Format = v.Format[:h]
	}

	if w > v.Width {
		for i := range v.Content {
			row := make([]rune, w)
			copy(row, v.Content[i])
			v.Content[i] = row
			format := make([]Format, w)
			copy(format, v.Format[i])
			v.Format[i] = format
			for j := v.Width; j < w; j++ {
				v.clear(i, j, Format{})
			}
		}
	} else if w < v.Width {
		for i := range v.Content {
			v.Content[i] = v.Content[i][:w]
			v.Format[i] = v.Format[i][:w]
		}
	}

	v.Height = h
	v.Width = w

	if v.Cursor.X >= v.Width {
		v.Cursor.X = v.Width - 1
	}
}

func (v *Screen) clear(y, x int, format Format) {
	if y >= len(v.Content) || x >= len(v.Content[0]) {
		return
	}
	v.Content[y][x] = ' '
	v.Format[y][x] = format
}
