#!/usr/bin/env python
#
# Any copyright is dedicated to the Public Domain.
# http://creativecommons.org/publicdomain/zero/1.0/
#
# Generates HTML test files with permutations for frame tree hierarchies to
# test page-name breaks.
# These should all have page-name-two-pages-ref.html as their ref case.
#
# The generated tests have the structure of two <p> elements that will have
# different page name values through various means:
# * Both <p> elements have different values for the `page` property.
# * One <p> element has a specified non-default `page` property and the other
#   does not.
# * One <p> element has a specified non-default `page` property and the other
#   has `page: auto`.
#
# Additionally, the <p> elements may be contained in a <div> element, which may
# also have the `page` property set on it.

import os
import sys

# Test count, used for file numbers
i = 1

# Generate tests that enumerate putting each paragraph element into a div or
# not, and applying the page-name to the div or the paragraph element.

# Data that is used to generate the structure and element attributes for a page.
ALL_DATA_COMBOS = (
    {"p_page": True, "use_div": False, "div_page": False},
    {"p_page": True, "use_div": True, "div_page": False},
    {"p_page": False, "use_div": True, "div_page": True},
)

# Process ALL_DATA_COMBOS to generate data combos for a page with a given name.


def gen_data_combos(name):
    combos = [{"p_page": False, "use_div": False, "div_page": False}]
    for data in ALL_DATA_COMBOS:
        data_copy = data.copy()
        data_copy["p_page"] = data["p_page"] and name
        data_copy["div_page"] = data["div_page"] and name
        combos.append(data_copy)
        # Make page: auto versions for parts with empty page values.
        for k in ("p_page", "div_page"):
            # Only care about div page when there is a div
            if k == "div_page" and not data["use_div"]:
                continue
            if not data[k]:
                data_copy_auto = data_copy.copy()
                data_copy_auto[k] = "auto"
                combos.append(data_copy_auto)
    return combos


A_DATA_COMBOS = gen_data_combos("a")
B_DATA_COMBOS = gen_data_combos("b")


def tag(name, page, inner=""):
    # Build the opening
    open_tag = "<" + name
    if page:
        open_tag += ' style="page:' + page + '"'
    open_tag += ">"

    close_tag = "</" + name + ">"
    return open_tag + inner + close_tag


def generate_page_html(txt, p_page, use_div, div_page):
    p = tag("p", p_page, txt)
    if use_div:
        return tag("div", div_page, p)
    return p


# Preamble to all test cases
BEGIN = """\
<!-- AUTOGENERATED BY generate-page-name-two-page-test.py, DO NOT MODIFY -->
<!DOCTYPE html>
<html class="reftest-paged">
<body>
"""

# Closing tags for all test cases.
END = """
</body>
</html>
"""

directory = os.path.dirname(sys.argv[0])


def data_has_no_page(d):
    return (not d["p_page"] or d["p_page"] == "auto") and (
        not d["div_page"] or d["div_page"] == "auto"
    )


for a in A_DATA_COMBOS:
    for b in B_DATA_COMBOS:
        # Test for the case of empty or auto page-names in both data
        if data_has_no_page(a) and data_has_no_page(b):
            continue
        file_name = "page-name-two-page-" + str(i).rjust(3, "0") + ".html"
        i += 1
        f = open(os.path.join(directory, file_name), "w")
        f.write(BEGIN)
        f.write(generate_page_html("a", **a))
        f.write("\n")
        f.write(generate_page_html("b", **b))
        f.write(END)
        f.close()
