/* keyboard functions

   Written by Matthias Hensler
   Copyright WSPse 1999-2004
   eMail: matthias@wspse.de

Created: 1999/06/06
Updated: 2004/04/15
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
   */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ncurses.h>
#include "mp3creat.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

extern void song_move_down(WINDOW *win, BOOL do_refresh);
extern void song_move_up(WINDOW *win, BOOL do_refresh);
extern void song_move_top(WINDOW *win, BOOL do_refresh);
extern void song_move_bottom(WINDOW *win, BOOL do_refresh);
extern void song_move_pup(WINDOW *win, BOOL do_refresh);
extern void song_move_pdown(WINDOW *win, BOOL do_refresh);
extern void draw_song_line(WINDOW *win, song_typ *song, BOOL marked, int line);
extern void draw_song_status(WINDOW *win, song_typ *song);
extern void win_effect(WINDOW *win, BOOL e_refresh, BOOL save_coor);
extern int play_track(int tr_begin, int tr_end);
extern int stop_cd();
extern int rip_non_fly(song_typ *track);
extern int enc_non_fly(song_typ *track, BOOL del_tmp_file);
extern int conv_on_fly(song_typ *track);
extern int set_mp3_inf(song_typ *track);
extern int select_genre();
extern int set_active_win(WINDOW *win);
extern char *req_get_file(const char *path, const char *description);
extern signed char config_fancy_colors;
extern char *input_line(int y_line, char *def_str, char *usage_str, BOOL select);
extern void fill_in_songs(WINDOW *win);
extern void fill_in_side(WINDOW *win, int side);
extern void filenm_generate(song_typ *track);
extern BOOL select_yesno_box(char *tx);
extern void wuuush(int);
extern void pop_up_help();
extern void setup_stat_win(int max_length);
extern void print_stat_win(char *text);
extern void destroy_stat_win();
extern void popup_error_win(char *tx);
extern void free_song_typ(song_typ **anchor);
extern int open_cdrom();
extern int close_cdrom();
extern int init_cd();
extern int build_data_tree(char *cddb_server, char *local_cddb_db,
			   song_typ **ret_tree, BOOL force_sampler);
extern int cddb_internet_lookup (char *addr, char *cddb_path, BOOL force);
extern void option_menu();
extern int output_batch(song_typ *anchor, char *filenm, BOOL ask_overwrite);
extern void calc_tot_frm();
extern int add_to_m3u(song_typ *song);
extern int eject_cd();
extern int close_tray();
extern void alter_track_infos_menu(song_typ *track);
extern void rhand_add_ripped(song_typ *track, WINDOW *win);
extern void rhand_do_enc(song_typ *track, WINDOW *win);
extern void rhand_del_actual(WINDOW *win);
extern void rhand_del_marked(WINDOW *win);
int rhand_save_ripped(char *file_name, BOOL clear_del, int sel_code);
extern int rhand_load_ripped(char *file_name, WINDOW *win);
extern void layout_update_cddb_flag(WINDOW *win);
extern void rhand_remove_dead_files(WINDOW *win);
extern int cddb_save_cddb_entry(song_typ *song_list, char *cddb_path);
extern int cddb_submit_cddb_entry(song_typ *song_list);
extern int cd_drv;
extern char *def_cddb_server;
extern char *def_cddb_bank;
extern char *def_cdrom_dev;
extern char *def_exp_file;
extern BOOL rip_enc_ordered;
extern song_typ *lay_global_anchor[2];
extern song_typ *lay_top_anchor[2];
extern song_typ *lay_curr_anchor[2];
extern int lay_select_line[2];
extern int lay_act_side;
extern unsigned long int lay_tot_frm[2];
extern BOOL clear_del_on_exp;
extern BOOL config_cddb_enbl;
extern BOOL config_open_tray;

extern struct {
  int min;
  int sec;
  int frame;
} cdtoc[100];

/* swap artist and title */
void a_t_swap(song_typ *song)
{
  char *pointer;

  pointer      = song->title;
  song->title  = song->artist;
  song->artist = pointer;
  filenm_generate(song);
}

/* reinitialize cd (and device if needed) */
void reinit_volume(WINDOW *win)
{
  song_typ *anchor;

  /* release data-structures not longer needed */
  free_song_typ(&(lay_global_anchor[0]));
  
  lay_global_anchor[0] = NULL;      /* no data present at the moment */
  lay_top_anchor[0]    = NULL;
  lay_curr_anchor[0]   = NULL;
  lay_select_line[0]   = 1;
  setup_stat_win(50);             /* open info-box */
  
  print_stat_win(_("reinitialising cd"));
  close_cdrom();   /* just make sure that there isn't any forgotten open file */
  if(init_cd() != 0) {
    if(cd_drv == -2) {       /* status = -2, opening was successful */
      popup_error_win(_("no cd-volume inserted"));
    } else {                 /* status = -1, opening failed */
      popup_error_win(_("failed to open cdrom device"));
    }
  } else {
    build_data_tree(def_cddb_server, def_cddb_bank, &anchor, FALSE);
    lay_top_anchor[0]    = anchor;
    lay_global_anchor[0] = anchor;
    calc_tot_frm();
  }

  destroy_stat_win();
  fill_in_songs(win);   
}

void init_sampler(WINDOW *win)
{
  song_typ *anchor;

  free_song_typ(&(lay_global_anchor[0]));

  lay_global_anchor[0] = NULL;
  lay_top_anchor[0]    = NULL;
  lay_curr_anchor[0]   = NULL;
  lay_select_line[0]   = 1;
  setup_stat_win(50);

  print_stat_win(_("interpreting CDDB as sampler"));
  build_data_tree(def_cddb_server, def_cddb_bank, &anchor, TRUE);
  lay_top_anchor[0]    = anchor;
  lay_global_anchor[0] = anchor;
  calc_tot_frm();
  
  destroy_stat_win();
  fill_in_songs(win);
}

/* open tray and do the neccessary things (clear list, etc) */
void keys_open_tray(WINDOW *win)
{
  setup_stat_win(50);
  print_stat_win(_("ejecting cdrom"));
  if(eject_cd()) {
    popup_error_win(_("ejecting failed..."));
  } else {
    print_stat_win(_("ok"));
  }
  
  free_song_typ(&(lay_global_anchor[0]));
  lay_global_anchor[0] = NULL;
  lay_top_anchor[0]    = NULL;
  lay_curr_anchor[0]   = NULL;
  lay_select_line[0]   = 1;
  
  destroy_stat_win();

  if(lay_global_anchor[1]) {
    lay_act_side = 1;
    fill_in_songs(win);
  } else {
    fill_in_side(win, 0);
  }
}

/* check if joblist is empty */
BOOL keys_joblist_is_empty(song_typ *list)
{
  song_typ *curr;

  curr = list;
  while(curr) {
    if(curr->convert) return FALSE;
    curr = curr->next;
  }

  return TRUE;
}

/* MAINLOOP */
void main_loop(WINDOW *win)
{
  int inp_ch;
  char *pointer1, *pointer3, *pointer4;
  song_typ *pointer2, *old_song;
  int i;
  BOOL abort_flag;
  
  cbreak();
  noecho();
  keypad(win, TRUE);
  if(config_fancy_colors)
    halfdelay(1);
  else
    nodelay(win, FALSE);

  while(1) {
    if(config_fancy_colors) win_effect(win, TRUE, FALSE);
    waitpid(0, NULL, WNOHANG);
    inp_ch = wgetch(win);
    if(lay_curr_anchor[lay_act_side] != NULL) {
      switch(inp_ch) {
	case KEY_DOWN:
	  song_move_down(win, TRUE);
	  break;
	  
	case KEY_UP: 
	  song_move_up(win, TRUE);
	  break;
	  
	case KEY_HOME:
	case 362:
	  song_move_top(win, TRUE);
	  break;
	  
	case KEY_END:
	case 385:
	  song_move_bottom(win, TRUE);
	  break;
	  
	case KEY_PPAGE:
	  song_move_pup(win, TRUE);
	  break;
	  
	case KEY_NPAGE:
	  song_move_pdown(win, TRUE);
	  break;
	  
	case KEY_DC:
	case ' ':
	  if(lay_curr_anchor[lay_act_side]->convert) {
	    lay_curr_anchor[lay_act_side]->convert = FALSE;
	    lay_tot_frm[lay_act_side] -= lay_curr_anchor[lay_act_side]->frame_len;
	  } else {
	    lay_curr_anchor[lay_act_side]->convert = TRUE;
	    lay_tot_frm[lay_act_side] += lay_curr_anchor[lay_act_side]->frame_len;
	  }
	  draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	  draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  song_move_down(win, FALSE);
	  wrefresh(win);
	  break;

	case 'L':
	  if(lay_curr_anchor[lay_act_side]->sampler) {
	    lay_curr_anchor[lay_act_side]->sampler = FALSE;
	  } else {
	    lay_curr_anchor[lay_act_side]->sampler = TRUE;
	  }
	  filenm_generate(lay_curr_anchor[lay_act_side]);
	  draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  song_move_down(win, FALSE);
	  wrefresh(win);
	  break;
	  
	case 's':
	case 'S':
	  a_t_swap(lay_curr_anchor[lay_act_side]);
	  draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	  draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  wrefresh(win);
	  break;
	  
	case 'o':
	case 'O':
	  if(lay_curr_anchor[lay_act_side]->on_fly) lay_curr_anchor[lay_act_side]->on_fly = FALSE;
	  else                                      lay_curr_anchor[lay_act_side]->on_fly = TRUE;
	  draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  wrefresh(win);
	  break;
	  
	case 'p':
	case 'P':
	  if(lay_act_side == 0)
	    play_track((lay_curr_anchor[lay_act_side]->toc)+1, (lay_curr_anchor[lay_act_side]->toc)+1);
	  break;

	case 'x':
	case 'X':
	  stop_cd();
	  break;

	case KEY_F(4):
	case '4':
	case KEY_F(7):
	case '7':
	  if(lay_act_side == 0) {
	    if(inp_ch == KEY_F(4) || inp_ch == '4' || lay_curr_anchor[lay_act_side]->on_fly) {
	      if(lay_curr_anchor[lay_act_side]->on_fly) {
		if(conv_on_fly(lay_curr_anchor[lay_act_side])) break;
		if(set_mp3_inf(lay_curr_anchor[lay_act_side])) break;
		add_to_m3u(lay_curr_anchor[lay_act_side]);
		if(lay_curr_anchor[lay_act_side]->convert) {
		  lay_curr_anchor[lay_act_side]->convert = FALSE;
		  lay_tot_frm[lay_act_side] -= lay_curr_anchor[lay_act_side]->frame_len;
		}
		song_move_down(win, TRUE);
		draw_song_status(win, lay_curr_anchor[lay_act_side]);
	      } else {
		if(rip_non_fly(lay_curr_anchor[lay_act_side])) break;
		if(enc_non_fly(lay_curr_anchor[lay_act_side], TRUE)) {
		  rhand_add_ripped(lay_curr_anchor[lay_act_side], win);
		  song_move_down(win, TRUE);
		  break;
		}
		if(set_mp3_inf(lay_curr_anchor[lay_act_side])) break;
		add_to_m3u(lay_curr_anchor[lay_act_side]);
		if(lay_curr_anchor[lay_act_side]->convert) {
		  lay_curr_anchor[lay_act_side]->convert = FALSE;
		  lay_tot_frm[lay_act_side] -= lay_curr_anchor[lay_act_side]->frame_len;
		}
		draw_song_status(win, lay_curr_anchor[lay_act_side]);
		song_move_down(win, TRUE);
	      }
	    } else {      /* only rip */
	      if(rip_non_fly(lay_curr_anchor[lay_act_side])) break;
	      rhand_add_ripped(lay_curr_anchor[lay_act_side], win);
	      song_move_down(win, TRUE);
	    }
	  } else {        /* do encoding */
	    rhand_do_enc(lay_curr_anchor[lay_act_side], win);
	  }
	  break;

	case KEY_F(8):
	case '8':
	  if(lay_act_side == 1) {
	    rhand_del_marked(win);
	  }
	  break;

	case KEY_F(9):
	case '9':
	  if(lay_act_side == 1) {
	    rhand_del_actual(win);
	  }
	  break;
	  
	case 'g':
	case 'G':
	  i = select_genre();
	  if(i != -1) {
	    lay_curr_anchor[lay_act_side]->genre = i;
	    draw_song_status(win, lay_curr_anchor[lay_act_side]);
	    wrefresh(win);
	  }
	  break;
	  
	case 't':
	case 'T':
	  pointer2 = lay_curr_anchor[lay_act_side];
	  pointer1 = input_line(7, lay_curr_anchor[lay_act_side]->title, _("Input title for this track"), TRUE);
	  if(pointer1) {
	    free(pointer2->title);
	    pointer2->title = pointer1;
	    filenm_generate(pointer2);
	  }
	  if(pointer2 == lay_curr_anchor[lay_act_side]) {
  	    draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	    draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  } else {
	    fill_in_songs(win);
	  }
	  break;
	  
	case 'a':
	case 'A':
	  pointer2 = lay_curr_anchor[lay_act_side];
	  pointer1 = input_line(7, lay_curr_anchor[lay_act_side]->artist, _("Input artist for this track"), TRUE);
	  if(pointer1) {
	    free(pointer2->artist);
	    pointer2->artist = pointer1;
	    filenm_generate(pointer2);
	  }
	  if(pointer2 == lay_curr_anchor[lay_act_side]) {
	    draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	    draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  } else fill_in_side(win, lay_act_side);
	  break;
	  
	case 'b':
	case 'B':
	  pointer2 = lay_curr_anchor[lay_act_side];
	  pointer1 = input_line(7, lay_curr_anchor[lay_act_side]->album, _("Input albumname for this track"), TRUE);
      	  if(pointer1) {
	    free(pointer2->album);
	    pointer2->album = pointer1;
	    filenm_generate(pointer2);
	  }
	  if(pointer2 == lay_curr_anchor[lay_act_side]) {
	    draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	    draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  } else fill_in_side(win, lay_act_side);
	  break;

	case 'c':
	case 'C':
	  pointer2 = lay_curr_anchor[lay_act_side];
	  pointer1 = input_line(7, lay_curr_anchor[lay_act_side]->comment, _("Input comment for this track"), TRUE);
	  if(pointer1) {
	    free(pointer2->comment);
	    pointer2->comment = pointer1;
	    filenm_generate(pointer2);
	  }
	  if(pointer2 == lay_curr_anchor[lay_act_side]) {
	    draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	    draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  } else fill_in_side(win, lay_act_side);
	  break;
	    
	case 'y':
	case 'Y':
	  pointer2 = lay_curr_anchor[lay_act_side];
	  pointer3 = (char *) malloc(sizeof(char) * 7);
	  if(! pointer3) {
	    perror("malloc");
	    wuuush(1);
	  }
	  sprintf(pointer3, "%d", lay_curr_anchor[lay_act_side]->year);
	  pointer1 = input_line(7, pointer3, _("Input year for this track"), TRUE);
	  free(pointer3);
	  if(pointer1) {
	    if(strlen(pointer1) <= 4) {
	      lay_curr_anchor[lay_act_side]->year = atoi(pointer1);
	      if((lay_curr_anchor[lay_act_side]->year < 0) || (lay_curr_anchor[lay_act_side]->year > 9999))
		lay_curr_anchor[lay_act_side]->year = 1999;
	    }
	    free(pointer1);
	    filenm_generate(pointer2);
	  }
	  if(pointer2 == lay_curr_anchor[lay_act_side]) {
	    draw_song_line(win, lay_curr_anchor[lay_act_side], TRUE, lay_select_line[lay_act_side]);
	    draw_song_status(win, lay_curr_anchor[lay_act_side]);
	  } else fill_in_songs(win);
	  break;

	case KEY_ENTER:
	case '\n':
	case '\r':
	  alter_track_infos_menu(lay_curr_anchor[lay_act_side]);
	  fill_in_songs(win);
	  break;

	case 'w':    /* write actual track */
	  if(lay_act_side == 1) {
	    rhand_save_ripped(def_exp_file, clear_del_on_exp, 0);
	  }
	  break;

	case 'W':    /* write marked track */
	  rhand_save_ripped(def_exp_file, clear_del_on_exp, 1);
	  break;
      }
    }
    switch(inp_ch) {
      case KEY_REFRESH:
      case '':
	set_active_win(win);
	break;

      case KEY_STAB:
      case KEY_CTAB:
      case '\t':
	if((lay_act_side == 0) && lay_global_anchor[1]) {
	  lay_act_side = 1;
	  fill_in_songs(win);
	} else if((lay_act_side == 1) && lay_global_anchor[0]) {
	  lay_act_side = 0;
	  fill_in_songs(win);
	}
	break;

      case 'e':
	keys_open_tray(win);
	break;

      case 'E':
      	close_tray();
	reinit_volume(win);
	break;
	
      case 'l':
      	if(config_cddb_enbl) config_cddb_enbl = FALSE;
	else                 config_cddb_enbl = TRUE;
	layout_update_cddb_flag(win);
	break;

      case 'd':
      case 'D':
	rhand_remove_dead_files(win);
	break;

      case 'm':
	 cddb_save_cddb_entry(lay_global_anchor[0], def_cddb_bank);
	 break;

      case 'M':
	 cddb_submit_cddb_entry(lay_global_anchor[0]);
	 break;

      case '*':
	pointer2 = lay_global_anchor[lay_act_side];
	while(pointer2) {
	  if(pointer2->convert) pointer2->convert = FALSE;
	  else                  pointer2->convert = TRUE;
	  pointer2 = pointer2->next;
	}
	calc_tot_frm();
	fill_in_side(win, lay_act_side);
	break;

      case 'v':
	reinit_volume(win);
	break;

      case 'V':
	init_sampler(win);
	break;

      case 'r':    /* read tracks */
	rhand_load_ripped(def_exp_file, win);
	break;

      case KEY_F(1):
      case 'h':
      case 'H':
      case '1':
	pop_up_help();
	break;

      case KEY_F(2):
      case '2':
	option_menu(win);
	break;

      case KEY_F(3):         /* start encoding */
      case '3':
      case KEY_F(6):
      case '6':
	pointer2 = lay_global_anchor[lay_act_side];
	if(lay_act_side == 1) {
	  while(pointer2) {
	    old_song = pointer2->next;
	    if(pointer2->convert) {
	      rhand_do_enc(pointer2, win);
	    }
	    pointer2 = old_song;
	  }
	  break;
	}
	/* act_side == left, do the same as in old versions */
	abort_flag = FALSE;
	old_song   = NULL;
	if(rip_enc_ordered || (inp_ch == KEY_F(6)) || (inp_ch == '6')) {
	  /* rip only is activated, or order is set to rip first,
	     nevertheless, we process our list now, and rip (maybe we must
	     do on-the-fly encoding). If rip-only was activated we can
	     leave after ripping is done, otherwise we process the list again
	     and try to encode the ripped tracks (if a break occurs here, we
	     just put all ripped tracks to our right-list) */
	  while(pointer2 && (! abort_flag)) {
	    i = 0;
	    if(pointer2->convert) {      /* track should converted */
	      if(pointer2->on_fly) {     /* and this on-the fly    */
		i = conv_on_fly(pointer2);
		if(! i) i = set_mp3_inf(pointer2);
		if(! i) {
		  add_to_m3u(pointer2);
		  pointer2->convert = FALSE;
		  lay_tot_frm[lay_act_side] -= pointer2->frame_len;
		  if(pointer2 == lay_curr_anchor[lay_act_side]) song_move_down(win, TRUE);
		}
	      } else {                   /* just rip track */
		i = rip_non_fly(pointer2);
		if(!i && (inp_ch == KEY_F(6) || inp_ch == '6')) {
		  /* track was ripped successfully, and we are in rip-only mode, so
		     add this track to right-list */
		  rhand_add_ripped(pointer2, win);
		  if(pointer2 == lay_curr_anchor[lay_act_side]) song_move_down(win, TRUE);
		}
	      }
	    }
	    if(i) {     /* so, error or abort occured */
      	      if(inp_ch == KEY_F(6) || inp_ch == '6') {      /* exit in rip-only mode           */
		abort_flag = TRUE;
		break;
	      }
	      old_song = pointer2->prev;                     /* just process tracks before this */
	      if(! old_song) {                               /* break if this was first track   */
		abort_flag = TRUE;
		break;
	      }
	      pointer2 = NULL;
	    } else {
	      pointer2 = pointer2->next;
	    }
	  }
	  if(inp_ch == KEY_F(6) || inp_ch == '6') {          /* if in rip-only mode, we are finish */
	    if(lay_act_side == 0 && config_open_tray &&      /* left list of ripping jobs is empty */
	       keys_joblist_is_empty(lay_global_anchor[0])) {
	      keys_open_tray(win);                           /* open the tray                      */
	    }
	    break;
	  }
	}
	if(abort_flag) break;
	
	/* we reached this point:
	   a. we were in rip-first mode and need now to encode ripped tracks
	      (maybe it was aborted, and only some tracks should be processed,
	      a break here would cause to swap ripped tracks to right-list)
	   b. we are in normal mode and process the list totally doing
	      - on-fly encoding (just break it if error or abort occurs)
	      - ripping and encoding (break here too if error occurs, but put
	        an already ripped track to right-list
	 */
	pointer2 = lay_global_anchor[lay_act_side];
	while(pointer2 && (! abort_flag)) {
	  if(pointer2->convert) {
	    if(rip_enc_ordered) {   /* possibility (a) */
	      i = enc_non_fly(pointer2, TRUE);
	      if(! i) i = set_mp3_inf(pointer2);
	      if(! i) {
		add_to_m3u(pointer2);
		pointer2->convert = FALSE;
		lay_tot_frm[lay_act_side] -= pointer2->frame_len;
		if(pointer2 == lay_curr_anchor[lay_act_side]) song_move_down(win, TRUE);
	      } else {              /* error occured, put ripped tracks to right list */
		abort_flag = TRUE;
		while(pointer2) {
		  if(pointer2->convert) rhand_add_ripped(pointer2, win);
		  if(pointer2 == old_song) break;
		  pointer2 = pointer2->next;
		}
	      }
	    } else {                /* possibility (b) */
	      if(pointer2->on_fly) {
		i = conv_on_fly(pointer2);
	      } else {
		i = rip_non_fly(pointer2);
		if(! i) {
		  i = enc_non_fly(pointer2, TRUE);
		  if(i) {
		    rhand_add_ripped(pointer2, win);
		    abort_flag = TRUE;
		  }
		}
	      }
	      if(! i) i = set_mp3_inf(pointer2);
	      if(! i) {
		add_to_m3u(pointer2);
		pointer2->convert = FALSE;
		lay_tot_frm[lay_act_side] -= pointer2->frame_len;
		if(pointer2 == lay_curr_anchor[lay_act_side]) song_move_down(win, TRUE);
	      } else {
		abort_flag = TRUE;
	      }
	    }
	  }
	  if(rip_enc_ordered && pointer2 == old_song) abort_flag = TRUE;
	  else pointer2 = pointer2->next;
	}

	if(lay_act_side == 0 && config_open_tray &&
	   keys_joblist_is_empty(lay_global_anchor[0])) {
	  keys_open_tray(win);
	} else {
	  fill_in_songs(win);
	}
	break;

      case KEY_F(5):         /* output batch-script */
      case '5':
	if(lay_global_anchor[lay_act_side]) {
	  /* there are datas for a batch-script */
	  pointer1 = req_get_file(NULL, _("select file for batch-script"));
	  if(pointer1) {
	    if(output_batch(lay_global_anchor[lay_act_side], pointer1, TRUE) == 0) {
	      popup_error_win(_("script created"));
	    } else {
	      popup_error_win(_("scriptcreation failed"));
	    }
	    free(pointer1);
	  }
	} else {
	  popup_error_win(_("no datas for scriptcreation"));
	}
	break;
	  
      case 'F':
	setup_stat_win(50);
	cddb_internet_lookup (def_cddb_server, def_cddb_bank, TRUE);
	destroy_stat_win();
	reinit_volume(win);
	break;
	
      case KEY_F(12):
      case 'q':
      case 'Q':
      case '\'':
	if(select_yesno_box(_("Really leave this program?"))) return;
	break;
	
      case 27:
	halfdelay(1);
	inp_ch = wgetch(win);
	switch(inp_ch) {
	  case 's':     /* swap artist and title for all entries */
	    pointer2 = lay_global_anchor[lay_act_side];
	    while(pointer2) {
	      if(pointer2->convert) a_t_swap(pointer2);
	      pointer2 = pointer2->next;
	    }
	    fill_in_side(win, lay_act_side);
	    break;
	    
	  case 'g':     /* input genre for selected entries */
	    pointer2 = lay_global_anchor[lay_act_side];
	    if(pointer2) i = select_genre();
	    else         break;
	    if(i == -1)  break;
	    while(pointer2) {
	      if(pointer2->convert) {
		pointer2->genre = i;
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    fill_in_side(win, lay_act_side);
	    break;
	    
	  case 'o':     /* swap on-fly flag */
	    pointer2 = lay_global_anchor[lay_act_side];
	    while(pointer2) {
	      if(pointer2->convert) {
		if(pointer2->on_fly) pointer2->on_fly = FALSE;
		else                 pointer2->on_fly = TRUE;
	      }
	      pointer2 = pointer2->next;
	    }
	    if(lay_curr_anchor[lay_act_side]) draw_song_status(win, lay_curr_anchor[lay_act_side]);
	    break;
	    
	  case 'a':     /* input artist for selected tracks */
	    pointer2 = lay_global_anchor[lay_act_side];
	    if(!pointer2) break;
	    if(lay_curr_anchor[lay_act_side]) pointer1 = input_line(8, lay_curr_anchor[lay_act_side]->
						     		    artist, _("input artistname for"
								    " selected tracks"), TRUE);
	    else pointer1 = input_line(8, NULL, _("input artistname for selected tracks"), TRUE);
	    if(pointer1 == NULL) break;
	    while(pointer2) {
      	      if(pointer2->convert) {
		if(pointer2->artist) free(pointer2->artist);
		pointer3 = (char *) malloc(sizeof(char) * (strlen(pointer1)+1));
		if(pointer3 == NULL) {
		  perror("malloc");
		  wuuush(1);
		}
		strcpy(pointer3, pointer1);
		pointer2->artist = pointer3;
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    free(pointer1);
	    fill_in_side(win, lay_act_side);
	    break;
	    
	  case 'b':     /* input albumname for selected tracks */
	    pointer2 = lay_global_anchor[lay_act_side];
      	    if(!pointer2) break;
	    if(lay_curr_anchor[lay_act_side]) pointer1 = input_line(8, lay_curr_anchor[lay_act_side]->
						     		    album, _("input albumname for"
								    " selected tracks"), TRUE);
	    else pointer1 = input_line(8, NULL, _("input albumname for selected tracks"), TRUE);
	    if(pointer1 == NULL) break;
	    while(pointer2) {
	      if(pointer2->convert) {
		if(pointer2->album) free(pointer2->album);
		pointer3 = (char *) malloc(sizeof(char) * (strlen(pointer1)+1));
		if(pointer3 == NULL) {
		  perror("malloc");
		  wuuush(1);
		}
		strcpy(pointer3, pointer1);
		pointer2->album = pointer3;
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    free(pointer1);
	    fill_in_side(win, lay_act_side);
	    break;
	    
	  case 'c':     /* input comment for selected tracks */
	    pointer2 = lay_global_anchor[lay_act_side];
	    if(! pointer2) break;
	    if(lay_curr_anchor[lay_act_side]) pointer1 = input_line(8, lay_curr_anchor[lay_act_side]->comment,
	     							    _("input comment for selected tracks"), TRUE);
	    else pointer1 = input_line(8, NULL, _("input comment for selected tracks"), TRUE);
	    if(pointer1 == NULL) break;
	    while(pointer2) {
	      if(pointer2->convert) {
		if(pointer2->comment) free(pointer2->comment);
		pointer2->comment = (char *) malloc(sizeof(char) * (strlen(pointer1)+1));
		if(! (pointer2->comment)) {
		  perror("malloc");
		  wuuush(1);
		}
		strcpy(pointer2->comment, pointer1);
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    free(pointer1);
	    fill_in_side(win, lay_act_side);
	    break;

	  case 'T':     /* input title for selected tracks */
	    pointer2 = lay_global_anchor[lay_act_side];
	    if(! pointer2) break;
	    if(lay_curr_anchor[lay_act_side]) pointer1 = input_line(8, lay_curr_anchor[lay_act_side]->title,
								    _("input title for selected tracks"), TRUE);
	    else pointer1 = input_line(8, NULL, _("input title for selected tracks"), TRUE);
	    if(pointer1 == NULL) break;
	    while(pointer2) {
	      if(pointer2->convert) {
		if(pointer2->title) free(pointer2->title);
		pointer2->title = (char *) malloc(sizeof(char) * (strlen(pointer1)+1));
		if(! (pointer2->title)) {
		  perror("malloc");
		  wuuush(1);
		}
		strcpy(pointer2->title, pointer1);
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    free(pointer1);
	    fill_in_side(win, lay_act_side);
	    break;

	  case 'Y':     /* input year for selected tracks */
	  case 'j':
	    pointer2 = lay_global_anchor[lay_act_side];
	    if(! pointer2) break;
	    pointer1 = input_line(8, NULL, _("input year for all selected tracks"), TRUE);
	    if(! pointer1) break;
	    if(strlen(pointer1) > 4) {
	      free(pointer1);
	      break;
	    }
	    i = atoi(pointer1);
	    free(pointer1);
	    if((i<0) || (i>9999)) i = 1999;
	    while(pointer2) {
	      if(pointer2->convert) {
		pointer2->year = i;
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    fill_in_side(win, lay_act_side);
	    break;
	    
	  case 't':     /* input title for each selected track */
	    pointer2 = lay_global_anchor[lay_act_side];
	    while(pointer2) {
	      if(pointer2->convert) {
		pointer1 = (char *) malloc(sizeof(char) * 100);
		if(pointer1 == NULL) {
		  perror("malloc");
		  wuuush(1);
		}
		sprintf(pointer1, _("input tracktitle for nr.%d"), (pointer2->toc)+1);
		pointer3 = input_line(8, pointer2->title, pointer1, TRUE);
		free(pointer1);
		if(! pointer3) break;
		if(pointer2->title) free(pointer2->title);
		pointer2->title = pointer3;
		filenm_generate(pointer2);
	      }
	      pointer2 = pointer2->next;
	    }
	    fill_in_side(win, lay_act_side);
	    break;

	  case 'y':     /* input year for each selected track */
	    pointer2 = lay_global_anchor[lay_act_side];
	    while(pointer2) {
	      if(pointer2->convert) {
		pointer1 = (char *) malloc(sizeof(char) * 100);
		pointer3 = (char *) malloc(sizeof(char) * 7);
		if((pointer1 == NULL) || (pointer3 == NULL)) {
		  perror("malloc");
		  wuuush(1);
		}
		sprintf(pointer1, _("input year for nr.%d"), (pointer2->toc)+1);
		sprintf(pointer3, "%d", pointer2->year);
		pointer4 = input_line(8, pointer3, pointer1, TRUE);
		free(pointer1);
		free(pointer3);
		if(! pointer4) break;
		if(strlen(pointer4) <= 4) {
		  pointer2->year = atoi(pointer4);
		  if((pointer2->year < 0) || (pointer2->year > 9999))
		    pointer2->year = 1999;
		  filenm_generate(pointer2);
		}
		free(pointer4);
	      }
	      pointer2 = pointer2->next;
	    }
	    fill_in_side(win, lay_act_side);
	    break;
	}
	if(! config_fancy_colors) nodelay(win, FALSE);
	break;
    }
  }
}

