/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

/**
 * The layout specifying the position of the board and the scores
 * on the playing table.
 */
struct Layout2D {
    // Coordinates (top-left) of the table.
    // (Not user specified. Computed for absolute layout)
    public int x;
    public int y;
    // width and height of the table using an arbitrary unit.
    public int width;
    public int height;
    // Coordinates (top-left) and size of the board
    public int board_x;
    public int board_y;
    public int board_size;
    // Coordinates (centre) and size of the scores
    public int score_x[2];
    public int score_y[2];
    public int score_font_size;
    public Rgb score_color[2];
    // Coordinates (centre) of the kases
    public int kas_x[2];
    public int kas_y[2];
    // Which kas frames to show
    public int kas_frame[2];
    // Coordinates (centre) and size for number of wins
    public int num_wins_x[2];
    public int num_wins_y[2];
    public int num_wins_font_size;
    public Rgb num_wins_color;
    // Coordinate (centre) and size for number of rounds
    public int num_rounds_x;
    public int num_rounds_y;
    public int num_rounds_format;   // 0 = "Round\n#",  1 = "Round #",  3 = "#"
    public int num_rounds_font_size;
    public Rgb num_rounds_color;
    // Filler to allow the table image to fill the edges without changing the aspect ratio.
    // Filler should not contain important images that must stay visible
    public int filler_left;
    public int filler_top;
    public int filler_right;
    public int filler_bottom;
    // Coordinates and size of the entire image (table + filler). Coordinates may be negative.
    // (Not user specified. Computed for absolute layout)
    public int image_x;
    public int image_y;
    public int image_width;
    public int image_height;
}

abstract class Theme2D : Theme {
    /**
     * Relative layout. Coordinates are relative to the (x,y).
     * DO NOT CHANGE the default layout. This is assumed by the
     * built-in themes.
     */
    protected Layout2D default_layout = Layout2D () {
        width = 1200, height = 1000,
        board_x = 0, board_y = 0,
        board_size = 1000,
        score_x = {1100, 1100},
        score_y = {300, 700},
        score_font_size = 40,
        score_color = {Rgb (0, 0, 0), Rgb(0, 0, 0)},
        kas_x = {1100, 1100},
        kas_y = {400, 600},
        kas_frame = {0, 0},
        num_wins_x = {1100, 1100},
        num_wins_y = {100, 900},
        num_wins_font_size = 50,
        num_wins_color = Rgb (0, 0, 0),
        num_rounds_x = 1100,
        num_rounds_y = 500,
        num_rounds_format = 0,
        num_rounds_font_size = 20,
        num_rounds_color = Rgb (0, 0, 0),
        filler_left = 0,
        filler_top = 0,
        filler_right = 0,
        filler_bottom = 0
    };

    protected Layout2D layout;

    construct {
        layout = default_layout;
        for (int i=0; i < Piece.COUNT; i++) num_rows_in_image[i] = 1;
    }

    /**
     * Absolute layout. Coordinates are relative to (0,0).
     * Rgb fields are ignored.
     */
    protected Layout2D table;

    /**
     * Set to true if the layout/theme has been changed
     */
    protected bool layout_changed = true;

    /**
     * Each image consists of square frames. Refer to the Wooden theme for example.
     */
    protected Cairo.ImageSurface[] images = new Cairo.ImageSurface[Piece.COUNT + 1];

    /**
     * Number of rows in images[]
     */
    protected int[] num_rows_in_image = new int[Piece.COUNT];

    public override void free () {
        for (int i=0; i < images.length; i++) images[i] = null;
    }

    /**
     * Set table according to layout and the allocated size.
     * Then resize images.
     */
    public override void resize (int allocated_width, int allocated_height) {
        // Compute a scale such that each cell on the board has an integer width
        int width_for_board_only = allocated_width * layout.board_size / layout.width;
        int height_for_board_only = allocated_height * layout.board_size / layout.height;
        int size = int.min (width_for_board_only, height_for_board_only) / BOARD_WIDTH * BOARD_WIDTH;
        double scale = (double) size / layout.board_size;

        layout_changed |= (size != table.board_size);
        table.board_size = size;

        // Horizontal position of the image
        table.width = (int) (scale * layout.width);
        table.filler_left = (int) (scale * layout.filler_left);
        table.filler_right = (int) (scale * layout.filler_right);
        table.image_width = table.width + table.filler_left + table.filler_right;
        var gap = allocated_width - table.image_width;
        if (gap > 0 || layout.filler_left == layout.filler_right) {
            // Image is fully visible, or table is centered (also takes care of filler == 0)
            // Divide borders around the image evenly to the left and right.
            table.image_x = gap / 2;
            table.x = table.image_x + table.filler_left;
        }
        else {
            // Image is partly obscured.
            var filler = allocated_width - table.width;
            table.x = filler * layout.filler_left / (layout.filler_left + layout.filler_right);
            table.image_x = table.x - table.filler_left;
        }
        table.board_x = table.x + (int) (scale * layout.board_x);

        // Vertical position of the image
        table.height = (int) (scale * layout.height);
        table.filler_top = (int) (scale * layout.filler_top);
        table.filler_bottom = (int) (scale * layout.filler_bottom);
        table.image_height = table.height + table.filler_top + table.filler_bottom;
        gap = allocated_height - table.image_height;
        if (gap > 0 || layout.filler_top == layout.filler_bottom) {
            table.image_y = gap / 2;
            table.y = table.image_y + table.filler_top;
        }
        else {
            var filler = allocated_height - table.height;
            table.y = filler * layout.filler_top / (layout.filler_top + layout.filler_bottom);
            table.image_y = table.y - table.filler_top;
        }
        table.board_y = table.y + (int) (scale * layout.board_y);

        for (int i=0; i < 2; i++) {
            table.score_x[i] = table.x + (int) (scale * layout.score_x[i]);
            table.score_y[i] = table.y + (int) (scale * layout.score_y[i]);
            table.kas_x[i] = table.x + (int) (scale * layout.kas_x[i]);
            table.kas_y[i] = table.y + (int) (scale * layout.kas_y[i]);
            table.kas_frame[i] = layout.kas_frame[i];
            table.num_wins_x[i] = table.x + (int) (scale * layout.num_wins_x[i]);
            table.num_wins_y[i] = table.y + (int) (scale * layout.num_wins_y[i]);
        }
        table.score_font_size = (int) (scale * layout.score_font_size);
        table.num_wins_font_size = (int) (scale * layout.num_wins_font_size);

        table.num_rounds_x = table.x + (int) (scale * layout.num_rounds_x);
        table.num_rounds_y = table.y + (int) (scale * layout.num_rounds_y);
        table.num_rounds_font_size = (int) (scale * layout.num_rounds_font_size);

        if (layout_changed) {
            layout_changed = false;
            resize_images ();
        }
    }

    /**
     * Resize images based on board_size.
     * Input: table.
     */
    public abstract void resize_images ();

    public override void draw (GameView game_view, Cairo.Context cr) {
        draw_table (cr);
        draw_scores (cr, game_view.game);
        draw_pieces (game_view, cr);
        draw_message (cr, game_view.message);
    }

    private void draw_pieces (GameView game_view, Cairo.Context cr) {
        GameSeries game = game_view.game;
        Stage stage = game_view.game.stage;
        Move move = game_view.selected_move;
        Point kas_point = game_view.kas_point;
        for (int layer=0; layer < 150; layer++) {
            for (int position=0; position < BOARD_SIZE; position++) {
                if (films[position].viewed != layer) continue;

                // Skip blanks and guards
                Piece piece = game.board[position];
                if (! piece.is_real ()) continue;
                
                // If we are making a kas, the source square should be shown empty
                // (actually a piece is still there on the board because the move
                // is not completed/submitted yet).
                if (stage == Stage.SELECT && move != null && position == move.position) continue;

                // If we are dragging a kas, the source square should be empty.
                // (actually the kas is still there on the board, but we are going to
                // show the kas being dragged after this loop).
                if (stage == Stage.MOVE && kas_point.x != -1 && piece == game.kas) continue;

                Point p = to_point (position, game.rotated);
                int radius = piece_image_width (piece) / 2;
                draw_piece (cr, piece, films[position].viewed, p.x - radius,  p.y - radius);
            }//endfor position
            // Draw dragged kas
            if (films[game.kas].viewed != layer) continue;
            if ((stage == Stage.SELECT && move != null) || 
                (stage == Stage.MOVE && kas_point.x != -1)) {
                Piece piece = game.kas;
                int radius = piece_image_width (piece) / 2;
                draw_piece (cr, piece, films[game.kas].viewed, kas_point.x - radius,  kas_point.y - radius);
            }
        }//endfor layer
    }

    private void draw_piece (Cairo.Context cr, Piece piece, int frame, int x, int y)
        requires (piece.is_real ()) {
        cr.save ();
        frame = int.min (frame, count_frames (piece) - 1);
        var frames_per_row = images[piece].get_width () / piece_image_width (piece);
        var frame_col = frame % frames_per_row;
        var frame_row = frame / frames_per_row;
        int width = piece_image_width (piece);
        cr.translate (x, y);
        cr.set_source_surface (images[piece], - frame_col * width, - frame_row * width);
        cr.rectangle (0, 0, width, width);
        cr.fill ();
        cr.restore ();
    }

    private void draw_table (Cairo.Context cr) {
        cr.save ();
        cr.translate(table.image_x, table.image_y);
        cr.set_source_surface (images[Piece.BOARD], 0, 0);
        cr.paint ();
        cr.restore ();
    }

    private void draw_scores (Cairo.Context cr, GameSeries game) {
        for (int player = 0; player < 2; player++) {       // 0 = first player, 1 = second
            int side = game.rotated ? 1 - player : player;  // 0 = top, 1 = bottom
            // Show kas
            Piece kas = player == 0 ? Piece.KAS_0 : Piece.KAS_1;
            int radius = piece_image_width (kas) / 2;
            int sx = table.kas_x[side] - radius;
            int sy = table.kas_y[side] - radius;
            draw_piece (cr, kas, table.kas_frame[player], sx, sy);
            // Show scores
            layout.score_color[player].set_source_for (cr);
            show_text (cr, "%d".printf (game.score[player]), table.score_font_size,
                table.score_x[side], table.score_y[side]);
            layout.num_wins_color.set_source_for (cr);
            show_text (cr, "%d".printf (game.num_wins[side]), table.num_wins_font_size,
                table.num_wins_x[side], table.num_wins_y[side]);
        }
        int round = game.stage == Stage.GAME_OVER ? game.num_rounds : game.num_rounds + 1;
        layout.num_rounds_color.set_source_for (cr);
        string format = layout.num_rounds_format == 0 ? _("Round\n%d") :
                         layout.num_rounds_format == 1 ? _("Round %d") : "%d";
        show_text (cr, format.printf (round), table.num_rounds_font_size,
            table.num_rounds_x, table.num_rounds_y);
    }

    private void draw_message (Cairo.Context cr, string? message) {
        if (message == null) return;
        cr.set_source_rgb (0, 0, 0);
        show_text (cr, message, table.board_size / BOARD_WIDTH / 3,
            table.board_x + table.board_size / 2, table.board_y + table.board_size / 4, true);
    }

    /**
     * Show text centred at x,y
     */
    protected void show_text (Cairo.Context cr, string text, int font_size,
        int cx, int cy, bool boxed = false)
    {
        var font = new Pango.FontDescription ();
        font.set_family ("Sans");
        font.set_size ((int)(font_size * Pango.SCALE));

        var pango_layout = Pango.cairo_create_layout (cr);
        pango_layout.set_font_description (font);
        pango_layout.set_alignment (Pango.Alignment.CENTER);
        pango_layout.set_text (text, -1);
        int width, height;
        pango_layout.get_pixel_size (out width, out height);
        int x = cx - width / 2;
        int y = cy - height / 2;
        if (boxed) {
            cr.save ();
            cr.set_source_rgba (1, 1, 1, 0.6);
            var gap = 0.8 * font_size;
            cr.rectangle (x - gap, y - gap, width + 2 * gap, height + 2 * gap);
            cr.fill ();
            cr.set_source_rgb (0.5, 0.5, 0.5);
            cr.move_to (x + 1, y + 1);
            Pango.cairo_show_layout (cr, pango_layout);
            cr.restore ();
        }
        cr.move_to (x, y);
        Pango.cairo_show_layout (cr, pango_layout);
    }

    public override void queue_draw_piece (Gtk.DrawingArea view, Piece piece, Point p) {
        if (p.x == -1) return;
        int width = piece == Piece.EMPTY ? cell_width () : piece_image_width (piece);
        view.queue_draw_area (p.x - width / 2, p.y - width / 2, width, width);
    }

    public override int count_frames (Piece piece) 
        requires (piece.is_real ()) {
        return images[piece].get_width () / piece_image_width (piece) * num_rows_in_image[piece];
    }

    int piece_image_width (Piece piece)
        requires (piece.is_real ()) {
        return images[piece].get_height () / num_rows_in_image[piece];
    }

    protected int cell_width () {
        return table.board_size / BOARD_WIDTH;
    }

    public override Point to_point (int pos, bool rotated)
        requires (pos >= 0 && pos < BOARD_SIZE) {
        if (rotated) pos = BOARD_SIZE -1 - pos;
        int border_width = cell_width () / 2;
        return Point.xy (
            table.board_x + pos % BOARD_WIDTH * cell_width () + border_width,
            table.board_y + pos / BOARD_WIDTH * cell_width () + border_width
        );
    }

    public override int to_position (Game game, Point raw_p, bool rotated, bool precise) {
        // p = point relative to top-left of the board.
        Point p = Point.xy (raw_p.x - table.board_x, raw_p.y - table.board_y);
        // Locate the nearest piece
        int col = p.x / cell_width ();
        int row = p.y / cell_width ();
        if (col < 1 || col > 11 || row < 1 || row > 11) return -1;
        int pos = row * BOARD_WIDTH + col;
        if (rotated) {
            pos = BOARD_SIZE - 1 - pos;
        }
        // Get the centre of this piece relative to top-left of the board
        Point c = to_point (pos, rotated);
        c.x -= table.board_x;
        c.y -= table.board_y;
        // Relative position of p to c
        int x = p.x - c.x;
        int y = p.y - c.y;
        Piece piece = game.board[pos];
        int snap_range = 1 + (precise ? 1 : cell_width () / 6);  
        if // p is near c, even if the piece has a hole in the centre.
           ((p.x - c.x) * (p.x - c.x) + (p.y - c.y)*(p.y - c.y) <= snap_range * snap_range ||
           // p is touching a piece.
           !precise && piece.is_real () && detect_piece (piece, 0, x, y) ||
           // p is near a vacant position and the dragged kas is touching c.
           !precise && piece == Piece.EMPTY &&
            (detect_piece (game.kas, 0, -x, -y) || detect_piece (game.kas, films[0].viewed, -x, -y))) {
            return pos;
        }
        else {
            return -1;
        }
    }

    bool detect_piece (Piece piece, int frame, int x, int y)
        requires (piece.is_real ()) {
        int width = piece_image_width (piece);
        x += width / 2;
        y += width / 2;
        frame = int.min (frame, count_frames (piece) - 1);
        var frames_per_row = images[piece].get_width () / piece_image_width (piece);
        var frame_col = frame % frames_per_row;
        var frame_row = frame / frames_per_row;
        weak uchar[] data = images[piece].get_data ();
        var n = 4 * ((frame_row * width + y) * images[piece].get_width () + frame_col * width + x);
        return (data[n] | data[n + 1] | data[n + 2] | data[n + 3]) != 0;
    }

    public override int to_side (Point p) {
        if (p.x >= table.board_x && p.x <= table.board_x + table.board_size) {
            if (p.y >= table.board_y && p.y <= table.board_y + table.board_size / 2)
                return 0;
            else if (p.y >= table.board_y + table.board_size / 2 && p.y <= table.board_y + table.board_size)
                return 1;
        }
        return 1;
    }
}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
