/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class PatternEditor : Gtk.Grid {
    private Gtk.DrawingArea editing_box = null;
    private PatternSelector pattern_selector;

    /**
     * List of custom patterns.
     */
    private CustomPattern[] patterns = {};

    /**
     * Container for list of custom patterns.
     */
    private Gtk.Grid pattern_button_box = new Gtk.Grid ();

    /**
     * File where custom patterns are saved.
     */
    private string custom_pattern_filename;

    /**
     * The pattern currently selected for editing.
     */
    private weak CustomPattern active_pattern;

    /**
     * Colour to use when the mouse is dragged.
     */
    private Piece active_color = Piece.NULL;

    /**
     * Symmetry type:
     *    S : least symmetry
     *    F : fan (90 degree rotation)
     *    H : axis fold
     *    / : diagonal fold
     *    X : axis and diagonal fold
     */
    private char symmetry = 'S';

    /**
     * The cursor position tracked by the mouse. Other cursor positions are derived by symmetry.
     */
    private Point cursor = Point.xy (0, 0);

    /**
     * Cursor positions derived from the one tracked by the mouse according to the currently
     * chosen symmetry.
     */
    private Point[] cursors = {};

    private Gtk.GestureDrag gesture;
    private Gtk.EventControllerMotion event_controller;

    public PatternEditor (PatternSelector ps) {
        pattern_selector = ps;
        attach (create_symmetry_box (), 0, 0, 2, 1);
        attach (new Gtk.Separator (Gtk.Orientation.HORIZONTAL), 0, 1, 2, 1);
        var scrolled_box = new Gtk.ScrolledWindow (null, null);
        scrolled_box.add (pattern_button_box);
        scrolled_box.set_policy (Gtk.PolicyType.NEVER, Gtk.PolicyType.ALWAYS);
        attach (scrolled_box, 0, 2, 1, 1);
        var editing_box = create_editing_box ();
        editing_box.hexpand = true;
        editing_box.vexpand = true;
        attach (editing_box, 1, 2, 1, 1);
        init_pattern_list ();
        apply (false);
    }

    /**
     * Create pattern list from pattern file. If the pattern file has not been created, then
     * populate the list with a few sample patterns.
     */
    private void init_pattern_list () {
        pattern_button_box.set_orientation (Gtk.Orientation.VERTICAL);
        custom_pattern_filename = Path.build_filename (Environment.get_home_dir (), Config.USER_DATADIR, "custom-patterns", null);
        // Load pattern from file
        if (FileUtils.test (custom_pattern_filename, FileTest.EXISTS)) {
            try {
                string s;
                FileUtils.get_contents (custom_pattern_filename, out s);
                var contents = s.split ("\n");
                foreach (var content in contents) {
                    // Check that the content consists of 30 H's, 30 P's and spaces only.
                    // Lines not following this rule are ignored.
                    if (Game.is_valid_pattern (content))
                        add_pattern (new CustomPattern.from_text (content));
                }
            }
            catch (Error e) {
                stderr.printf ("File error: %s\n", e.message);
            }
        }
        // No pattern file found. Create samples.
        else { 
            add_pattern (new CustomPattern.from_text ("PPPPPHPPPPPPHHPPHHHHHPPHPHHHHPPHPPHPHHHPHHPPPHHPHPHHHPPHHHHP"));
            add_pattern (new CustomPattern.from_text ("PPPPHHHPPPPPHHHHHHHHHPPHPPPPPPPHPPHPHHHHHPHPHHPHPPPHPHHHHPHP"));
            add_pattern (new CustomPattern.from_text ("HHHHPPPHHHHPPPHPHPHPPPPHPPPHPPPHPPHHHHHHHHHPPPPPHHHPPPPHHHPH"));
        }
        // Add a blank pattern and select it
        add_pattern (new CustomPattern ());
        active_pattern = patterns[patterns.length - 1];
        active_pattern.button.active = true;
    }

    /**
     * Ensure that user datadir exists.
     */
    private void ensure_user_datadir () {
        var dir_name = Path.build_filename (Environment.get_home_dir (), Config.USER_DATADIR, null);
        if (! FileUtils.test (dir_name, FileTest.EXISTS)) {
            Posix.mkdir (dir_name, 0777);
        }
    }

    /**
     * Submit all valid patterns (those with 60 whites & 60 blacks) to the pattern selector.
     * Save all valid patterns.
     */
    public void apply (bool must_save = true) {
        var text_to_save = "Custom patterns for Pasang Emas\nH=background (beHind), P=pattern\n";
        Pattern[] valid_patterns = {};
        foreach (var p in patterns) {
            if (p.count_whites () != 60) continue;
            valid_patterns += new CustomPattern.copy (p);
            text_to_save += p.dots + "\n";
        }
        pattern_selector.add_patterns (valid_patterns);
        if (must_save) {
            ensure_user_datadir ();
            try {
                FileUtils.set_contents (custom_pattern_filename, text_to_save);
            }
            catch (Error e) {
                stderr.printf ("File error: %s\n", e.message);
            }
        }
    }

    /**
     * Add a new pattern.
     * Side effect: patterns, pattern_buttons, pattern_button_box
     */
    private void add_pattern (CustomPattern pattern) {
        var button = pattern.button;
        if (patterns.length != 0) {
            button.set_group (patterns[0].button.get_group ());
        }
        button.set_mode (false);  // On/off only. No intermediate state.
        button.border_width = 0;
        button.relief = Gtk.ReliefStyle.NONE;
        button.clicked.connect (pattern_button_clicked);
        pattern_button_box.add (button);
        pattern_button_box.show_all ();
        patterns += pattern;
    }

    /**
     * The user wants to edit another pattern.
     */
    private void pattern_button_clicked (Gtk.Button source) {
        foreach (var pattern in patterns) {
            if (source == pattern.button) {
                active_pattern = pattern;
                editing_box.queue_draw ();
                return;
            }
        }
        assert_not_reached ();
    }

    /**
     * Create a visual list of symmetry types.
     */
    private Gtk.Widget create_symmetry_box () {
        var box = new Gtk.Grid ();
        box.set_orientation (Gtk.Orientation.HORIZONTAL);
        var symmetry_codes = "SFH/X";
        Gtk.RadioButton representative = null;
        var dot = 2;
        for (int i=0; i < symmetry_codes.length; i++) {
            var button = representative == null ? new Gtk.RadioButton (null) : new Gtk.RadioButton.from_widget (representative);
            representative = representative ?? button;
            button.set_mode (false);  // On/off only. No intermediate state.
            button.relief = Gtk.ReliefStyle.NONE;

            // Icon
            char code = (char)(symmetry_codes[i]);
            var icon = new Gtk.DrawingArea ();
            icon.set_size_request (11 * dot, 11 * dot);
            button.image = icon;
            icon.draw.connect ((cr) => {
                cr.rectangle (0, 0, 11 * dot, 11 * dot);
                cr.set_source (new Cairo.Pattern.rgb (0.2, 0.2, 0.2));
                cr.fill ();
                cr.rectangle (1 * dot, 3 * dot, dot, dot);
                cr.rectangle (9 * dot, 7 * dot, dot, dot);
                if (code == '/' || code == 'X') {
                    cr.rectangle (3 * dot, 1 * dot, dot, dot);
                    cr.rectangle (7 * dot, 9 * dot, dot, dot);
                }
                if (code == 'F' || code == 'X') {
                    cr.rectangle (7 * dot, 1 * dot, dot, dot);
                    cr.rectangle (3 * dot, 9 * dot, dot, dot);
                }
                if (code == 'H' || code == 'X') {
                    cr.rectangle (9 * dot, 3 * dot, dot, dot);
                    cr.rectangle (1 * dot, 7 * dot, dot, dot);
                }
                cr.set_source (new Cairo.Pattern.rgb (0.8, 0.8, 1));
                cr.fill ();
                return true;
            });

            // Response to user selection
            button.toggled.connect ((s) => {
                symmetry = code;
                set_cursors ();
            });
            box.add (button);
        }
        return box;
    }

    private Gtk.Widget create_editing_box () {
        editing_box = new Gtk.DrawingArea ();
        editing_box.set_size_request (BOARD_WIDTH * 25, BOARD_WIDTH * 25);
        editing_box.draw.connect ((cr) => {
            draw_editing_box (cr);
            return true;
        });

        // Non-dragging motion
        editing_box.add_events (Gdk.EventMask.POINTER_MOTION_MASK);
        event_controller = new Gtk.EventControllerMotion (editing_box);
        event_controller.motion.connect ((x, y) => {
            track ((int)x, (int)y);
        });

        // Drag gesture
        gesture = new Gtk.GestureDrag (editing_box);
        gesture.set_propagation_phase (Gtk.PropagationPhase.BUBBLE);
        gesture.set_button (1);
        gesture.drag_begin.connect ((x, y) => {
            track ((int)x, (int)y);
            toggle_color ();
        });
        gesture.drag_update.connect ((x, y) => {
            gesture.get_point (null, out x, out y);
            track ((int)x, (int)y);
            if (active_color != Piece.NULL) apply_color ();
        });
        gesture.drag_end.connect ((x, y) => {
            active_color = Piece.NULL;
        });

        return editing_box;
    }

    /**
     * Calculate the width of a cell, maximizing the board
     * to fill the allocated visual area.
     */
    private double cell_width {
        get {
            var alloc = Gtk.Allocation ();
            editing_box.get_allocation (out alloc);
            var width = int.min (alloc.width, alloc.height);
            return 1.0 * width / BOARD_WIDTH;
        }
    }

    /**
     * Show all cells that will change upon the next mouse click.
     * Side effect: cursors (computed from cursor and symmetry)
     */
    private void set_cursors () {
        redraw_cursors ();  // Erase old cursor positions
        cursors = {};
        // Translate origin to the centre
        var x = cursor.x - 6;
        var y = cursor.y - 6;
        if (x == 0 && y == 0 || x < -5 || x > 5 || y < -5 || y > 5) return;

        // Derive cursor positions based on chosen symmetry.
        // There are 8 possible positions:
        //
        //      |\1|2/|  0 : tracked cursor
        //      |0\|/3|  1 : reflection along diagonal
        //      |--+--|  2 : 90 degree rotation
        //      |7/|\4|  3 : reflection along vertical axis
        //      |/6|5\|  etc
        var s = symmetry.to_string ();
        cursors = {Point.xy (x, y)};                            // 0 in the diagram above
        if ("/X".contains (s)) cursors += Point.xy (y, x);      // 1 in the diagram above
        if ("FX".contains (s)) cursors += Point.xy (-y, x);     // 2
        if ("HX".contains (s)) cursors += Point.xy (-x, y);     // 3
        cursors += Point.xy (-x, -y);                           // 4
        if ("/X".contains (s)) cursors += Point.xy (-y, -x);    // 5
        if ("FX".contains (s)) cursors += Point.xy (y, -x);     // 6
        if ("HX".contains (s)) cursors += Point.xy (x, -y);     // 7

        // Undo "translate origin"
        for (int i=0; i < cursors.length; i++) {
            cursors[i].x += 6;
            cursors[i].y += 6;
        }
        redraw_cursors ();
    }

    /**
     * Convert mouse position (pixelx, pixely) into board position (row, column).
     * Side effect: cursor and (indirectly) cursors.
     */
    private void track (int x, int y) {
        var p = Point.xy ((int)(x / cell_width), (int)(y / cell_width));
        if (p.x == cursor.x && p.y == cursor.y) return;
        cursor = p;
        set_cursors ();
    }

    private void redraw_cursors () {
        foreach (var c in cursors) {
            editing_box.queue_draw_area ((int)(c.x * cell_width), (int)(c.y * cell_width),
                                          (int)(cell_width), (int)(cell_width));
        }
    }

    /**
     * Toggle the colour on the mouse-tracking cursor. 
     * Copy the new colour to the other cursors.
     */
    private void toggle_color () {
        if (cursor.x < 1 || cursor.x > 11 || cursor.y < 1 || cursor.y > 11 || cursor.x == 6 && cursor.y == 6) return;
        active_color = active_pattern.board[cursor.y * BOARD_WIDTH + cursor.x];
        active_color = active_color == Piece.WHITE ? Piece.BLACK : Piece.WHITE;
        apply_color ();
    }

    /**
     * Apply current_color on the board positions indicated by cursors.
     * If this is the last pattern in the list, and a valid pattern has been produced, then
     * add a new pattern button.
     */
    private void apply_color () {
        foreach (var c in cursors) {
            active_pattern.board[c.y * BOARD_WIDTH + c.x] = active_color;
        }
        active_pattern.update ();
        redraw_cursors ();
        if (active_pattern == patterns[patterns.length - 1] && active_pattern.count_whites () == 60)
            add_pattern (new CustomPattern ());
    }

    private void draw_editing_box (Cairo.Context cr) {
        cr.rectangle (0, 0, BOARD_WIDTH * cell_width, BOARD_WIDTH * cell_width);
        cr.set_source (new Cairo.Pattern.rgb (0.0, 0.5, 0.5));
        cr.fill ();

        for (int position=0; position < BOARD_SIZE; position++) {
            var x = position % BOARD_WIDTH;
            var y = position / BOARD_WIDTH;
            if (x < 1 || x > 11 || y < 1 || y > 11 || x == 6 && y == 6) continue;
            if (active_pattern.board[position] == Piece.WHITE) {
                // Back surface to hide imperfection
                cr.rectangle (x * cell_width, y * cell_width, cell_width, cell_width);
                cr.set_source (new Cairo.Pattern.rgb (1, 1, 0.8));
                cr.fill ();
                // Highlighted surfaces
                cr.move_to ((x + 1) * cell_width, y * cell_width);
                cr.line_to ((x + 0.5) * cell_width, (y + 0.5) * cell_width);
                cr.line_to (x * cell_width, y * cell_width);
                cr.close_path ();
                cr.set_source (new Cairo.Pattern.rgb (1, 1, 1));
                cr.fill ();
                cr.move_to (x * cell_width, y * cell_width);
                cr.line_to ((x + 0.5) * cell_width, (y + 0.5) * cell_width);
                cr.line_to (x * cell_width, (y + 1) * cell_width);
                cr.close_path ();
                cr.set_source (new Cairo.Pattern.rgb (0.95, 0.95, 0.95));
                cr.fill ();
                // Shadowed surfaces
                cr.move_to (x * cell_width, (y + 1) * cell_width);
                cr.line_to ((x + 0.5) * cell_width, (y + 0.5) * cell_width);
                cr.line_to ((x + 1) * cell_width, (y + 1) * cell_width);
                cr.close_path ();
                cr.set_source (new Cairo.Pattern.rgb (0.7, 0.7, 0.6));
                cr.fill ();
                cr.move_to ((x + 1) * cell_width, (y + 1) * cell_width);
                cr.line_to ((x + 0.5) * cell_width, (y + 0.5) * cell_width);
                cr.line_to ((x + 1) * cell_width, y * cell_width);
                cr.close_path ();
                cr.set_source (new Cairo.Pattern.rgb (0.8, 0.8, 0.7));
                cr.fill ();
                // Top surface
                cr.rectangle ((x + 0.1) * cell_width, (y + 0.1) * cell_width, 0.8 * cell_width, 0.8 * cell_width);
                cr.set_source (new Cairo.Pattern.rgb (1, 1, 0.7));
                cr.fill ();
            }
            else {
                cr.rectangle ((x + 0.05) * cell_width, (y + 0.05) * cell_width, 0.9 * cell_width, 0.9 * cell_width);
                cr.set_source ( new Cairo.Pattern.rgb (0, 0.4, 0.4));
                cr.fill ();
            }
        }
        foreach (var c in cursors) {
            if (c.x == 0) continue;
            cr.rectangle ((c.x + 0.2) * cell_width, (c.y + 0.2) * cell_width, 0.6 * cell_width, 0.6 * cell_width);
            cr.set_source (new Cairo.Pattern.rgb (0.5, 0.5, 1));
            cr.fill ();
        }
    }
}//class PatternEditor

}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
