/*
 * Copyright (C) 2002-2008 The Warp Rogue Team
 * Part of the Warp Rogue Project
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY.
 *
 * See the license.txt file for more details.
 */

/*
 * Module: AI
 */

#include "wrogue.h"

#include "summer.h"


/* Function prototypes
 */
static SM_RULE_FUNCTION(rule_seek);
static SM_RULE_FUNCTION(rule_alternate_tactics);

static SM_RULE_FUNCTION(rule_drug_antidote);
static SM_RULE_FUNCTION(rule_drug_stoic);

static SM_RULE_FUNCTION(rule_stealth);

static SM_RULE_FUNCTION(rule_flee);

static SM_RULE_FUNCTION(rule_stay_close);

static SM_RULE_FUNCTION(rule_choose_target);

static SM_RULE_FUNCTION(rule_find_path);

static SM_RULE_FUNCTION(rule_plan_a);

static SM_RULE_FUNCTION(rule_choose_option);

static SM_RULE_FUNCTION(rule_destroy_obstacles);

static SM_RULE_FUNCTION(rule_fight_unarmed);
static SM_RULE_FUNCTION(rule_switch_weapons);
static SM_RULE_FUNCTION(rule_unjam_weapon);
static SM_RULE_FUNCTION(rule_reload_weapon);

static SM_RULE_FUNCTION(rule_move);

static SM_RULE_FUNCTION(rule_use_option);

static SM_RULE_FUNCTION(rule_recover);


static void reset_ai_data(CHARACTER *);




/* AI rules
 */
static const SM_RULE    AI_Rules[AI_MAX_RULES] = {
        
        rule_seek,
        
        rule_alternate_tactics,

        rule_drug_antidote,
        rule_drug_stoic,

        rule_stealth,
        
        rule_flee,

        rule_stay_close,

        rule_choose_target,
        
        rule_find_path,

        rule_choose_option,

        rule_plan_a,

        rule_destroy_obstacles,
        
        rule_fight_unarmed,
        rule_switch_weapons,
        rule_unjam_weapon,
        rule_reload_weapon,

        rule_move,

        rule_use_option,

        rule_recover
};


/* Pointer to the AI controlled character */
static CHARACTER *      Self;

/* Temporary decision making data */
static bool             StealthTried;
static bool             TargetChosen;
static bool             PathChosen;
static bool             OptionChosen;
static AI_OPTION        ChosenOption;
static bool             WeaponsSwitched;
static bool             ObstacleHandled;
static bool             TacticsAlternated;



/* Uses the AI to determine a character's action
 */
void ai_control(CHARACTER *character)
{
        reset_ai_data(character);

        sm_apply_rules(AI_Rules, AI_MAX_RULES);
}



/* Rule: Seek
 */
static SM_RULE_FUNCTION(rule_seek)
{
        SM_CONDITIONS {
        
                if (character_target_type(Self) == TT_SEEK &&
                        character_target_distance(Self) == 0) {
                                
                        SM_RULE_MATCHED;
                }
                
                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                character_erase_target(Self);
                
                SM_RULE_FIRED;
        }
}



/* Rule: Alternate tactics
 */
static SM_RULE_FUNCTION(rule_alternate_tactics)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        !TacticsAlternated &&
                        (character_tactic(Self) == TACTIC_PATROL ||
                        character_tactic(Self) == TACTIC_GUARD)) {
                                
                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                if (character_tactic(Self) == TACTIC_PATROL) {
                        
                        if (random_choice(AI_SWITCH_TO_GUARD_PROBABILITY)) {
                                character_set_tactic(Self, TACTIC_GUARD);
                        }

                } else if (character_tactic(Self) == TACTIC_GUARD) {
                        
                        if (random_choice(AI_SWITCH_TO_PATROL_PROBABILITY)) {
                                character_set_tactic(Self, TACTIC_PATROL);
                        }
                }
                
                TacticsAlternated = true;

                SM_RULE_FIRED;
        }
}



/* Rule: Drug - Antidote
 */
static SM_RULE_FUNCTION(rule_drug_antidote)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_attribute(Self, CA_POISONED) &&
                        inventory_find_object(Self, "Antidote") != NULL) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                action_use_drug(Self, 
                        inventory_find_object(Self, "Antidote")
                );

                SM_RULE_FIRED;
        }
}



/* Rule: Drug - Stoic
 */
static SM_RULE_FUNCTION(rule_drug_stoic)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_attribute(Self, CA_BROKEN) &&
                        inventory_find_object(Self, "Stoic") != NULL) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                action_use_drug(Self, 
                        inventory_find_object(Self, "Stoic")
                );

                SM_RULE_FIRED;
        }
}



/* Rule: Stealth
 */
static SM_RULE_FUNCTION(rule_stealth)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_perk(Self, PK_STEALTH) &&
                        !character_unnoticed(Self) &&
                        !StealthTried) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }        
         
        SM_EFFECTS {

                action_stealth(Self); 

                StealthTried = true; 

                SM_RULE_FIRED;
        }
}



/* Rule: Flee
 */
static SM_RULE_FUNCTION(rule_flee)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_attribute(Self, CA_BROKEN)) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }        
         
        SM_EFFECTS {
               
                ai_flee(Self);

                SM_RULE_FIRED;
        }
}



/* Rule: Stay close
 */
static SM_RULE_FUNCTION(rule_stay_close)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        !TargetChosen &&
                        character_tactic(Self) == TACTIC_STAY_CLOSE &&
                        !character_close_to_pcc(Self)) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                character_set_target(Self, TT_FOLLOW,
                        &player_controlled_character()->location
                );

                TargetChosen = true;

                SM_RULE_FIRED;
        }
}



/* Rule: Choose target
 */
static SM_RULE_FUNCTION(rule_choose_target)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        !TargetChosen) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                if (enemies_noticed(Self)) {
                        
                        character_set_target(Self, TT_COMBAT,
                                &nearest_noticed_enemy(Self)->location
                        );

                } else if (character_target_type(Self) == TT_COMBAT) {
                        
                        if (Self->party == PARTY_PLAYER) {
                                character_erase_target(Self);
                        } else {
                                Self->target.type = TT_SEEK;
                        }

                        SM_RULE_FIRED;
                        
                } else if (character_target_type(Self) == TT_SEEK) {

                        /* DO NOTHING */
        
                } else if (character_tactic(Self) == TACTIC_PATROL) {

                        if (character_target_type(Self) == TT_PATROL &&
                                character_target_distance(Self) == 0) {
        
                                character_erase_target(Self);
                        }

                        if (character_target_type(Self) != TT_PATROL) {
                                const AREA_POINT * p; 

                                p = random_sector(SC_MOVE_TARGET_SAFE);

                                if (p == NULL) {
                                        action_recover(Self);
                                        SM_RULE_FIRED;
                                }

                                character_set_target(Self, TT_PATROL, p);
                        }

                } else {

                        character_erase_target(Self);
                }

                TargetChosen = true; 
        
                SM_RULE_FIRED;
        }
}



/* Rule: Find path
 */
static SM_RULE_FUNCTION(rule_find_path)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_target(Self) &&
                        !PathChosen) {

                        SM_RULE_MATCHED;
                }
                
                SM_RULE_NOT_MATCHED;
        }
        
        SM_EFFECTS {
                
                if (character_target_type(Self) == TT_PATROL) {
                        
                        find_patrol_path(Self);

                        if (!character_target_reachable(Self)) {
                                
                                character_erase_target(Self);
                                SM_RULE_FIRED;
                        }
        
                } else if (character_target_type(Self) == TT_COMBAT) {

                        find_attack_path(Self);

                } else if (character_target_type(Self) == TT_SEEK ||
                        character_target_type(Self) == TT_FOLLOW) {

                        find_seek_path(Self);
                }

                PathChosen = true;

                SM_RULE_FIRED;
        }
}



/* Rule: Choose option
 */
static SM_RULE_FUNCTION(rule_choose_option)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_target_type(Self) == TT_COMBAT &&
                        !OptionChosen) {
                                
                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {
                
                ChosenOption = ai_choose_option(Self);

                OptionChosen = true;

                SM_RULE_FIRED;
        }
}



/* Rule: Plan A
 */
static SM_RULE_FUNCTION(rule_plan_a)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_tactic(Self) == TACTIC_PLAN_A &&
                        !character_close_to_pcc(Self) &&
                        !character_has_path(Self) &&
                        !(OptionChosen && 
                                ai_required_position_reached(
                        Self, ChosenOption))) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                character_set_target(Self, TT_FOLLOW,
                        &player_controlled_character()->location
                );

                TargetChosen = true;
                PathChosen = false;

                SM_RULE_FIRED;
        }
}



/* Rule: Destroy obstacles
 */
static SM_RULE_FUNCTION(rule_destroy_obstacles)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_path(Self) &&
                        destructable_obstacle_at(path_first_step()) &&
                        !ObstacleHandled &&
                        !(OptionChosen && 
                                ai_required_position_reached(
                        Self, ChosenOption))) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }
        
        SM_EFFECTS {
        
                ObstacleHandled = true;

                character_set_target(Self, TT_COMBAT, path_first_step());
                OptionChosen = false;

                SM_RULE_FIRED;
        }
}



/* Rule: Fight unarmed
 */
static SM_RULE_FUNCTION(rule_fight_unarmed)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        OptionChosen &&
                        ChosenOption == AI_OPTION_UNARMED &&
                        Self->weapon != NULL) {
                                
                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                action_unequip_object(Self, Self->weapon);

                SM_RULE_FIRED;
        }
}



/* Rule: Switch weapons
 */
static SM_RULE_FUNCTION(rule_switch_weapons)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        OptionChosen &&
                        ChosenOption == AI_OPTION_SECONDARY_WEAPON &&
                        !WeaponsSwitched) {
                                
                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {
        
                action_switch_weapons(Self);

                WeaponsSwitched = true ;

                SM_RULE_FIRED;
        }
}



/* Rule: Unjam weapon
 */
static SM_RULE_FUNCTION(rule_unjam_weapon)
{
        SM_CONDITIONS {
                
                if (!character_action_spent(Self) &&
                        Self->weapon != NULL &&
                        object_has_attribute(Self->weapon, OA_JAMMED)) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {
                
                action_unjam_weapon(Self);

                SM_RULE_FIRED;
        }
}



/* Rule: Reload weapon
 */
static SM_RULE_FUNCTION(rule_reload_weapon)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        Self->weapon != NULL &&
                        Self->weapon->charge == 0) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {

                if (object_has_attribute(Self->weapon, 
                        OA_AUTOMATIC_RECHARGE)) {

                        action_recover(Self);

                } else {

                        action_reload_weapon(Self);
                }

                SM_RULE_FIRED;
        }
}



/* Rule: Move
 */
static SM_RULE_FUNCTION(rule_move)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        character_has_path(Self) &&
                        movement_speed_max(Self) > 0 &&
                        character_tactic(Self) != TACTIC_HOLD_POSITION &&
                        !(OptionChosen && 
                                ai_required_position_reached(
                        Self, ChosenOption))) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {
                
                if (Self->target.type == TT_PATROL) {
                
                        if (random_choice(50)) {
                                action_move(Self, path_first_step());
                        } else {
                                action_recover(Self);
                        }

                } else if (Self->target.type == TT_COMBAT) {

                        action_run(Self, run_factor(Self));
                
                } else if (Self->target.type == TT_FOLLOW ||
                        Self->target.type == TT_SEEK) {

                        action_run(Self, run_factor(Self));
                }

                SM_RULE_FIRED;
        }
}



/* Rule: Use option
 */
static SM_RULE_FUNCTION(rule_use_option)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self) &&
                        OptionChosen &&
                        ai_required_position_reached(Self, ChosenOption)) {

                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }
        
        SM_EFFECTS {
                
                ai_use_option(Self, ChosenOption);

                SM_RULE_FIRED;
        }
}



/* Rule: Recover
 */
static SM_RULE_FUNCTION(rule_recover)
{
        SM_CONDITIONS {

                if (!character_action_spent(Self)) {
                        
                        SM_RULE_MATCHED;
                }

                SM_RULE_NOT_MATCHED;
        }

        SM_EFFECTS {
        
                action_recover(Self);

                SM_RULE_FIRED;
        }
}



/* Resets AI data
 */
static void reset_ai_data(CHARACTER *character)
{
        Self = character;
        Self->target.path = false;
        Self->target.reachable = false;

        StealthTried = false;
        TargetChosen = false;
        PathChosen = false;
        OptionChosen = false;
        ChosenOption = AI_OPTION_NIL;
        WeaponsSwitched = false;
        ObstacleHandled = false;
        TacticsAlternated = false;
}


