/* SPDX-License-Identifier: GPL-2.0-or-later */

#include "superiotool.h"

#define DEVICE_ID_REG		0x20	/* Super I/O ID (SID) / family */
#define DEVICE_REV_REG		0x27	/* Super I/O revision ID (SRID) */

static uint8_t regread(uint16_t port, uint8_t reg)
{
	OUTB(reg, port);
	return INB(port + 1);
}

/* For Nuvoton EC space */
static void set_page(uint16_t port, uint8_t page)
{
	/*
	 * INDEX reg can be written if PAGE reg is not 0xff
	 * PAGE reg can be written if value or writing data is 0xff
	 */
	OUTB(0xff, port);
	OUTB(page, port);
}

static void dump_page_index_data(uint16_t iobase)
{
	uint16_t i,j ;

	for (i = 0; i < 255; i++) {
		printf("Page %d:\n", i);
		for (j = 0; j < 256; j++) {
			if (j % 16 == 0)
				printf("\n%02x: ", j);
			/* PAGE must be selected before each data read */
			set_page(iobase, i);
			printf("%02x ", regread(iobase + 1, j));
		}
		printf("\n");
	}
	printf("\n");
}

static const struct superio_registers reg_table[] = {
	{0xfc, "WPCE775x / NPCE781x", {
		{NOLDN, NULL,
			{0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
			 0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,EOT},
			{0xfc,0x11,RSVD,RSVD,RSVD,0x00,0x00,MISC,0x00,
			 0x04,RSVD,RSVD,RSVD,0x00,RSVD,RSVD,EOT}},
		{0x03, "CIR Port (CIRP)",	/* where supported */
			{0x30,0x60,0x61,0x70,0x71,0x74,0x75,0xf0,EOT},
			{0x00,0x03,0xf8,0x04,0x03,0x04,0x04,0x02,EOT}},
		{0x04, "Mobile System Wake-Up Control Config (MSWC)",
			{0x30,0x60,0x61,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x00,0x00,0x00,0x03,0x04,0x04,EOT}},
		{0x05, "Mouse config (KBC)",
			{0x30,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x0c,0x03,0x04,0x04,EOT}},
		{0x06, "Keyboard config (KBC)",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x00,0x60,0x00,0x64,0x01,0x03,0x04,0x04,EOT}},
		{0x0f, "Shared memory (SHM)",
			{0x30,0x60,0x61,0x70,0x71,0x74,0x75,0xf0,0xf1,0xf2,
			0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x04,0x04,MISC,0x07,RSVD,
			RSVD,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x11, "Power management I/F Channel 1 (PM1)",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x00,0x62,0x00,0x66,0x01,0x03,0x04,0x04,EOT}},
		{0x12, "Power management I/F Channel 2 (PM2)",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x00,0x68,0x00,0x6c,0x01,0x03,0x04,0x04,EOT}},
		{0x15, "Enhanced Wake On CIR (EWOC)",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x04,0x04,EOT}},
		{0x17, "Power Management I/F Channel 3 (PM3)",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x71,0x74,0x75,EOT},
			{0x00,0x00,0x6a,0x00,0x6e,0x01,0x03,0x04,0x04,EOT}},
		{0x1a, "Serial Port with Fast Infrared Port (FIR)",
			{0x30,0x60,0x61,0x70,0x71,0x74,0x75,0xf0,EOT},
			{0x00,0x02,0xf8,0x03,0x03,0x04,0x04,0x02,EOT}},
		{EOT}}},
	{0x1a, "WPCM450", {
		{EOT}}},
	{0xd592, "NCT6687D-W", {
		{NOLDN, "Global Configuration",
			{0x10,0x11,0x13,0x14,0x15,0x1a,0x1b,0x1d,0x1e,
			 0x1f,0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,
			 0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2f,EOT},
			{0xff,0xff,0x00,0x00,0x00,0x00,0x10,0x00,0x00,
			 0x00,0xd5,0x92,0x00,0x80,0x67,0x01,0x00,0x3e,
			 0x00,0x03,0x0f,0x00,0x00,0x00,MISC,EOT}},
		{0x02, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO0-7",
			{0x30,0x60,0x61,0x70,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,
			 0xe6,0xe7,0xe8,0xe9,0xeb,0xec,0xed,0xee,0xef,0xf0,
			 0xf1,EOT},
			{0x00,0x00,0x00,0x00,NANA,NANA,NANA,NANA,NANA,NANA,
			 NANA,NANA,NANA,NANA,NANA,NANA,NANA,NANA,NANA,NANA,
			 0x01,EOT}},
		{0x08, "PORT80 UART",
			{0xe0,0xe1,0xe2,0xe3,0xe4,EOT},
			{0x80,0x00,0x00,0x10,0x00,EOT}},
		{0x09, "GPIO8-9, GPIO1-8 Alternate Function",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0x30,0x60,0x61,0x70,0xe0,0xe1,0xe2,0xe3,0xe4,0xe6,
			 0xe7,0xe8,0xea,0xeb,0xec,0xee,0xf0,0xf1,0xf2,0xf3,
			 0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,EOT},
			{0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,
			 0xef,0x80,0x2e,0x00,0x01,0x00,0x00,0x00,0x00,0x00,
			 0x0d,0x0d,0x01,0x00,0x04,0x00,0x00,0x00,0x04,EOT}},
		{0x0b, "EC",
			{0x30,0x60,0x61,0x70,0xe0,0xe3,0xe4,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x0c, "RTC",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,0xf0,EOT},
			{0x00,NANA,NANA,NANA,NANA,NANA,NANA,NANA,0x00,0x00,
			 0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x80,EOT}},
		{0x0d, "Deep Sleep, Power Fault",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xf0,0xf1,0xf3,EOT},
			{0xa0,0x20,0x04,0x05,0x6e,0x00,0x00,0x00,0x88,0x77,
			 0x70,0xaa,0x01,EOT}},
		{0x0e, "TACHIN/PWMOUT Assignment",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,
			 0xea,0xeb,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
			 0x00,0x00,EOT}},
		{0x0f, "Function Register",
			{0xe3,0xe4,0xe5,0xe8,0xe9,0xea,EOT},
			{0x80,0x01,0x00,0x00,0x00,0x00,EOT}},
		{EOT}}},
	{0xb472, "NCT6775F (A)", {
		{NOLDN, NULL,
			{0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
			 0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,EOT},
			{0xb4,0x72,0xff,0x78,0x40,0x00,0x00,0x7d,0x00,
			 0x00,0x58,0x77,0xfc,0x04,0x00,MISC,EOT}},
		{0x00, "FDC",
			{0x30,0x60,0x61,0x70,0x74,0xf0,0xf1,0xf2,0xf4,
			 0xf5,EOT},
			{0x03,0x03,0xf0,0x06,0x02,0x8e,0x00,0xff,0x00,
			 0x00,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,EOT},
			{0x01,0x03,0xf8,0x04,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x07, "GPIO6, GPIO7, GPIO8, GPIO9",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xf4,0xf5,
			 0xf6,0xf7,0xf8,EOT},
			{0x18,0xff,0x00,0x00,0x00,0xef,0x00,0x00,0x00,
			 0xff,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x00,
			 0x00,0x00,0x00,EOT}},
		{0x08, "WDT1, GPIO0, GPIO1",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,
			 0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x00,0xff,0x00,0x00,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO2, GPIO3, GPIO4, GPIO5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,
			 0xf5,0xf6,0xf7,0xfe,EOT},
			{0x05,0xff,0x00,0x00,0x00,0xff,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0xff,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xed,0xf2,0xf3,0xf4,0xf6,
			 0xf7,0xfe,EOT},
			{0x00,0x01,0x00,0x00,0x00,0x00,0x02,0x1c,0x00,
			 0x00,0x00,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,
			 0x20,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xf0,0xf1,0xf2,
			 0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,
			 0xfc,0xfd,0xfe,
			 EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0xc1,0x00,0x00,
			 0x00,0x00,0x10,0x00,0x87,0x47,0x00,0x00,0x00,
			 0x00,0x00,0x00,
			 EOT}},
		{0x0c, "PECI, SST",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,
			 0xee,0xef,0xf1,0xf2,0xf3,0xfa,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,
			 0x47,0x5b,0x40,0x50,0x10,0x00,EOT}},
		{0x0d, "VID, VIDDAC, BUSSEL",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xeb,
			 0xec,0xed,0xee,0xef,0xf3,0xf4,0xf5,0xf6,0xf7,
			 0xf8,0xf9,EOT},
			{0x00,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x14,
			 0x01,0x00,0x88,0x00,0x00,0x00,0x00,0x07,0x00,
			 0x00,0x00,EOT}},
		{0x0e, "CIR WAKE-UP",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-drain",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xf0,0xf2,0xf3,0xf4,0xf8,0xfe,EOT},
			{0xff,0xff,0xf7,0xff,0xfb,0xcb,0xff,0xff,0xff,
			 0xff,0x00,0x00,0x00,0x00,0x70,0xff,EOT}},
		{EOT}}},
	{0xb473, "NCT6775F (B) / NCT5572D (B) (not all LDNs supported)", {
		{NOLDN, NULL,
			{0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
			 0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,EOT},
			{0xb4,0x73,0xff,0x78,0x40,0x00,0x00,0x7d,0x00,
			 0x00,0x58,0x77,0xfc,0x04,0x00,MISC,EOT}},
		{0x00, "FDC",
			{0x30,0x60,0x61,0x70,0x74,0xf0,0xf1,0xf2,0xf4,
			 0xf5,EOT},
			{0x03,0x03,0xf0,0x06,0x02,0x8e,0x00,0xff,0x00,
			 0x00,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,EOT},
			{0x01,0x03,0xf8,0x04,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x07, "GPIO6, GPIO7, GPIO8, GPIO9",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xf4,0xf5,
			 0xf6,0xf7,0xf8,EOT},
			{0x18,0xff,0x00,0x00,0x00,0xef,0x00,0x00,0x00,
			 0xff,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x00,
			 0x00,0x00,0x00,EOT}},
		{0x08, "WDT1, GPIO0, GPIO1",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,
			 0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x00,0xff,0x00,0x00,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO2, GPIO3, GPIO4, GPIO5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,
			 0xf5,0xf6,0xf7,0xfe,EOT},
			{0x05,0xff,0x00,0x00,0x00,0xff,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0xff,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xed,0xf2,0xf3,0xf4,0xf6,
			 0xf7,0xfe,EOT},
			{0x00,0x01,0x00,0x00,0x00,0x00,0x02,0x1c,0x00,
			 0x00,0x00,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,
			 0x20,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xf0,0xf1,0xf2,
			 0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,
			 0xfc,0xfd,0xfe,
			 EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0xc1,0x00,0x00,
			 0x00,0x00,0x10,0x00,0x87,0x47,0x00,0x00,0x00,
			 0x00,0x00,0x00,
			 EOT}},
		{0x0c, "PECI, SST",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,
			 0xee,0xef,0xf1,0xf2,0xf3,0xfa,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,
			 0x47,0x5b,0x40,0x50,0x10,0x00,EOT}},
		{0x0d, "VID, VIDDAC, BUSSEL",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xeb,
			 0xec,0xed,0xee,0xef,0xf3,0xf4,0xf5,0xf6,0xf7,
			 0xf8,0xf9,EOT},
			{0x00,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x14,
			 0x01,0x00,0x88,0x00,0x00,0x00,0x00,0x07,0x00,
			 0x00,0x00,EOT}},
		{0x0e, "CIR WAKE-UP",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-drain",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xf0,0xf2,0xf3,0xf4,0xf8,0xfe,EOT},
			{0xff,0xff,0xf7,0xff,0xfb,0xcb,0xff,0xff,0xff,
			 0xff,0x00,0x00,0x00,0x00,0x70,0xff,EOT}},
		{EOT}}},
	{0xc332, "NCT6776F (B)", {
		{NOLDN, NULL,
			{0x10,0x11,0x13,0x14,0x16,0x17,0x18,0x19,0x1a,
			 0x1b,0x1c,0x1d,0x1e,0x1f,0x20,0x21,0x22,0x23,
			 0x24,0x25,0x26,0x27,0x28,0x2a,0x2b,0x2c,0x2d,
			 0x2e,0x2f,EOT},
			{0xff,0xff,0x00,0x00,0xff,0xff,0xff,0xff,0xf0,
			 0x78,0x00,0x00,0xff,0xff,0xc3,0x32,0xff,0x00,
			 0x64,0x00,MISC,0x00,0x00,0xc0,0x00,0x81,0x00,
			 0x00,MISC,EOT}},
		{0x00, "FDC",
			{0x30,0x60,0x61,0x70,0x74,0xf0,0xf1,0xf2,0xf4,
			 0xf5,EOT},
			{0x01,0x03,0xf0,0x06,0x02,0x0e,0x00,0xff,0x00,
			 0x00,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO6, GPIO7, GPIO8, GPIO9",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xf4,0xf5,
			 0xf6,0xf7,0xf8,EOT},
			{0x03,0xff,0x00,0x00,0x00,0xef,0x00,0x00,0x00,
			 0xff,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x00,
			 0x00,0x00,0x00,EOT}},
		{0x08, "WDT1, GPIO0, GPIO1, GPIOA",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,
			 0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x02,0x00,0x00,0xff,0x00,0x00,0x00,0xef,0xff,
			 0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO2, GPIO3, GPIO4, GPIO5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,
			 0xf5,0xf6,0xf7,0xfe,EOT},
			{0x04,0xdf,0x00,0x00,0x00,0xff,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0xff,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,
			 0xee,0xf0,0xf2,0xf3,0xf4,0xf6,0xf7,0xfe,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1c,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xe0,0xe1,0xe2,
			 0xf0,0xf1,0xf2,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,
			 EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0x7f,0xff,
			 0x00,0x00,0x00,0x10,0x00,0x87,0x47,0x00,0x00,
			 EOT}},
		{0x0d, "VID",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe9,0xee,
			 0xef,0xf0,0xf4,0xf5,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x88,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0e, "CIR WAKE-UP",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-drain",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xf0,0xf1,0xf2,EOT},
			{0xff,0xdf,0xff,0xfe,0xf6,0xff,0xff,0xd3,0xff,
			 0x9f,0x00,0x00,0x00,EOT}},
		{0x14, "SVID",
			{0xe0,0xe1,0xe3,0xe4,EOT},
			{0x00,0x80,0x00,0x00,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,EOT},
			{0x20,0x20,0x04,0x05,EOT}},
		{0x17, "GPIOA",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,EOT},
			{0x01,0x00,0x00,0x00,0x01,0x00,EOT}},
		{EOT}}},
	{0xc333, "NCT6776F/D (C)", {
		{NOLDN, NULL,
			{0x10,0x11,0x13,0x14,0x16,0x17,0x18,0x19,0x1a,
			 0x1b,0x1c,0x1d,0x1e,0x1f,0x20,0x21,0x22,0x23,
			 0x24,0x25,0x26,0x27,0x28,0x2a,0x2b,0x2c,0x2d,
			 0x2e,0x2f,EOT},
			{0xff,0xff,0x00,0x00,0xff,0xff,0xff,0xff,0xf0,
			 0x78,0x00,0x00,0xff,0xff,0xc3,0x33,0xff,0x00,
			 0x64,0x00,MISC,0x00,0x00,0xc0,0x00,0x81,0x00,
			 0x00,MISC,EOT}},
		{0x00, "FDC",
			{0x30,0x60,0x61,0x70,0x74,0xf0,0xf1,0xf2,0xf4,
			 0xf5,EOT},
			{0x01,0x03,0xf0,0x06,0x02,0x0e,0x00,0xff,0x00,
			 0x00,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO6, GPIO7, GPIO8, GPIO9",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xf4,0xf5,
			 0xf6,0xf7,0xf8,EOT},
			{0x03,0xff,0x00,0x00,0x00,0xef,0x00,0x00,0x00,
			 0xff,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x00,
			 0x00,0x00,0x00,EOT}},
		{0x08, "WDT1, GPIO0, GPIO1, GPIOA",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,
			 0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x02,0x00,0x00,0xff,0x00,0x00,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO2, GPIO3, GPIO4, GPIO5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,
			 0xf5,0xf6,0xf7,0xfe,EOT},
			{0x04,0xdf,0x00,0x00,0x00,0xff,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0xff,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,
			 0xee,0xf0,0xf2,0xf3,0xf4,0xf6,0xf7,0xfe,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1c,0x00,0x00,
			 0x00,0x00,0x5c,0x00,0x00,0x00,0xc0,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xe0,0xe1,0xe2,
			 0xf0,0xf1,0xf2,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,
			 EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0x7f,0xff,
			 0x00,0x00,0x00,0x10,0x00,0x87,0x47,0x00,0x00,
			 EOT}},
		{0x0d, "VID",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe9,0xee,
			 0xef,0xf0,0xf4,0xf5,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x88,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0e, "CIR WAKE-UP",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-drain",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xf0,0xf1,0xf2,EOT},
			{0xff,0xdf,0xff,0xfe,0xf6,0xff,0xff,0xd3,0xff,
			 0x9f,0x00,0x00,0x00,EOT}},
		{0x14, "SVID",
			{0xe0,0xe1,0xe3,0xe4,EOT},
			{0x00,0x80,0x00,0x00,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,EOT},
			{0x20,0x20,0x04,0x05,EOT}},
		{0x17, "GPIOA",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,EOT},
			{0x01,0x00,0x00,0x00,0x01,0x00,EOT}},
		{EOT}}},
	{0xc562, "NCT6779D", {
		{NOLDN, NULL,
			{0x10,0x11,0x13,0x14,0x1a,0x1b,0x1c,0x1d,0x20,
			 0x21,0x22,0x24,0x25,0x26,0x27,0x28,0x2a,0x2b,
			 0x2c,0x2f,EOT},
			{0xff,0xff,0x00,0x00,0x30,0x70,0x10,0x00,0xc5,
			 0x62,0xff,0x04,0x00,MISC,0x00,0x00,0xc0,0x00,
			 0x01,MISC,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO6, GPIO7, GPIO8",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xec,0xed,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x0f,0x00,0x00,0x00,0xff,0x00,0x00,0x00,
			 0x00,0x00,0xff,0x00,0x00,0x00,0x00,EOT}},
		{0x08, "WDT1, GPIO0, GPIO1",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,
			 0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x00,0x00,0x00,0xff,0x00,0x00,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO1, GPIO2, GPIO3, GPIO4, GPIO5, GPIO6, GPIO7, GPIO8",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
			 0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,
			 0xf5,0xf6,0xf7,0xfe,EOT},
			{0x00,0xff,0x00,0x00,0x00,0x7f,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00,0xff,0x00,0x00,0xff,
			 0x00,0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,
			 0xee,0xf0,0xf2,0xf3,0xf4,0xf6,0xf7,0xfe,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1c,0x00,0x00,
			 0x00,0x10,0x5c,0x00,0x00,0x00,0xc0,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xe0,0xe1,0xe2,
			 0xe4,0xf0,0xf1,0xf2,0xf5,0xf6,0xf7,0xf8,0xf9,
			 0xfa,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0x7f,0xff,
			 0xff,0x00,0x00,0x00,0x10,0x00,0x87,0x47,0x00,
			 0x00,EOT}},
		{0x0d, "WDT1",
			{0xf0,EOT},
			{0x00,EOT}},
		{0x0e, "CIR WAKE-UP",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-drain",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,
			 0xf0,0xf1,0xf2,EOT},
			{0xff,0xff,0x7f,0xff,0xff,0xff,0x0f,0xff,0xff,
			 0x9d,0x00,0x00,EOT}},
		{0x14, "Port 80 UART",
			{0xe0,0xe1,0xe2,0xe3,0xe4,EOT},
			{0x80,0x00,0x00,0x10,0x00,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,EOT},
			{0x20,0x20,0x04,0x05,EOT}},
		{EOT}}},
	{0xc452, "NCT6102D / NCT6106D", {
		{NOLDN, NULL,
			{0x07,0x10,0x11,0x13,0x14,0x1a,0x1b,0x20,0x21,0x22,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2f,EOT},
			{0x00,0xff,0xff,0x00,0x00,0xcc,0x03,0x10,0x61,0x7F,0x00,0x00,MISC,0x00,0x00,0xf0,0x00,MISC,EOT}},
		{0x00, "FDC",
			{0x30,0x60,0x61,0x70,0x74,0xf0,0xf1,0xf2,0xf4,0xf5,EOT},
			{0x01,0x03,0xf0,0x06,0x02,0x0e,0x00,0xff,0x00,0x00,EOT}},
		{0x01, "PRT",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,0x00,0xff,0xff,0x02,0x00,0x00,EOT}},
		{0x03, "UART B",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,0x00,0xff,0xff,0x02,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller (KBC)",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "CIR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff,EOT},
			{0xdf,0xff,0x00,0x00,0x00,0xef,0x00,0x00,0x00,0xff,0x00,0x00,0x00,0xff,0x00,0x00,0x00,0xff,0xff,0x00,0x00,0xff,0x00,0x00,0x00,0xff,0x00,0x00,0x00,0x02,0x00,0x00,0x00,EOT}},
		{0x08, "GPIO,WDT1",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xf0,0xf1,0xf2,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,EOT}},
		{0x09, "GPIO",
			{0xe0,0xe1,0xe2,EOT},
			{0x00,0x00,0x00,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xfa,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1c,0x00,0x80,0x00,0x40,0x00,0x00,0x00,0x00,0xc0,0x00,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xe0,0xe1,0xe2,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0x7f,0xff,0x00,0x00,0x00,0x00,0x00,0x87,0x47,0x00,EOT}},
		{0x0d, "WDT2",
			{0xe0,0xe1,0xe2,0xe3,0xe4,EOT},
			{0x00,0x32,0x14,0x00,0x00,EOT}},
		{0x0e, "CIR WAKE-UP",
			{0x30,0x60,0x61,0x70,0xe0,0xe1,EOT},
			{0x00,0x00,0x00,0x00,0x25,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-drain",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xf0,0xf1,0xf2,EOT},
			{0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x03,0x9d,0x00,0x00,EOT}},
		{0x10, "UARTC",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x00,0x03,0x00,0x04,0x00,0x00,0x00,0xff,0xff,0x02,0x00,0x00,EOT}},
		{0x11, "UARTD",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x00,0x02,0x00,0x03,0x00,0x00,0x00,0xff,0xff,0x02,0x00,0x00,EOT}},
		{0x12, "UARTE",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x00,0x03,0xe8,0x04,0x00,0x00,0x00,0xff,0xff,0x02,0x00,0x00,EOT}},
		{0x13, "UARTF",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x00,0x02,0xe8,0x03,0x00,0x00,0x00,0xff,0xff,0x02,0x00,0x00,EOT}},
		{0x14, "PORT80 IR",
			{0x30,0x60,0x61,0x70,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,0xf1,EOT},
			{0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x10,0x00,0x00,0x00,EOT}},
		{0x15, "FADING LED",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,EOT},
			{0x00,0x3f,0x01,0x11,0x11,0x02,0x00,0x00,0x01,0x00,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,EOT},
			{0x20,0x20,0x04,0x05,EOT}},
		{EOT}}},
	{0xc803, "NCT6791D", {
		{NOLDN, NULL,
			{0x07,0x10,0x11,0x13,0x14,0x1a,0x1b,0x1c,0x1d,0x20,0x21,0x22,0x24,0x25,0x26,0x27,0x28,0x2a,0x2b,0x2c,0x2d,0x2f,EOT},
			{0x00,0xff,0xff,0x00,0x00,0x30,0x70,0x10,0x00,0xc8,0x03,0xff,0x04,0x00,MISC,0x00,0x00,0xc0,0x00,0x01,0x00,MISC,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "Consumer IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO 6, GPIO 7, GPIO 8",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xec,0xed,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x00,0x7f,NANA,0x00,NANA,0xff,NANA,0x00,NANA,0x00,0x00,0xff,NANA,0x00,NANA,0x00,EOT}},
		{0x08, "WDT1, WDT_MEM, GPIO 0, GPIO 1",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfe,0xff,EOT},
			{0x00,0x00,0x00,0xff,NANA,0x00,NANA,0x00,0xff,NANA,0x00,NANA,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO 2, GPIO 3, GPIO 4, GPIO 5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,0xf5,0xf6,0xf7,0xfe,EOT},
			{0x00,0xff,NANA,0x00,NANA,0x7f,NANA,0x00,NANA,NANA,0x00,0x00,0x00,0x00,0xff,NANA,0x00,0xff,NANA,0x00,NANA,0x00,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,0xec,0xed,0xee,0xf0,0xf2,0xf3,0xf4,0xf6,0xf7,0xfc,0xfe,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1a,0x00,0x00,0x00,0x00,0x00,0x10,0x5c,0x00,0x00,0x00,0xc0,0x00,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xe0,0xe1,0xe2,0xe3,0xe4,0xe6,0xe7,0xf0,0xf1,0xf2,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0x7f,0xff,0xff,0xff,0x08,0xff,0x00,0x00,0x00,0x10,0x00,0x87,0x47,0x00,0x00,0x00,EOT}},
		{0x0d, "BCLK, WDT2, WDT_MEM",
			{0xe0,0xe1,0xe7,0xe8,0xeb,0xed,0xf0,0xf3,EOT},
			{0x00,MISC,0x00,0x32,0x14,0x00,0x00,0x00,EOT}},
		{0x0e, "CIR Wake-Up",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-Drain selection",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,0xf0,0xf1,0xf2,EOT},
			{0xff,0xff,0x7f,0xff,0xff,0xff,0x0f,0xff,0xff,0x9d,0x00,0x00,EOT}},
		{0x14, "Port 80 UART",
			{0xe0,0xe1,0xe2,0xe3,0xe4,EOT},
			{0x80,0x00,0x00,0x10,0x00,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,0xe3,EOT},
			{0x20,0x20,0x04,0x05,0x01,EOT}},
		{EOT}}},
	{0xd451, "NCT6797D (experimental)", {
		{NOLDN, NULL,
			{0x07,0x10,0x11,0x13,0x14,0x1a,0x1b,0x1c,0x1d,0x20,0x21,0x22,0x24,0x25,0x26,0x27,0x28,0x2a,0x2b,0x2c,0x2d,0x2f,EOT},
			{0x00,0xff,0xff,0x00,0x00,0x30,0x70,0x10,0x00,0xc8,0x03,0xff,0x04,0x00,MISC,0x00,0x00,0xc0,0x00,0x01,0x00,MISC,EOT}},
		{0x01, "Parallel Port",
			{0x30,0x60,0x61,0x70,0x74,0xf0,EOT},
			{0x01,0x03,0x78,0x07,0x04,0x3f,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,EOT}},
		{0x03, "UART B, IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,EOT},
			{0x01,0x02,0xf8,0x03,0x00,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "Consumer IR",
			{0x30,0x60,0x61,0x70,0xf0,0xf1,0xf2,0xf3,EOT},
			{0x00,0x00,0x00,0x00,0x08,0x09,0x32,0x00,EOT}},
		{0x07, "GPIO 6, GPIO 7, GPIO 8",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xec,0xed,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0x00,0x7f,NANA,0x00,NANA,0xff,NANA,0x00,NANA,0x00,0x00,0xff,NANA,0x00,NANA,0x00,EOT}},
		{0x08, "WDT1, WDT_MEM, GPIO 0, GPIO 1",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfe,0xff,EOT},
			{0x00,0x00,0x00,0xff,NANA,0x00,NANA,0x00,0xff,NANA,0x00,NANA,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,EOT}},
		{0x09, "GPIO 2, GPIO 3, GPIO 4, GPIO 5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xee,0xf0,0xf1,0xf2,0xf4,0xf5,0xf6,0xf7,0xfe,EOT},
			{0x00,0xff,NANA,0x00,NANA,0x7f,NANA,0x00,NANA,NANA,0x00,0x00,0x00,0x00,0xff,NANA,0x00,0xff,NANA,0x00,NANA,0x00,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,0xec,0xed,0xee,0xf0,0xf2,0xf3,0xf4,0xf6,0xf7,0xfc,0xfe,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1a,0x00,0x00,0x00,0x00,0x00,0x10,0x5c,0x00,0x00,0x00,0xc0,0x00,0x00,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x62,0x63,0x70,0xe0,0xe1,0xe2,0xe3,0xe4,0xe6,0xe7,0xf0,0xf1,0xf2,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0x7f,0xff,0xff,0xff,0x08,0xff,0x00,0x00,0x00,0x10,0x00,0x87,0x47,0x00,0x00,0x00,EOT}},
		{0x0d, "BCLK, WDT2, WDT_MEM",
			{0xe0,0xe1,0xe7,0xe8,0xeb,0xed,0xf0,0xf3,EOT},
			{0x00,MISC,0x00,0x32,0x14,0x00,0x00,0x00,EOT}},
		{0x0e, "CIR Wake-Up",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-Drain selection",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,0xf0,0xf1,0xf2,EOT},
			{0xff,0xff,0x7f,0xff,0xff,0xff,0x0f,0xff,0xff,0x9d,0x00,0x00,EOT}},
		{0x14, "Port 80 UART",
			{0xe0,0xe1,0xe2,0xe3,0xe4,EOT},
			{0x80,0x00,0x00,0x10,0x00,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,0xe3,EOT},
			{0x20,0x20,0x04,0x05,0x01,EOT}},
		{EOT}}},
	{0xd121, "NCT5539D", {
		{NOLDN, NULL,
			{0x10,0x11,0x13,0x14,0x1a,0x1b,0x1c,0x1d,0x22,0x24,
			 0x25,0x26,0x27,0x28,0x2a,0x2b,0x2c,0x2d,0x2f,EOT},
			{0xff,0xff,0x00,0x00,0x30,0x50,0x10,0x00,0xff,0x04,
			 0x00,MISC,0x03,0x00,0xc0,0x00,0x01,MISC,MISC,EOT}},
		{0x02, "UART A",
			{0x30,0x60,0x61,0x70,0xf0,0xf2,EOT},
			{0x01,0x03,0xf8,0x04,0x00,0x00,EOT}},
		{0x05, "Keyboard Controller",
			{0x30,0x60,0x61,0x62,0x63,0x70,0x72,0xf0,EOT},
			{0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,EOT}},
		{0x06, "Consumer IR",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x07, "GPIO 7, GPIO 8",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xec,0xed,EOT},
			{0x00,0xff,NANA,0x00,NANA,0xff,NANA,0x00,NANA,0x00,0x00,EOT}},
		{0x08, "WDT1, WDT3, GPIO 0, KBC P20",
			{0x30,0x60,0x61,0xe0,0xe1,0xe2,0xe3,0xe4,0xf5,0xf6,
			 0xf7,0xf8,0xf9,0xfa,0xfe,0xff,EOT},
			{0x00,0x00,0x00,0xff,NANA,0x00,NANA,0x00,0x00,0x00,
			 0x00,0x00,0x00,0x00,0x00, 0x00,EOT}},
		{0x09, "GPIO 2, GPIO 3, GPIO 4, GPIO 5",
			{0x30,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
			 0xe9,0xea,0xeb,0xee, 0xf0,0xf1,0xf2,0xf4,0xf5,0xf6,0xf7,EOT},
			{0x00,0xff,NANA,0x00,NANA,0x7f,NANA,0x00,NANA,NANA,
			 0x00,0x00,0x00,0x00,0xff,NANA,0x00,0xff,NANA,0x00,NANA,EOT}},
		{0x0a, "ACPI",
			{0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe9,0xec,
			 0xed,0xee,0xf0,0xf1,0xf2,0xf3,0xf6,0xf7,0xfc,EOT},
			{0x01,0x00,0x00,0x00,0x00,0x02,0x1a,0x00,0x00,0x00,
			 0x01,0x00,0x10,0x48,0x5c,0x00,0x00,0xc0,0x80,EOT}},
		{0x0b, "Hardware Monitor, Front Panel LED",
			{0x30,0x60,0x61,0x70,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,
			 0xe6,0xe7,0xec,0xed,0xf0,0xf1,0xf2,0xf6,0xf7,0xf8,
			 0xf9,0xfa,0xfb,0xfc,EOT},
			{0x00,0x00,0x00,0x00,0x7f,0x7f,0x7f,0x7f,0x7f,0xa8,
			 0x08,0x7f,0x00,0x81,0x00,0x00,0x00,0x00,0x87,0x47,
			 0x00,0x00,0x00,0x02,EOT}},
		{0x0d, "WDT2",
			{0xe7,0xe8,0xeb,0xed,0xf0,0xf3,EOT},
			{0x10,0x32,0x14,0x00,0x00,0x00,EOT}},
		{0x0e, "CIR Wake-Up",
			{0x30,0x60,0x61,0x70,EOT},
			{0x00,0x00,0x00,0x00,EOT}},
		{0x0f, "GPIO Push-Pull or Open-Drain selection",
			{0xe1,0xe2,0xe3,0xe4,0xe6,0xe7,0xe9,EOT},
			{0xff,0x7f,0xff,0xff,0xff,0xff,0xff,EOT}},
		{0x11, "GPIO, RI PSOUT Wake-Up Status",
			{0xe1,0xe2,0xe9,EOT},
			{0x00,0x00,0x00,EOT}},
		{0x12, "SW Error Control",
			{0xe1,0xe7,0xea,0xeb,0xec,EOT},
			{0x00,0x00,0x10,0x00,0x00,EOT}},
		{0x15, "Fading LED",
			{0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,EOT},
			{0xff,0x7f,0x00,0x56,0x56,0x17,0x00,0x0f,0x10,EOT}},
		{0x16, "Deep Sleep",
			{0x30,0xe0,0xe1,0xe2,0xe3,EOT},
			{0xa0,0x20,0x04,0x05,0x01,EOT}},
		{EOT}}},
	{EOT}
};

static void dump_nct6687d_gpios(uint16_t port)
{
	uint8_t group, sel;

	const char *gpio_groups[] ={
		"0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
		"Reserved", /* Does not exist */
		"EN0", /* Enhance 0 */
		"EN1", /* Enhance 1 */
	};

	/* Used by reg at 0xd offset to select which registers to reflect */
	const char *gpio_sel[] ={
		"GPIO Data",
		"GPIO Interrupt Enable",
		"GPIO Status",
		"GPIO I/O Control",
		"GPIO Inversion Control",
		"GPIO PP/OD Control",
		"GPIO Interrupt Type",
		"GPIO Output Data Reflection",
		"GPIO Internal Pull Down Control",
		"GPIO Reset Source Control",
		"Reserved",
		/* Below are valid only for GPIO Enhance Group 0 and 1 */
		"GPIO De-bounce Clock Option",
		"GPIO De-bounce Type 0",
		"GPIO De-bounce Type 1",
		"GPIO De-bounce Time Option 0",
		"GPIO De-bounce Time Option 1",
	};

	enter_conf_mode_winbond_fintek_ite_8787(port);
	regwrite(port, LDN_SEL, 0x07);

	printf("\nDumping GPIO configuration...\n\n");

	printf("%-35s", "GPIO Group");
	for (group = 0; group < ARRAY_SIZE(gpio_groups); group++) {
		if (group == 10)
			continue;
		printf("%-5s", gpio_groups[group]);
	}

	printf("\n");

	for (sel = 0; sel < ARRAY_SIZE(gpio_sel); sel++) {
		if (sel == 10)
			continue;
		printf("%-35s", gpio_sel[sel]);
		for (group = 0; group < ARRAY_SIZE(gpio_groups); group++) {
			if (group == 10)
				continue;
			/* Select GPIO group */
			regwrite(port, 0xf0, group);
			if (group < 11 && sel > 10)
				printf("XX   ");
			else
				/* GPIO registers start at LDN 7 offset 0xe0 */
				printf("%02x   ", regval(port, 0xe0 + sel));
		}
		printf("\n");
	}

	printf("\n");
	exit_conf_mode_winbond_fintek_ite_8787(port);
}

void probe_idregs_nuvoton(uint16_t port)
{
	uint8_t sid, srid;
	uint16_t chip_id = 0;
	uint8_t chip_rev = 0;
	uint16_t iobase = 0;
	int i;

	/* Probe for the 16bit IDs first to avoid collisions */
	probing_for("Nuvoton", "", port);
	enter_conf_mode_winbond_fintek_ite_8787(port);
	chip_id = (regval(port, DEVICE_ID_REG) << 8) |
	      regval(port, DEVICE_ID_REG + 1);
	regwrite(port, LDN_SEL, 0x0b);
	iobase = (regval(port, 0x60) << 8) | (regval(port, 0x61) & ~7);

	exit_conf_mode_winbond_fintek_ite_8787(port);

	if (!superio_unknown(reg_table, chip_id)) {
		printf("Found Nuvoton %s (id=0x%02x) at 0x%x\n",
		       get_superio_name(reg_table, chip_id), chip_id, port);
		chip_found = 1;
		enter_conf_mode_winbond_fintek_ite_8787(port);
		dump_superio("Nuvoton", reg_table, port, chip_id, LDN_SEL);
		exit_conf_mode_winbond_fintek_ite_8787(port);
		goto extra;
	}

	if (verbose)
		printf(NOTFOUND "chip_id=0x%04x\n", chip_id);

	probing_for("Nuvoton", "(sid=0xfc) ", port);

	sid = regval(port, DEVICE_ID_REG);
	srid = regval(port, DEVICE_REV_REG);

	if (sid == 0xfc) {	/* WPCE775xL family */
		/*
		 * bits 7-5: Chip ID
		 * bits 4-0: Chip revision
		 */
		chip_id = srid >> 5;
		chip_rev = srid & 0x1f;
	}

	if (superio_unknown(reg_table, sid)) {
		if (verbose)
			printf(NOTFOUND
			       "sid=0x%02x, id=0x%02x, rev=0x%02x\n", sid,
			       chip_id, chip_rev);
		return;
	}

	printf("Found Nuvoton %s (id=0x%02x, rev=0x%02x) at 0x%x\n",
	       get_superio_name(reg_table, sid), chip_id, chip_rev, port);
	chip_found = 1;
	dump_superio("Nuvoton", reg_table, port, sid, LDN_SEL);

extra:
	if (extra_dump && iobase) {
		switch (chip_id & 0xfff0) {
		case 0xb470:	/* NCT6775F */
			for (i = 0; i < 7; i++)
				dump_data(iobase + 5, i);
			dump_data(iobase + 5, 0xa);
			dump_data(iobase + 5, 0xc);
			dump_data(iobase + 5, 0xd);
			break;
		case 0xc330:	/* NCT6776F */
			for (i = 0; i < 8; i++)
				dump_data(iobase + 5, i);
			break;
		case 0xc560:	/* NCT6779D */
			for (i = 0; i < 10; i++)
				dump_data(iobase + 5, i);
			break;
		case 0xd590: /* NCT6687D-W */
			dump_nct6687d_gpios(port);
			/* One can use the APCI/BIOS register set, although the
			 * resulting data is still the same when using software
			 * register set.
			 * printf("EC I/O base for ACPI/BIOS: 0x%x\n", iobase);
			 * dump_page_index_data(iobase);
			*/
			printf("EC I/O base for software: 0x%x\n", iobase + 4);
			dump_page_index_data(iobase + 4);
			break;
		}
	}
}

void print_nuvoton_chips(void)
{
	print_vendor_chips("Nuvoton", reg_table);
}
