/* Test clear_cache.

   Copyright 2023-2024 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <https://www.gnu.org/licenses/>.  */

#include <config.h>

#include <jit/cache.h>

#include <pagealign_alloc.h>
#include <string.h>
#include <unistd.h>

#if HAVE_SYS_MMAN_H && HAVE_MPROTECT
# include <sys/mman.h>
#endif

#include "macros.h"

/*  On most platforms, function pointers are just a pointer to the
    code, i.e. to the first instruction to be executed.  This,
    however, is not universally true, see:
    https://git.savannah.gnu.org/gitweb/?p=libffcall.git;a=blob;f=porting-tools/abis/function-pointer.txt.  */

#define CODE(fn) (((struct func *) (fn))->code_address)
#if ((defined __powerpc__ || defined __powerpc64__) && defined _AIX) || (defined __powerpc64__ && defined __linux__)
struct func
{
  void *code_address;
  void *toc_pointer;
  void *static_chain;
};
#elif defined __ia64__
struct func
{
  void *code_address;
  void *global_pointer;
};
#elif defined __ia64_ilp32__
struct func
{
  void *code_address;
  void *unused1;
  void *global_pointer;
  void *unused2;
};
#elif __hppa__
struct func
{
  void *code_address;
  void *pic_base;
};
#elif __hppa64__
struct func
{
  void *some_other_code_address;
  void *some_other_pic_base;
  void *code_address;
  void *pic_base;
};
#else
# undef CODE
# define CODE(fn) ((*(void **) (&fn)))
#endif

/* This test assumes that the code generated by the compiler for the
   procedures `return1' and `return2' is position independent.  It
   also assumes that data pointers are bit-compatible to integers.  */

static int
return1 (void)
{
  return 1;
}

static int
return2 (void)
{
  return 2;
}

int
main ()
{
#if !(HAVE_SYS_MMAN_H && HAVE_MPROTECT)
  return 77;
#else
  int const pagesize = getpagesize ();
  unsigned char *start = pagealign_xalloc (pagesize);
  unsigned char *end = start + pagesize;

  /* We have to call `mprotect' before the tests because on some
     platforms `mprotect' invalidates the caches.  */
  mprotect (start, end - start, PROT_READ | PROT_WRITE | PROT_EXEC);

  int (*f) (void) = return1;
  CODE (f) = start;

  /* We assume that the code is not longer than 64 bytes and that we
     can access the full 64 bytes for reading.  */
  memcpy (start, return1, 64);
  clear_cache (start, end);
  ASSERT (f () == 1);

  memcpy (start, return2, 64);
  clear_cache (start, end);
  ASSERT (f () == 2);

  return 0;
#endif  /* !(HAVE_SYS_MMAN_H && HAVE_PROTECT) */
}
