import type { Schema } from "./ecmascript/schema.js";
import type { FileInfo } from "./ecmascript/generated-file.js";
import type { Plugin } from "./plugin.js";
import type { FeatureSetDefaults } from "@bufbuild/protobuf";
interface PluginInit {
    /**
     * Name of this code generator plugin.
     */
    name: string;
    /**
     * Version of this code generator plugin.
     */
    version: string;
    /**
     * An optional parsing function which can be used to parse your own plugin
     * options.
     */
    parseOption?: (key: string, value: string) => void;
    /**
     * Whether the plugin supports editions.
     */
    supportsEditions?: boolean;
    /**
     * By default, plugins support all editions supported by createDescriptorSet()
     * from @bufbuild/protobuf, if supportsEditions is enabled.
     *
     * This option can be used to limit support for specific editions, by providing
     * your own google.protobuf.FeatureSetDefaults generated by protoc with the
     * flags --experimental_edition_defaults_out, --experimental_edition_defaults_minimum,
     * and --experimental_edition_defaults_maximum.
     */
    featureSetDefaults?: FeatureSetDefaults;
    /**
     * A function which will generate TypeScript files based on proto input.
     * This function will be invoked by the plugin framework when the plugin runs.
     *
     * Note that this is required to be provided for plugin initialization.
     */
    generateTs: (schema: Schema, target: "ts") => void;
    /**
     * A optional function which will generate JavaScript files based on proto
     * input.  This function will be invoked by the  plugin framework when the
     * plugin runs.
     *
     * If this function is not provided, the plugin framework will then check if
     * a transpile function is provided.  If so, it will be invoked to transpile
     * JavaScript files.  If not, the plugin framework will transpile the files
     * itself.
     */
    generateJs?: (schema: Schema, target: "js") => void;
    /**
     * A optional function which will generate TypeScript declaration files
     * based on proto input.  This function will be invoked by the plugin
     * framework when the plugin runs.
     *
     * If this function is not provided, the plugin framework will then check if
     * a transpile function is provided.  If so, it will be invoked to transpile
     * declaration files.  If not, the plugin framework will transpile the files
     * itself.
     */
    generateDts?: (schema: Schema, target: "dts") => void;
    /**
     * An optional function which will transpile a given set of files.
     *
     * This function is meant to be used in place of either generateJs,
     * generateDts, or both.  However, those functions will take precedence.
     * This means that if generateJs, generateDts, and this transpile function
     * are all provided, this transpile function will be ignored.
     *
     * If jsImportStyle is "module" (the standard behavior), the function is
     * expected to use ECMAScript module import and export statements when
     * transpiling to JS. If jsImportStyle is "legacy_commonjs", the function is
     * expected to use CommonJs require() and exports when transpiling to JS.
     */
    transpile?: (files: FileInfo[], transpileJs: boolean, transpileDts: boolean, jsImportStyle: "module" | "legacy_commonjs") => FileInfo[];
}
/**
 * Create a new code generator plugin for ECMAScript.
 * The plugin can generate JavaScript, TypeScript, or TypeScript declaration
 * files.
 */
export declare function createEcmaScriptPlugin(init: PluginInit): Plugin;
export {};
