/*
subsync - A subtitle syncronizer
Copyright (C) 2003 Vanraes Maarten

This file is part of subsync package.

The file is provided as-is, no warranty is supplied. Use it at your
own risk.

subsync is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

subsync is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABLILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with subsync; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <sys/time.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/*
#define LOG
*/

#define HELP		"subsync version %s written by alien999999999 AT users DOT sourceforge DOT net\n\nusage:\tsubsync source1 dest1 source2 dest2 < input.srt > output.srt\n\nsource1,dest1,source2,dest2 are timings of the following format:\n    HH:MM:SS,nnn (nnn means milliseconds) (numbers must be padded with zeros)\n\nThe source1 time will be mapped to dest1 time, and the source2 time will be\nmapped to the dest2 time. All times in between are synchronized in a linear way.\n\n"
#define VERSION		"0.0.1"
#define USPS		1000000
#define GROWSIZE	256
#define SIGN_NEG	'-'
#define SIGN_POS	' '

typedef struct
{
	unsigned int s,us;
	int sign;/* if true then a '-' is added in front */
} t_t;

typedef struct
{
	t_t t1,t2;
	int nr;
	char *text;
} srt_t;
typedef srt_t* srt_p;
typedef srt_p* srts_t;
typedef srts_t* srts_p;

static t_t timediff(t_t,t_t);
static t_t timeadd(t_t,t_t);

static srt_p mksrt(t_t t1,t_t t2,int nr,char *text)
{
	srt_p s=(srt_p)malloc(sizeof(*s));
	s->t1=t1;
	s->t2=t2;
	s->nr=nr;
	s->text=text;
	return s;
}

static void freesrt(srt_p *s)
{
	if(s)
	{
		if(*s)
			free((*s)->text);
		free(*s);
		*s=0;
	}
}

static char getsign(int s)
{
	if(s)
		return SIGN_NEG;
	return SIGN_POS;
}

static char* timetostr(t_t t)
{
	char *s=malloc(13);
	sprintf(s,"%02u:%02u:%02u,%03u",(t.s/3600)%24,(t.s/60)%60,t.s%60,(t.us/1000)%1000);
	return s;
}

static char* timetostrsign(t_t t)
{
	char *s=malloc(14);
	sprintf(s,"%c%02u:%02u:%02u,%03u",getsign(t.sign),(t.s/3600)%24,(t.s/60)%60,t.s%60,(t.us/1000)%1000);
	return s;
}

static t_t strtotime(char *str)
{
	t_t t;
	int i,j,k,l;
	sscanf(str,"%2u:%2u:%2u,%3u",&i,&j,&k,&l);
	t.s=i*3600+j*60+k;
	t.us=l*1000;
	t.sign=0;
	return t;
}

static t_t readtime(FILE *f)
{
	t_t t;
	int i,j,k,l;
	fscanf(f,"%2u:%2u:%2u,%3u",&i,&j,&k,&l);
	t.s=i*3600+j*60+k;
	t.us=l*1000;
	t.sign=0;
	return t;
}

static t_t chsign(t_t a)
{
	t_t r=a;
	r.sign=!r.sign;
	return r;
}

static t_t timediff(t_t a,t_t b)
{
	t_t r;
	if(a.sign!=b.sign)
		return timeadd(a,chsign(b));
	if(a.sign)
		return chsign(timediff(chsign(a),chsign(b)));
	if((a.s<b.s)||((a.s==b.s)&&(a.us<b.us)))
		return chsign(timediff(b,a));
	r.sign=0;
	r.s=a.s-b.s;
	r.us=a.us-b.us;
	if(a.us<b.us)
	{
		r.us+=USPS;
		r.s--;
	}
	return r;
}

static t_t timeadd(t_t a,t_t b)
{
	t_t r;
	if(a.sign!=b.sign)
		return timediff(a,chsign(b));
	if(a.sign)
		return chsign(timeadd(chsign(a),chsign(b)));
	r.sign=0;
	r.s=a.s+b.s;
	r.us=a.us+b.us;
	if(r.us>USPS)
	{
		r.us-=USPS;
		r.s++;
	}
	return r;
}

static t_t timemult(t_t a,double scale)
{
	t_t r;
	double s=a.s*scale,us=a.us*scale;
	if(scale<0)
		return timemult(chsign(a),-scale);
	r.sign=a.sign;
	us=us+(remainder(s,1)*USPS);
	r.us=lround(floor(us))%USPS;
	r.s=floor(s+us/USPS);
	return r;
}

static double timediv(t_t a,t_t b)
{
	double scale=(a.us+((double)USPS)*a.s)/(b.us+((double)USPS)*b.s);
	if(a.sign!=b.sign)
		scale*=-1;
	return scale;
}

static t_t calcnewtime(t_t source,double scale,t_t sourcedisp,t_t destdisp)
{
	return timeadd(source,timeadd(timemult(timediff(source,sourcedisp),scale),timediff(destdisp,sourcedisp)));
}

static int readsrt(FILE *f,srt_p srt)
{
	char *s;
	if(EOF==fscanf(f,"%d\n",&srt->nr))
		return 0;
	s=(char*)malloc(256);
	srt->t1=readtime(f);
	fscanf(f," --> ");
	srt->t2=readtime(f);
	srt->text=0;
	fscanf(f,"\n");
	do
	{
		fgets(s,256,f);
		if(s&&(s[0]!='\r')&&(s[0]!='\n')&&s[0])
		{
			if(srt->text)
			{
				srt->text=realloc(srt->text,strlen(srt->text)+strlen(s)+1);
				strcat(srt->text,s);
			}
			else
			{
				srt->text=(char *)malloc(strlen(s)+1);
				strcpy(srt->text,s);
			}
		}
	} while(s&&(s[0]!='\r')&&(s[0]!='\n')&&s[0]);
	free(s);
	return 1;
}

static int writesrt(FILE *f,srt_p p)
{
	char *i=timetostr(p->t1),*j=timetostr(p->t2);
	fprintf(f,"%d\n%s --> %s\n%s\n",p->nr,i,j,p->text);
	free(i);
	free(j);
}

int main(int argc,char **argv)
{
	t_t source1,dest1,source2,dest2;
	srt_t cur;
	double scale;
	if(argc!=5)
	{
		fprintf(stderr,HELP,VERSION);
		return 1;
	}
	source1=strtotime(argv[1]);
	dest1=strtotime(argv[2]);
	source2=strtotime(argv[3]);
	dest2=strtotime(argv[4]);
	scale=timediv(timediff(timediff(dest2,source2),timediff(dest1,source1)),timediff(source2,source1));
	while(readsrt(stdin,&cur))
	{
		cur.t1=calcnewtime(cur.t1,scale,source1,dest1);
		cur.t2=calcnewtime(cur.t2,scale,source1,dest1);
		writesrt(stdout,&cur);
		if(cur.text)
			free(cur.text);
	}
	return 0;
}
