/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

/*
 * idncmp -- Encode two internatinalized domain names and compare them.
 */

#include <config.h>

#include <stddef.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <idn/result.h>
#include <idn/punycode.h>
#include <idn/res.h>
#include <idn/resconf.h>
#include <idn/util.h>
#include <idn/utf8.h>
#include <idn/version.h>

/*
 * Exit codes.
 */
#define IDNCMP_EXITCODE_SUCCESS		0
#define IDNCMP_EXITCODE_NEQ		1
#define IDNCMP_EXITCODE_CHECKERROR	2
#define IDNCMP_EXITCODE_FAILURE		3

/*
 * Domain name registration / lookup procotols.
 */
typedef enum idncmp_protocol {
	idncmp_lookup = 0,
	idncmp_registration = 1
} idncmp_protocol_t;

/*
 * Parsing result of command line arguments.
 */
typedef struct {
	char *in_code;
	char *conf_file;
	int no_conf; 
	idncmp_protocol_t protocol;
	char *localcheck_file;
	idn_action_t skip_actions;
	char *name1;
	char *name2;
	int quiet;
	int test;
}  idncmp_option_t;

/*
 * Quiet mode flag.
 */
int quiet_mode = 0;

static void	idncmp_option_init(idncmp_option_t *option);
static int	parse_command_line(int ac, char **av,
				   idncmp_option_t *option);
static int	parse_skip(const char *s, idn_action_t *action);
static int	create_resconf(idn_resconf_t *resconf, int no_conf,
			       const char *conf_file,
			       const char *localencoding,
			       const char *localcheck_file);
static void	print_test_status(idn_resconf_t conf);
static void	print_version(void);
static void	print_usage(void);
static void	errormsg(const char *fmt, ...);

int
main(int ac, char **av) {
	idn_result_t r;
	int exit_code = IDNCMP_EXITCODE_SUCCESS;
	idncmp_option_t option;
	idn_action_t actions;
	idn_resconf_t resconf = NULL;

#ifdef HAVE_SETLOCALE
	setlocale(LC_ALL, "");
#endif

	/*
	 * Parse command line options.
	 */
	idncmp_option_init(&option);
	if (!parse_command_line(ac, av, &option)) {
		errormsg("try 'idncmp -help' for more information.\n");
		exit_code = IDNCMP_EXITCODE_FAILURE;
		goto ret;
	}

	quiet_mode = option.quiet;

	/*
	 * Initialize idnkit library.
	 */
	r = idn_resconf_initialize();
	if (r != idn_success) {
		errormsg("error initializing library\n");
		exit_code = IDNCMP_EXITCODE_FAILURE;
		goto ret;
	}

	/*
	 * Create a resource context.
	 */
	if (!create_resconf(&resconf, option.no_conf, option.conf_file, 
			    option.in_code, option.localcheck_file)) {
		exit_code = IDNCMP_EXITCODE_FAILURE;
		goto ret;
	}

	/*
	 * Determine main actions.
	 */
	if (option.protocol == idncmp_registration)
		actions = IDN_COMPARE_REGIST & ~option.skip_actions;
	else
		actions = IDN_COMPARE_LOOKUP & ~option.skip_actions;
	if (option.localcheck_file != NULL &&
	    !(option.skip_actions & ~IDN_LOCALCHECK)) {
		actions |= IDN_LOCALCHECK;
	}

	/*
	 * Test mode.
	 */
	if (option.test) {
		print_test_status(resconf);
		goto ret;
	}

	/*
	 * Encode two names and compare them.
	 */
	r = idn_res_comparenames(resconf, actions, option.name1, option.name2);

	if (r == idn_success) {
		errormsg("matched.\n");
	} else if (r == idn_neq) {
		errormsg("unmatched.\n");
		exit_code = IDNCMP_EXITCODE_NEQ;
	} else if (r == idn_prohcheck_error ||
		   r == idn_unascheck_error ||
		   r == idn_nfccheck_error ||
		   r == idn_prefcheck_error ||
		   r == idn_hyphcheck_error ||
		   r == idn_combcheck_error ||
		   r == idn_ctxjcheck_error ||
		   r == idn_ctxocheck_error ||
		   r == idn_bidicheck_error ||
		   r == idn_localcheck_error ||
		   r == idn_lencheck_error ||
		   r == idn_rtcheck_error ||
		   r == idn_tr46check_error) {
		errormsg("%s\n", idn_result_tostring(r));
		exit_code = IDNCMP_EXITCODE_CHECKERROR;
	} else if (r != idn_success) {
		errormsg("%s\n", idn_result_tostring(r));
		exit_code = IDNCMP_EXITCODE_FAILURE;
	} 

ret:
	if (resconf != NULL)
		idn_resconf_destroy(resconf);

	return (exit_code);
}

static void
idncmp_option_init(idncmp_option_t *option) {
	option->in_code         = NULL;
	option->conf_file       = NULL;
	option->no_conf         = 0;
	option->protocol        = idncmp_lookup;
	option->localcheck_file = NULL;
	option->skip_actions    = 0;
	option->name1           = NULL;
	option->name2           = NULL;
	option->quiet           = 0;
	option->test            = 0;
}

/*
 * Parse command line arguments.
 */
static int
parse_command_line(int ac, char **av, idncmp_option_t *option) {
	ac--;
	av++;
	while (ac > 0 && **av == '-') {
		if (strcmp(*av, "--") == 0) {
			ac--;
			av++;
			break;
		} else if (strcmp(*av, "-in") == 0 ||
		    strcmp(*av, "-i") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->in_code = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-conf") == 0 ||
			   strcmp(*av, "-c") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->conf_file = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-noconf") == 0 ||
			   strcmp(*av, "-C") == 0) {
			option->no_conf = 1;
		} else if (strcmp(*av, "-registration") == 0 ||
			   strcmp(*av, "-g") == 0) {
			option->protocol = idncmp_registration;
		} else if (strcmp(*av, "-lookup") == 0 ||
			   strcmp(*av, "-l") == 0) {
			option->protocol = idncmp_lookup;
		} else if (strcmp(*av, "-nomap") == 0 ||
			   strcmp(*av, "-M") == 0) {
			if (!parse_skip("map", &option->skip_actions))
				return (0);
		} else if (strcmp(*av, "-skip") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			if (!parse_skip(av[1], &option->skip_actions)) {
				errormsg("invalid argument to option '-skip'.\n", *av);
				return (0);
			}
			ac--;
			av++;
		} else if (strcmp(*av, "-localcheck") == 0 ||
			   strcmp(*av, "-e") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->localcheck_file = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-quiet") == 0 ||
			   strcmp(*av, "-q") == 0) {
			option->quiet = 1;
		} else if (strcmp(*av, "-test") == 0 ||
			   strcmp(*av, "-t") == 0) {
			option->test = 1;
		} else if (strcmp(*av, "-version") == 0 ||
			   strcmp(*av, "-v") == 0) {
			print_version();
			exit(IDNCMP_EXITCODE_SUCCESS);
		} else if (strcmp(*av, "-help") == 0 ||
			   strcmp(*av, "-h") == 0) {
			print_usage();
			exit(IDNCMP_EXITCODE_SUCCESS);
		} else {
			errormsg("unrecognized option '%s'.\n", *av);
			return (0);
		}
		ac--;
		av++;
	}

	if (option->test) {
		option->name1 = "";
		option->name2 = "";
	} else if (ac > 2) {
		errormsg("too many arguments.\n");
		return (0);
	} else if (ac < 2) {
		errormsg("too few argument.\n");
		return (0);
	} else {
		option->name1 = av[0];
		option->name2 = av[1];
	}

	/*
	 * Resolve conflicts between options.
	 */
	if (option->no_conf && option->conf_file != NULL)
		option->conf_file = NULL;

	return (1);
}

struct idncmp_actionmap {
	char *name;
	idn_action_t value;
};

static const struct idncmp_actionmap action_maps[] = {
	{"map", 		IDN_MAP}, 
	{"asclower", 		IDN_ASCLOWER}, 
	{"rtconv", 		IDN_RTCONV}, 
	{"prohcheck", 		IDN_PROHCHECK}, 
	{"unascheck", 		IDN_UNASCHECK}, 
	{"nfccheck", 		IDN_NFCCHECK}, 
	{"prefcheck", 		IDN_PREFCHECK}, 
	{"hyphcheck", 		IDN_HYPHCHECK}, 
	{"combcheck", 		IDN_COMBCHECK}, 
	{"ctxjcheck", 		IDN_CTXJCHECK}, 
	{"ctxocheck", 		IDN_CTXOCHECK}, 
	{"ctxolitecheck", 	IDN_CTXOLITECHECK},
	{"bidicheck", 		IDN_BIDICHECK}, 
	{"idnconv",		IDN_IDNCONV}, 
	{"lencheck",		IDN_LENCHECK}, 
	{"rtcheck",		IDN_RTCHECK}, 
	{NULL, 			0}
};

static int
parse_skip(const char *arg, idn_action_t *action) {
	const struct idncmp_actionmap *mapp;
	int found;

	while (*arg != '\0') {
		found = 0;
		for (mapp = action_maps; mapp->name != NULL; mapp++) {
			const char *s1 = arg;
			const char *s2 = mapp->name;

			while (*s1 != ',' && *s1 != '\0') {
				if (*s1 != *s2 || *s2 == '\0')
					break;
				s1++;
				s2++;
			}
			if (*s1 == ',' && *s2 == '\0') {
				if (*(s1 + 1) == '\0')
					return (0);
				*action |= mapp->value;
				arg = s1 + 1;
				found = 1;
				break;
			} else if (*s1 == '\0' && *s2 == '\0') {
				*action |= mapp->value;
				arg = s1;
				found = 1;
				break;
			}
		}
		if (!found)
			return (0);
	}
	return (1);
}

static int
create_resconf(idn_resconf_t *resconf, int no_conf,
	       const char *conf_file, const char *localencoding,
	       const char *localcheck_file) {
	idn_result_t r;

	r = idn_resconf_create(resconf);
	if (r != idn_success) {
		errormsg("failed to initialize configuration "
			"contexts, %s\n", idn_result_tostring(r));
		return (0);
	}

	if (!no_conf) {
		r = idn_resconf_loadfile(*resconf, conf_file);
		if (r != idn_success &&
		    (r != idn_nofile || conf_file != NULL)) {
			errormsg("failed to read a configuration file, "
				"%s\n", idn_result_tostring(r));
			return (0);
		}
	}

	/*
	 * For backward compatibility to idnkit-1.0, 'Punycode' specified
	 * as local encoding, we assumes it as UTF-8.
	 */
	if (localencoding != NULL &&
	    strcmp(localencoding, IDN__PUNYCODE_ACENAME) == 0)
		localencoding = IDN__UTF8_ENCODINGNAME;
		
	r = idn_resconf_setlocalencoding(*resconf, localencoding);
	if (r != idn_success) {
		errormsg("failed to set the local encoding, %s\n",
			idn_result_tostring(r));
		return (0);
	}

	r = idn_resconf_setlocalcheckfile(*resconf, localcheck_file);
	if (r != idn_success) {
		errormsg("failed to set the localcheck file, %s\n",
			idn_result_tostring(r));
		return (0);
	}

	return (1);
}

static char *options[] = {
	"Usage: idncmp [options..] name1 name2",
	"  -in INPUT-CODESET   : specifies input codeset name.",
	"  -i INPUT-CODESET    : synonym for -in",
	"  -conf CONF-FILE     : specifies idnkit configuration file.",
	"  -c CONF-FILE        : synonym for -conf",
	"  -noconf             : do not load idnkit configuration file.",
	"  -C                  : synonym for -noconf",
	"  -registration       : convert regions with the Domain Name Registration",
	"                        protocol.",
	"  -g                  : synonym for -registration",
	"  -lookup             : convert regions with the Domain Name Lookup",
	"                        protocol. (default)",
	"  -l                  : synonym for -lookup",
	"  -nomap              : do not perform mappings.",
	"  -M                  : synonym for -nomap",
	"  -skip ACTION,...    : do not perform ACTION.",
	"                        the following action names are recognized:",
	"                        map         : mappings",
	"                        asclower    : ASCII uppercase letters to lowercase",
	"                        rtconv      : conversion from Punycode to Unicode",
	"                                      for round trip check",
	"                        prohcheck   : prohibited code point check",
	"                        unascheck   : unassigned code point check",
	"                        nfccheck    : NFC conformance check",
	"                        prefcheck   : ACE prefix check",
	"                        hyphcheck   : hyphen check",
	"                        combcheck   : combining character check",
	"                        ctxjcheck   : CONTEXTJ code point check",
	"                        ctxocheck   : CONTEXTO code point check for the",
	"                                      Domain Name Registration protocol",
	"                        ctxolitecheck",
	"                                    : CONTEXTO code point check for the",
	"                                      Domain Name Lookup protocol",
	"                        bidicheck   : Bidi check",
	"                        idnconv     : conversion from/to Punycode",
	"                        lencheck    : label length check",
	"                        rtcheck     : round trip check",
	"  -quiet              : suppress a result message; yield exit status only.",
	"  -q                  : synonym for -quiet",
	"  -localcheck FILE    : perform local check with a code point table FILE.",
	"  -e FILE             : synonym for -localcheck",
	"  -test               : output language and local encoding status,",
	"                        then exit.",
	"  -t                    synonym for -test",
	"  -version            : print version number, then exit.",
	"  -v                  : synonym for -version",
	"  -help               : print this help, then exit.",
	"  -h                  : synonym for -help",
	NULL
};

static void
print_test_status(idn_resconf_t conf) {
	printf("language = %s\n", idn_resconf_getlanguage(conf));
	printf("input encoding = %s\n", idn_resconf_getlocalencoding(conf));
}

static void
print_version(void) {
	printf("idncmp (idnkit) version: %s\n", IDNKIT_VERSION);
	printf("%s\n", idn_version_getstring());
}

static void
print_usage(void) {
	int i;

	for (i = 0; options[i] != NULL; i++)
		printf("%s\n", options[i]);
}

static void
errormsg(const char *fmt, ...) {
	va_list args;

	if (quiet_mode)
		return;

	va_start(args, fmt);
	vfprintf(stderr, fmt, args);
	va_end(args);
}
