#ifndef OPENMM_CUSTOMCPPFORCEIMPL_H_
#define OPENMM_CUSTOMCPPFORCEIMPL_H_

/* -------------------------------------------------------------------------- *
 *                                   OpenMM                                   *
 * -------------------------------------------------------------------------- *
 * This is part of the OpenMM molecular simulation toolkit originating from   *
 * Simbios, the NIH National Center for Physics-Based Simulation of           *
 * Biological Structures at Stanford, funded under the NIH Roadmap for        *
 * Medical Research, grant U54 GM072970. See https://simtk.org.               *
 *                                                                            *
 * Portions copyright (c) 2023 Stanford University and the Authors.           *
 * Authors: Peter Eastman                                                     *
 * Contributors:                                                              *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS, CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,    *
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR      *
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE  *
 * USE OR OTHER DEALINGS IN THE SOFTWARE.                                     *
 * -------------------------------------------------------------------------- */

#include "ForceImpl.h"
#include "openmm/Kernel.h"
#include "openmm/Vec3.h"
#include <string>
#include <vector>

namespace OpenMM {

/**
 * This class provides an easy way of writing Forces that are implemented entirely
 * in C++, and do not involve any platform-specific code.  To use it, have your
 * ForceImpl extend this class, implementing computeForce() to compute the forces
 * and energy in a platform-independent way.  You can optionally override other
 * methods as well, such as if your Force defines global parameters or bonds
 * between atoms.
 * 
 * Most Forces should not be implemented with this class.  Writing platform-specific
 * code that can run on a GPU and interact directly with internal data structures
 * usually leads to much faster performance.  This class is useful for special
 * situations, such as plugins that interact with external libraries through their
 * own public APIs.  In that case, there is little opportunity for optimization.
 * This class allows you to write a single implementation that is automatically
 * used for all platforms.
 */

class OPENMM_EXPORT CustomCPPForceImpl : public ForceImpl {
public:
    CustomCPPForceImpl(const Force& owner);
    /**
     * Subclasses may override this to do their own initialization, but they should
     * also call CustomCPPForceImpl::initialize(context) to let the parent class
     * initialize its kernel.
     * 
     * @param context    the context in which the system is being simulated
     */
    void initialize(ContextImpl& context);
    /**
     * Calculate the force on each particle generated by this ForceImpl and/or this ForceImpl's
     * contribution to the potential energy of the system.
     * 
     * @param context    the context in which the system is being simulated
     * @param positions  the position of every particle
     * @param forces     store the force on each particle into this array
     * @return this force's contribution to the potential energy of the system, or 0 if this
     * force does not contribute to potential energy
     */
    virtual double computeForce(ContextImpl& context, const std::vector<Vec3>& positions, std::vector<Vec3>& forces) = 0;
    /**
     * Override this if the force updates the context state directly.  In general
     * it should only make changes by invoking public methods of the ContextImpl.
     * If you need to access platform-specific data structures, this class is
     * probably not appropriate.
     * 
     * @param context        the context in which the system is being simulated
     * @param forcesInvalid  if the state was modified in any way that might cause previously
     *                       calculated forces to no longer be valid (such as modifying
     *                       positions or parameters), the method should set this to true.
     */
    void updateContextState(ContextImpl& context, bool& forcesInvalid) {
    }
    /**
     * Override this if the force defines global parameters.
     */
    std::map<std::string, double> getDefaultParameters() {
        return {};
    }
    /**
     * Override this if the force defines bonds between particles.
     */
    std::vector<std::pair<int, int> > getBondedParticles() const {
        return {};
    }
    /**
     * Override this if the force defines an updateParametersInContext() method.
     */
    void updateParametersInContext(ContextImpl& context) {
    }
    double calcForcesAndEnergy(ContextImpl& context, bool includeForces, bool includeEnergy, int groups);
    std::vector<std::string> getKernelNames();
private:
    Kernel kernel;
};

} // namespace OpenMM

#endif /*OPENMM_CUSTOMCPPFORCEIMPL_H_*/
