#!/usr/bin/env python

"""
The handler invocation mechanism.

Copyright (C) 2014, 2015, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.config import settings
from imiptools.data import Object, parse_object, get_value

try:
    from cStringIO import StringIO
except ImportError:
    from StringIO import StringIO

IMIP_COUNTER_AS_REQUEST = settings["IMIP_COUNTER_AS_REQUEST"]

def handle_itip_part(part, handlers):

    """
    Handle the given iTIP 'part' using the given 'handlers' dictionary.

    Return a list of responses, each response being a tuple of the form
    (outgoing-recipients, message-part).
    """

    method = part.get_param("method")
    method = method and method.upper()

    # Decode the data and parse it.

    f = StringIO(part.get_payload(decode=True))

    itip = parse_object(f, part.get_content_charset(), "VCALENDAR")

    # Ignore the part if not a calendar object.

    if not itip:
        return

    # Require consistency between declared and employed methods.

    itip_method = get_value(itip, "METHOD")

    if itip_method == method or \
       IMIP_COUNTER_AS_REQUEST and itip_method == "COUNTER" and method == "REQUEST":

        # Assert the object's method as the definitive one.

        method = itip_method

        # Look for different kinds of sections.

        all_results = []

        for name, items in itip.items():

            # Get a handler for the given section.

            handler = handlers.get(name)
            if not handler:
                continue

            for item in items:

                # Dispatch to a handler and obtain any response.

                handler.set_object(Object({name : item}))
                handler.set_identity(method)

                if handler.is_usable(method):

                    # Perform the method in a critical section.

                    handler.acquire_lock()
                    try:
                        methods[method](handler)()
                    finally:
                        handler.release_lock()

# Handler registry.

methods = {
    "ADD"            : lambda handler: handler.add,
    "CANCEL"         : lambda handler: handler.cancel,
    "COUNTER"        : lambda handler: handler.counter,
    "DECLINECOUNTER" : lambda handler: handler.declinecounter,
    "PUBLISH"        : lambda handler: handler.publish,
    "REFRESH"        : lambda handler: handler.refresh,
    "REPLY"          : lambda handler: handler.reply,
    "REQUEST"        : lambda handler: handler.request,
    }

# vim: tabstop=4 expandtab shiftwidth=4
