# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.

# Make coding more python3-ish
from __future__ import (absolute_import, division, print_function)
__metaclass__ = type


from ansible_collections.infoblox.nios_modules.plugins.modules import nios_member
from ansible_collections.infoblox.nios_modules.plugins.module_utils import api
from ansible_collections.infoblox.nios_modules.tests.unit.compat.mock import patch, MagicMock, Mock
from .test_nios_module import TestNiosModule, load_fixture


class TestNiosMemberModule(TestNiosModule):

    module = nios_member

    def setUp(self):
        super(TestNiosMemberModule, self).setUp()
        self.module = MagicMock(name='ansible_collections.infoblox.nios_modules.plugins.modules.nios_member.WapiModule')
        self.module.check_mode = False
        self.module.params = {'provider': None}
        self.mock_wapi = patch('ansible_collections.infoblox.nios_modules.plugins.modules.nios_member.WapiModule')
        self.exec_command = self.mock_wapi.start()
        self.mock_wapi_run = patch('ansible_collections.infoblox.nios_modules.plugins.modules.nios_member.WapiModule.run')
        self.load_config = self.mock_wapi_run.start()

    def tearDown(self):
        super(TestNiosMemberModule, self).tearDown()
        self.mock_wapi.stop()
        self.mock_wapi_run.stop()

    def load_fixtures(self, commands=None):
        self.exec_command.return_value = (0, load_fixture('nios_result.txt').strip(), None)
        self.load_config.return_value = dict(diff=None, session='session')

    def _get_wapi(self, test_object):
        wapi = api.WapiModule(self.module)
        wapi.get_object = Mock(name='get_object', return_value=test_object)
        wapi.create_object = Mock(name='create_object')
        wapi.update_object = Mock(name='update_object')
        wapi.delete_object = Mock(name='delete_object')
        return wapi

    def test_nios_member_create(self):
        self.module.params = {'provider': None, 'state': 'present', 'host_name': 'test_member',
                              'vip_setting': {'address': '192.168.1.110', 'subnet_mask': '255.255.255.0', 'gateway': '192.168.1.1'},
                              'config_addr_type': 'IPV4', 'platform': 'VNIOS', 'comment': None, 'extattrs': None}

        test_object = None
        test_spec = {
            "host_name": {"ib_req": True},
            "vip_setting": {},
            "config_addr_type": {},
            "platform": {},
            "comment": {},
            "extattrs": {}
        }

        wapi = self._get_wapi(test_object)
        res = wapi.run('testobject', test_spec)

        self.assertTrue(res['changed'])
        wapi.create_object.assert_called_once_with('testobject', {'host_name': 'test_member',
                                                                  'vip_setting': {'address': '192.168.1.110', 'subnet_mask': '255.255.255.0',
                                                                                  'gateway': '192.168.1.1'},
                                                                  'config_addr_type': 'IPV4', 'platform': 'VNIOS'})

    def test_nios_member_update(self):
        self.module.params = {'provider': None, 'state': 'present', 'host_name': 'test_member',
                              'vip_setting': {'address': '192.168.1.110', 'subnet_mask': '255.255.255.0', 'gateway': '192.168.1.1'},
                              'config_addr_type': 'IPV4', 'platform': 'VNIOS', 'comment': 'updated comment', 'extattrs': None}
        ref = "member/b25lLnZpcnR1YWxfbm9kZSQ3:member01.ansible-dev.com"
        test_object = [
            {
                "comment": "Created with Ansible",
                "_ref": ref,
                "config_addr_type": "IPV4",
                "host_name": "member01.ansible-dev.com",
                "platform": "VNIOS",
                "service_type_configuration": "ALL_V4",
                "vip_setting":
                    {
                        "address": "192.168.1.100",
                        "dscp": 0,
                        "gateway": "192.168.1.1",
                        "primary": True,
                        "subnet_mask": "255.255.255.0",
                        "use_dscp": False
                    }
            }
        ]

        test_spec = {
            "host_name": {"ib_req": True},
            "vip_setting": {},
            "config_addr_type": {},
            "platform": {},
            "comment": {},
            "extattrs": {}
        }

        wapi = self._get_wapi(test_object)
        res = wapi.run('testobject', test_spec)

        self.assertTrue(res['changed'])
        wapi.update_object.assert_called_once_with(
            ref,
            {'comment': 'updated comment', 'host_name': 'test_member',
             'vip_setting': {'address': '192.168.1.110', 'subnet_mask': '255.255.255.0', 'gateway': '192.168.1.1'},
             'config_addr_type': 'IPV4', 'platform': 'VNIOS'}
        )

    def test_nios_member_remove(self):
        self.module.params = {'provider': None, 'state': 'absent', 'host_name': 'test_member',
                              'vip_setting': {'address': '192.168.1.110', 'subnet_mask': '255.255.255.0', 'gateway': '192.168.1.1'},
                              'config_addr_type': 'IPV4', 'platform': 'VNIOS', 'comment': 'updated comment', 'extattrs': None}

        ref = "member/b25lLnZpcnR1YWxfbm9kZSQ3:member01.ansible-dev.com"

        test_object = [
            {
                "comment": "Created with Ansible",
                "_ref": "member/b25lLnZpcnR1YWxfbm9kZSQ3:member01.ansible-dev.com",
                "config_addr_type": "IPV4",
                "host_name": "member01.ansible-dev.com",
                "platform": "VNIOS",
                "service_type_configuration": "ALL_V4",
                "vip_setting":
                    {
                        "address": "192.168.1.100",
                        "dscp": 0,
                        "gateway": "192.168.1.1",
                        "primary": True,
                        "subnet_mask": "255.255.255.0",
                        "use_dscp": False
                    }
            }
        ]

        test_spec = {
            "host_name": {"ib_req": True},
            "vip_setting": {},
            "config_addr_type": {},
            "platform": {},
            "comment": {},
            "extattrs": {}
        }

        wapi = self._get_wapi(test_object)
        res = wapi.run('testobject', test_spec)

        self.assertTrue(res['changed'])
        wapi.delete_object.assert_called_once_with(ref)
