"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.prometheusManager = void 0;
var _base_connection_manager = require("./base_connection_manager");
var _constants = require("../../../common/constants");
var _prometheus_connection_client = require("../clients/prometheus_connection_client");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const BASE_RESOURCE_API = 'api/v1';
const BASE_ALERT_MANAGER_API = 'alertmanager/api/v2';

// docs: https://prometheus.io/docs/concepts/metric_types/#metric-types

/**
 * Parameters for PromQL query execution
 */

/**
 * Result of a single metric series
 */

/**
 * Response from a PromQL query
 */

// We want to ensure valid resourceType and resourceName parameters in the type system

class PrometheusManager extends _base_connection_manager.BaseConnectionManager {
  constructor() {
    super();
    // Set up the client factory for resource fetching
    const clientFactory = (context, request) => new _prometheus_connection_client.PrometheusConnectionClient(context, request);
    this.setClientFactory(clientFactory);
  }

  /**
   * Initialize the default query executor using the legacy OpenSearch client.
   * This should be called during plugin setup.
   * @param client The legacy cluster client for OpenSearch
   */
  initializeDefaultQueryExecutor(client) {
    const defaultQueryExecutor = {
      execute: async (context, request, params) => {
        // Support multi-datasource routing
        const dataSourceId = request.dataSourceId;
        const callAPI = dataSourceId ? context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI : client.asScoped(request).callAsCurrentUser;
        return callAPI('enhancements.promqlQuery', params);
      }
    };
    this.setQueryExecutor(defaultQueryExecutor);
  }
  getResourceURI(query) {
    const {
      resourceType,
      resourceName
    } = query;
    switch (resourceType) {
      case _constants.RESOURCE_TYPES.PROMETHEUS.LABELS:
        {
          return `${BASE_RESOURCE_API}/labels`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.ALERTS:
        {
          return `${BASE_RESOURCE_API}/alerts`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.LABEL_VALUES:
        {
          return `${BASE_RESOURCE_API}/label/${resourceName}/values`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.METRICS:
        {
          return `${BASE_RESOURCE_API}/label/__name__/values`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.METRIC_METADATA:
        {
          return `${BASE_RESOURCE_API}/metadata`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.ALERTS_GROUPS:
        {
          return `${BASE_ALERT_MANAGER_API}/alerts/groups`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.RULES:
        {
          return `${BASE_RESOURCE_API}/rules`;
        }
      case _constants.RESOURCE_TYPES.PROMETHEUS.SERIES:
        {
          return `${BASE_RESOURCE_API}/series`;
        }
      default:
        {
          throw Error(`unknown resource type: ${resourceType}`);
        }
    }
  }
  getResources(context, request, query) {
    return this.getClient(context, request).getResources({
      path: `${_constants.URI.DIRECT_QUERY.RESOURCES}/${query.dataSourceName}/${this.getResourceURI(query)}`,
      querystring: new URLSearchParams(query.query).toString()
    });
  }
  handlePostRequest(context, request) {
    const {
      id: dataSourceName
    } = request.body.connection;
    const {
      type: resourceType,
      name: resourceName
    } = request.body.resource;
    const content = request.body.content;
    const queryParams = {};
    if (resourceType === _constants.RESOURCE_TYPES.PROMETHEUS.LABELS && resourceName) {
      queryParams['match[]'] = resourceName;
    } else if (resourceType === _constants.RESOURCE_TYPES.PROMETHEUS.METRIC_METADATA && resourceName) {
      queryParams.metric = resourceName;
    } else if (resourceType === _constants.RESOURCE_TYPES.PROMETHEUS.SERIES && resourceName) {
      queryParams['match[]'] = resourceName;
    }
    if ((content === null || content === void 0 ? void 0 : content.start) !== undefined) {
      queryParams.start = String(content.start);
    }
    if ((content === null || content === void 0 ? void 0 : content.end) !== undefined) {
      queryParams.end = String(content.end);
    }
    const query = {
      dataSourceName,
      resourceType,
      resourceName,
      query: queryParams
    };
    return this.getResources(context, request, query);
  }
}

// we must export as singleton so all changes to clientFactory are respected across plugins
const prometheusManager = exports.prometheusManager = new PrometheusManager();