﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesis/KinesisRequest.h>
#include <aws/kinesis/Kinesis_EXPORTS.h>
#include <aws/kinesis/model/EncryptionType.h>

#include <utility>

namespace Aws {
namespace Kinesis {
namespace Model {

/**
 */
class StopStreamEncryptionRequest : public KinesisRequest {
 public:
  AWS_KINESIS_API StopStreamEncryptionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StopStreamEncryption"; }

  AWS_KINESIS_API Aws::String SerializePayload() const override;

  AWS_KINESIS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_KINESIS_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The name of the stream on which to stop encrypting records.</p>
   */
  inline const Aws::String& GetStreamName() const { return m_streamName; }
  inline bool StreamNameHasBeenSet() const { return m_streamNameHasBeenSet; }
  template <typename StreamNameT = Aws::String>
  void SetStreamName(StreamNameT&& value) {
    m_streamNameHasBeenSet = true;
    m_streamName = std::forward<StreamNameT>(value);
  }
  template <typename StreamNameT = Aws::String>
  StopStreamEncryptionRequest& WithStreamName(StreamNameT&& value) {
    SetStreamName(std::forward<StreamNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption type. The only valid value is <code>KMS</code>.</p>
   */
  inline EncryptionType GetEncryptionType() const { return m_encryptionType; }
  inline bool EncryptionTypeHasBeenSet() const { return m_encryptionTypeHasBeenSet; }
  inline void SetEncryptionType(EncryptionType value) {
    m_encryptionTypeHasBeenSet = true;
    m_encryptionType = value;
  }
  inline StopStreamEncryptionRequest& WithEncryptionType(EncryptionType value) {
    SetEncryptionType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The GUID for the customer-managed Amazon Web Services KMS key to use for
   * encryption. This value can be a globally unique identifier, a fully specified
   * Amazon Resource Name (ARN) to either an alias or a key, or an alias name
   * prefixed by "alias/".You can also use a master key owned by Kinesis Data Streams
   * by specifying the alias <code>aws/kinesis</code>.</p> <ul> <li> <p>Key ARN
   * example:
   * <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
   * </p> </li> <li> <p>Alias ARN example:
   * <code>arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code> </p> </li>
   * <li> <p>Globally unique key ID example:
   * <code>12345678-1234-1234-1234-123456789012</code> </p> </li> <li> <p>Alias name
   * example: <code>alias/MyAliasName</code> </p> </li> <li> <p>Master key owned by
   * Kinesis Data Streams: <code>alias/aws/kinesis</code> </p> </li> </ul>
   */
  inline const Aws::String& GetKeyId() const { return m_keyId; }
  inline bool KeyIdHasBeenSet() const { return m_keyIdHasBeenSet; }
  template <typename KeyIdT = Aws::String>
  void SetKeyId(KeyIdT&& value) {
    m_keyIdHasBeenSet = true;
    m_keyId = std::forward<KeyIdT>(value);
  }
  template <typename KeyIdT = Aws::String>
  StopStreamEncryptionRequest& WithKeyId(KeyIdT&& value) {
    SetKeyId(std::forward<KeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the stream.</p>
   */
  inline const Aws::String& GetStreamARN() const { return m_streamARN; }
  inline bool StreamARNHasBeenSet() const { return m_streamARNHasBeenSet; }
  template <typename StreamARNT = Aws::String>
  void SetStreamARN(StreamARNT&& value) {
    m_streamARNHasBeenSet = true;
    m_streamARN = std::forward<StreamARNT>(value);
  }
  template <typename StreamARNT = Aws::String>
  StopStreamEncryptionRequest& WithStreamARN(StreamARNT&& value) {
    SetStreamARN(std::forward<StreamARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Not Implemented. Reserved for future use.</p>
   */
  inline const Aws::String& GetStreamId() const { return m_streamId; }
  inline bool StreamIdHasBeenSet() const { return m_streamIdHasBeenSet; }
  template <typename StreamIdT = Aws::String>
  void SetStreamId(StreamIdT&& value) {
    m_streamIdHasBeenSet = true;
    m_streamId = std::forward<StreamIdT>(value);
  }
  template <typename StreamIdT = Aws::String>
  StopStreamEncryptionRequest& WithStreamId(StreamIdT&& value) {
    SetStreamId(std::forward<StreamIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_streamName;

  EncryptionType m_encryptionType{EncryptionType::NOT_SET};

  Aws::String m_keyId;

  Aws::String m_streamARN;

  Aws::String m_streamId;
  bool m_streamNameHasBeenSet = false;
  bool m_encryptionTypeHasBeenSet = false;
  bool m_keyIdHasBeenSet = false;
  bool m_streamARNHasBeenSet = false;
  bool m_streamIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kinesis
}  // namespace Aws
