/*
 * CRC.m
 *
 * Copyright (c) 2017-present, MacPaw Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301  USA
 */
#import "Checksums.h"

uint32_t XADCRC(uint32_t prevcrc,uint8_t byte,const uint32_t *table)
{
	return table[(prevcrc^byte)&0xff]^(prevcrc>>8);
}

uint32_t XADCalculateCRC(uint32_t prevcrc,const uint8_t *buffer,int length,const uint32_t *table)
{
	uint32_t crc=prevcrc;
	for(int i=0;i<length;i++) crc=XADCRC(crc,buffer[i],table);
	return crc;
}

#if XAD_BYTE_ORDER_BIG_ENDIAN
static inline uint32_t swap(uint32_t x)
{
#if defined(__GNUC__) || defined(__clang__)
    return __builtin_bswap32(x);
#else
    return (x >> 24) |
    ((x >>  8) & 0x0000FF00) |
    ((x <<  8) & 0x00FF0000) |
    (x << 24);
#endif
}
#endif

uint32_t XADCalculateCRCFast(uint32_t prevcrc,const uint8_t *buffer,int length, const uint32_t (*table)[256])
{
    uint32_t crc = prevcrc;
    const uint32_t* pos = (const uint32_t*) buffer;
    
    const size_t Unroll = 4;
    const size_t processedBytes = 16 * Unroll;
    
    while (length >= processedBytes)
    {
        for (size_t unrolling = 0; unrolling < Unroll; unrolling++)
        {
#if XAD_BYTE_ORDER_BIG_ENDIAN
            uint32_t a   = *pos++ ^ swap(crc);
            uint32_t b   = *pos++;
            uint32_t c   = *pos++;
            uint32_t d   = *pos++;
            crc  = table[ 0][ d        & 0xFF] ^
                   table[ 1][(d >>  8) & 0xFF] ^
                   table[ 2][(d >> 16) & 0xFF] ^
                   table[ 3][(d >> 24) & 0xFF] ^
                   table[ 4][ c        & 0xFF] ^
                   table[ 5][(c >>  8) & 0xFF] ^
                   table[ 6][(c >> 16) & 0xFF] ^
                   table[ 7][(c >> 24) & 0xFF] ^
                   table[ 8][ b        & 0xFF] ^
                   table[ 9][(b >>  8) & 0xFF] ^
                   table[10][(b >> 16) & 0xFF] ^
                   table[11][(b >> 24) & 0xFF] ^
                   table[12][ a        & 0xFF] ^
                   table[13][(a >>  8) & 0xFF] ^
                   table[14][(a >> 16) & 0xFF] ^
                   table[15][(a >> 24) & 0xFF];
#else
            uint32_t a = *pos++ ^ crc;
            uint32_t b = *pos++;
            uint32_t c = *pos++;
            uint32_t d = *pos++;
            crc  = table[ 0][(d >> 24) & 0xFF] ^
                   table[ 1][(d >> 16) & 0xFF] ^
                   table[ 2][(d >>  8) & 0xFF] ^
                   table[ 3][ d        & 0xFF] ^
                   table[ 4][(c >> 24) & 0xFF] ^
                   table[ 5][(c >> 16) & 0xFF] ^
                   table[ 6][(c >>  8) & 0xFF] ^
                   table[ 7][ c        & 0xFF] ^
                   table[ 8][(b >> 24) & 0xFF] ^
                   table[ 9][(b >> 16) & 0xFF] ^
                   table[10][(b >>  8) & 0xFF] ^
                   table[11][ b        & 0xFF] ^
                   table[12][(a >> 24) & 0xFF] ^
                   table[13][(a >> 16) & 0xFF] ^
                   table[14][(a >>  8) & 0xFF] ^
                   table[15][ a        & 0xFF];
#endif
        }
        
        length -= processedBytes;
    }
    
    const uint8_t* currByte = (const uint8_t*) pos;
    while (length-- != 0)
        crc = (crc >> 8) ^ table[0][(crc & 0xFF) ^ *currByte++];
    
    return crc;
}

uint64_t XADCRC64(uint64_t prevcrc,uint8_t byte,const uint64_t *table)
{
	return table[(prevcrc^byte)&0xff]^(prevcrc>>8);
}

uint64_t XADCalculateCRC64(uint64_t prevcrc,const uint8_t *buffer,int length,const uint64_t *table)
{
	uint64_t crc=prevcrc;
	for(int i=0;i<length;i++) crc=XADCRC64(crc,buffer[i],table);
	return crc;
}

int XADUnReverseCRC16(int val)
{
	val=((val>>8)&0x00FF)|((val&0x00FF)<<8);
	return val;
}

const uint32_t XADCRCTable_a001[256]=
{
	0x00000000,0x0000c0c1,0x0000c181,0x00000140,0x0000c301,0x000003c0,0x00000280,0x0000c241,
	0x0000c601,0x000006c0,0x00000780,0x0000c741,0x00000500,0x0000c5c1,0x0000c481,0x00000440,
	0x0000cc01,0x00000cc0,0x00000d80,0x0000cd41,0x00000f00,0x0000cfc1,0x0000ce81,0x00000e40,
	0x00000a00,0x0000cac1,0x0000cb81,0x00000b40,0x0000c901,0x000009c0,0x00000880,0x0000c841,
	0x0000d801,0x000018c0,0x00001980,0x0000d941,0x00001b00,0x0000dbc1,0x0000da81,0x00001a40,
	0x00001e00,0x0000dec1,0x0000df81,0x00001f40,0x0000dd01,0x00001dc0,0x00001c80,0x0000dc41,
	0x00001400,0x0000d4c1,0x0000d581,0x00001540,0x0000d701,0x000017c0,0x00001680,0x0000d641,
	0x0000d201,0x000012c0,0x00001380,0x0000d341,0x00001100,0x0000d1c1,0x0000d081,0x00001040,
	0x0000f001,0x000030c0,0x00003180,0x0000f141,0x00003300,0x0000f3c1,0x0000f281,0x00003240,
	0x00003600,0x0000f6c1,0x0000f781,0x00003740,0x0000f501,0x000035c0,0x00003480,0x0000f441,
	0x00003c00,0x0000fcc1,0x0000fd81,0x00003d40,0x0000ff01,0x00003fc0,0x00003e80,0x0000fe41,
	0x0000fa01,0x00003ac0,0x00003b80,0x0000fb41,0x00003900,0x0000f9c1,0x0000f881,0x00003840,
	0x00002800,0x0000e8c1,0x0000e981,0x00002940,0x0000eb01,0x00002bc0,0x00002a80,0x0000ea41,
	0x0000ee01,0x00002ec0,0x00002f80,0x0000ef41,0x00002d00,0x0000edc1,0x0000ec81,0x00002c40,
	0x0000e401,0x000024c0,0x00002580,0x0000e541,0x00002700,0x0000e7c1,0x0000e681,0x00002640,
	0x00002200,0x0000e2c1,0x0000e381,0x00002340,0x0000e101,0x000021c0,0x00002080,0x0000e041,
	0x0000a001,0x000060c0,0x00006180,0x0000a141,0x00006300,0x0000a3c1,0x0000a281,0x00006240,
	0x00006600,0x0000a6c1,0x0000a781,0x00006740,0x0000a501,0x000065c0,0x00006480,0x0000a441,
	0x00006c00,0x0000acc1,0x0000ad81,0x00006d40,0x0000af01,0x00006fc0,0x00006e80,0x0000ae41,
	0x0000aa01,0x00006ac0,0x00006b80,0x0000ab41,0x00006900,0x0000a9c1,0x0000a881,0x00006840,
	0x00007800,0x0000b8c1,0x0000b981,0x00007940,0x0000bb01,0x00007bc0,0x00007a80,0x0000ba41,
	0x0000be01,0x00007ec0,0x00007f80,0x0000bf41,0x00007d00,0x0000bdc1,0x0000bc81,0x00007c40,
	0x0000b401,0x000074c0,0x00007580,0x0000b541,0x00007700,0x0000b7c1,0x0000b681,0x00007640,
	0x00007200,0x0000b2c1,0x0000b381,0x00007340,0x0000b101,0x000071c0,0x00007080,0x0000b041,
	0x00005000,0x000090c1,0x00009181,0x00005140,0x00009301,0x000053c0,0x00005280,0x00009241,
	0x00009601,0x000056c0,0x00005780,0x00009741,0x00005500,0x000095c1,0x00009481,0x00005440,
	0x00009c01,0x00005cc0,0x00005d80,0x00009d41,0x00005f00,0x00009fc1,0x00009e81,0x00005e40,
	0x00005a00,0x00009ac1,0x00009b81,0x00005b40,0x00009901,0x000059c0,0x00005880,0x00009841,
	0x00008801,0x000048c0,0x00004980,0x00008941,0x00004b00,0x00008bc1,0x00008a81,0x00004a40,
	0x00004e00,0x00008ec1,0x00008f81,0x00004f40,0x00008d01,0x00004dc0,0x00004c80,0x00008c41,
	0x00004400,0x000084c1,0x00008581,0x00004540,0x00008701,0x000047c0,0x00004680,0x00008641,
	0x00008201,0x000042c0,0x00004380,0x00008341,0x00004100,0x000081c1,0x00008081,0x00004040,
};

const uint32_t XADCRCReverseTable_1021[256]=
{
	0x00000000,0x00002110,0x00004220,0x00006330,0x00008440,0x0000a550,0x0000c660,0x0000e770,
	0x00000881,0x00002991,0x00004aa1,0x00006bb1,0x00008cc1,0x0000add1,0x0000cee1,0x0000eff1,
	0x00003112,0x00001002,0x00007332,0x00005222,0x0000b552,0x00009442,0x0000f772,0x0000d662,
	0x00003993,0x00001883,0x00007bb3,0x00005aa3,0x0000bdd3,0x00009cc3,0x0000fff3,0x0000dee3,
	0x00006224,0x00004334,0x00002004,0x00000114,0x0000e664,0x0000c774,0x0000a444,0x00008554,
	0x00006aa5,0x00004bb5,0x00002885,0x00000995,0x0000eee5,0x0000cff5,0x0000acc5,0x00008dd5,
	0x00005336,0x00007226,0x00001116,0x00003006,0x0000d776,0x0000f666,0x00009556,0x0000b446,
	0x00005bb7,0x00007aa7,0x00001997,0x00003887,0x0000dff7,0x0000fee7,0x00009dd7,0x0000bcc7,
	0x0000c448,0x0000e558,0x00008668,0x0000a778,0x00004008,0x00006118,0x00000228,0x00002338,
	0x0000ccc9,0x0000edd9,0x00008ee9,0x0000aff9,0x00004889,0x00006999,0x00000aa9,0x00002bb9,
	0x0000f55a,0x0000d44a,0x0000b77a,0x0000966a,0x0000711a,0x0000500a,0x0000333a,0x0000122a,
	0x0000fddb,0x0000dccb,0x0000bffb,0x00009eeb,0x0000799b,0x0000588b,0x00003bbb,0x00001aab,
	0x0000a66c,0x0000877c,0x0000e44c,0x0000c55c,0x0000222c,0x0000033c,0x0000600c,0x0000411c,
	0x0000aeed,0x00008ffd,0x0000eccd,0x0000cddd,0x00002aad,0x00000bbd,0x0000688d,0x0000499d,
	0x0000977e,0x0000b66e,0x0000d55e,0x0000f44e,0x0000133e,0x0000322e,0x0000511e,0x0000700e,
	0x00009fff,0x0000beef,0x0000dddf,0x0000fccf,0x00001bbf,0x00003aaf,0x0000599f,0x0000788f,
	0x00008891,0x0000a981,0x0000cab1,0x0000eba1,0x00000cd1,0x00002dc1,0x00004ef1,0x00006fe1,
	0x00008010,0x0000a100,0x0000c230,0x0000e320,0x00000450,0x00002540,0x00004670,0x00006760,
	0x0000b983,0x00009893,0x0000fba3,0x0000dab3,0x00003dc3,0x00001cd3,0x00007fe3,0x00005ef3,
	0x0000b102,0x00009012,0x0000f322,0x0000d232,0x00003542,0x00001452,0x00007762,0x00005672,
	0x0000eab5,0x0000cba5,0x0000a895,0x00008985,0x00006ef5,0x00004fe5,0x00002cd5,0x00000dc5,
	0x0000e234,0x0000c324,0x0000a014,0x00008104,0x00006674,0x00004764,0x00002454,0x00000544,
	0x0000dba7,0x0000fab7,0x00009987,0x0000b897,0x00005fe7,0x00007ef7,0x00001dc7,0x00003cd7,
	0x0000d326,0x0000f236,0x00009106,0x0000b016,0x00005766,0x00007676,0x00001546,0x00003456,
	0x00004cd9,0x00006dc9,0x00000ef9,0x00002fe9,0x0000c899,0x0000e989,0x00008ab9,0x0000aba9,
	0x00004458,0x00006548,0x00000678,0x00002768,0x0000c018,0x0000e108,0x00008238,0x0000a328,
	0x00007dcb,0x00005cdb,0x00003feb,0x00001efb,0x0000f98b,0x0000d89b,0x0000bbab,0x00009abb,
	0x0000754a,0x0000545a,0x0000376a,0x0000167a,0x0000f10a,0x0000d01a,0x0000b32a,0x0000923a,
	0x00002efd,0x00000fed,0x00006cdd,0x00004dcd,0x0000aabd,0x00008bad,0x0000e89d,0x0000c98d,
	0x0000267c,0x0000076c,0x0000645c,0x0000454c,0x0000a23c,0x0000832c,0x0000e01c,0x0000c10c,
	0x00001fef,0x00003eff,0x00005dcf,0x00007cdf,0x00009baf,0x0000babf,0x0000d98f,0x0000f89f,
	0x0000176e,0x0000367e,0x0000554e,0x0000745e,0x0000932e,0x0000b23e,0x0000d10e,0x0000f01e,
};

const uint32_t XADCRCTable_usb[256] =
{
    0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280, 0xC241, 0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481, 0x0440,
    0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81, 0x0E40, 0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880, 0xC841,
    0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81, 0x1A40, 0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80, 0xDC41,
    0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680, 0xD641, 0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081, 0x1040,
    0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281, 0x3240, 0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480, 0xF441,
    0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80, 0xFE41, 0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881, 0x3840,
    0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80, 0xEA41, 0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81, 0x2C40,
    0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681, 0x2640, 0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080, 0xE041,
    0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281, 0x6240, 0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480, 0xA441,
    0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80, 0xAE41, 0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881, 0x6840,
    0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80, 0xBA41, 0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81, 0x7C40,
    0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681, 0x7640, 0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080, 0xB041,
    0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280, 0x9241, 0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481, 0x5440,
    0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81, 0x5E40, 0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880, 0x9841,
    0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81, 0x4A40, 0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80, 0x8C41,
    0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680, 0x8641, 0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081, 0x4040,
};


const uint32_t XADCRCTable_edb88320[256]=
{
	0x00000000,0x77073096,0xee0e612c,0x990951ba,0x076dc419,0x706af48f,0xe963a535,0x9e6495a3,
	0x0edb8832,0x79dcb8a4,0xe0d5e91e,0x97d2d988,0x09b64c2b,0x7eb17cbd,0xe7b82d07,0x90bf1d91,
	0x1db71064,0x6ab020f2,0xf3b97148,0x84be41de,0x1adad47d,0x6ddde4eb,0xf4d4b551,0x83d385c7,
	0x136c9856,0x646ba8c0,0xfd62f97a,0x8a65c9ec,0x14015c4f,0x63066cd9,0xfa0f3d63,0x8d080df5,
	0x3b6e20c8,0x4c69105e,0xd56041e4,0xa2677172,0x3c03e4d1,0x4b04d447,0xd20d85fd,0xa50ab56b,
	0x35b5a8fa,0x42b2986c,0xdbbbc9d6,0xacbcf940,0x32d86ce3,0x45df5c75,0xdcd60dcf,0xabd13d59,
	0x26d930ac,0x51de003a,0xc8d75180,0xbfd06116,0x21b4f4b5,0x56b3c423,0xcfba9599,0xb8bda50f,
	0x2802b89e,0x5f058808,0xc60cd9b2,0xb10be924,0x2f6f7c87,0x58684c11,0xc1611dab,0xb6662d3d,
	0x76dc4190,0x01db7106,0x98d220bc,0xefd5102a,0x71b18589,0x06b6b51f,0x9fbfe4a5,0xe8b8d433,
	0x7807c9a2,0x0f00f934,0x9609a88e,0xe10e9818,0x7f6a0dbb,0x086d3d2d,0x91646c97,0xe6635c01,
	0x6b6b51f4,0x1c6c6162,0x856530d8,0xf262004e,0x6c0695ed,0x1b01a57b,0x8208f4c1,0xf50fc457,
	0x65b0d9c6,0x12b7e950,0x8bbeb8ea,0xfcb9887c,0x62dd1ddf,0x15da2d49,0x8cd37cf3,0xfbd44c65,
	0x4db26158,0x3ab551ce,0xa3bc0074,0xd4bb30e2,0x4adfa541,0x3dd895d7,0xa4d1c46d,0xd3d6f4fb,
	0x4369e96a,0x346ed9fc,0xad678846,0xda60b8d0,0x44042d73,0x33031de5,0xaa0a4c5f,0xdd0d7cc9,
	0x5005713c,0x270241aa,0xbe0b1010,0xc90c2086,0x5768b525,0x206f85b3,0xb966d409,0xce61e49f,
	0x5edef90e,0x29d9c998,0xb0d09822,0xc7d7a8b4,0x59b33d17,0x2eb40d81,0xb7bd5c3b,0xc0ba6cad,
	0xedb88320,0x9abfb3b6,0x03b6e20c,0x74b1d29a,0xead54739,0x9dd277af,0x04db2615,0x73dc1683,
	0xe3630b12,0x94643b84,0x0d6d6a3e,0x7a6a5aa8,0xe40ecf0b,0x9309ff9d,0x0a00ae27,0x7d079eb1,
	0xf00f9344,0x8708a3d2,0x1e01f268,0x6906c2fe,0xf762575d,0x806567cb,0x196c3671,0x6e6b06e7,
	0xfed41b76,0x89d32be0,0x10da7a5a,0x67dd4acc,0xf9b9df6f,0x8ebeeff9,0x17b7be43,0x60b08ed5,
	0xd6d6a3e8,0xa1d1937e,0x38d8c2c4,0x4fdff252,0xd1bb67f1,0xa6bc5767,0x3fb506dd,0x48b2364b,
	0xd80d2bda,0xaf0a1b4c,0x36034af6,0x41047a60,0xdf60efc3,0xa867df55,0x316e8eef,0x4669be79,
	0xcb61b38c,0xbc66831a,0x256fd2a0,0x5268e236,0xcc0c7795,0xbb0b4703,0x220216b9,0x5505262f,
	0xc5ba3bbe,0xb2bd0b28,0x2bb45a92,0x5cb36a04,0xc2d7ffa7,0xb5d0cf31,0x2cd99e8b,0x5bdeae1d,
	0x9b64c2b0,0xec63f226,0x756aa39c,0x026d930a,0x9c0906a9,0xeb0e363f,0x72076785,0x05005713,
	0x95bf4a82,0xe2b87a14,0x7bb12bae,0x0cb61b38,0x92d28e9b,0xe5d5be0d,0x7cdcefb7,0x0bdbdf21,
	0x86d3d2d4,0xf1d4e242,0x68ddb3f8,0x1fda836e,0x81be16cd,0xf6b9265b,0x6fb077e1,0x18b74777,
	0x88085ae6,0xff0f6a70,0x66063bca,0x11010b5c,0x8f659eff,0xf862ae69,0x616bffd3,0x166ccf45,
	0xa00ae278,0xd70dd2ee,0x4e048354,0x3903b3c2,0xa7672661,0xd06016f7,0x4969474d,0x3e6e77db,
	0xaed16a4a,0xd9d65adc,0x40df0b66,0x37d83bf0,0xa9bcae53,0xdebb9ec5,0x47b2cf7f,0x30b5ffe9,
	0xbdbdf21c,0xcabac28a,0x53b39330,0x24b4a3a6,0xbad03605,0xcdd70693,0x54de5729,0x23d967bf,
	0xb3667a2e,0xc4614ab8,0x5d681b02,0x2a6f2b94,0xb40bbe37,0xc30c8ea1,0x5a05df1b,0x2d02ef8d,
};

const uint64_t XADCRCTable_c96c5795d7870f42[256]=
{
	0x0000000000000000,0xb32e4cbe03a75f6f,0xf4843657a840a05b,0x47aa7ae9abe7ff34,
	0x7bd0c384ff8f5e33,0xc8fe8f3afc28015c,0x8f54f5d357cffe68,0x3c7ab96d5468a107,
	0xf7a18709ff1ebc66,0x448fcbb7fcb9e309,0x0325b15e575e1c3d,0xb00bfde054f94352,
	0x8c71448d0091e255,0x3f5f08330336bd3a,0x78f572daa8d1420e,0xcbdb3e64ab761d61,
	0x7d9ba13851336649,0xceb5ed8652943926,0x891f976ff973c612,0x3a31dbd1fad4997d,
	0x064b62bcaebc387a,0xb5652e02ad1b6715,0xf2cf54eb06fc9821,0x41e11855055bc74e,
	0x8a3a2631ae2dda2f,0x39146a8fad8a8540,0x7ebe1066066d7a74,0xcd905cd805ca251b,
	0xf1eae5b551a2841c,0x42c4a90b5205db73,0x056ed3e2f9e22447,0xb6409f5cfa457b28,
	0xfb374270a266cc92,0x48190ecea1c193fd,0x0fb374270a266cc9,0xbc9d3899098133a6,
	0x80e781f45de992a1,0x33c9cd4a5e4ecdce,0x7463b7a3f5a932fa,0xc74dfb1df60e6d95,
	0x0c96c5795d7870f4,0xbfb889c75edf2f9b,0xf812f32ef538d0af,0x4b3cbf90f69f8fc0,
	0x774606fda2f72ec7,0xc4684a43a15071a8,0x83c230aa0ab78e9c,0x30ec7c140910d1f3,
	0x86ace348f355aadb,0x3582aff6f0f2f5b4,0x7228d51f5b150a80,0xc10699a158b255ef,
	0xfd7c20cc0cdaf4e8,0x4e526c720f7dab87,0x09f8169ba49a54b3,0xbad65a25a73d0bdc,
	0x710d64410c4b16bd,0xc22328ff0fec49d2,0x85895216a40bb6e6,0x36a71ea8a7ace989,
	0x0adda7c5f3c4488e,0xb9f3eb7bf06317e1,0xfe5991925b84e8d5,0x4d77dd2c5823b7ba,
	0x64b62bcaebc387a1,0xd7986774e864d8ce,0x90321d9d438327fa,0x231c512340247895,
	0x1f66e84e144cd992,0xac48a4f017eb86fd,0xebe2de19bc0c79c9,0x58cc92a7bfab26a6,
	0x9317acc314dd3bc7,0x2039e07d177a64a8,0x67939a94bc9d9b9c,0xd4bdd62abf3ac4f3,
	0xe8c76f47eb5265f4,0x5be923f9e8f53a9b,0x1c4359104312c5af,0xaf6d15ae40b59ac0,
	0x192d8af2baf0e1e8,0xaa03c64cb957be87,0xeda9bca512b041b3,0x5e87f01b11171edc,
	0x62fd4976457fbfdb,0xd1d305c846d8e0b4,0x96797f21ed3f1f80,0x2557339fee9840ef,
	0xee8c0dfb45ee5d8e,0x5da24145464902e1,0x1a083bacedaefdd5,0xa9267712ee09a2ba,
	0x955cce7fba6103bd,0x267282c1b9c65cd2,0x61d8f8281221a3e6,0xd2f6b4961186fc89,
	0x9f8169ba49a54b33,0x2caf25044a02145c,0x6b055fede1e5eb68,0xd82b1353e242b407,
	0xe451aa3eb62a1500,0x577fe680b58d4a6f,0x10d59c691e6ab55b,0xa3fbd0d71dcdea34,
	0x6820eeb3b6bbf755,0xdb0ea20db51ca83a,0x9ca4d8e41efb570e,0x2f8a945a1d5c0861,
	0x13f02d374934a966,0xa0de61894a93f609,0xe7741b60e174093d,0x545a57dee2d35652,
	0xe21ac88218962d7a,0x5134843c1b317215,0x169efed5b0d68d21,0xa5b0b26bb371d24e,
	0x99ca0b06e7197349,0x2ae447b8e4be2c26,0x6d4e3d514f59d312,0xde6071ef4cfe8c7d,
	0x15bb4f8be788911c,0xa6950335e42fce73,0xe13f79dc4fc83147,0x521135624c6f6e28,
	0x6e6b8c0f1807cf2f,0xdd45c0b11ba09040,0x9aefba58b0476f74,0x29c1f6e6b3e0301b,
	0xc96c5795d7870f42,0x7a421b2bd420502d,0x3de861c27fc7af19,0x8ec62d7c7c60f076,
	0xb2bc941128085171,0x0192d8af2baf0e1e,0x4638a2468048f12a,0xf516eef883efae45,
	0x3ecdd09c2899b324,0x8de39c222b3eec4b,0xca49e6cb80d9137f,0x7967aa75837e4c10,
	0x451d1318d716ed17,0xf6335fa6d4b1b278,0xb199254f7f564d4c,0x02b769f17cf11223,
	0xb4f7f6ad86b4690b,0x07d9ba1385133664,0x4073c0fa2ef4c950,0xf35d8c442d53963f,
	0xcf273529793b3738,0x7c0979977a9c6857,0x3ba3037ed17b9763,0x888d4fc0d2dcc80c,
	0x435671a479aad56d,0xf0783d1a7a0d8a02,0xb7d247f3d1ea7536,0x04fc0b4dd24d2a59,
	0x3886b22086258b5e,0x8ba8fe9e8582d431,0xcc0284772e652b05,0x7f2cc8c92dc2746a,
	0x325b15e575e1c3d0,0x8175595b76469cbf,0xc6df23b2dda1638b,0x75f16f0cde063ce4,
	0x498bd6618a6e9de3,0xfaa59adf89c9c28c,0xbd0fe036222e3db8,0x0e21ac88218962d7,
	0xc5fa92ec8aff7fb6,0x76d4de52895820d9,0x317ea4bb22bfdfed,0x8250e80521188082,
	0xbe2a516875702185,0x0d041dd676d77eea,0x4aae673fdd3081de,0xf9802b81de97deb1,
	0x4fc0b4dd24d2a599,0xfceef8632775faf6,0xbb44828a8c9205c2,0x086ace348f355aad,
	0x34107759db5dfbaa,0x873e3be7d8faa4c5,0xc094410e731d5bf1,0x73ba0db070ba049e,
	0xb86133d4dbcc19ff,0x0b4f7f6ad86b4690,0x4ce50583738cb9a4,0xffcb493d702be6cb,
	0xc3b1f050244347cc,0x709fbcee27e418a3,0x3735c6078c03e797,0x841b8ab98fa4b8f8,
	0xadda7c5f3c4488e3,0x1ef430e13fe3d78c,0x595e4a08940428b8,0xea7006b697a377d7,
	0xd60abfdbc3cbd6d0,0x6524f365c06c89bf,0x228e898c6b8b768b,0x91a0c532682c29e4,
	0x5a7bfb56c35a3485,0xe955b7e8c0fd6bea,0xaeffcd016b1a94de,0x1dd181bf68bdcbb1,
	0x21ab38d23cd56ab6,0x9285746c3f7235d9,0xd52f0e859495caed,0x6601423b97329582,
	0xd041dd676d77eeaa,0x636f91d96ed0b1c5,0x24c5eb30c5374ef1,0x97eba78ec690119e,
	0xab911ee392f8b099,0x18bf525d915feff6,0x5f1528b43ab810c2,0xec3b640a391f4fad,
	0x27e05a6e926952cc,0x94ce16d091ce0da3,0xd3646c393a29f297,0x604a2087398eadf8,
	0x5c3099ea6de60cff,0xef1ed5546e415390,0xa8b4afbdc5a6aca4,0x1b9ae303c601f3cb,
	0x56ed3e2f9e224471,0xe5c372919d851b1e,0xa26908783662e42a,0x114744c635c5bb45,
	0x2d3dfdab61ad1a42,0x9e13b115620a452d,0xd9b9cbfcc9edba19,0x6a978742ca4ae576,
	0xa14cb926613cf817,0x1262f598629ba778,0x55c88f71c97c584c,0xe6e6c3cfcadb0723,
	0xda9c7aa29eb3a624,0x69b2361c9d14f94b,0x2e184cf536f3067f,0x9d36004b35545910,
	0x2b769f17cf112238,0x9858d3a9ccb67d57,0xdff2a94067518263,0x6cdce5fe64f6dd0c,
	0x50a65c93309e7c0b,0xe388102d33392364,0xa4226ac498dedc50,0x170c267a9b79833f,
	0xdcd7181e300f9e5e,0x6ff954a033a8c131,0x28532e49984f3e05,0x9b7d62f79be8616a,
	0xa707db9acf80c06d,0x14299724cc279f02,0x5383edcd67c06036,0xe0ada17364673f59,
};

const uint32_t XADCRCTable_sliced16_edb88320[16][256] =
{
    {
        0x00000000,0x77073096,0xee0e612c,0x990951ba,0x076dc419,0x706af48f,0xe963a535,0x9e6495a3,
        0x0edb8832,0x79dcb8a4,0xe0d5e91e,0x97d2d988,0x09b64c2b,0x7eb17cbd,0xe7b82d07,0x90bf1d91,
        0x1db71064,0x6ab020f2,0xf3b97148,0x84be41de,0x1adad47d,0x6ddde4eb,0xf4d4b551,0x83d385c7,
        0x136c9856,0x646ba8c0,0xfd62f97a,0x8a65c9ec,0x14015c4f,0x63066cd9,0xfa0f3d63,0x8d080df5,
        0x3b6e20c8,0x4c69105e,0xd56041e4,0xa2677172,0x3c03e4d1,0x4b04d447,0xd20d85fd,0xa50ab56b,
        0x35b5a8fa,0x42b2986c,0xdbbbc9d6,0xacbcf940,0x32d86ce3,0x45df5c75,0xdcd60dcf,0xabd13d59,
        0x26d930ac,0x51de003a,0xc8d75180,0xbfd06116,0x21b4f4b5,0x56b3c423,0xcfba9599,0xb8bda50f,
        0x2802b89e,0x5f058808,0xc60cd9b2,0xb10be924,0x2f6f7c87,0x58684c11,0xc1611dab,0xb6662d3d,
        0x76dc4190,0x01db7106,0x98d220bc,0xefd5102a,0x71b18589,0x06b6b51f,0x9fbfe4a5,0xe8b8d433,
        0x7807c9a2,0x0f00f934,0x9609a88e,0xe10e9818,0x7f6a0dbb,0x086d3d2d,0x91646c97,0xe6635c01,
        0x6b6b51f4,0x1c6c6162,0x856530d8,0xf262004e,0x6c0695ed,0x1b01a57b,0x8208f4c1,0xf50fc457,
        0x65b0d9c6,0x12b7e950,0x8bbeb8ea,0xfcb9887c,0x62dd1ddf,0x15da2d49,0x8cd37cf3,0xfbd44c65,
        0x4db26158,0x3ab551ce,0xa3bc0074,0xd4bb30e2,0x4adfa541,0x3dd895d7,0xa4d1c46d,0xd3d6f4fb,
        0x4369e96a,0x346ed9fc,0xad678846,0xda60b8d0,0x44042d73,0x33031de5,0xaa0a4c5f,0xdd0d7cc9,
        0x5005713c,0x270241aa,0xbe0b1010,0xc90c2086,0x5768b525,0x206f85b3,0xb966d409,0xce61e49f,
        0x5edef90e,0x29d9c998,0xb0d09822,0xc7d7a8b4,0x59b33d17,0x2eb40d81,0xb7bd5c3b,0xc0ba6cad,
        0xedb88320,0x9abfb3b6,0x03b6e20c,0x74b1d29a,0xead54739,0x9dd277af,0x04db2615,0x73dc1683,
        0xe3630b12,0x94643b84,0x0d6d6a3e,0x7a6a5aa8,0xe40ecf0b,0x9309ff9d,0x0a00ae27,0x7d079eb1,
        0xf00f9344,0x8708a3d2,0x1e01f268,0x6906c2fe,0xf762575d,0x806567cb,0x196c3671,0x6e6b06e7,
        0xfed41b76,0x89d32be0,0x10da7a5a,0x67dd4acc,0xf9b9df6f,0x8ebeeff9,0x17b7be43,0x60b08ed5,
        0xd6d6a3e8,0xa1d1937e,0x38d8c2c4,0x4fdff252,0xd1bb67f1,0xa6bc5767,0x3fb506dd,0x48b2364b,
        0xd80d2bda,0xaf0a1b4c,0x36034af6,0x41047a60,0xdf60efc3,0xa867df55,0x316e8eef,0x4669be79,
        0xcb61b38c,0xbc66831a,0x256fd2a0,0x5268e236,0xcc0c7795,0xbb0b4703,0x220216b9,0x5505262f,
        0xc5ba3bbe,0xb2bd0b28,0x2bb45a92,0x5cb36a04,0xc2d7ffa7,0xb5d0cf31,0x2cd99e8b,0x5bdeae1d,
        0x9b64c2b0,0xec63f226,0x756aa39c,0x026d930a,0x9c0906a9,0xeb0e363f,0x72076785,0x05005713,
        0x95bf4a82,0xe2b87a14,0x7bb12bae,0x0cb61b38,0x92d28e9b,0xe5d5be0d,0x7cdcefb7,0x0bdbdf21,
        0x86d3d2d4,0xf1d4e242,0x68ddb3f8,0x1fda836e,0x81be16cd,0xf6b9265b,0x6fb077e1,0x18b74777,
        0x88085ae6,0xff0f6a70,0x66063bca,0x11010b5c,0x8f659eff,0xf862ae69,0x616bffd3,0x166ccf45,
        0xa00ae278,0xd70dd2ee,0x4e048354,0x3903b3c2,0xa7672661,0xd06016f7,0x4969474d,0x3e6e77db,
        0xaed16a4a,0xd9d65adc,0x40df0b66,0x37d83bf0,0xa9bcae53,0xdebb9ec5,0x47b2cf7f,0x30b5ffe9,
        0xbdbdf21c,0xcabac28a,0x53b39330,0x24b4a3a6,0xbad03605,0xcdd70693,0x54de5729,0x23d967bf,
        0xb3667a2e,0xc4614ab8,0x5d681b02,0x2a6f2b94,0xb40bbe37,0xc30c8ea1,0x5a05df1b,0x2d02ef8d,
    }
    
    ,{
        0x00000000,0x191b3141,0x32366282,0x2b2d53c3,0x646cc504,0x7d77f445,0x565aa786,0x4f4196c7,
        0xc8d98a08,0xd1c2bb49,0xfaefe88a,0xe3f4d9cb,0xacb54f0c,0xb5ae7e4d,0x9e832d8e,0x87981ccf,
        0x4ac21251,0x53d92310,0x78f470d3,0x61ef4192,0x2eaed755,0x37b5e614,0x1c98b5d7,0x05838496,
        0x821b9859,0x9b00a918,0xb02dfadb,0xa936cb9a,0xe6775d5d,0xff6c6c1c,0xd4413fdf,0xcd5a0e9e,
        0x958424a2,0x8c9f15e3,0xa7b24620,0xbea97761,0xf1e8e1a6,0xe8f3d0e7,0xc3de8324,0xdac5b265,
        0x5d5daeaa,0x44469feb,0x6f6bcc28,0x7670fd69,0x39316bae,0x202a5aef,0x0b07092c,0x121c386d,
        0xdf4636f3,0xc65d07b2,0xed705471,0xf46b6530,0xbb2af3f7,0xa231c2b6,0x891c9175,0x9007a034,
        0x179fbcfb,0x0e848dba,0x25a9de79,0x3cb2ef38,0x73f379ff,0x6ae848be,0x41c51b7d,0x58de2a3c,
        0xf0794f05,0xe9627e44,0xc24f2d87,0xdb541cc6,0x94158a01,0x8d0ebb40,0xa623e883,0xbf38d9c2,
        0x38a0c50d,0x21bbf44c,0x0a96a78f,0x138d96ce,0x5ccc0009,0x45d73148,0x6efa628b,0x77e153ca,
        0xbabb5d54,0xa3a06c15,0x888d3fd6,0x91960e97,0xded79850,0xc7cca911,0xece1fad2,0xf5facb93,
        0x7262d75c,0x6b79e61d,0x4054b5de,0x594f849f,0x160e1258,0x0f152319,0x243870da,0x3d23419b,
        0x65fd6ba7,0x7ce65ae6,0x57cb0925,0x4ed03864,0x0191aea3,0x188a9fe2,0x33a7cc21,0x2abcfd60,
        0xad24e1af,0xb43fd0ee,0x9f12832d,0x8609b26c,0xc94824ab,0xd05315ea,0xfb7e4629,0xe2657768,
        0x2f3f79f6,0x362448b7,0x1d091b74,0x04122a35,0x4b53bcf2,0x52488db3,0x7965de70,0x607eef31,
        0xe7e6f3fe,0xfefdc2bf,0xd5d0917c,0xcccba03d,0x838a36fa,0x9a9107bb,0xb1bc5478,0xa8a76539,
        0x3b83984b,0x2298a90a,0x09b5fac9,0x10aecb88,0x5fef5d4f,0x46f46c0e,0x6dd93fcd,0x74c20e8c,
        0xf35a1243,0xea412302,0xc16c70c1,0xd8774180,0x9736d747,0x8e2de606,0xa500b5c5,0xbc1b8484,
        0x71418a1a,0x685abb5b,0x4377e898,0x5a6cd9d9,0x152d4f1e,0x0c367e5f,0x271b2d9c,0x3e001cdd,
        0xb9980012,0xa0833153,0x8bae6290,0x92b553d1,0xddf4c516,0xc4eff457,0xefc2a794,0xf6d996d5,
        0xae07bce9,0xb71c8da8,0x9c31de6b,0x852aef2a,0xca6b79ed,0xd37048ac,0xf85d1b6f,0xe1462a2e,
        0x66de36e1,0x7fc507a0,0x54e85463,0x4df36522,0x02b2f3e5,0x1ba9c2a4,0x30849167,0x299fa026,
        0xe4c5aeb8,0xfdde9ff9,0xd6f3cc3a,0xcfe8fd7b,0x80a96bbc,0x99b25afd,0xb29f093e,0xab84387f,
        0x2c1c24b0,0x350715f1,0x1e2a4632,0x07317773,0x4870e1b4,0x516bd0f5,0x7a468336,0x635db277,
        0xcbfad74e,0xd2e1e60f,0xf9ccb5cc,0xe0d7848d,0xaf96124a,0xb68d230b,0x9da070c8,0x84bb4189,
        0x03235d46,0x1a386c07,0x31153fc4,0x280e0e85,0x674f9842,0x7e54a903,0x5579fac0,0x4c62cb81,
        0x8138c51f,0x9823f45e,0xb30ea79d,0xaa1596dc,0xe554001b,0xfc4f315a,0xd7626299,0xce7953d8,
        0x49e14f17,0x50fa7e56,0x7bd72d95,0x62cc1cd4,0x2d8d8a13,0x3496bb52,0x1fbbe891,0x06a0d9d0,
        0x5e7ef3ec,0x4765c2ad,0x6c48916e,0x7553a02f,0x3a1236e8,0x230907a9,0x0824546a,0x113f652b,
        0x96a779e4,0x8fbc48a5,0xa4911b66,0xbd8a2a27,0xf2cbbce0,0xebd08da1,0xc0fdde62,0xd9e6ef23,
        0x14bce1bd,0x0da7d0fc,0x268a833f,0x3f91b27e,0x70d024b9,0x69cb15f8,0x42e6463b,0x5bfd777a,
        0xdc656bb5,0xc57e5af4,0xee530937,0xf7483876,0xb809aeb1,0xa1129ff0,0x8a3fcc33,0x9324fd72,
    },
    
    {
        0x00000000,0x01c26a37,0x0384d46e,0x0246be59,0x0709a8dc,0x06cbc2eb,0x048d7cb2,0x054f1685,
        0x0e1351b8,0x0fd13b8f,0x0d9785d6,0x0c55efe1,0x091af964,0x08d89353,0x0a9e2d0a,0x0b5c473d,
        0x1c26a370,0x1de4c947,0x1fa2771e,0x1e601d29,0x1b2f0bac,0x1aed619b,0x18abdfc2,0x1969b5f5,
        0x1235f2c8,0x13f798ff,0x11b126a6,0x10734c91,0x153c5a14,0x14fe3023,0x16b88e7a,0x177ae44d,
        0x384d46e0,0x398f2cd7,0x3bc9928e,0x3a0bf8b9,0x3f44ee3c,0x3e86840b,0x3cc03a52,0x3d025065,
        0x365e1758,0x379c7d6f,0x35dac336,0x3418a901,0x3157bf84,0x3095d5b3,0x32d36bea,0x331101dd,
        0x246be590,0x25a98fa7,0x27ef31fe,0x262d5bc9,0x23624d4c,0x22a0277b,0x20e69922,0x2124f315,
        0x2a78b428,0x2bbade1f,0x29fc6046,0x283e0a71,0x2d711cf4,0x2cb376c3,0x2ef5c89a,0x2f37a2ad,
        0x709a8dc0,0x7158e7f7,0x731e59ae,0x72dc3399,0x7793251c,0x76514f2b,0x7417f172,0x75d59b45,
        0x7e89dc78,0x7f4bb64f,0x7d0d0816,0x7ccf6221,0x798074a4,0x78421e93,0x7a04a0ca,0x7bc6cafd,
        0x6cbc2eb0,0x6d7e4487,0x6f38fade,0x6efa90e9,0x6bb5866c,0x6a77ec5b,0x68315202,0x69f33835,
        0x62af7f08,0x636d153f,0x612bab66,0x60e9c151,0x65a6d7d4,0x6464bde3,0x662203ba,0x67e0698d,
        0x48d7cb20,0x4915a117,0x4b531f4e,0x4a917579,0x4fde63fc,0x4e1c09cb,0x4c5ab792,0x4d98dda5,
        0x46c49a98,0x4706f0af,0x45404ef6,0x448224c1,0x41cd3244,0x400f5873,0x4249e62a,0x438b8c1d,
        0x54f16850,0x55330267,0x5775bc3e,0x56b7d609,0x53f8c08c,0x523aaabb,0x507c14e2,0x51be7ed5,
        0x5ae239e8,0x5b2053df,0x5966ed86,0x58a487b1,0x5deb9134,0x5c29fb03,0x5e6f455a,0x5fad2f6d,
        0xe1351b80,0xe0f771b7,0xe2b1cfee,0xe373a5d9,0xe63cb35c,0xe7fed96b,0xe5b86732,0xe47a0d05,
        0xef264a38,0xeee4200f,0xeca29e56,0xed60f461,0xe82fe2e4,0xe9ed88d3,0xebab368a,0xea695cbd,
        0xfd13b8f0,0xfcd1d2c7,0xfe976c9e,0xff5506a9,0xfa1a102c,0xfbd87a1b,0xf99ec442,0xf85cae75,
        0xf300e948,0xf2c2837f,0xf0843d26,0xf1465711,0xf4094194,0xf5cb2ba3,0xf78d95fa,0xf64fffcd,
        0xd9785d60,0xd8ba3757,0xdafc890e,0xdb3ee339,0xde71f5bc,0xdfb39f8b,0xddf521d2,0xdc374be5,
        0xd76b0cd8,0xd6a966ef,0xd4efd8b6,0xd52db281,0xd062a404,0xd1a0ce33,0xd3e6706a,0xd2241a5d,
        0xc55efe10,0xc49c9427,0xc6da2a7e,0xc7184049,0xc25756cc,0xc3953cfb,0xc1d382a2,0xc011e895,
        0xcb4dafa8,0xca8fc59f,0xc8c97bc6,0xc90b11f1,0xcc440774,0xcd866d43,0xcfc0d31a,0xce02b92d,
        0x91af9640,0x906dfc77,0x922b422e,0x93e92819,0x96a63e9c,0x976454ab,0x9522eaf2,0x94e080c5,
        0x9fbcc7f8,0x9e7eadcf,0x9c381396,0x9dfa79a1,0x98b56f24,0x99770513,0x9b31bb4a,0x9af3d17d,
        0x8d893530,0x8c4b5f07,0x8e0de15e,0x8fcf8b69,0x8a809dec,0x8b42f7db,0x89044982,0x88c623b5,
        0x839a6488,0x82580ebf,0x801eb0e6,0x81dcdad1,0x8493cc54,0x8551a663,0x8717183a,0x86d5720d,
        0xa9e2d0a0,0xa820ba97,0xaa6604ce,0xaba46ef9,0xaeeb787c,0xaf29124b,0xad6fac12,0xacadc625,
        0xa7f18118,0xa633eb2f,0xa4755576,0xa5b73f41,0xa0f829c4,0xa13a43f3,0xa37cfdaa,0xa2be979d,
        0xb5c473d0,0xb40619e7,0xb640a7be,0xb782cd89,0xb2cddb0c,0xb30fb13b,0xb1490f62,0xb08b6555,
        0xbbd72268,0xba15485f,0xb853f606,0xb9919c31,0xbcde8ab4,0xbd1ce083,0xbf5a5eda,0xbe9834ed,
    },
    
    {
        0x00000000,0xb8bc6765,0xaa09c88b,0x12b5afee,0x8f629757,0x37def032,0x256b5fdc,0x9dd738b9,
        0xc5b428ef,0x7d084f8a,0x6fbde064,0xd7018701,0x4ad6bfb8,0xf26ad8dd,0xe0df7733,0x58631056,
        0x5019579f,0xe8a530fa,0xfa109f14,0x42acf871,0xdf7bc0c8,0x67c7a7ad,0x75720843,0xcdce6f26,
        0x95ad7f70,0x2d111815,0x3fa4b7fb,0x8718d09e,0x1acfe827,0xa2738f42,0xb0c620ac,0x087a47c9,
        0xa032af3e,0x188ec85b,0x0a3b67b5,0xb28700d0,0x2f503869,0x97ec5f0c,0x8559f0e2,0x3de59787,
        0x658687d1,0xdd3ae0b4,0xcf8f4f5a,0x7733283f,0xeae41086,0x525877e3,0x40edd80d,0xf851bf68,
        0xf02bf8a1,0x48979fc4,0x5a22302a,0xe29e574f,0x7f496ff6,0xc7f50893,0xd540a77d,0x6dfcc018,
        0x359fd04e,0x8d23b72b,0x9f9618c5,0x272a7fa0,0xbafd4719,0x0241207c,0x10f48f92,0xa848e8f7,
        0x9b14583d,0x23a83f58,0x311d90b6,0x89a1f7d3,0x1476cf6a,0xaccaa80f,0xbe7f07e1,0x06c36084,
        0x5ea070d2,0xe61c17b7,0xf4a9b859,0x4c15df3c,0xd1c2e785,0x697e80e0,0x7bcb2f0e,0xc377486b,
        0xcb0d0fa2,0x73b168c7,0x6104c729,0xd9b8a04c,0x446f98f5,0xfcd3ff90,0xee66507e,0x56da371b,
        0x0eb9274d,0xb6054028,0xa4b0efc6,0x1c0c88a3,0x81dbb01a,0x3967d77f,0x2bd27891,0x936e1ff4,
        0x3b26f703,0x839a9066,0x912f3f88,0x299358ed,0xb4446054,0x0cf80731,0x1e4da8df,0xa6f1cfba,
        0xfe92dfec,0x462eb889,0x549b1767,0xec277002,0x71f048bb,0xc94c2fde,0xdbf98030,0x6345e755,
        0x6b3fa09c,0xd383c7f9,0xc1366817,0x798a0f72,0xe45d37cb,0x5ce150ae,0x4e54ff40,0xf6e89825,
        0xae8b8873,0x1637ef16,0x048240f8,0xbc3e279d,0x21e91f24,0x99557841,0x8be0d7af,0x335cb0ca,
        0xed59b63b,0x55e5d15e,0x47507eb0,0xffec19d5,0x623b216c,0xda874609,0xc832e9e7,0x708e8e82,
        0x28ed9ed4,0x9051f9b1,0x82e4565f,0x3a58313a,0xa78f0983,0x1f336ee6,0x0d86c108,0xb53aa66d,
        0xbd40e1a4,0x05fc86c1,0x1749292f,0xaff54e4a,0x322276f3,0x8a9e1196,0x982bbe78,0x2097d91d,
        0x78f4c94b,0xc048ae2e,0xd2fd01c0,0x6a4166a5,0xf7965e1c,0x4f2a3979,0x5d9f9697,0xe523f1f2,
        0x4d6b1905,0xf5d77e60,0xe762d18e,0x5fdeb6eb,0xc2098e52,0x7ab5e937,0x680046d9,0xd0bc21bc,
        0x88df31ea,0x3063568f,0x22d6f961,0x9a6a9e04,0x07bda6bd,0xbf01c1d8,0xadb46e36,0x15080953,
        0x1d724e9a,0xa5ce29ff,0xb77b8611,0x0fc7e174,0x9210d9cd,0x2aacbea8,0x38191146,0x80a57623,
        0xd8c66675,0x607a0110,0x72cfaefe,0xca73c99b,0x57a4f122,0xef189647,0xfdad39a9,0x45115ecc,
        0x764dee06,0xcef18963,0xdc44268d,0x64f841e8,0xf92f7951,0x41931e34,0x5326b1da,0xeb9ad6bf,
        0xb3f9c6e9,0x0b45a18c,0x19f00e62,0xa14c6907,0x3c9b51be,0x842736db,0x96929935,0x2e2efe50,
        0x2654b999,0x9ee8defc,0x8c5d7112,0x34e11677,0xa9362ece,0x118a49ab,0x033fe645,0xbb838120,
        0xe3e09176,0x5b5cf613,0x49e959fd,0xf1553e98,0x6c820621,0xd43e6144,0xc68bceaa,0x7e37a9cf,
        0xd67f4138,0x6ec3265d,0x7c7689b3,0xc4caeed6,0x591dd66f,0xe1a1b10a,0xf3141ee4,0x4ba87981,
        0x13cb69d7,0xab770eb2,0xb9c2a15c,0x017ec639,0x9ca9fe80,0x241599e5,0x36a0360b,0x8e1c516e,
        0x866616a7,0x3eda71c2,0x2c6fde2c,0x94d3b949,0x090481f0,0xb1b8e695,0xa30d497b,0x1bb12e1e,
        0x43d23e48,0xfb6e592d,0xe9dbf6c3,0x516791a6,0xccb0a91f,0x740cce7a,0x66b96194,0xde0506f1,
    }
    ,{
        0x00000000,0x3d6029b0,0x7ac05360,0x47a07ad0,0xf580a6c0,0xc8e08f70,0x8f40f5a0,0xb220dc10,
        0x30704bc1,0x0d106271,0x4ab018a1,0x77d03111,0xc5f0ed01,0xf890c4b1,0xbf30be61,0x825097d1,
        0x60e09782,0x5d80be32,0x1a20c4e2,0x2740ed52,0x95603142,0xa80018f2,0xefa06222,0xd2c04b92,
        0x5090dc43,0x6df0f5f3,0x2a508f23,0x1730a693,0xa5107a83,0x98705333,0xdfd029e3,0xe2b00053,
        0xc1c12f04,0xfca106b4,0xbb017c64,0x866155d4,0x344189c4,0x0921a074,0x4e81daa4,0x73e1f314,
        0xf1b164c5,0xccd14d75,0x8b7137a5,0xb6111e15,0x0431c205,0x3951ebb5,0x7ef19165,0x4391b8d5,
        0xa121b886,0x9c419136,0xdbe1ebe6,0xe681c256,0x54a11e46,0x69c137f6,0x2e614d26,0x13016496,
        0x9151f347,0xac31daf7,0xeb91a027,0xd6f18997,0x64d15587,0x59b17c37,0x1e1106e7,0x23712f57,
        0x58f35849,0x659371f9,0x22330b29,0x1f532299,0xad73fe89,0x9013d739,0xd7b3ade9,0xead38459,
        0x68831388,0x55e33a38,0x124340e8,0x2f236958,0x9d03b548,0xa0639cf8,0xe7c3e628,0xdaa3cf98,
        0x3813cfcb,0x0573e67b,0x42d39cab,0x7fb3b51b,0xcd93690b,0xf0f340bb,0xb7533a6b,0x8a3313db,
        0x0863840a,0x3503adba,0x72a3d76a,0x4fc3feda,0xfde322ca,0xc0830b7a,0x872371aa,0xba43581a,
        0x9932774d,0xa4525efd,0xe3f2242d,0xde920d9d,0x6cb2d18d,0x51d2f83d,0x167282ed,0x2b12ab5d,
        0xa9423c8c,0x9422153c,0xd3826fec,0xeee2465c,0x5cc29a4c,0x61a2b3fc,0x2602c92c,0x1b62e09c,
        0xf9d2e0cf,0xc4b2c97f,0x8312b3af,0xbe729a1f,0x0c52460f,0x31326fbf,0x7692156f,0x4bf23cdf,
        0xc9a2ab0e,0xf4c282be,0xb362f86e,0x8e02d1de,0x3c220dce,0x0142247e,0x46e25eae,0x7b82771e,
        0xb1e6b092,0x8c869922,0xcb26e3f2,0xf646ca42,0x44661652,0x79063fe2,0x3ea64532,0x03c66c82,
        0x8196fb53,0xbcf6d2e3,0xfb56a833,0xc6368183,0x74165d93,0x49767423,0x0ed60ef3,0x33b62743,
        0xd1062710,0xec660ea0,0xabc67470,0x96a65dc0,0x248681d0,0x19e6a860,0x5e46d2b0,0x6326fb00,
        0xe1766cd1,0xdc164561,0x9bb63fb1,0xa6d61601,0x14f6ca11,0x2996e3a1,0x6e369971,0x5356b0c1,
        0x70279f96,0x4d47b626,0x0ae7ccf6,0x3787e546,0x85a73956,0xb8c710e6,0xff676a36,0xc2074386,
        0x4057d457,0x7d37fde7,0x3a978737,0x07f7ae87,0xb5d77297,0x88b75b27,0xcf1721f7,0xf2770847,
        0x10c70814,0x2da721a4,0x6a075b74,0x576772c4,0xe547aed4,0xd8278764,0x9f87fdb4,0xa2e7d404,
        0x20b743d5,0x1dd76a65,0x5a7710b5,0x67173905,0xd537e515,0xe857cca5,0xaff7b675,0x92979fc5,
        0xe915e8db,0xd475c16b,0x93d5bbbb,0xaeb5920b,0x1c954e1b,0x21f567ab,0x66551d7b,0x5b3534cb,
        0xd965a31a,0xe4058aaa,0xa3a5f07a,0x9ec5d9ca,0x2ce505da,0x11852c6a,0x562556ba,0x6b457f0a,
        0x89f57f59,0xb49556e9,0xf3352c39,0xce550589,0x7c75d999,0x4115f029,0x06b58af9,0x3bd5a349,
        0xb9853498,0x84e51d28,0xc34567f8,0xfe254e48,0x4c059258,0x7165bbe8,0x36c5c138,0x0ba5e888,
        0x28d4c7df,0x15b4ee6f,0x521494bf,0x6f74bd0f,0xdd54611f,0xe03448af,0xa794327f,0x9af41bcf,
        0x18a48c1e,0x25c4a5ae,0x6264df7e,0x5f04f6ce,0xed242ade,0xd044036e,0x97e479be,0xaa84500e,
        0x4834505d,0x755479ed,0x32f4033d,0x0f942a8d,0xbdb4f69d,0x80d4df2d,0xc774a5fd,0xfa148c4d,
        0x78441b9c,0x4524322c,0x028448fc,0x3fe4614c,0x8dc4bd5c,0xb0a494ec,0xf704ee3c,0xca64c78c,
    },
    
    {
        0x00000000,0xcb5cd3a5,0x4dc8a10b,0x869472ae,0x9b914216,0x50cd91b3,0xd659e31d,0x1d0530b8,
        0xec53826d,0x270f51c8,0xa19b2366,0x6ac7f0c3,0x77c2c07b,0xbc9e13de,0x3a0a6170,0xf156b2d5,
        0x03d6029b,0xc88ad13e,0x4e1ea390,0x85427035,0x9847408d,0x531b9328,0xd58fe186,0x1ed33223,
        0xef8580f6,0x24d95353,0xa24d21fd,0x6911f258,0x7414c2e0,0xbf481145,0x39dc63eb,0xf280b04e,
        0x07ac0536,0xccf0d693,0x4a64a43d,0x81387798,0x9c3d4720,0x57619485,0xd1f5e62b,0x1aa9358e,
        0xebff875b,0x20a354fe,0xa6372650,0x6d6bf5f5,0x706ec54d,0xbb3216e8,0x3da66446,0xf6fab7e3,
        0x047a07ad,0xcf26d408,0x49b2a6a6,0x82ee7503,0x9feb45bb,0x54b7961e,0xd223e4b0,0x197f3715,
        0xe82985c0,0x23755665,0xa5e124cb,0x6ebdf76e,0x73b8c7d6,0xb8e41473,0x3e7066dd,0xf52cb578,
        0x0f580a6c,0xc404d9c9,0x4290ab67,0x89cc78c2,0x94c9487a,0x5f959bdf,0xd901e971,0x125d3ad4,
        0xe30b8801,0x28575ba4,0xaec3290a,0x659ffaaf,0x789aca17,0xb3c619b2,0x35526b1c,0xfe0eb8b9,
        0x0c8e08f7,0xc7d2db52,0x4146a9fc,0x8a1a7a59,0x971f4ae1,0x5c439944,0xdad7ebea,0x118b384f,
        0xe0dd8a9a,0x2b81593f,0xad152b91,0x6649f834,0x7b4cc88c,0xb0101b29,0x36846987,0xfdd8ba22,
        0x08f40f5a,0xc3a8dcff,0x453cae51,0x8e607df4,0x93654d4c,0x58399ee9,0xdeadec47,0x15f13fe2,
        0xe4a78d37,0x2ffb5e92,0xa96f2c3c,0x6233ff99,0x7f36cf21,0xb46a1c84,0x32fe6e2a,0xf9a2bd8f,
        0x0b220dc1,0xc07ede64,0x46eaacca,0x8db67f6f,0x90b34fd7,0x5bef9c72,0xdd7beedc,0x16273d79,
        0xe7718fac,0x2c2d5c09,0xaab92ea7,0x61e5fd02,0x7ce0cdba,0xb7bc1e1f,0x31286cb1,0xfa74bf14,
        0x1eb014d8,0xd5ecc77d,0x5378b5d3,0x98246676,0x852156ce,0x4e7d856b,0xc8e9f7c5,0x03b52460,
        0xf2e396b5,0x39bf4510,0xbf2b37be,0x7477e41b,0x6972d4a3,0xa22e0706,0x24ba75a8,0xefe6a60d,
        0x1d661643,0xd63ac5e6,0x50aeb748,0x9bf264ed,0x86f75455,0x4dab87f0,0xcb3ff55e,0x006326fb,
        0xf135942e,0x3a69478b,0xbcfd3525,0x77a1e680,0x6aa4d638,0xa1f8059d,0x276c7733,0xec30a496,
        0x191c11ee,0xd240c24b,0x54d4b0e5,0x9f886340,0x828d53f8,0x49d1805d,0xcf45f2f3,0x04192156,
        0xf54f9383,0x3e134026,0xb8873288,0x73dbe12d,0x6eded195,0xa5820230,0x2316709e,0xe84aa33b,
        0x1aca1375,0xd196c0d0,0x5702b27e,0x9c5e61db,0x815b5163,0x4a0782c6,0xcc93f068,0x07cf23cd,
        0xf6999118,0x3dc542bd,0xbb513013,0x700de3b6,0x6d08d30e,0xa65400ab,0x20c07205,0xeb9ca1a0,
        0x11e81eb4,0xdab4cd11,0x5c20bfbf,0x977c6c1a,0x8a795ca2,0x41258f07,0xc7b1fda9,0x0ced2e0c,
        0xfdbb9cd9,0x36e74f7c,0xb0733dd2,0x7b2fee77,0x662adecf,0xad760d6a,0x2be27fc4,0xe0beac61,
        0x123e1c2f,0xd962cf8a,0x5ff6bd24,0x94aa6e81,0x89af5e39,0x42f38d9c,0xc467ff32,0x0f3b2c97,
        0xfe6d9e42,0x35314de7,0xb3a53f49,0x78f9ecec,0x65fcdc54,0xaea00ff1,0x28347d5f,0xe368aefa,
        0x16441b82,0xdd18c827,0x5b8cba89,0x90d0692c,0x8dd55994,0x46898a31,0xc01df89f,0x0b412b3a,
        0xfa1799ef,0x314b4a4a,0xb7df38e4,0x7c83eb41,0x6186dbf9,0xaada085c,0x2c4e7af2,0xe712a957,
        0x15921919,0xdececabc,0x585ab812,0x93066bb7,0x8e035b0f,0x455f88aa,0xc3cbfa04,0x089729a1,
        0xf9c19b74,0x329d48d1,0xb4093a7f,0x7f55e9da,0x6250d962,0xa90c0ac7,0x2f987869,0xe4c4abcc,
    },
    
    {
        0x00000000,0xa6770bb4,0x979f1129,0x31e81a9d,0xf44f2413,0x52382fa7,0x63d0353a,0xc5a73e8e,
        0x33ef4e67,0x959845d3,0xa4705f4e,0x020754fa,0xc7a06a74,0x61d761c0,0x503f7b5d,0xf64870e9,
        0x67de9cce,0xc1a9977a,0xf0418de7,0x56368653,0x9391b8dd,0x35e6b369,0x040ea9f4,0xa279a240,
        0x5431d2a9,0xf246d91d,0xc3aec380,0x65d9c834,0xa07ef6ba,0x0609fd0e,0x37e1e793,0x9196ec27,
        0xcfbd399c,0x69ca3228,0x582228b5,0xfe552301,0x3bf21d8f,0x9d85163b,0xac6d0ca6,0x0a1a0712,
        0xfc5277fb,0x5a257c4f,0x6bcd66d2,0xcdba6d66,0x081d53e8,0xae6a585c,0x9f8242c1,0x39f54975,
        0xa863a552,0x0e14aee6,0x3ffcb47b,0x998bbfcf,0x5c2c8141,0xfa5b8af5,0xcbb39068,0x6dc49bdc,
        0x9b8ceb35,0x3dfbe081,0x0c13fa1c,0xaa64f1a8,0x6fc3cf26,0xc9b4c492,0xf85cde0f,0x5e2bd5bb,
        0x440b7579,0xe27c7ecd,0xd3946450,0x75e36fe4,0xb044516a,0x16335ade,0x27db4043,0x81ac4bf7,
        0x77e43b1e,0xd19330aa,0xe07b2a37,0x460c2183,0x83ab1f0d,0x25dc14b9,0x14340e24,0xb2430590,
        0x23d5e9b7,0x85a2e203,0xb44af89e,0x123df32a,0xd79acda4,0x71edc610,0x4005dc8d,0xe672d739,
        0x103aa7d0,0xb64dac64,0x87a5b6f9,0x21d2bd4d,0xe47583c3,0x42028877,0x73ea92ea,0xd59d995e,
        0x8bb64ce5,0x2dc14751,0x1c295dcc,0xba5e5678,0x7ff968f6,0xd98e6342,0xe86679df,0x4e11726b,
        0xb8590282,0x1e2e0936,0x2fc613ab,0x89b1181f,0x4c162691,0xea612d25,0xdb8937b8,0x7dfe3c0c,
        0xec68d02b,0x4a1fdb9f,0x7bf7c102,0xdd80cab6,0x1827f438,0xbe50ff8c,0x8fb8e511,0x29cfeea5,
        0xdf879e4c,0x79f095f8,0x48188f65,0xee6f84d1,0x2bc8ba5f,0x8dbfb1eb,0xbc57ab76,0x1a20a0c2,
        0x8816eaf2,0x2e61e146,0x1f89fbdb,0xb9fef06f,0x7c59cee1,0xda2ec555,0xebc6dfc8,0x4db1d47c,
        0xbbf9a495,0x1d8eaf21,0x2c66b5bc,0x8a11be08,0x4fb68086,0xe9c18b32,0xd82991af,0x7e5e9a1b,
        0xefc8763c,0x49bf7d88,0x78576715,0xde206ca1,0x1b87522f,0xbdf0599b,0x8c184306,0x2a6f48b2,
        0xdc27385b,0x7a5033ef,0x4bb82972,0xedcf22c6,0x28681c48,0x8e1f17fc,0xbff70d61,0x198006d5,
        0x47abd36e,0xe1dcd8da,0xd034c247,0x7643c9f3,0xb3e4f77d,0x1593fcc9,0x247be654,0x820cede0,
        0x74449d09,0xd23396bd,0xe3db8c20,0x45ac8794,0x800bb91a,0x267cb2ae,0x1794a833,0xb1e3a387,
        0x20754fa0,0x86024414,0xb7ea5e89,0x119d553d,0xd43a6bb3,0x724d6007,0x43a57a9a,0xe5d2712e,
        0x139a01c7,0xb5ed0a73,0x840510ee,0x22721b5a,0xe7d525d4,0x41a22e60,0x704a34fd,0xd63d3f49,
        0xcc1d9f8b,0x6a6a943f,0x5b828ea2,0xfdf58516,0x3852bb98,0x9e25b02c,0xafcdaab1,0x09baa105,
        0xfff2d1ec,0x5985da58,0x686dc0c5,0xce1acb71,0x0bbdf5ff,0xadcafe4b,0x9c22e4d6,0x3a55ef62,
        0xabc30345,0x0db408f1,0x3c5c126c,0x9a2b19d8,0x5f8c2756,0xf9fb2ce2,0xc813367f,0x6e643dcb,
        0x982c4d22,0x3e5b4696,0x0fb35c0b,0xa9c457bf,0x6c636931,0xca146285,0xfbfc7818,0x5d8b73ac,
        0x03a0a617,0xa5d7ada3,0x943fb73e,0x3248bc8a,0xf7ef8204,0x519889b0,0x6070932d,0xc6079899,
        0x304fe870,0x9638e3c4,0xa7d0f959,0x01a7f2ed,0xc400cc63,0x6277c7d7,0x539fdd4a,0xf5e8d6fe,
        0x647e3ad9,0xc209316d,0xf3e12bf0,0x55962044,0x90311eca,0x3646157e,0x07ae0fe3,0xa1d90457,
        0x579174be,0xf1e67f0a,0xc00e6597,0x66796e23,0xa3de50ad,0x05a95b19,0x34414184,0x92364a30,
    },
    
    {
        0x00000000,0xccaa009e,0x4225077d,0x8e8f07e3,0x844a0efa,0x48e00e64,0xc66f0987,0x0ac50919,
        0xd3e51bb5,0x1f4f1b2b,0x91c01cc8,0x5d6a1c56,0x57af154f,0x9b0515d1,0x158a1232,0xd92012ac,
        0x7cbb312b,0xb01131b5,0x3e9e3656,0xf23436c8,0xf8f13fd1,0x345b3f4f,0xbad438ac,0x767e3832,
        0xaf5e2a9e,0x63f42a00,0xed7b2de3,0x21d12d7d,0x2b142464,0xe7be24fa,0x69312319,0xa59b2387,
        0xf9766256,0x35dc62c8,0xbb53652b,0x77f965b5,0x7d3c6cac,0xb1966c32,0x3f196bd1,0xf3b36b4f,
        0x2a9379e3,0xe639797d,0x68b67e9e,0xa41c7e00,0xaed97719,0x62737787,0xecfc7064,0x205670fa,
        0x85cd537d,0x496753e3,0xc7e85400,0x0b42549e,0x01875d87,0xcd2d5d19,0x43a25afa,0x8f085a64,
        0x562848c8,0x9a824856,0x140d4fb5,0xd8a74f2b,0xd2624632,0x1ec846ac,0x9047414f,0x5ced41d1,
        0x299dc2ed,0xe537c273,0x6bb8c590,0xa712c50e,0xadd7cc17,0x617dcc89,0xeff2cb6a,0x2358cbf4,
        0xfa78d958,0x36d2d9c6,0xb85dde25,0x74f7debb,0x7e32d7a2,0xb298d73c,0x3c17d0df,0xf0bdd041,
        0x5526f3c6,0x998cf358,0x1703f4bb,0xdba9f425,0xd16cfd3c,0x1dc6fda2,0x9349fa41,0x5fe3fadf,
        0x86c3e873,0x4a69e8ed,0xc4e6ef0e,0x084cef90,0x0289e689,0xce23e617,0x40ace1f4,0x8c06e16a,
        0xd0eba0bb,0x1c41a025,0x92cea7c6,0x5e64a758,0x54a1ae41,0x980baedf,0x1684a93c,0xda2ea9a2,
        0x030ebb0e,0xcfa4bb90,0x412bbc73,0x8d81bced,0x8744b5f4,0x4beeb56a,0xc561b289,0x09cbb217,
        0xac509190,0x60fa910e,0xee7596ed,0x22df9673,0x281a9f6a,0xe4b09ff4,0x6a3f9817,0xa6959889,
        0x7fb58a25,0xb31f8abb,0x3d908d58,0xf13a8dc6,0xfbff84df,0x37558441,0xb9da83a2,0x7570833c,
        0x533b85da,0x9f918544,0x111e82a7,0xddb48239,0xd7718b20,0x1bdb8bbe,0x95548c5d,0x59fe8cc3,
        0x80de9e6f,0x4c749ef1,0xc2fb9912,0x0e51998c,0x04949095,0xc83e900b,0x46b197e8,0x8a1b9776,
        0x2f80b4f1,0xe32ab46f,0x6da5b38c,0xa10fb312,0xabcaba0b,0x6760ba95,0xe9efbd76,0x2545bde8,
        0xfc65af44,0x30cfafda,0xbe40a839,0x72eaa8a7,0x782fa1be,0xb485a120,0x3a0aa6c3,0xf6a0a65d,
        0xaa4de78c,0x66e7e712,0xe868e0f1,0x24c2e06f,0x2e07e976,0xe2ade9e8,0x6c22ee0b,0xa088ee95,
        0x79a8fc39,0xb502fca7,0x3b8dfb44,0xf727fbda,0xfde2f2c3,0x3148f25d,0xbfc7f5be,0x736df520,
        0xd6f6d6a7,0x1a5cd639,0x94d3d1da,0x5879d144,0x52bcd85d,0x9e16d8c3,0x1099df20,0xdc33dfbe,
        0x0513cd12,0xc9b9cd8c,0x4736ca6f,0x8b9ccaf1,0x8159c3e8,0x4df3c376,0xc37cc495,0x0fd6c40b,
        0x7aa64737,0xb60c47a9,0x3883404a,0xf42940d4,0xfeec49cd,0x32464953,0xbcc94eb0,0x70634e2e,
        0xa9435c82,0x65e95c1c,0xeb665bff,0x27cc5b61,0x2d095278,0xe1a352e6,0x6f2c5505,0xa386559b,
        0x061d761c,0xcab77682,0x44387161,0x889271ff,0x825778e6,0x4efd7878,0xc0727f9b,0x0cd87f05,
        0xd5f86da9,0x19526d37,0x97dd6ad4,0x5b776a4a,0x51b26353,0x9d1863cd,0x1397642e,0xdf3d64b0,
        0x83d02561,0x4f7a25ff,0xc1f5221c,0x0d5f2282,0x079a2b9b,0xcb302b05,0x45bf2ce6,0x89152c78,
        0x50353ed4,0x9c9f3e4a,0x121039a9,0xdeba3937,0xd47f302e,0x18d530b0,0x965a3753,0x5af037cd,
        0xff6b144a,0x33c114d4,0xbd4e1337,0x71e413a9,0x7b211ab0,0xb78b1a2e,0x39041dcd,0xf5ae1d53,
        0x2c8e0fff,0xe0240f61,0x6eab0882,0xa201081c,0xa8c40105,0x646e019b,0xeae10678,0x264b06e6,
    }
    ,{
        0x00000000,0x177b1443,0x2ef62886,0x398d3cc5,0x5dec510c,0x4a97454f,0x731a798a,0x64616dc9,
        0xbbd8a218,0xaca3b65b,0x952e8a9e,0x82559edd,0xe634f314,0xf14fe757,0xc8c2db92,0xdfb9cfd1,
        0xacc04271,0xbbbb5632,0x82366af7,0x954d7eb4,0xf12c137d,0xe657073e,0xdfda3bfb,0xc8a12fb8,
        0x1718e069,0x0063f42a,0x39eec8ef,0x2e95dcac,0x4af4b165,0x5d8fa526,0x640299e3,0x73798da0,
        0x82f182a3,0x958a96e0,0xac07aa25,0xbb7cbe66,0xdf1dd3af,0xc866c7ec,0xf1ebfb29,0xe690ef6a,
        0x392920bb,0x2e5234f8,0x17df083d,0x00a41c7e,0x64c571b7,0x73be65f4,0x4a335931,0x5d484d72,
        0x2e31c0d2,0x394ad491,0x00c7e854,0x17bcfc17,0x73dd91de,0x64a6859d,0x5d2bb958,0x4a50ad1b,
        0x95e962ca,0x82927689,0xbb1f4a4c,0xac645e0f,0xc80533c6,0xdf7e2785,0xe6f31b40,0xf1880f03,
        0xde920307,0xc9e91744,0xf0642b81,0xe71f3fc2,0x837e520b,0x94054648,0xad887a8d,0xbaf36ece,
        0x654aa11f,0x7231b55c,0x4bbc8999,0x5cc79dda,0x38a6f013,0x2fdde450,0x1650d895,0x012bccd6,
        0x72524176,0x65295535,0x5ca469f0,0x4bdf7db3,0x2fbe107a,0x38c50439,0x014838fc,0x16332cbf,
        0xc98ae36e,0xdef1f72d,0xe77ccbe8,0xf007dfab,0x9466b262,0x831da621,0xba909ae4,0xadeb8ea7,
        0x5c6381a4,0x4b1895e7,0x7295a922,0x65eebd61,0x018fd0a8,0x16f4c4eb,0x2f79f82e,0x3802ec6d,
        0xe7bb23bc,0xf0c037ff,0xc94d0b3a,0xde361f79,0xba5772b0,0xad2c66f3,0x94a15a36,0x83da4e75,
        0xf0a3c3d5,0xe7d8d796,0xde55eb53,0xc92eff10,0xad4f92d9,0xba34869a,0x83b9ba5f,0x94c2ae1c,
        0x4b7b61cd,0x5c00758e,0x658d494b,0x72f65d08,0x169730c1,0x01ec2482,0x38611847,0x2f1a0c04,
        0x6655004f,0x712e140c,0x48a328c9,0x5fd83c8a,0x3bb95143,0x2cc24500,0x154f79c5,0x02346d86,
        0xdd8da257,0xcaf6b614,0xf37b8ad1,0xe4009e92,0x8061f35b,0x971ae718,0xae97dbdd,0xb9eccf9e,
        0xca95423e,0xddee567d,0xe4636ab8,0xf3187efb,0x97791332,0x80020771,0xb98f3bb4,0xaef42ff7,
        0x714de026,0x6636f465,0x5fbbc8a0,0x48c0dce3,0x2ca1b12a,0x3bdaa569,0x025799ac,0x152c8def,
        0xe4a482ec,0xf3df96af,0xca52aa6a,0xdd29be29,0xb948d3e0,0xae33c7a3,0x97befb66,0x80c5ef25,
        0x5f7c20f4,0x480734b7,0x718a0872,0x66f11c31,0x029071f8,0x15eb65bb,0x2c66597e,0x3b1d4d3d,
        0x4864c09d,0x5f1fd4de,0x6692e81b,0x71e9fc58,0x15889191,0x02f385d2,0x3b7eb917,0x2c05ad54,
        0xf3bc6285,0xe4c776c6,0xdd4a4a03,0xca315e40,0xae503389,0xb92b27ca,0x80a61b0f,0x97dd0f4c,
        0xb8c70348,0xafbc170b,0x96312bce,0x814a3f8d,0xe52b5244,0xf2504607,0xcbdd7ac2,0xdca66e81,
        0x031fa150,0x1464b513,0x2de989d6,0x3a929d95,0x5ef3f05c,0x4988e41f,0x7005d8da,0x677ecc99,
        0x14074139,0x037c557a,0x3af169bf,0x2d8a7dfc,0x49eb1035,0x5e900476,0x671d38b3,0x70662cf0,
        0xafdfe321,0xb8a4f762,0x8129cba7,0x9652dfe4,0xf233b22d,0xe548a66e,0xdcc59aab,0xcbbe8ee8,
        0x3a3681eb,0x2d4d95a8,0x14c0a96d,0x03bbbd2e,0x67dad0e7,0x70a1c4a4,0x492cf861,0x5e57ec22,
        0x81ee23f3,0x969537b0,0xaf180b75,0xb8631f36,0xdc0272ff,0xcb7966bc,0xf2f45a79,0xe58f4e3a,
        0x96f6c39a,0x818dd7d9,0xb800eb1c,0xaf7bff5f,0xcb1a9296,0xdc6186d5,0xe5ecba10,0xf297ae53,
        0x2d2e6182,0x3a5575c1,0x03d84904,0x14a35d47,0x70c2308e,0x67b924cd,0x5e341808,0x494f0c4b,
    },
    {
        0x00000000,0xefc26b3e,0x04f5d03d,0xeb37bb03,0x09eba07a,0xe629cb44,0x0d1e7047,0xe2dc1b79,
        0x13d740f4,0xfc152bca,0x172290c9,0xf8e0fbf7,0x1a3ce08e,0xf5fe8bb0,0x1ec930b3,0xf10b5b8d,
        0x27ae81e8,0xc86cead6,0x235b51d5,0xcc993aeb,0x2e452192,0xc1874aac,0x2ab0f1af,0xc5729a91,
        0x3479c11c,0xdbbbaa22,0x308c1121,0xdf4e7a1f,0x3d926166,0xd2500a58,0x3967b15b,0xd6a5da65,
        0x4f5d03d0,0xa09f68ee,0x4ba8d3ed,0xa46ab8d3,0x46b6a3aa,0xa974c894,0x42437397,0xad8118a9,
        0x5c8a4324,0xb348281a,0x587f9319,0xb7bdf827,0x5561e35e,0xbaa38860,0x51943363,0xbe56585d,
        0x68f38238,0x8731e906,0x6c065205,0x83c4393b,0x61182242,0x8eda497c,0x65edf27f,0x8a2f9941,
        0x7b24c2cc,0x94e6a9f2,0x7fd112f1,0x901379cf,0x72cf62b6,0x9d0d0988,0x763ab28b,0x99f8d9b5,
        0x9eba07a0,0x71786c9e,0x9a4fd79d,0x758dbca3,0x9751a7da,0x7893cce4,0x93a477e7,0x7c661cd9,
        0x8d6d4754,0x62af2c6a,0x89989769,0x665afc57,0x8486e72e,0x6b448c10,0x80733713,0x6fb15c2d,
        0xb9148648,0x56d6ed76,0xbde15675,0x52233d4b,0xb0ff2632,0x5f3d4d0c,0xb40af60f,0x5bc89d31,
        0xaac3c6bc,0x4501ad82,0xae361681,0x41f47dbf,0xa32866c6,0x4cea0df8,0xa7ddb6fb,0x481fddc5,
        0xd1e70470,0x3e256f4e,0xd512d44d,0x3ad0bf73,0xd80ca40a,0x37cecf34,0xdcf97437,0x333b1f09,
        0xc2304484,0x2df22fba,0xc6c594b9,0x2907ff87,0xcbdbe4fe,0x24198fc0,0xcf2e34c3,0x20ec5ffd,
        0xf6498598,0x198beea6,0xf2bc55a5,0x1d7e3e9b,0xffa225e2,0x10604edc,0xfb57f5df,0x14959ee1,
        0xe59ec56c,0x0a5cae52,0xe16b1551,0x0ea97e6f,0xec756516,0x03b70e28,0xe880b52b,0x0742de15,
        0xe6050901,0x09c7623f,0xe2f0d93c,0x0d32b202,0xefeea97b,0x002cc245,0xeb1b7946,0x04d91278,
        0xf5d249f5,0x1a1022cb,0xf12799c8,0x1ee5f2f6,0xfc39e98f,0x13fb82b1,0xf8cc39b2,0x170e528c,
        0xc1ab88e9,0x2e69e3d7,0xc55e58d4,0x2a9c33ea,0xc8402893,0x278243ad,0xccb5f8ae,0x23779390,
        0xd27cc81d,0x3dbea323,0xd6891820,0x394b731e,0xdb976867,0x34550359,0xdf62b85a,0x30a0d364,
        0xa9580ad1,0x469a61ef,0xadaddaec,0x426fb1d2,0xa0b3aaab,0x4f71c195,0xa4467a96,0x4b8411a8,
        0xba8f4a25,0x554d211b,0xbe7a9a18,0x51b8f126,0xb364ea5f,0x5ca68161,0xb7913a62,0x5853515c,
        0x8ef68b39,0x6134e007,0x8a035b04,0x65c1303a,0x871d2b43,0x68df407d,0x83e8fb7e,0x6c2a9040,
        0x9d21cbcd,0x72e3a0f3,0x99d41bf0,0x761670ce,0x94ca6bb7,0x7b080089,0x903fbb8a,0x7ffdd0b4,
        0x78bf0ea1,0x977d659f,0x7c4ade9c,0x9388b5a2,0x7154aedb,0x9e96c5e5,0x75a17ee6,0x9a6315d8,
        0x6b684e55,0x84aa256b,0x6f9d9e68,0x805ff556,0x6283ee2f,0x8d418511,0x66763e12,0x89b4552c,
        0x5f118f49,0xb0d3e477,0x5be45f74,0xb426344a,0x56fa2f33,0xb938440d,0x520fff0e,0xbdcd9430,
        0x4cc6cfbd,0xa304a483,0x48331f80,0xa7f174be,0x452d6fc7,0xaaef04f9,0x41d8bffa,0xae1ad4c4,
        0x37e20d71,0xd820664f,0x3317dd4c,0xdcd5b672,0x3e09ad0b,0xd1cbc635,0x3afc7d36,0xd53e1608,
        0x24354d85,0xcbf726bb,0x20c09db8,0xcf02f686,0x2ddeedff,0xc21c86c1,0x292b3dc2,0xc6e956fc,
        0x104c8c99,0xff8ee7a7,0x14b95ca4,0xfb7b379a,0x19a72ce3,0xf66547dd,0x1d52fcde,0xf29097e0,
        0x039bcc6d,0xec59a753,0x076e1c50,0xe8ac776e,0x0a706c17,0xe5b20729,0x0e85bc2a,0xe147d714,
    },
    {
        0x00000000,0xc18edfc0,0x586cb9c1,0x99e26601,0xb0d97382,0x7157ac42,0xe8b5ca43,0x293b1583,
        0xbac3e145,0x7b4d3e85,0xe2af5884,0x23218744,0x0a1a92c7,0xcb944d07,0x52762b06,0x93f8f4c6,
        0xaef6c4cb,0x6f781b0b,0xf69a7d0a,0x3714a2ca,0x1e2fb749,0xdfa16889,0x46430e88,0x87cdd148,
        0x1435258e,0xd5bbfa4e,0x4c599c4f,0x8dd7438f,0xa4ec560c,0x656289cc,0xfc80efcd,0x3d0e300d,
        0x869c8fd7,0x47125017,0xdef03616,0x1f7ee9d6,0x3645fc55,0xf7cb2395,0x6e294594,0xafa79a54,
        0x3c5f6e92,0xfdd1b152,0x6433d753,0xa5bd0893,0x8c861d10,0x4d08c2d0,0xd4eaa4d1,0x15647b11,
        0x286a4b1c,0xe9e494dc,0x7006f2dd,0xb1882d1d,0x98b3389e,0x593de75e,0xc0df815f,0x01515e9f,
        0x92a9aa59,0x53277599,0xcac51398,0x0b4bcc58,0x2270d9db,0xe3fe061b,0x7a1c601a,0xbb92bfda,
        0xd64819ef,0x17c6c62f,0x8e24a02e,0x4faa7fee,0x66916a6d,0xa71fb5ad,0x3efdd3ac,0xff730c6c,
        0x6c8bf8aa,0xad05276a,0x34e7416b,0xf5699eab,0xdc528b28,0x1ddc54e8,0x843e32e9,0x45b0ed29,
        0x78bedd24,0xb93002e4,0x20d264e5,0xe15cbb25,0xc867aea6,0x09e97166,0x900b1767,0x5185c8a7,
        0xc27d3c61,0x03f3e3a1,0x9a1185a0,0x5b9f5a60,0x72a44fe3,0xb32a9023,0x2ac8f622,0xeb4629e2,
        0x50d49638,0x915a49f8,0x08b82ff9,0xc936f039,0xe00de5ba,0x21833a7a,0xb8615c7b,0x79ef83bb,
        0xea17777d,0x2b99a8bd,0xb27bcebc,0x73f5117c,0x5ace04ff,0x9b40db3f,0x02a2bd3e,0xc32c62fe,
        0xfe2252f3,0x3fac8d33,0xa64eeb32,0x67c034f2,0x4efb2171,0x8f75feb1,0x169798b0,0xd7194770,
        0x44e1b3b6,0x856f6c76,0x1c8d0a77,0xdd03d5b7,0xf438c034,0x35b61ff4,0xac5479f5,0x6ddaa635,
        0x77e1359f,0xb66fea5f,0x2f8d8c5e,0xee03539e,0xc738461d,0x06b699dd,0x9f54ffdc,0x5eda201c,
        0xcd22d4da,0x0cac0b1a,0x954e6d1b,0x54c0b2db,0x7dfba758,0xbc757898,0x25971e99,0xe419c159,
        0xd917f154,0x18992e94,0x817b4895,0x40f59755,0x69ce82d6,0xa8405d16,0x31a23b17,0xf02ce4d7,
        0x63d41011,0xa25acfd1,0x3bb8a9d0,0xfa367610,0xd30d6393,0x1283bc53,0x8b61da52,0x4aef0592,
        0xf17dba48,0x30f36588,0xa9110389,0x689fdc49,0x41a4c9ca,0x802a160a,0x19c8700b,0xd846afcb,
        0x4bbe5b0d,0x8a3084cd,0x13d2e2cc,0xd25c3d0c,0xfb67288f,0x3ae9f74f,0xa30b914e,0x62854e8e,
        0x5f8b7e83,0x9e05a143,0x07e7c742,0xc6691882,0xef520d01,0x2edcd2c1,0xb73eb4c0,0x76b06b00,
        0xe5489fc6,0x24c64006,0xbd242607,0x7caaf9c7,0x5591ec44,0x941f3384,0x0dfd5585,0xcc738a45,
        0xa1a92c70,0x6027f3b0,0xf9c595b1,0x384b4a71,0x11705ff2,0xd0fe8032,0x491ce633,0x889239f3,
        0x1b6acd35,0xdae412f5,0x430674f4,0x8288ab34,0xabb3beb7,0x6a3d6177,0xf3df0776,0x3251d8b6,
        0x0f5fe8bb,0xced1377b,0x5733517a,0x96bd8eba,0xbf869b39,0x7e0844f9,0xe7ea22f8,0x2664fd38,
        0xb59c09fe,0x7412d63e,0xedf0b03f,0x2c7e6fff,0x05457a7c,0xc4cba5bc,0x5d29c3bd,0x9ca71c7d,
        0x2735a3a7,0xe6bb7c67,0x7f591a66,0xbed7c5a6,0x97ecd025,0x56620fe5,0xcf8069e4,0x0e0eb624,
        0x9df642e2,0x5c789d22,0xc59afb23,0x041424e3,0x2d2f3160,0xeca1eea0,0x754388a1,0xb4cd5761,
        0x89c3676c,0x484db8ac,0xd1afdead,0x1021016d,0x391a14ee,0xf894cb2e,0x6176ad2f,0xa0f872ef,
        0x33008629,0xf28e59e9,0x6b6c3fe8,0xaae2e028,0x83d9f5ab,0x42572a6b,0xdbb54c6a,0x1a3b93aa,
    },
    {
        0x00000000,0x9ba54c6f,0xec3b9e9f,0x779ed2f0,0x03063b7f,0x98a37710,0xef3da5e0,0x7498e98f,
        0x060c76fe,0x9da93a91,0xea37e861,0x7192a40e,0x050a4d81,0x9eaf01ee,0xe931d31e,0x72949f71,
        0x0c18edfc,0x97bda193,0xe0237363,0x7b863f0c,0x0f1ed683,0x94bb9aec,0xe325481c,0x78800473,
        0x0a149b02,0x91b1d76d,0xe62f059d,0x7d8a49f2,0x0912a07d,0x92b7ec12,0xe5293ee2,0x7e8c728d,
        0x1831dbf8,0x83949797,0xf40a4567,0x6faf0908,0x1b37e087,0x8092ace8,0xf70c7e18,0x6ca93277,
        0x1e3dad06,0x8598e169,0xf2063399,0x69a37ff6,0x1d3b9679,0x869eda16,0xf10008e6,0x6aa54489,
        0x14293604,0x8f8c7a6b,0xf812a89b,0x63b7e4f4,0x172f0d7b,0x8c8a4114,0xfb1493e4,0x60b1df8b,
        0x122540fa,0x89800c95,0xfe1ede65,0x65bb920a,0x11237b85,0x8a8637ea,0xfd18e51a,0x66bda975,
        0x3063b7f0,0xabc6fb9f,0xdc58296f,0x47fd6500,0x33658c8f,0xa8c0c0e0,0xdf5e1210,0x44fb5e7f,
        0x366fc10e,0xadca8d61,0xda545f91,0x41f113fe,0x3569fa71,0xaeccb61e,0xd95264ee,0x42f72881,
        0x3c7b5a0c,0xa7de1663,0xd040c493,0x4be588fc,0x3f7d6173,0xa4d82d1c,0xd346ffec,0x48e3b383,
        0x3a772cf2,0xa1d2609d,0xd64cb26d,0x4de9fe02,0x3971178d,0xa2d45be2,0xd54a8912,0x4eefc57d,
        0x28526c08,0xb3f72067,0xc469f297,0x5fccbef8,0x2b545777,0xb0f11b18,0xc76fc9e8,0x5cca8587,
        0x2e5e1af6,0xb5fb5699,0xc2658469,0x59c0c806,0x2d582189,0xb6fd6de6,0xc163bf16,0x5ac6f379,
        0x244a81f4,0xbfefcd9b,0xc8711f6b,0x53d45304,0x274cba8b,0xbce9f6e4,0xcb772414,0x50d2687b,
        0x2246f70a,0xb9e3bb65,0xce7d6995,0x55d825fa,0x2140cc75,0xbae5801a,0xcd7b52ea,0x56de1e85,
        0x60c76fe0,0xfb62238f,0x8cfcf17f,0x1759bd10,0x63c1549f,0xf86418f0,0x8ffaca00,0x145f866f,
        0x66cb191e,0xfd6e5571,0x8af08781,0x1155cbee,0x65cd2261,0xfe686e0e,0x89f6bcfe,0x1253f091,
        0x6cdf821c,0xf77ace73,0x80e41c83,0x1b4150ec,0x6fd9b963,0xf47cf50c,0x83e227fc,0x18476b93,
        0x6ad3f4e2,0xf176b88d,0x86e86a7d,0x1d4d2612,0x69d5cf9d,0xf27083f2,0x85ee5102,0x1e4b1d6d,
        0x78f6b418,0xe353f877,0x94cd2a87,0x0f6866e8,0x7bf08f67,0xe055c308,0x97cb11f8,0x0c6e5d97,
        0x7efac2e6,0xe55f8e89,0x92c15c79,0x09641016,0x7dfcf999,0xe659b5f6,0x91c76706,0x0a622b69,
        0x74ee59e4,0xef4b158b,0x98d5c77b,0x03708b14,0x77e8629b,0xec4d2ef4,0x9bd3fc04,0x0076b06b,
        0x72e22f1a,0xe9476375,0x9ed9b185,0x057cfdea,0x71e41465,0xea41580a,0x9ddf8afa,0x067ac695,
        0x50a4d810,0xcb01947f,0xbc9f468f,0x273a0ae0,0x53a2e36f,0xc807af00,0xbf997df0,0x243c319f,
        0x56a8aeee,0xcd0de281,0xba933071,0x21367c1e,0x55ae9591,0xce0bd9fe,0xb9950b0e,0x22304761,
        0x5cbc35ec,0xc7197983,0xb087ab73,0x2b22e71c,0x5fba0e93,0xc41f42fc,0xb381900c,0x2824dc63,
        0x5ab04312,0xc1150f7d,0xb68bdd8d,0x2d2e91e2,0x59b6786d,0xc2133402,0xb58de6f2,0x2e28aa9d,
        0x489503e8,0xd3304f87,0xa4ae9d77,0x3f0bd118,0x4b933897,0xd03674f8,0xa7a8a608,0x3c0dea67,
        0x4e997516,0xd53c3979,0xa2a2eb89,0x3907a7e6,0x4d9f4e69,0xd63a0206,0xa1a4d0f6,0x3a019c99,
        0x448dee14,0xdf28a27b,0xa8b6708b,0x33133ce4,0x478bd56b,0xdc2e9904,0xabb04bf4,0x3015079b,
        0x428198ea,0xd924d485,0xaeba0675,0x351f4a1a,0x4187a395,0xda22effa,0xadbc3d0a,0x36197165,
    },
    {
        0x00000000,0xdd96d985,0x605cb54b,0xbdca6cce,0xc0b96a96,0x1d2fb313,0xa0e5dfdd,0x7d730658,
        0x5a03d36d,0x87950ae8,0x3a5f6626,0xe7c9bfa3,0x9abab9fb,0x472c607e,0xfae60cb0,0x2770d535,
        0xb407a6da,0x69917f5f,0xd45b1391,0x09cdca14,0x74becc4c,0xa92815c9,0x14e27907,0xc974a082,
        0xee0475b7,0x3392ac32,0x8e58c0fc,0x53ce1979,0x2ebd1f21,0xf32bc6a4,0x4ee1aa6a,0x937773ef,
        0xb37e4bf5,0x6ee89270,0xd322febe,0x0eb4273b,0x73c72163,0xae51f8e6,0x139b9428,0xce0d4dad,
        0xe97d9898,0x34eb411d,0x89212dd3,0x54b7f456,0x29c4f20e,0xf4522b8b,0x49984745,0x940e9ec0,
        0x0779ed2f,0xdaef34aa,0x67255864,0xbab381e1,0xc7c087b9,0x1a565e3c,0xa79c32f2,0x7a0aeb77,
        0x5d7a3e42,0x80ece7c7,0x3d268b09,0xe0b0528c,0x9dc354d4,0x40558d51,0xfd9fe19f,0x2009381a,
        0xbd8d91ab,0x601b482e,0xddd124e0,0x0047fd65,0x7d34fb3d,0xa0a222b8,0x1d684e76,0xc0fe97f3,
        0xe78e42c6,0x3a189b43,0x87d2f78d,0x5a442e08,0x27372850,0xfaa1f1d5,0x476b9d1b,0x9afd449e,
        0x098a3771,0xd41ceef4,0x69d6823a,0xb4405bbf,0xc9335de7,0x14a58462,0xa96fe8ac,0x74f93129,
        0x5389e41c,0x8e1f3d99,0x33d55157,0xee4388d2,0x93308e8a,0x4ea6570f,0xf36c3bc1,0x2efae244,
        0x0ef3da5e,0xd36503db,0x6eaf6f15,0xb339b690,0xce4ab0c8,0x13dc694d,0xae160583,0x7380dc06,
        0x54f00933,0x8966d0b6,0x34acbc78,0xe93a65fd,0x944963a5,0x49dfba20,0xf415d6ee,0x29830f6b,
        0xbaf47c84,0x6762a501,0xdaa8c9cf,0x073e104a,0x7a4d1612,0xa7dbcf97,0x1a11a359,0xc7877adc,
        0xe0f7afe9,0x3d61766c,0x80ab1aa2,0x5d3dc327,0x204ec57f,0xfdd81cfa,0x40127034,0x9d84a9b1,
        0xa06a2517,0x7dfcfc92,0xc036905c,0x1da049d9,0x60d34f81,0xbd459604,0x008ffaca,0xdd19234f,
        0xfa69f67a,0x27ff2fff,0x9a354331,0x47a39ab4,0x3ad09cec,0xe7464569,0x5a8c29a7,0x871af022,
        0x146d83cd,0xc9fb5a48,0x74313686,0xa9a7ef03,0xd4d4e95b,0x094230de,0xb4885c10,0x691e8595,
        0x4e6e50a0,0x93f88925,0x2e32e5eb,0xf3a43c6e,0x8ed73a36,0x5341e3b3,0xee8b8f7d,0x331d56f8,
        0x13146ee2,0xce82b767,0x7348dba9,0xaede022c,0xd3ad0474,0x0e3bddf1,0xb3f1b13f,0x6e6768ba,
        0x4917bd8f,0x9481640a,0x294b08c4,0xf4ddd141,0x89aed719,0x54380e9c,0xe9f26252,0x3464bbd7,
        0xa713c838,0x7a8511bd,0xc74f7d73,0x1ad9a4f6,0x67aaa2ae,0xba3c7b2b,0x07f617e5,0xda60ce60,
        0xfd101b55,0x2086c2d0,0x9d4cae1e,0x40da779b,0x3da971c3,0xe03fa846,0x5df5c488,0x80631d0d,
        0x1de7b4bc,0xc0716d39,0x7dbb01f7,0xa02dd872,0xdd5ede2a,0x00c807af,0xbd026b61,0x6094b2e4,
        0x47e467d1,0x9a72be54,0x27b8d29a,0xfa2e0b1f,0x875d0d47,0x5acbd4c2,0xe701b80c,0x3a976189,
        0xa9e01266,0x7476cbe3,0xc9bca72d,0x142a7ea8,0x695978f0,0xb4cfa175,0x0905cdbb,0xd493143e,
        0xf3e3c10b,0x2e75188e,0x93bf7440,0x4e29adc5,0x335aab9d,0xeecc7218,0x53061ed6,0x8e90c753,
        0xae99ff49,0x730f26cc,0xcec54a02,0x13539387,0x6e2095df,0xb3b64c5a,0x0e7c2094,0xd3eaf911,
        0xf49a2c24,0x290cf5a1,0x94c6996f,0x495040ea,0x342346b2,0xe9b59f37,0x547ff3f9,0x89e92a7c,
        0x1a9e5993,0xc7088016,0x7ac2ecd8,0xa754355d,0xda273305,0x07b1ea80,0xba7b864e,0x67ed5fcb,
        0x409d8afe,0x9d0b537b,0x20c13fb5,0xfd57e630,0x8024e068,0x5db239ed,0xe0785523,0x3dee8ca6,
    },
    {
        0x00000000,0x9d0fe176,0xe16ec4ad,0x7c6125db,0x19ac8f1b,0x84a36e6d,0xf8c24bb6,0x65cdaac0,
        0x33591e36,0xae56ff40,0xd237da9b,0x4f383bed,0x2af5912d,0xb7fa705b,0xcb9b5580,0x5694b4f6,
        0x66b23c6c,0xfbbddd1a,0x87dcf8c1,0x1ad319b7,0x7f1eb377,0xe2115201,0x9e7077da,0x037f96ac,
        0x55eb225a,0xc8e4c32c,0xb485e6f7,0x298a0781,0x4c47ad41,0xd1484c37,0xad2969ec,0x3026889a,
        0xcd6478d8,0x506b99ae,0x2c0abc75,0xb1055d03,0xd4c8f7c3,0x49c716b5,0x35a6336e,0xa8a9d218,
        0xfe3d66ee,0x63328798,0x1f53a243,0x825c4335,0xe791e9f5,0x7a9e0883,0x06ff2d58,0x9bf0cc2e,
        0xabd644b4,0x36d9a5c2,0x4ab88019,0xd7b7616f,0xb27acbaf,0x2f752ad9,0x53140f02,0xce1bee74,
        0x988f5a82,0x0580bbf4,0x79e19e2f,0xe4ee7f59,0x8123d599,0x1c2c34ef,0x604d1134,0xfd42f042,
        0x41b9f7f1,0xdcb61687,0xa0d7335c,0x3dd8d22a,0x581578ea,0xc51a999c,0xb97bbc47,0x24745d31,
        0x72e0e9c7,0xefef08b1,0x938e2d6a,0x0e81cc1c,0x6b4c66dc,0xf64387aa,0x8a22a271,0x172d4307,
        0x270bcb9d,0xba042aeb,0xc6650f30,0x5b6aee46,0x3ea74486,0xa3a8a5f0,0xdfc9802b,0x42c6615d,
        0x1452d5ab,0x895d34dd,0xf53c1106,0x6833f070,0x0dfe5ab0,0x90f1bbc6,0xec909e1d,0x719f7f6b,
        0x8cdd8f29,0x11d26e5f,0x6db34b84,0xf0bcaaf2,0x95710032,0x087ee144,0x741fc49f,0xe91025e9,
        0xbf84911f,0x228b7069,0x5eea55b2,0xc3e5b4c4,0xa6281e04,0x3b27ff72,0x4746daa9,0xda493bdf,
        0xea6fb345,0x77605233,0x0b0177e8,0x960e969e,0xf3c33c5e,0x6eccdd28,0x12adf8f3,0x8fa21985,
        0xd936ad73,0x44394c05,0x385869de,0xa55788a8,0xc09a2268,0x5d95c31e,0x21f4e6c5,0xbcfb07b3,
        0x8373efe2,0x1e7c0e94,0x621d2b4f,0xff12ca39,0x9adf60f9,0x07d0818f,0x7bb1a454,0xe6be4522,
        0xb02af1d4,0x2d2510a2,0x51443579,0xcc4bd40f,0xa9867ecf,0x34899fb9,0x48e8ba62,0xd5e75b14,
        0xe5c1d38e,0x78ce32f8,0x04af1723,0x99a0f655,0xfc6d5c95,0x6162bde3,0x1d039838,0x800c794e,
        0xd698cdb8,0x4b972cce,0x37f60915,0xaaf9e863,0xcf3442a3,0x523ba3d5,0x2e5a860e,0xb3556778,
        0x4e17973a,0xd318764c,0xaf795397,0x3276b2e1,0x57bb1821,0xcab4f957,0xb6d5dc8c,0x2bda3dfa,
        0x7d4e890c,0xe041687a,0x9c204da1,0x012facd7,0x64e20617,0xf9ede761,0x858cc2ba,0x188323cc,
        0x28a5ab56,0xb5aa4a20,0xc9cb6ffb,0x54c48e8d,0x3109244d,0xac06c53b,0xd067e0e0,0x4d680196,
        0x1bfcb560,0x86f35416,0xfa9271cd,0x679d90bb,0x02503a7b,0x9f5fdb0d,0xe33efed6,0x7e311fa0,
        0xc2ca1813,0x5fc5f965,0x23a4dcbe,0xbeab3dc8,0xdb669708,0x4669767e,0x3a0853a5,0xa707b2d3,
        0xf1930625,0x6c9ce753,0x10fdc288,0x8df223fe,0xe83f893e,0x75306848,0x09514d93,0x945eace5,
        0xa478247f,0x3977c509,0x4516e0d2,0xd81901a4,0xbdd4ab64,0x20db4a12,0x5cba6fc9,0xc1b58ebf,
        0x97213a49,0x0a2edb3f,0x764ffee4,0xeb401f92,0x8e8db552,0x13825424,0x6fe371ff,0xf2ec9089,
        0x0fae60cb,0x92a181bd,0xeec0a466,0x73cf4510,0x1602efd0,0x8b0d0ea6,0xf76c2b7d,0x6a63ca0b,
        0x3cf77efd,0xa1f89f8b,0xdd99ba50,0x40965b26,0x255bf1e6,0xb8541090,0xc435354b,0x593ad43d,
        0x691c5ca7,0xf413bdd1,0x8872980a,0x157d797c,0x70b0d3bc,0xedbf32ca,0x91de1711,0x0cd1f667,
        0x5a454291,0xc74aa3e7,0xbb2b863c,0x2624674a,0x43e9cd8a,0xdee62cfc,0xa2870927,0x3f88e851,
    },
    {
        0x00000000,0xb9fbdbe8,0xa886b191,0x117d6a79,0x8a7c6563,0x3387be8b,0x22fad4f2,0x9b010f1a,
        0xcf89cc87,0x7672176f,0x670f7d16,0xdef4a6fe,0x45f5a9e4,0xfc0e720c,0xed731875,0x5488c39d,
        0x44629f4f,0xfd9944a7,0xece42ede,0x551ff536,0xce1efa2c,0x77e521c4,0x66984bbd,0xdf639055,
        0x8beb53c8,0x32108820,0x236de259,0x9a9639b1,0x019736ab,0xb86ced43,0xa911873a,0x10ea5cd2,
        0x88c53e9e,0x313ee576,0x20438f0f,0x99b854e7,0x02b95bfd,0xbb428015,0xaa3fea6c,0x13c43184,
        0x474cf219,0xfeb729f1,0xefca4388,0x56319860,0xcd30977a,0x74cb4c92,0x65b626eb,0xdc4dfd03,
        0xcca7a1d1,0x755c7a39,0x64211040,0xdddacba8,0x46dbc4b2,0xff201f5a,0xee5d7523,0x57a6aecb,
        0x032e6d56,0xbad5b6be,0xaba8dcc7,0x1253072f,0x89520835,0x30a9d3dd,0x21d4b9a4,0x982f624c,
        0xcafb7b7d,0x7300a095,0x627dcaec,0xdb861104,0x40871e1e,0xf97cc5f6,0xe801af8f,0x51fa7467,
        0x0572b7fa,0xbc896c12,0xadf4066b,0x140fdd83,0x8f0ed299,0x36f50971,0x27886308,0x9e73b8e0,
        0x8e99e432,0x37623fda,0x261f55a3,0x9fe48e4b,0x04e58151,0xbd1e5ab9,0xac6330c0,0x1598eb28,
        0x411028b5,0xf8ebf35d,0xe9969924,0x506d42cc,0xcb6c4dd6,0x7297963e,0x63eafc47,0xda1127af,
        0x423e45e3,0xfbc59e0b,0xeab8f472,0x53432f9a,0xc8422080,0x71b9fb68,0x60c49111,0xd93f4af9,
        0x8db78964,0x344c528c,0x253138f5,0x9ccae31d,0x07cbec07,0xbe3037ef,0xaf4d5d96,0x16b6867e,
        0x065cdaac,0xbfa70144,0xaeda6b3d,0x1721b0d5,0x8c20bfcf,0x35db6427,0x24a60e5e,0x9d5dd5b6,
        0xc9d5162b,0x702ecdc3,0x6153a7ba,0xd8a87c52,0x43a97348,0xfa52a8a0,0xeb2fc2d9,0x52d41931,
        0x4e87f0bb,0xf77c2b53,0xe601412a,0x5ffa9ac2,0xc4fb95d8,0x7d004e30,0x6c7d2449,0xd586ffa1,
        0x810e3c3c,0x38f5e7d4,0x29888dad,0x90735645,0x0b72595f,0xb28982b7,0xa3f4e8ce,0x1a0f3326,
        0x0ae56ff4,0xb31eb41c,0xa263de65,0x1b98058d,0x80990a97,0x3962d17f,0x281fbb06,0x91e460ee,
        0xc56ca373,0x7c97789b,0x6dea12e2,0xd411c90a,0x4f10c610,0xf6eb1df8,0xe7967781,0x5e6dac69,
        0xc642ce25,0x7fb915cd,0x6ec47fb4,0xd73fa45c,0x4c3eab46,0xf5c570ae,0xe4b81ad7,0x5d43c13f,
        0x09cb02a2,0xb030d94a,0xa14db333,0x18b668db,0x83b767c1,0x3a4cbc29,0x2b31d650,0x92ca0db8,
        0x8220516a,0x3bdb8a82,0x2aa6e0fb,0x935d3b13,0x085c3409,0xb1a7efe1,0xa0da8598,0x19215e70,
        0x4da99ded,0xf4524605,0xe52f2c7c,0x5cd4f794,0xc7d5f88e,0x7e2e2366,0x6f53491f,0xd6a892f7,
        0x847c8bc6,0x3d87502e,0x2cfa3a57,0x9501e1bf,0x0e00eea5,0xb7fb354d,0xa6865f34,0x1f7d84dc,
        0x4bf54741,0xf20e9ca9,0xe373f6d0,0x5a882d38,0xc1892222,0x7872f9ca,0x690f93b3,0xd0f4485b,
        0xc01e1489,0x79e5cf61,0x6898a518,0xd1637ef0,0x4a6271ea,0xf399aa02,0xe2e4c07b,0x5b1f1b93,
        0x0f97d80e,0xb66c03e6,0xa711699f,0x1eeab277,0x85ebbd6d,0x3c106685,0x2d6d0cfc,0x9496d714,
        0x0cb9b558,0xb5426eb0,0xa43f04c9,0x1dc4df21,0x86c5d03b,0x3f3e0bd3,0x2e4361aa,0x97b8ba42,
        0xc33079df,0x7acba237,0x6bb6c84e,0xd24d13a6,0x494c1cbc,0xf0b7c754,0xe1caad2d,0x583176c5,
        0x48db2a17,0xf120f1ff,0xe05d9b86,0x59a6406e,0xc2a74f74,0x7b5c949c,0x6a21fee5,0xd3da250d,
        0x8752e690,0x3ea93d78,0x2fd45701,0x962f8ce9,0x0d2e83f3,0xb4d5581b,0xa5a83262,0x1c53e98a,
    },
    {
        0x00000000,0xae689191,0x87a02563,0x29c8b4f2,0xd4314c87,0x7a59dd16,0x539169e4,0xfdf9f875,
        0x73139f4f,0xdd7b0ede,0xf4b3ba2c,0x5adb2bbd,0xa722d3c8,0x094a4259,0x2082f6ab,0x8eea673a,
        0xe6273e9e,0x484faf0f,0x61871bfd,0xcfef8a6c,0x32167219,0x9c7ee388,0xb5b6577a,0x1bdec6eb,
        0x9534a1d1,0x3b5c3040,0x129484b2,0xbcfc1523,0x4105ed56,0xef6d7cc7,0xc6a5c835,0x68cd59a4,
        0x173f7b7d,0xb957eaec,0x909f5e1e,0x3ef7cf8f,0xc30e37fa,0x6d66a66b,0x44ae1299,0xeac68308,
        0x642ce432,0xca4475a3,0xe38cc151,0x4de450c0,0xb01da8b5,0x1e753924,0x37bd8dd6,0x99d51c47,
        0xf11845e3,0x5f70d472,0x76b86080,0xd8d0f111,0x25290964,0x8b4198f5,0xa2892c07,0x0ce1bd96,
        0x820bdaac,0x2c634b3d,0x05abffcf,0xabc36e5e,0x563a962b,0xf85207ba,0xd19ab348,0x7ff222d9,
        0x2e7ef6fa,0x8016676b,0xa9ded399,0x07b64208,0xfa4fba7d,0x54272bec,0x7def9f1e,0xd3870e8f,
        0x5d6d69b5,0xf305f824,0xdacd4cd6,0x74a5dd47,0x895c2532,0x2734b4a3,0x0efc0051,0xa09491c0,
        0xc859c864,0x663159f5,0x4ff9ed07,0xe1917c96,0x1c6884e3,0xb2001572,0x9bc8a180,0x35a03011,
        0xbb4a572b,0x1522c6ba,0x3cea7248,0x9282e3d9,0x6f7b1bac,0xc1138a3d,0xe8db3ecf,0x46b3af5e,
        0x39418d87,0x97291c16,0xbee1a8e4,0x10893975,0xed70c100,0x43185091,0x6ad0e463,0xc4b875f2,
        0x4a5212c8,0xe43a8359,0xcdf237ab,0x639aa63a,0x9e635e4f,0x300bcfde,0x19c37b2c,0xb7abeabd,
        0xdf66b319,0x710e2288,0x58c6967a,0xf6ae07eb,0x0b57ff9e,0xa53f6e0f,0x8cf7dafd,0x229f4b6c,
        0xac752c56,0x021dbdc7,0x2bd50935,0x85bd98a4,0x784460d1,0xd62cf140,0xffe445b2,0x518cd423,
        0x5cfdedf4,0xf2957c65,0xdb5dc897,0x75355906,0x88cca173,0x26a430e2,0x0f6c8410,0xa1041581,
        0x2fee72bb,0x8186e32a,0xa84e57d8,0x0626c649,0xfbdf3e3c,0x55b7afad,0x7c7f1b5f,0xd2178ace,
        0xbadad36a,0x14b242fb,0x3d7af609,0x93126798,0x6eeb9fed,0xc0830e7c,0xe94bba8e,0x47232b1f,
        0xc9c94c25,0x67a1ddb4,0x4e696946,0xe001f8d7,0x1df800a2,0xb3909133,0x9a5825c1,0x3430b450,
        0x4bc29689,0xe5aa0718,0xcc62b3ea,0x620a227b,0x9ff3da0e,0x319b4b9f,0x1853ff6d,0xb63b6efc,
        0x38d109c6,0x96b99857,0xbf712ca5,0x1119bd34,0xece04541,0x4288d4d0,0x6b406022,0xc528f1b3,
        0xade5a817,0x038d3986,0x2a458d74,0x842d1ce5,0x79d4e490,0xd7bc7501,0xfe74c1f3,0x501c5062,
        0xdef63758,0x709ea6c9,0x5956123b,0xf73e83aa,0x0ac77bdf,0xa4afea4e,0x8d675ebc,0x230fcf2d,
        0x72831b0e,0xdceb8a9f,0xf5233e6d,0x5b4baffc,0xa6b25789,0x08dac618,0x211272ea,0x8f7ae37b,
        0x01908441,0xaff815d0,0x8630a122,0x285830b3,0xd5a1c8c6,0x7bc95957,0x5201eda5,0xfc697c34,
        0x94a42590,0x3accb401,0x130400f3,0xbd6c9162,0x40956917,0xeefdf886,0xc7354c74,0x695ddde5,
        0xe7b7badf,0x49df2b4e,0x60179fbc,0xce7f0e2d,0x3386f658,0x9dee67c9,0xb426d33b,0x1a4e42aa,
        0x65bc6073,0xcbd4f1e2,0xe21c4510,0x4c74d481,0xb18d2cf4,0x1fe5bd65,0x362d0997,0x98459806,
        0x16afff3c,0xb8c76ead,0x910fda5f,0x3f674bce,0xc29eb3bb,0x6cf6222a,0x453e96d8,0xeb560749,
        0x839b5eed,0x2df3cf7c,0x043b7b8e,0xaa53ea1f,0x57aa126a,0xf9c283fb,0xd00a3709,0x7e62a698,
        0xf088c1a2,0x5ee05033,0x7728e4c1,0xd9407550,0x24b98d25,0x8ad11cb4,0xa319a846,0x0d7139d7,
    }
};
