const char msgpackffi_lua[] =
"-- msgpackffi.lua (internal file)\n"
"\n"
"local ffi = require('ffi')\n"
"local buffer = require('buffer')\n"
"local builtin = ffi.C\n"
"local msgpack = require('msgpack') -- .NULL, .array_mt, .map_mt, .cfg\n"
"local int8_ptr_t = ffi.typeof('int8_t *')\n"
"local uint8_ptr_t = ffi.typeof('uint8_t *')\n"
"local uint16_ptr_t = ffi.typeof('uint16_t *')\n"
"local uint32_ptr_t = ffi.typeof('uint32_t *')\n"
"local uint64_ptr_t = ffi.typeof('uint64_t *')\n"
"local char_ptr_t = ffi.typeof('char *')\n"
"local cord_ibuf_take = buffer.internal.cord_ibuf_take\n"
"local cord_ibuf_drop = buffer.internal.cord_ibuf_drop\n"
"\n"
"ffi.cdef([[\n"
"char *\n"
"tnt_mp_encode_float(char *data, float num);\n"
"char *\n"
"tnt_mp_encode_double(char *data, double num);\n"
"char *\n"
"tnt_mp_encode_decimal(char *data, decimal_t *dec);\n"
"uint32_t\n"
"tnt_mp_sizeof_decimal(const decimal_t *dec);\n"
"char *\n"
"tnt_mp_encode_uuid(char *data, const struct tt_uuid *uuid);\n"
"uint32_t\n"
"tnt_mp_sizeof_uuid();\n"
"uint32_t\n"
"tnt_mp_sizeof_error(const struct error *error);\n"
"char *\n"
"tnt_mp_encode_error(char *data, const struct error *error);\n"
"uint32_t\n"
"tnt_mp_sizeof_datetime(const struct datetime *date);\n"
"char *\n"
"tnt_mp_encode_datetime(char *data, const struct datetime *date);\n"
"char *\n"
"tnt_mp_encode_interval(char *data, const struct interval *itv);\n"
"uint32_t\n"
"tnt_mp_sizeof_interval(const struct interval *itv);\n"
"float\n"
"tnt_mp_decode_float(const char **data);\n"
"double\n"
"tnt_mp_decode_double(const char **data);\n"
"uint32_t\n"
"tnt_mp_decode_extl(const char **data, int8_t *type);\n"
"decimal_t *\n"
"decimal_unpack(const char **data, uint32_t len, decimal_t *dec);\n"
"struct tt_uuid *\n"
"uuid_unpack(const char **data, uint32_t len, struct tt_uuid *uuid);\n"
"struct error *\n"
"error_unpack(const char **data, uint32_t len);\n"
"void\n"
"error_unref(struct error *e);\n"
"struct datetime *\n"
"tnt_datetime_unpack(const char **data, uint32_t len, struct datetime *date);\n"
"struct interval *\n"
"tnt_interval_unpack(const char **data, uint32_t len, struct interval *itv);\n"
"]])\n"
"\n"
"local strict_alignment = (jit.arch == 'arm')\n"
"\n"
"local function bswap_u16(num)\n"
"    return bit.rshift(bit.bswap(tonumber(num)), 16)\n"
"end\n"
"\n"
"--------------------------------------------------------------------------------\n"
"-- Encoder\n"
"--------------------------------------------------------------------------------\n"
"\n"
"local encode_ext_cdata = {}\n"
"\n"
"-- Set trigger that called when encoding cdata\n"
"local function on_encode(ctype_or_udataname, callback)\n"
"    if type(ctype_or_udataname) ~= \"cdata\" or type(callback) ~= \"function\" then\n"
"        error(\"Usage: on_encode(ffi.typeof('mytype'), function(buf, obj)\")\n"
"    end\n"
"    local ctypeid = tonumber(ffi.typeof(ctype_or_udataname))\n"
"    local prev = encode_ext_cdata[ctypeid]\n"
"    encode_ext_cdata[ctypeid] = callback\n"
"    return prev\n"
"end\n"
"\n"
"local function encode_fix(buf, code, num)\n"
"    local p = buf:alloc(1)\n"
"    p[0] = bit.bor(code, tonumber(num))\n"
"end\n"
"\n"
"local function encode_u8(buf, code, num)\n"
"    local p = buf:alloc(2)\n"
"    p[0] = code\n"
"    ffi.cast(uint8_ptr_t, p + 1)[0] = num\n"
"end\n"
"\n"
"local encode_u16\n"
"if strict_alignment then\n"
"    encode_u16 = function(buf, code, num)\n"
"        local u16 = ffi.new('uint16_t[1]', bswap_u16(num))\n"
"        local p = buf:alloc(3)\n"
"        p[0] = code\n"
"        ffi.copy(p + 1, u16, 2)\n"
"    end\n"
"else\n"
"    encode_u16 = function(buf, code, num)\n"
"        local p = buf:alloc(3)\n"
"        p[0] = code\n"
"        ffi.cast(uint16_ptr_t, p + 1)[0] = bswap_u16(num)\n"
"    end\n"
"end\n"
"\n"
"local encode_u32\n"
"if strict_alignment then\n"
"    encode_u32 = function(buf, code, num)\n"
"        local u32 = ffi.new('uint32_t[1]',\n"
"                            ffi.cast('uint32_t', bit.bswap(tonumber(num))))\n"
"        local p = buf:alloc(5)\n"
"        p[0] = code\n"
"        ffi.copy(p + 1, u32, 4)\n"
"    end\n"
"else\n"
"    encode_u32 = function(buf, code, num)\n"
"        local p = buf:alloc(5)\n"
"        p[0] = code\n"
"        ffi.cast(uint32_ptr_t, p + 1)[0] =\n"
"            ffi.cast('uint32_t', bit.bswap(tonumber(num)))\n"
"    end\n"
"end\n"
"\n"
"local encode_u64\n"
"if strict_alignment then\n"
"    encode_u64 = function(buf, code, num)\n"
"        local u64 = ffi.new('uint64_t[1]', bit.bswap(ffi.cast('uint64_t', num)))\n"
"        local p = buf:alloc(9)\n"
"        p[0] = code\n"
"        ffi.copy(p + 1, u64, 8)\n"
"    end\n"
"else\n"
"    encode_u64 = function(buf, code, num)\n"
"        local p = buf:alloc(9)\n"
"        p[0] = code\n"
"        ffi.cast(uint64_ptr_t, p + 1)[0] = bit.bswap(ffi.cast('uint64_t', num))\n"
"    end\n"
"end\n"
"\n"
"local function encode_float(buf, num)\n"
"    local p = buf:alloc(5)\n"
"    builtin.tnt_mp_encode_float(p, num)\n"
"end\n"
"\n"
"local function encode_double(buf, num)\n"
"    local p = buf:alloc(9)\n"
"    builtin.tnt_mp_encode_double(p, num)\n"
"end\n"
"\n"
"local function encode_decimal(buf, num)\n"
"    local p = buf:alloc(builtin.tnt_mp_sizeof_decimal(num))\n"
"    builtin.tnt_mp_encode_decimal(p, num)\n"
"end\n"
"\n"
"local function encode_uuid(buf, uuid)\n"
"    local p = buf:alloc(builtin.tnt_mp_sizeof_uuid())\n"
"    builtin.tnt_mp_encode_uuid(p, uuid)\n"
"end\n"
"\n"
"local function encode_datetime(buf, date)\n"
"    local p = buf:alloc(builtin.tnt_mp_sizeof_datetime(date))\n"
"    builtin.tnt_mp_encode_datetime(p, date)\n"
"end\n"
"\n"
"local function encode_interval(buf, itv)\n"
"    local p = buf:alloc(builtin.tnt_mp_sizeof_interval(itv))\n"
"    builtin.tnt_mp_encode_interval(p, itv)\n"
"end\n"
"\n"
"local function encode_int(buf, num)\n"
"    if num >= 0 then\n"
"        if num <= 0x7f then\n"
"            encode_fix(buf, 0, num)\n"
"        elseif num <= 0xff then\n"
"            encode_u8(buf, 0xcc, num)\n"
"        elseif num <= 0xffff then\n"
"            encode_u16(buf, 0xcd, num)\n"
"        elseif num <= 0xffffffff then\n"
"            encode_u32(buf, 0xce, num)\n"
"        else\n"
"            encode_u64(buf, 0xcf, 0ULL + num)\n"
"        end\n"
"    else\n"
"        if num >= -0x20 then\n"
"            encode_fix(buf, 0xe0, num)\n"
"        elseif num >= -0x80 then\n"
"            encode_u8(buf, 0xd0, num)\n"
"        elseif num >= -0x8000 then\n"
"            encode_u16(buf, 0xd1, num)\n"
"        elseif num >= -0x80000000 then\n"
"            encode_u32(buf, 0xd2, num)\n"
"        else\n"
"            encode_u64(buf, 0xd3, 0LL + num)\n"
"        end\n"
"    end\n"
"end\n"
"\n"
"local function encode_str(buf, str)\n"
"    local len = #str\n"
"    buf:reserve(5 + len)\n"
"    if len <= 31 then\n"
"        encode_fix(buf, 0xa0, len)\n"
"    elseif len <= 0xff then\n"
"        encode_u8(buf, 0xd9, len)\n"
"    elseif len <= 0xffff then\n"
"        encode_u16(buf, 0xda, len)\n"
"    else\n"
"        encode_u32(buf, 0xdb, len)\n"
"    end\n"
"    local p = buf:alloc(len)\n"
"    ffi.copy(p, str, len)\n"
"end\n"
"\n"
"local function encode_array(buf, size)\n"
"    if size <= 0xf then\n"
"        encode_fix(buf, 0x90, size)\n"
"    elseif size <= 0xffff then\n"
"        encode_u16(buf, 0xdc, size)\n"
"    else\n"
"        encode_u32(buf, 0xdd, size)\n"
"    end\n"
"end\n"
"\n"
"local function encode_map(buf, size)\n"
"    if size <= 0xf then\n"
"        encode_fix(buf, 0x80, size)\n"
"    elseif size <= 0xffff then\n"
"        encode_u16(buf, 0xde, size)\n"
"    else\n"
"        encode_u32(buf, 0xdf, size)\n"
"    end\n"
"end\n"
"\n"
"local function encode_bool(buf, val)\n"
"    encode_fix(buf, 0xc2, val and 1 or 0)\n"
"end\n"
"\n"
"local function encode_bool_cdata(buf, val)\n"
"    encode_fix(buf, 0xc2, val ~= 0 and 1 or 0)\n"
"end\n"
"\n"
"local function encode_nil(buf)\n"
"    local p = buf:alloc(1)\n"
"    p[0] = 0xc0\n"
"end\n"
"\n"
"local function encode_error(buf, err)\n"
"    if msgpack.cfg.encode_error_as_ext then\n"
"        local p = buf:alloc(builtin.tnt_mp_sizeof_error(err))\n"
"        builtin.tnt_mp_encode_error(p, err)\n"
"    else\n"
"        encode_str(buf, err.message)\n"
"    end\n"
"end\n"
"\n"
"local function encode_r(buf, obj, level)\n"
"::restart::\n"
"    if type(obj) == \"number\" then\n"
"        -- Lua-way to check that number is an integer\n"
"        if obj % 1 == 0 and obj >= -2^63 and obj < 2^64 then\n"
"            encode_int(buf, obj)\n"
"        else\n"
"            encode_double(buf, obj)\n"
"        end\n"
"    elseif type(obj) == \"string\" then\n"
"        encode_str(buf, obj)\n"
"    elseif type(obj) == \"table\" then\n"
"        if level >= msgpack.cfg.encode_max_depth then\n"
"            if not msgpack.cfg.encode_deep_as_nil then\n"
"                error(string.format('Too high nest level - %d',\n"
"                                    msgpack.cfg.encode_max_depth + 1))\n"
"            end\n"
"            encode_nil(buf)\n"
"            return\n"
"        end\n"
"        local serialize = nil\n"
"        local mt = getmetatable(obj)\n"
"        if mt ~= nil then\n"
"            serialize = mt.__serialize\n"
"        end\n"
"        -- calculate the number of array and map elements in the table\n"
"        -- TODO: pairs() aborts JIT\n"
"        local array_count, map_count = 0, 0\n"
"        for key in pairs(obj) do\n"
"            if type(key) == 'number' and key >= 1 and\n"
"               key == math.floor(key) and key == array_count + 1 then\n"
"                array_count = array_count + 1\n"
"            else\n"
"                map_count = map_count + 1\n"
"            end\n"
"        end\n"
"        if (serialize == nil and map_count == 0) or serialize == 'array' or\n"
"            serialize == 'seq' or serialize == 'sequence' then\n"
"            encode_array(buf, array_count)\n"
"            for i=1,array_count,1 do\n"
"                encode_r(buf, obj[i], level + 1)\n"
"            end\n"
"        elseif (serialize == nil and map_count > 0) or\n"
"            serialize == 'map' or serialize == 'mapping' then\n"
"            encode_map(buf, array_count + map_count)\n"
"            for key, val in pairs(obj) do\n"
"                encode_r(buf, key, level + 1)\n"
"                encode_r(buf, val, level + 1)\n"
"            end\n"
"        elseif type(serialize) == 'function' then\n"
"            obj = serialize(obj)\n"
"            goto restart\n"
"        else\n"
"            error(\"Invalid __serialize value\")\n"
"        end\n"
"    elseif obj == nil then\n"
"        encode_nil(buf)\n"
"    elseif type(obj) == \"boolean\" then\n"
"        encode_bool(buf, obj)\n"
"    elseif type(obj) == \"cdata\" then\n"
"        if obj == nil then -- a workaround for nil\n"
"            encode_nil(buf, obj)\n"
"            return\n"
"        end\n"
"        local ctypeid = tonumber(ffi.typeof(obj))\n"
"        local fun = encode_ext_cdata[ctypeid]\n"
"        if fun ~= nil then\n"
"            fun(buf, obj)\n"
"        else\n"
"            error(\"can not encode FFI type: '\"..ffi.typeof(obj)..\"'\")\n"
"        end\n"
"    else\n"
"        error(\"can not encode Lua type: '\"..type(obj)..\"'\")\n"
"    end\n"
"end\n"
"\n"
"local function encode(obj)\n"
"    local tmpbuf = cord_ibuf_take()\n"
"    encode_r(tmpbuf, obj, 0)\n"
"    local r = ffi.string(tmpbuf.rpos, tmpbuf:size())\n"
"    cord_ibuf_drop(tmpbuf)\n"
"    return r\n"
"end\n"
"\n"
"on_encode(ffi.typeof('uint8_t'), encode_int)\n"
"on_encode(ffi.typeof('uint16_t'), encode_int)\n"
"on_encode(ffi.typeof('uint32_t'), encode_int)\n"
"on_encode(ffi.typeof('uint64_t'), encode_int)\n"
"on_encode(ffi.typeof('int8_t'), encode_int)\n"
"on_encode(ffi.typeof('int16_t'), encode_int)\n"
"on_encode(ffi.typeof('int32_t'), encode_int)\n"
"on_encode(ffi.typeof('int64_t'), encode_int)\n"
"on_encode(ffi.typeof('char'), encode_int)\n"
"on_encode(ffi.typeof('const char'), encode_int)\n"
"on_encode(ffi.typeof('unsigned char'), encode_int)\n"
"on_encode(ffi.typeof('const unsigned char'), encode_int)\n"
"on_encode(ffi.typeof('bool'), encode_bool_cdata)\n"
"on_encode(ffi.typeof('float'), encode_float)\n"
"on_encode(ffi.typeof('double'), encode_double)\n"
"on_encode(ffi.typeof('decimal_t'), encode_decimal)\n"
"on_encode(ffi.typeof('struct tt_uuid'), encode_uuid)\n"
"on_encode(ffi.typeof('const struct error &'), encode_error)\n"
"on_encode(ffi.typeof('struct datetime'), encode_datetime)\n"
"on_encode(ffi.typeof('struct interval'), encode_interval)\n"
"\n"
"--------------------------------------------------------------------------------\n"
"-- Decoder\n"
"--------------------------------------------------------------------------------\n"
"\n"
"local decode_r\n"
"\n"
"-- See similar constants in utils.cc\n"
"local DBL_INT_MAX = 1e14 - 1\n"
"\n"
"local function decode_u8(data)\n"
"    local num = ffi.cast(uint8_ptr_t, data[0])[0]\n"
"    data[0] = data[0] + 1\n"
"    return tonumber(num)\n"
"end\n"
"\n"
"local decode_u16\n"
"if strict_alignment then\n"
"    decode_u16 = function(data)\n"
"        local u16 = ffi.new('uint16_t[1]')\n"
"        ffi.copy(u16, data[0], 2)\n"
"        data[0] = data[0] + 2\n"
"        return tonumber(bswap_u16(u16[0]))\n"
"    end\n"
"else\n"
"    decode_u16 = function(data)\n"
"        local num = bswap_u16(ffi.cast(uint16_ptr_t, data[0])[0])\n"
"        data[0] = data[0] + 2\n"
"        return tonumber(num)\n"
"    end\n"
"end\n"
"\n"
"local decode_u32\n"
"if strict_alignment then\n"
"    decode_u32 = function(data)\n"
"        local u32 = ffi.new('uint32_t[1]')\n"
"        ffi.copy(u32, data[0], 4)\n"
"        data[0] = data[0] + 4\n"
"        return tonumber(\n"
"            ffi.cast('uint32_t', bit.bswap(tonumber(u32[0]))))\n"
"    end\n"
"else\n"
"    decode_u32 = function(data)\n"
"        local num = ffi.cast('uint32_t',\n"
"            bit.bswap(tonumber(ffi.cast(uint32_ptr_t, data[0])[0])))\n"
"        data[0] = data[0] + 4\n"
"        return tonumber(num)\n"
"    end\n"
"end\n"
"\n"
"local decode_u64\n"
"if strict_alignment then\n"
"    decode_u64 = function(data)\n"
"        local u64 = ffi.new('uint64_t[1]')\n"
"        ffi.copy(u64, data[0], 8);\n"
"        data[0] = data[0] + 8\n"
"        local num = bit.bswap(u64[0])\n"
"        if num <= DBL_INT_MAX then\n"
"            return tonumber(num) -- return as 'number'\n"
"        end\n"
"        return num -- return as 'cdata'\n"
"    end\n"
"else\n"
"    decode_u64 = function(data)\n"
"        local num =\n"
"            bit.bswap(ffi.cast(uint64_ptr_t, data[0])[0])\n"
"        data[0] = data[0] + 8\n"
"        if num <= DBL_INT_MAX then\n"
"            return tonumber(num) -- return as 'number'\n"
"        end\n"
"        return num -- return as 'cdata'\n"
"    end\n"
"end\n"
"\n"
"local function decode_i8(data)\n"
"    local num = ffi.cast(int8_ptr_t, data[0])[0]\n"
"    data[0] = data[0] + 1\n"
"    return tonumber(num)\n"
"end\n"
"\n"
"local decode_i16\n"
"if strict_alignment then\n"
"    decode_i16 = function(data)\n"
"        local u16 = ffi.new('uint16_t[1]')\n"
"        ffi.copy(u16, data[0], 2)\n"
"        local num = bswap_u16(u16[0])\n"
"        data[0] = data[0] + 2\n"
"        -- note: this double cast is actually necessary\n"
"        return tonumber(ffi.cast('int16_t', ffi.cast('uint16_t', num)))\n"
"    end\n"
"else\n"
"    decode_i16 = function(data)\n"
"        local num = bswap_u16(ffi.cast(uint16_ptr_t, data[0])[0])\n"
"        data[0] = data[0] + 2\n"
"        -- note: this double cast is actually necessary\n"
"        return tonumber(ffi.cast('int16_t', ffi.cast('uint16_t', num)))\n"
"    end\n"
"end\n"
"\n"
"local decode_i32\n"
"if strict_alignment then\n"
"    decode_i32 = function(data)\n"
"        local u32 = ffi.new('uint32_t[1]')\n"
"        ffi.copy(u32, data[0], 4)\n"
"        local num = bit.bswap(tonumber(u32[0]))\n"
"        data[0] = data[0] + 4\n"
"        return num\n"
"    end\n"
"else\n"
"    decode_i32 = function(data)\n"
"        local num = bit.bswap(tonumber(ffi.cast(uint32_ptr_t, data[0])[0]))\n"
"        data[0] = data[0] + 4\n"
"        return num\n"
"    end\n"
"end\n"
"\n"
"local decode_i64\n"
"if strict_alignment then\n"
"    decode_i64 = function(data)\n"
"        local i64 = ffi.new('int64_t[1]')\n"
"        ffi.copy(i64, data[0], 8)\n"
"        data[0] = data[0] + 8\n"
"        local num = bit.bswap(i64[0])\n"
"        if num >= -DBL_INT_MAX and num <= DBL_INT_MAX then\n"
"            return tonumber(num) -- return as 'number'\n"
"        end\n"
"        return num -- return as 'cdata'\n"
"    end\n"
"else\n"
"    decode_i64 = function(data)\n"
"        local num = bit.bswap(ffi.cast('int64_t',\n"
"                ffi.cast(uint64_ptr_t, data[0])[0]))\n"
"        data[0] = data[0] + 8\n"
"        if num >= -DBL_INT_MAX and num <= DBL_INT_MAX then\n"
"            return tonumber(num) -- return as 'number'\n"
"        end\n"
"        return num -- return as 'cdata'\n"
"    end\n"
"end\n"
"\n"
"local function decode_float(data)\n"
"    data[0] = data[0] - 1 -- tnt_mp_decode_float need type code\n"
"    return tonumber(builtin.tnt_mp_decode_float(data))\n"
"end\n"
"\n"
"local function decode_double(data)\n"
"    data[0] = data[0] - 1 -- tnt_mp_decode_double need type code\n"
"    return tonumber(builtin.tnt_mp_decode_double(data))\n"
"end\n"
"\n"
"local function decode_str(data, size)\n"
"    local ret = ffi.string(data[0], size)\n"
"    data[0] = data[0] + size\n"
"    return ret\n"
"end\n"
"\n"
"local function decode_array(data, size)\n"
"    assert (type(size) == \"number\")\n"
"    local arr = {}\n"
"    for _ = 1, size do\n"
"        table.insert(arr, decode_r(data))\n"
"    end\n"
"    if not msgpack.cfg.decode_save_metatables then\n"
"        return arr\n"
"    end\n"
"    return setmetatable(arr, msgpack.array_mt)\n"
"end\n"
"\n"
"local function decode_map(data, size)\n"
"    assert (type(size) == \"number\")\n"
"    local map = {}\n"
"    for _ = 1, size do\n"
"        local key = decode_r(data);\n"
"        local val = decode_r(data);\n"
"        map[key] = val\n"
"    end\n"
"    if not msgpack.cfg.decode_save_metatables then\n"
"        return map\n"
"    end\n"
"    return setmetatable(map, msgpack.map_mt)\n"
"end\n"
"\n"
"local ext_decoder = {\n"
"    -- MP_UNKNOWN_EXTENSION\n"
"    [0] = function(data, len) error(\"unsupported extension type\") end, -- luacheck: no unused args\n"
"    -- MP_DECIMAL\n"
"    [1] = function(data, len) local num = ffi.new(\"decimal_t\") builtin.decimal_unpack(data, len, num) return num end,\n"
"    -- MP_UUID\n"
"    [2] = function(data, len)\n"
"        local uuid = ffi.new(\"struct tt_uuid\")\n"
"        builtin.uuid_unpack(data, len, uuid)\n"
"        return uuid\n"
"    end,\n"
"    -- MP_ERROR\n"
"    [3] = function(data, len)\n"
"        local err = builtin.error_unpack(data, len)\n"
"        if err ~= nil then\n"
"            err._refs = err._refs + 1\n"
"            -- From FFI it is returned as 'struct error *', which is\n"
"            -- not considered equal to 'const struct error &', and is\n"
"            -- is not accepted by functions like box.error(). Need to\n"
"            -- cast explicitly.\n"
"            err = ffi.cast('const struct error &', err)\n"
"            err = ffi.gc(err, builtin.error_unref)\n"
"        end\n"
"        return err\n"
"    end,\n"
"    -- MP_DATETIME\n"
"    [4] = function(data, len)\n"
"        local dt = ffi.new(\"struct datetime\")\n"
"        builtin.tnt_datetime_unpack(data, len, dt)\n"
"        return dt\n"
"    end,\n"
"    -- MP_INTERVAL\n"
"    [6] = function(data, len)\n"
"        local itv = ffi.new(\"struct interval\")\n"
"        builtin.tnt_interval_unpack(data, len, itv)\n"
"        return itv\n"
"    end,\n"
"}\n"
"\n"
"local function decode_ext(data)\n"
"    local t = ffi.new(\"int8_t[1]\")\n"
"    -- tnt_mp_decode_extl and tnt_mp_decode_decimal\n"
"    -- need type code\n"
"    data[0] = data[0] - 1\n"
"    local len = builtin.tnt_mp_decode_extl(data, t)\n"
"    local fun = ext_decoder[t[0]]\n"
"    if type(fun) == 'function' then\n"
"        return fun(data, len)\n"
"    else\n"
"        error(\"Unsupported extension type\")\n"
"    end\n"
"end\n"
"\n"
"local decoder_hint = {\n"
"    --[[{{{ MP_BIN]]\n"
"    [0xc4] = function(data) return decode_str(data, decode_u8(data)) end;\n"
"    [0xc5] = function(data) return decode_str(data, decode_u16(data)) end;\n"
"    [0xc6] = function(data) return decode_str(data, decode_u32(data)) end;\n"
"\n"
"    --[[MP_FLOAT, MP_DOUBLE]]\n"
"    [0xca] = decode_float;\n"
"    [0xcb] = decode_double;\n"
"\n"
"    --[[MP_UINT]]\n"
"    [0xcc] = decode_u8;\n"
"    [0xcd] = decode_u16;\n"
"    [0xce] = decode_u32;\n"
"    [0xcf] = decode_u64;\n"
"\n"
"    --[[MP_INT]]\n"
"    [0xd0] = decode_i8;\n"
"    [0xd1] = decode_i16;\n"
"    [0xd2] = decode_i32;\n"
"    [0xd3] = decode_i64;\n"
"\n"
"    --[[MP_STR]]\n"
"    [0xd9] = function(data) return decode_str(data, decode_u8(data)) end;\n"
"    [0xda] = function(data) return decode_str(data, decode_u16(data)) end;\n"
"    [0xdb] = function(data) return decode_str(data, decode_u32(data)) end;\n"
"\n"
"    --[[MP_ARRAY]]\n"
"    [0xdc] = function(data) return decode_array(data, decode_u16(data)) end;\n"
"    [0xdd] = function(data) return decode_array(data, decode_u32(data)) end;\n"
"\n"
"    --[[MP_MAP]]\n"
"    [0xde] = function(data) return decode_map(data, decode_u16(data)) end;\n"
"    [0xdf] = function(data) return decode_map(data, decode_u32(data)) end;\n"
"}\n"
"\n"
"decode_r = function(data)\n"
"    local c = data[0][0]\n"
"    data[0] = data[0] + 1\n"
"    if c <= 0x7f then\n"
"        return tonumber(c) -- fixint\n"
"    elseif c >= 0xa0 and c <= 0xbf then\n"
"        return decode_str(data, bit.band(c, 0x1f)) -- fixstr\n"
"    elseif c >= 0x90 and c <= 0x9f then\n"
"        return decode_array(data, bit.band(c, 0xf)) -- fixarray\n"
"    elseif c >= 0x80 and c <= 0x8f then\n"
"        return decode_map(data, bit.band(c, 0xf)) -- fixmap\n"
"    elseif c >= 0xe0 then\n"
"        return tonumber(ffi.cast('signed char',c)) -- negfixint\n"
"    elseif c == 0xc0 then\n"
"        return msgpack.NULL\n"
"    elseif c == 0xc2 then\n"
"        return false\n"
"    elseif c == 0xc3 then\n"
"        return true\n"
"    elseif c >= 0xd4 and c <= 0xd8 or c >= 0xc7 and c <= 0xc9 then\n"
"        return decode_ext(data)\n"
"    else\n"
"        local fun = decoder_hint[c];\n"
"        assert (type(fun) == \"function\")\n"
"        return fun(data)\n"
"    end\n"
"end\n"
"\n"
"---\n"
"-- All decode_XXX functions accept const char **data as its first argument,\n"
"-- like libmsgpuck does. After decoding data[0] position is changed to the next\n"
"-- element. It is significally faster on LuaJIT to use double pointer than\n"
"-- return result, newpos.\n"
"--\n"
"\n"
"local function check_offset(offset, len)\n"
"    if offset == nil then\n"
"        return 1\n"
"    end\n"
"    local offset = ffi.cast('ptrdiff_t', offset)\n"
"    if offset < 1 or offset > len then\n"
"        error(string.format(\"offset = %d is out of bounds [1..%d]\",\n"
"            tonumber(offset), len))\n"
"    end\n"
"    return offset\n"
"end\n"
"\n"
"-- decode_unchecked(str, offset) -> res, new_offset\n"
"-- decode_unchecked(buf) -> res, new_buf\n"
"local function decode_unchecked(str, offset)\n"
"    if type(str) == \"string\" then\n"
"        offset = check_offset(offset, #str)\n"
"        local buf = ffi.cast(char_ptr_t, str)\n"
"        local bufp = ffi.new('const unsigned char *[1]', buf + offset - 1)\n"
"        local r = decode_r(bufp)\n"
"        return r, bufp[0] - buf + 1\n"
"    elseif ffi.istype(char_ptr_t, str) then\n"
"        -- Note: ffi.istype() ignores the const qualifier, so both\n"
"        -- (char *) and (const char *) buffers are valid.\n"
"        local bufp = ffi.new('const unsigned char *[1]', str)\n"
"        local r = decode_r(bufp)\n"
"        return r, ffi.cast(ffi.typeof(str), bufp[0])\n"
"    else\n"
"        error(\"msgpackffi.decode_unchecked(str, offset) -> res, new_offset | \"..\n"
"              \"msgpackffi.decode_unchecked([const] char *buf) -> res, new_buf\")\n"
"    end\n"
"end\n"
"\n"
"--------------------------------------------------------------------------------\n"
"-- exports\n"
"--------------------------------------------------------------------------------\n"
"\n"
"return {\n"
"    NULL = msgpack.NULL;\n"
"    array_mt = msgpack.array_mt;\n"
"    map_mt = msgpack.map_mt;\n"
"    encode = encode;\n"
"    on_encode = on_encode;\n"
"    decode_unchecked = decode_unchecked;\n"
"    decode = decode_unchecked; -- just for tests\n"
"    internal = {\n"
"        encode_fix = encode_fix;\n"
"        encode_array = encode_array;\n"
"        encode_r = encode_r;\n"
"    }\n"
"}\n"
""
;
