use glib::Enum;
use serde::{Deserialize, Serialize};
use std::fmt::Display;
use std::slice::Iter;

use crate::themes::ThemeInfo;

#[derive(Clone, Default, Debug, Serialize, Deserialize, Eq, PartialEq, Copy, Enum)]
#[repr(u32)]
#[enum_type(name = "NewsFlashTheme")]
pub enum NewsflashTheme {
    #[default]
    Default,
    Spring,
    Midnight,
    Parchment,
    Gruvbox,
    Zorin,
}

impl From<&str> for NewsflashTheme {
    fn from(value: &str) -> Self {
        match value {
            "theme dark" | "theme default" | "dark" | "default" | "adwaita" => Self::Default,
            "spring" | "theme spring" => Self::Spring,
            "midnight" | "theme midnight" => Self::Midnight,
            "parchment" | "theme parchment" => Self::Parchment,
            "gruvbox" | "theme gruvbox" => Self::Gruvbox,
            "zorin" | "theme zorin" => Self::Zorin,

            _ => Self::Default,
        }
    }
}

impl Display for NewsflashTheme {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            NewsflashTheme::Default => write!(f, "Adwaita"),
            NewsflashTheme::Spring => write!(f, "Spring"),
            NewsflashTheme::Midnight => write!(f, "Midnight"),
            NewsflashTheme::Parchment => write!(f, "Parchment"),
            NewsflashTheme::Gruvbox => write!(f, "Gruvbox"),
            NewsflashTheme::Zorin => write!(f, "Zorin"),
        }
    }
}

impl NewsflashTheme {
    pub fn as_str(&self) -> &'static str {
        match self {
            Self::Default => "theme default",
            Self::Spring => "theme spring",
            Self::Midnight => "theme midnight",
            Self::Parchment => "theme parchment",
            Self::Gruvbox => "theme gruvbox",
            Self::Zorin => "theme zorin",
        }
    }

    pub fn info(&self) -> ThemeInfo {
        ThemeInfo::get(self)
    }

    pub fn iterator() -> Iter<'static, NewsflashTheme> {
        static THEMES: [NewsflashTheme; 6] = [
            NewsflashTheme::Default,
            NewsflashTheme::Spring,
            NewsflashTheme::Midnight,
            NewsflashTheme::Parchment,
            NewsflashTheme::Gruvbox,
            NewsflashTheme::Zorin,
        ];
        THEMES.iter()
    }
}
