use news_flash::models::{CategoryID, FeedMapping};
use serde::{Deserialize, Serialize};
use std::fmt;

#[derive(Eq, Clone, Debug, Hash, Serialize, Deserialize)]
pub enum FeedListItemID {
    Category(CategoryID),
    Feed(FeedMapping),
}

impl FeedListItemID {
    pub fn is_category(&self) -> bool {
        matches!(self, FeedListItemID::Category(..))
    }

    pub fn is_feed(&self) -> bool {
        matches!(self, FeedListItemID::Feed(..))
    }
}

impl fmt::Display for FeedListItemID {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Category(id) => write!(f, "{id}"),
            Self::Feed(mapping) => write!(f, "{} - parent {}", mapping.feed_id, mapping.category_id),
        }
    }
}

impl PartialEq for FeedListItemID {
    fn eq(&self, other: &FeedListItemID) -> bool {
        match other {
            FeedListItemID::Category(other_category) => match self {
                FeedListItemID::Category(self_category) => other_category == self_category,
                FeedListItemID::Feed(_) => false,
            },
            FeedListItemID::Feed(other_mapping) => match self {
                FeedListItemID::Feed(self_mapping) => other_mapping == self_mapping,
                FeedListItemID::Category(_) => false,
            },
        }
    }
}

impl PartialEq<FeedMapping> for FeedListItemID {
    fn eq(&self, other: &FeedMapping) -> bool {
        match self {
            FeedListItemID::Category(_) => false,
            FeedListItemID::Feed(mapping) => mapping == other,
        }
    }
}

impl PartialEq<CategoryID> for FeedListItemID {
    fn eq(&self, other: &CategoryID) -> bool {
        match self {
            FeedListItemID::Category(category_id) => category_id == other,
            FeedListItemID::Feed(_) => false,
        }
    }
}
