use crate::app::App;
use eyre::{Result, eyre};
use gio::Cancellable;
use glib::{Object, Properties, clone, prelude::*, subclass::prelude::*, subclass::*};
use gtk4::{
    Adjustment, Allocation, CompositeTemplate, Grid, Orientation, Scrollable, ScrollablePolicy, SizeRequestMode,
    Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::{Bin, Clamp, prelude::*};
use std::cell::{Cell, RefCell};
use std::marker::PhantomData;
use std::time::Duration;
use webkit6::{WebView, prelude::*};

mod imp {
    use super::*;

    #[derive(Default, Debug, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ArticleScrollable)]
    #[template(file = "data/resources/ui_templates/article_view/scrollable.blp")]
    pub struct ArticleScrollable {
        #[template_child]
        pub clamp: TemplateChild<Clamp>,
        #[template_child]
        pub grid: TemplateChild<Grid>,
        #[template_child]
        pub header_bin: TemplateChild<Bin>,
        #[template_child]
        pub content_bin: TemplateChild<Bin>,

        #[property(override_interface = Scrollable, get, set = Self::set_vadjustment)]
        pub(super) vadjustment: RefCell<Adjustment>,

        #[property(override_interface = Scrollable, get, set = Self::set_hadjustment)]
        pub(super) hadjustment: RefCell<Adjustment>,

        #[property(override_interface = Scrollable, get = Self::scroll_policy, set = Self::set_ignore_scroll_policy)]
        pub(super) _vscroll_policy: PhantomData<ScrollablePolicy>,

        #[property(override_interface = Scrollable, get = Self::scroll_policy, set = Self::set_ignore_scroll_policy)]
        pub(super) _hscroll_policy: PhantomData<ScrollablePolicy>,

        #[property(get, set = Self::set_header, nullable)]
        pub header: RefCell<Option<Widget>>,

        #[property(get, set = Self::set_content, nullable)]
        pub content: RefCell<Option<WebView>>,

        #[property(get)]
        pub header_height: Cell<f64>,

        #[property(get)]
        pub content_height: Cell<i32>,

        #[property(get, set)]
        pub header_visible: Cell<bool>,

        #[property(get, set)]
        pub content_visible: Cell<bool>,

        #[property(get, set, name = "content-width")]
        pub content_width: Cell<i32>,

        #[property(get, set = Self::set_is_fullscreen)]
        pub is_fullscreen: Cell<bool>,

        pub prev_height: Cell<i32>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleScrollable {
        const NAME: &'static str = "ArticleScrollable";
        type ParentType = Widget;
        type Type = super::ArticleScrollable;
        type Interfaces = (Scrollable,);

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleScrollable {
        fn constructed(&self) {
            App::default()
                .settings()
                .article_view()
                .bind_property("content-width", &*self.obj(), "content-width")
                .sync_create()
                .build();
        }
    }

    impl WidgetImpl for ArticleScrollable {
        fn snapshot(&self, snapshot: &gtk4::Snapshot) {
            let obj = self.obj();

            snapshot.push_clip(&gtk4::graphene::Rect::new(
                0.0,
                0.0,
                obj.width() as f32,
                obj.height() as f32,
            ));
            self.parent_snapshot(snapshot);
            snapshot.pop();
        }

        fn compute_expand(&self, hexpand: &mut bool, vexpand: &mut bool) {
            *hexpand = true;
            *vexpand = true;
        }

        fn request_mode(&self) -> SizeRequestMode {
            SizeRequestMode::HeightForWidth
        }

        fn size_allocate(&self, width: i32, height: i32, baseline: i32) {
            let value = self.vadjustment.borrow().value();
            let header_height = self.header_height.get();
            let clamped = f64::min(value, header_height);

            let current_height = self.obj().height();
            let prev_height = self.prev_height.get();
            if prev_height > 0 && prev_height != current_height {
                self.update_size(None);
            }
            self.prev_height.set(current_height);

            self.clamp
                .size_allocate(&Allocation::new(0, -clamped as i32, width, height + clamped as i32), -1);
            self.parent_size_allocate(width, height, baseline);
        }

        fn measure(&self, orientation: Orientation, for_size: i32) -> (i32, i32, i32, i32) {
            self.clamp.measure(orientation, for_size)
        }
    }

    impl ScrollableImpl for ArticleScrollable {}

    impl ArticleScrollable {
        fn set_is_fullscreen(&self, is_fullscreen: bool) {
            if is_fullscreen {
                self.clamp.set_tightening_threshold(999999);
                self.clamp.set_maximum_size(999999);
                self.clamp.remove_css_class("article-clamp");
            } else {
                let content_width = self.content_width.get();
                self.clamp.set_tightening_threshold(content_width);
                self.clamp.set_maximum_size(content_width);
                self.clamp.add_css_class("article-clamp");
            }
            self.is_fullscreen.set(is_fullscreen);
        }

        fn set_header(&self, header: Option<Widget>) {
            self.header_bin.set_child(header.as_ref());
            self.header.replace(header);
        }

        fn set_content(&self, content: Option<WebView>) {
            self.content_bin.set_child(content.as_ref());
            self.content.replace(content);
        }

        pub fn set_vadjustment(&self, vadjustment: Option<Adjustment>) {
            let adjustment = vadjustment.unwrap_or_default();
            adjustment.connect_value_changed(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |adj| {
                    let header_height = imp.header_height.get();
                    let value = adj.value();
                    let clamped = f64::max(0.0, value - header_height);
                    imp.set_webview_scroll_pos(clamped, false);
                    imp.obj().queue_allocate();
                }
            ));
            self.vadjustment.replace(adjustment);
        }

        pub fn set_hadjustment(&self, hadjustment: Option<Adjustment>) {
            let adjustment = hadjustment.unwrap_or_default();
            self.hadjustment.replace(adjustment);
        }

        pub fn set_ignore_scroll_policy(&self, scroll_policy: ScrollablePolicy) {
            log::error!("Ignored setting new scroll policy {scroll_policy:?}");
        }

        pub fn scroll_policy(&self) -> ScrollablePolicy {
            ScrollablePolicy::Minimum
        }

        pub fn update_size(&self, content_height: Option<i32>) {
            let is_resize = content_height.is_none();
            let content_height = content_height.unwrap_or(self.content_height.get());
            let view_height = self.obj().height();

            if let Some(content) = self.content.borrow().as_ref() {
                let clamped = i32::min(content_height, view_height);
                if content.height_request() != clamped {
                    content.set_height_request(clamped);
                }
            }

            glib::timeout_add_local_once(
                Duration::from_millis(300),
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    move || {
                        let header_height = imp.header.borrow().as_ref().map(|header| header.height()).unwrap_or(0);
                        imp.header_height.set(header_height as f64);
                        imp.content_height.set(content_height);

                        let zoom = imp.content.borrow().as_ref().map(WebViewExt::zoom_level).unwrap_or(1.0);
                        let upper = header_height + content_height;
                        let page_size = i32::min(upper, view_height);
                        let page_size = (page_size as f64 / zoom) as i32;

                        log::debug!("widget height: {view_height}");
                        log::debug!("header height: {header_height}");
                        log::debug!("webview height: {content_height}");
                        log::debug!("upper: {upper}");
                        log::debug!("page_size: {page_size}");

                        imp.vadjustment.borrow().set_upper(upper as f64);
                        imp.vadjustment.borrow().set_page_size(page_size as f64);

                        let val = imp.vadjustment.borrow().value();

                        if is_resize && page_size <= upper && val > 0.0 {
                            let new_val = (upper - page_size) as f64;
                            if new_val != val {
                                log::warn!("value {val} set to {new_val}");
                                imp.vadjustment.borrow().set_value(new_val);
                            }
                        }
                    }
                ),
            );
        }

        fn set_webview_scroll_pos(&self, pos: f64, animate: bool) {
            let behavior = if animate { "smooth" } else { "instant" };
            let js = format!(
                "window.scrollTo({{
                    top: {pos},
                    behavior: \"{behavior}\",
                }});"
            );
            self.evaluate_js(&js);
        }

        fn evaluate_js(&self, js: &str) {
            if let Some(view) = self.content.borrow().as_ref() {
                view.evaluate_javascript(js, None, None, Cancellable::NONE, |res| {
                    if let Err(error) = res {
                        log::error!("failed to evaluate js {error}");
                    }
                });
            }
        }

        pub(super) async fn webview_js_get_f64(&self, java_script: &str) -> Result<f64> {
            let Some(view) = self.content.borrow().clone() else {
                return Err(eyre!("no webview"));
            };

            let value = view.evaluate_javascript_future(java_script, None, None).await?;
            Ok(value.to_double())
        }
    }
}

glib::wrapper! {
    pub struct ArticleScrollable(ObjectSubclass<imp::ArticleScrollable>)
        @extends Widget,
        @implements Scrollable;
}

impl Default for ArticleScrollable {
    fn default() -> Self {
        Object::new::<Self>()
    }
}

impl ArticleScrollable {
    pub fn update_size(&self, upper: i32) {
        self.imp().update_size(Some(upper))
    }

    pub fn scroll_to_element_by_id(&self, id: &str) {
        let js = format!(
            r#"
            var node = document.getElementById('{id}');
            node.focus();
            node.getBoundingClientRect().top + window.scrollY - 100
        "#
        );

        glib::spawn_future_local(clone!(
            #[weak(rename_to = imp)]
            self.imp(),
            async move {
                if let Ok(y) = imp.webview_js_get_f64(&js).await {
                    log::debug!("element y: {y}");
                    imp.vadjustment.borrow().set_value(imp.header_height.get() + y);
                }
            }
        ));
    }
}
