/* src/services.rs
 *
 * Copyright 2025 Mission Center Developers
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

use parking_lot::Mutex;
use rayon::ThreadPool;
use std::num::NonZero;
use std::sync::atomic::{AtomicBool, Ordering};

pub type Service = types::services::Service;

pub type ServiceList = types::services::services_response::ServiceList;
pub type ServicesRequest = types::services::ServicesRequest;
pub type ServicesRequestKind = types::services::services_request::Request;
pub type ServicesResponse = types::services::ServicesResponse;
pub type ServiceResponseKind = types::services::services_response::Response;
pub type ServicesResponseError = types::services::ServicesResponseError;
pub type ServicesResponseErrorKind = types::services::services_response_error::Error;

pub trait ServiceCache {
    fn new() -> Self
    where
        Self: Sized;

    fn refresh(&mut self);
    fn cached_entries(&self) -> &[Service];
}

pub trait ServiceManager {
    fn new() -> Self
    where
        Self: Sized;

    fn logs(&self, id: &str, pid: Option<NonZero<u32>>) -> Option<String>;
    fn start(&self, id: &str);
    fn stop(&self, id: &str);
    fn restart(&self, id: &str);
    fn enable(&self, id: &str);
    fn disable(&self, id: &str);
}

pub struct ServicesHandler<SC, SM>
where
    SC: ServiceCache,
    SM: ServiceManager,
{
    pub(crate) services: Mutex<SC>,
    pub(crate) service_manager: SM,
    pub(crate) local_cache: Mutex<Vec<Service>>,
    refreshing: AtomicBool,
}

impl<SC, SM> ServicesHandler<SC, SM>
where
    SC: ServiceCache + Send,
    SM: ServiceManager + Send + Sync,
{
    pub fn new() -> Self {
        Self {
            services: Mutex::new(SC::new()),
            service_manager: SM::new(),
            local_cache: Mutex::new(Vec::new()),
            refreshing: AtomicBool::new(false),
        }
    }

    pub fn refresh(&self) {
        let mut services = self.services.lock();
        services.refresh();
        *self.local_cache.lock() = services.cached_entries().to_vec();
    }

    pub fn refresh_async(&'static self, thread_pool: &ThreadPool) {
        if self.refreshing.fetch_or(true, Ordering::AcqRel) {
            return;
        }

        thread_pool.spawn(move || {
            let start = std::time::Instant::now();

            self.refresh();
            self.refreshing.store(false, Ordering::Release);

            log::debug!("PERF: Refreshed services in {:?}", start.elapsed());
        });
    }
}
