{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GApplication@ is the core class for application support.
-- 
-- A @GApplication@ is the foundation of an application. It wraps some
-- low-level platform-specific services and is intended to act as the
-- foundation for higher-level application classes such as
-- @GtkApplication@ or @MxApplication@. In general, you should not use
-- this class outside of a higher level framework.
-- 
-- @GApplication@ provides convenient life-cycle management by maintaining
-- a \"use count\" for the primary application instance. The use count can
-- be changed using 'GI.Gio.Objects.Application.applicationHold' and
-- 'GI.Gio.Objects.Application.applicationRelease'. If it drops to zero, the application
-- exits. Higher-level classes such as @GtkApplication@ employ the use count
-- to ensure that the application stays alive as long as it has any opened
-- windows.
-- 
-- Another feature that @GApplication@ (optionally) provides is process
-- uniqueness. Applications can make use of this functionality by
-- providing a unique application ID. If given, only one application
-- with this ID can be running at a time per session. The session
-- concept is platform-dependent, but corresponds roughly to a graphical
-- desktop login. When your application is launched again, its
-- arguments are passed through platform communication to the already
-- running program. The already running instance of the program is
-- called the \"primary instance\"; for non-unique applications this is
-- always the current instance. On Linux, the D-Bus session bus
-- is used for communication.
-- 
-- The use of @GApplication@ differs from some other commonly-used
-- uniqueness libraries (such as libunique) in important ways. The
-- application is not expected to manually register itself and check
-- if it is the primary instance. Instead, the @/main()/@ function of a
-- @GApplication@ should do very little more than instantiating the
-- application instance, possibly connecting signal handlers, then
-- calling 'GI.Gio.Objects.Application.applicationRun'. All checks for uniqueness are done
-- internally. If the application is the primary instance then the
-- startup signal is emitted and the mainloop runs. If the application
-- is not the primary instance then a signal is sent to the primary
-- instance and 'GI.Gio.Objects.Application.applicationRun' promptly returns. See the code
-- examples below.
-- 
-- If used, the expected form of an application identifier is the
-- same as that of a
-- <https://dbus.freedesktop.org/doc/dbus-specification.html#message-protocol-names-bus D-Bus well-known bus name>.
-- Examples include: @com.example.MyApp@, @org.example.internal_apps.Calculator@,
-- @org._7_zip.Archiver@.
-- For details on valid application identifiers, see 'GI.Gio.Objects.Application.applicationIdIsValid'.
-- 
-- On Linux, the application identifier is claimed as a well-known bus name
-- on the user\'s session bus. This means that the uniqueness of your
-- application is scoped to the current session. It also means that your
-- application may provide additional services (through registration of other
-- object paths) at that bus name. The registration of these object paths
-- should be done with the shared GDBus session bus. Note that due to the
-- internal architecture of GDBus, method calls can be dispatched at any time
-- (even if a main loop is not running). For this reason, you must ensure that
-- any object paths that you wish to register are registered before t'GI.Gio.Objects.Application.Application'
-- attempts to acquire the bus name of your application (which happens in
-- 'GI.Gio.Objects.Application.applicationRegister'). Unfortunately, this means that you cannot
-- use [Application:isRemote]("GI.Gio.Objects.Application#g:attr:isRemote") to decide if you want to register
-- object paths.
-- 
-- @GApplication@ also implements the t'GI.Gio.Interfaces.ActionGroup.ActionGroup' and t'GI.Gio.Interfaces.ActionMap.ActionMap'
-- interfaces and lets you easily export actions by adding them with
-- 'GI.Gio.Interfaces.ActionMap.actionMapAddAction'. When invoking an action by calling
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupActivateAction' on the application, it is always
-- invoked in the primary instance. The actions are also exported on
-- the session bus, and GIO provides the t'GI.Gio.Objects.DBusActionGroup.DBusActionGroup' wrapper to
-- conveniently access them remotely. GIO provides a t'GI.Gio.Objects.DBusMenuModel.DBusMenuModel' wrapper
-- for remote access to exported t'GI.Gio.Objects.MenuModel.MenuModel's.
-- 
-- Note: Due to the fact that actions are exported on the session bus,
-- using @maybe@ parameters is not supported, since D-Bus does not support
-- @maybe@ types.
-- 
-- There is a number of different entry points into a @GApplication@:
-- 
-- * via \'Activate\' (i.e. just starting the application)
-- * via \'Open\' (i.e. opening some files)
-- * by handling a command-line
-- * via activating an action
-- 
-- 
-- The [Application::startup]("GI.Gio.Objects.Application#g:signal:startup") signal lets you handle the application
-- initialization for all of these in a single place.
-- 
-- Regardless of which of these entry points is used to start the
-- application, @GApplication@ passes some ‘platform data’ from the
-- launching instance to the primary instance, in the form of a
-- t'GVariant' dictionary mapping strings to variants. To use platform
-- data, override the t'GI.Gio.Objects.Application.Application'.@/before_emit/@() or
-- t'GI.Gio.Objects.Application.Application'.@/after_emit/@() virtual functions
-- in your @GApplication@ subclass. When dealing with
-- t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine' objects, the platform data is
-- directly available via 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetCwd',
-- 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetEnviron' and
-- 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetPlatformData'.
-- 
-- As the name indicates, the platform data may vary depending on the
-- operating system, but it always includes the current directory (key
-- @cwd@), and optionally the environment (ie the set of environment
-- variables and their values) of the calling process (key @environ@).
-- The environment is only added to the platform data if the
-- @G_APPLICATION_SEND_ENVIRONMENT@ flag is set. @GApplication@ subclasses
-- can add their own platform data by overriding the
-- t'GI.Gio.Objects.Application.Application'.@/add_platform_data/@() virtual function. For instance,
-- @GtkApplication@ adds startup notification data in this way.
-- 
-- To parse commandline arguments you may handle the
-- [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") signal or override the
-- t'GI.Gio.Objects.Application.Application'.@/local_command_line/@() virtual function, to parse them in
-- either the primary instance or the local instance, respectively.
-- 
-- For an example of opening files with a @GApplication@, see
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-open.c gapplication-example-open.c>.
-- 
-- For an example of using actions with @GApplication@, see
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-actions.c gapplication-example-actions.c>.
-- 
-- For an example of using extra D-Bus hooks with @GApplication@, see
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-dbushooks.c gapplication-example-dbushooks.c>.
-- 
-- /Since: 2.28/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.Application
    ( 

-- * Exported types
    Application(..)                         ,
    IsApplication                           ,
    toApplication                           ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionAdded]("GI.Gio.Interfaces.ActionGroup#g:method:actionAdded"), [actionEnabledChanged]("GI.Gio.Interfaces.ActionGroup#g:method:actionEnabledChanged"), [actionRemoved]("GI.Gio.Interfaces.ActionGroup#g:method:actionRemoved"), [actionStateChanged]("GI.Gio.Interfaces.ActionGroup#g:method:actionStateChanged"), [activate]("GI.Gio.Objects.Application#g:method:activate"), [activateAction]("GI.Gio.Interfaces.ActionGroup#g:method:activateAction"), [addAction]("GI.Gio.Interfaces.ActionMap#g:method:addAction"), [addActionEntries]("GI.Gio.Interfaces.ActionMap#g:method:addActionEntries"), [addMainOption]("GI.Gio.Objects.Application#g:method:addMainOption"), [addMainOptionEntries]("GI.Gio.Objects.Application#g:method:addMainOptionEntries"), [addOptionGroup]("GI.Gio.Objects.Application#g:method:addOptionGroup"), [bindBusyProperty]("GI.Gio.Objects.Application#g:method:bindBusyProperty"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [changeActionState]("GI.Gio.Interfaces.ActionGroup#g:method:changeActionState"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasAction]("GI.Gio.Interfaces.ActionGroup#g:method:hasAction"), [hold]("GI.Gio.Objects.Application#g:method:hold"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [listActions]("GI.Gio.Interfaces.ActionGroup#g:method:listActions"), [lookupAction]("GI.Gio.Interfaces.ActionMap#g:method:lookupAction"), [markBusy]("GI.Gio.Objects.Application#g:method:markBusy"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [open]("GI.Gio.Objects.Application#g:method:open"), [queryAction]("GI.Gio.Interfaces.ActionGroup#g:method:queryAction"), [quit]("GI.Gio.Objects.Application#g:method:quit"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [register]("GI.Gio.Objects.Application#g:method:register"), [release]("GI.Gio.Objects.Application#g:method:release"), [removeAction]("GI.Gio.Interfaces.ActionMap#g:method:removeAction"), [removeActionEntries]("GI.Gio.Interfaces.ActionMap#g:method:removeActionEntries"), [run]("GI.Gio.Objects.Application#g:method:run"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [sendNotification]("GI.Gio.Objects.Application#g:method:sendNotification"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unbindBusyProperty]("GI.Gio.Objects.Application#g:method:unbindBusyProperty"), [unmarkBusy]("GI.Gio.Objects.Application#g:method:unmarkBusy"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure"), [withdrawNotification]("GI.Gio.Objects.Application#g:method:withdrawNotification").
-- 
-- ==== Getters
-- [getActionEnabled]("GI.Gio.Interfaces.ActionGroup#g:method:getActionEnabled"), [getActionParameterType]("GI.Gio.Interfaces.ActionGroup#g:method:getActionParameterType"), [getActionState]("GI.Gio.Interfaces.ActionGroup#g:method:getActionState"), [getActionStateHint]("GI.Gio.Interfaces.ActionGroup#g:method:getActionStateHint"), [getActionStateType]("GI.Gio.Interfaces.ActionGroup#g:method:getActionStateType"), [getApplicationId]("GI.Gio.Objects.Application#g:method:getApplicationId"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDbusConnection]("GI.Gio.Objects.Application#g:method:getDbusConnection"), [getDbusObjectPath]("GI.Gio.Objects.Application#g:method:getDbusObjectPath"), [getFlags]("GI.Gio.Objects.Application#g:method:getFlags"), [getInactivityTimeout]("GI.Gio.Objects.Application#g:method:getInactivityTimeout"), [getIsBusy]("GI.Gio.Objects.Application#g:method:getIsBusy"), [getIsRegistered]("GI.Gio.Objects.Application#g:method:getIsRegistered"), [getIsRemote]("GI.Gio.Objects.Application#g:method:getIsRemote"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getResourceBasePath]("GI.Gio.Objects.Application#g:method:getResourceBasePath"), [getVersion]("GI.Gio.Objects.Application#g:method:getVersion").
-- 
-- ==== Setters
-- [setActionGroup]("GI.Gio.Objects.Application#g:method:setActionGroup"), [setApplicationId]("GI.Gio.Objects.Application#g:method:setApplicationId"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDefault]("GI.Gio.Objects.Application#g:method:setDefault"), [setFlags]("GI.Gio.Objects.Application#g:method:setFlags"), [setInactivityTimeout]("GI.Gio.Objects.Application#g:method:setInactivityTimeout"), [setOptionContextDescription]("GI.Gio.Objects.Application#g:method:setOptionContextDescription"), [setOptionContextParameterString]("GI.Gio.Objects.Application#g:method:setOptionContextParameterString"), [setOptionContextSummary]("GI.Gio.Objects.Application#g:method:setOptionContextSummary"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setResourceBasePath]("GI.Gio.Objects.Application#g:method:setResourceBasePath"), [setVersion]("GI.Gio.Objects.Application#g:method:setVersion").

#if defined(ENABLE_OVERLOADING)
    ResolveApplicationMethod                ,
#endif

-- ** activate #method:activate#

#if defined(ENABLE_OVERLOADING)
    ApplicationActivateMethodInfo           ,
#endif
    applicationActivate                     ,


-- ** addMainOption #method:addMainOption#

#if defined(ENABLE_OVERLOADING)
    ApplicationAddMainOptionMethodInfo      ,
#endif
    applicationAddMainOption                ,


-- ** addMainOptionEntries #method:addMainOptionEntries#

#if defined(ENABLE_OVERLOADING)
    ApplicationAddMainOptionEntriesMethodInfo,
#endif
    applicationAddMainOptionEntries         ,


-- ** addOptionGroup #method:addOptionGroup#

#if defined(ENABLE_OVERLOADING)
    ApplicationAddOptionGroupMethodInfo     ,
#endif
    applicationAddOptionGroup               ,


-- ** bindBusyProperty #method:bindBusyProperty#

#if defined(ENABLE_OVERLOADING)
    ApplicationBindBusyPropertyMethodInfo   ,
#endif
    applicationBindBusyProperty             ,


-- ** getApplicationId #method:getApplicationId#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetApplicationIdMethodInfo   ,
#endif
    applicationGetApplicationId             ,


-- ** getDbusConnection #method:getDbusConnection#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetDbusConnectionMethodInfo  ,
#endif
    applicationGetDbusConnection            ,


-- ** getDbusObjectPath #method:getDbusObjectPath#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetDbusObjectPathMethodInfo  ,
#endif
    applicationGetDbusObjectPath            ,


-- ** getDefault #method:getDefault#

    applicationGetDefault                   ,


-- ** getFlags #method:getFlags#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetFlagsMethodInfo           ,
#endif
    applicationGetFlags                     ,


-- ** getInactivityTimeout #method:getInactivityTimeout#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetInactivityTimeoutMethodInfo,
#endif
    applicationGetInactivityTimeout         ,


-- ** getIsBusy #method:getIsBusy#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetIsBusyMethodInfo          ,
#endif
    applicationGetIsBusy                    ,


-- ** getIsRegistered #method:getIsRegistered#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetIsRegisteredMethodInfo    ,
#endif
    applicationGetIsRegistered              ,


-- ** getIsRemote #method:getIsRemote#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetIsRemoteMethodInfo        ,
#endif
    applicationGetIsRemote                  ,


-- ** getResourceBasePath #method:getResourceBasePath#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetResourceBasePathMethodInfo,
#endif
    applicationGetResourceBasePath          ,


-- ** getVersion #method:getVersion#

#if defined(ENABLE_OVERLOADING)
    ApplicationGetVersionMethodInfo         ,
#endif
    applicationGetVersion                   ,


-- ** hold #method:hold#

#if defined(ENABLE_OVERLOADING)
    ApplicationHoldMethodInfo               ,
#endif
    applicationHold                         ,


-- ** idIsValid #method:idIsValid#

    applicationIdIsValid                    ,


-- ** markBusy #method:markBusy#

#if defined(ENABLE_OVERLOADING)
    ApplicationMarkBusyMethodInfo           ,
#endif
    applicationMarkBusy                     ,


-- ** new #method:new#

    applicationNew                          ,


-- ** open #method:open#

#if defined(ENABLE_OVERLOADING)
    ApplicationOpenMethodInfo               ,
#endif
    applicationOpen                         ,


-- ** quit #method:quit#

#if defined(ENABLE_OVERLOADING)
    ApplicationQuitMethodInfo               ,
#endif
    applicationQuit                         ,


-- ** register #method:register#

#if defined(ENABLE_OVERLOADING)
    ApplicationRegisterMethodInfo           ,
#endif
    applicationRegister                     ,


-- ** release #method:release#

#if defined(ENABLE_OVERLOADING)
    ApplicationReleaseMethodInfo            ,
#endif
    applicationRelease                      ,


-- ** run #method:run#

#if defined(ENABLE_OVERLOADING)
    ApplicationRunMethodInfo                ,
#endif
    applicationRun                          ,


-- ** sendNotification #method:sendNotification#

#if defined(ENABLE_OVERLOADING)
    ApplicationSendNotificationMethodInfo   ,
#endif
    applicationSendNotification             ,


-- ** setActionGroup #method:setActionGroup#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetActionGroupMethodInfo     ,
#endif
    applicationSetActionGroup               ,


-- ** setApplicationId #method:setApplicationId#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetApplicationIdMethodInfo   ,
#endif
    applicationSetApplicationId             ,


-- ** setDefault #method:setDefault#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetDefaultMethodInfo         ,
#endif
    applicationSetDefault                   ,


-- ** setFlags #method:setFlags#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetFlagsMethodInfo           ,
#endif
    applicationSetFlags                     ,


-- ** setInactivityTimeout #method:setInactivityTimeout#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetInactivityTimeoutMethodInfo,
#endif
    applicationSetInactivityTimeout         ,


-- ** setOptionContextDescription #method:setOptionContextDescription#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetOptionContextDescriptionMethodInfo,
#endif
    applicationSetOptionContextDescription  ,


-- ** setOptionContextParameterString #method:setOptionContextParameterString#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetOptionContextParameterStringMethodInfo,
#endif
    applicationSetOptionContextParameterString,


-- ** setOptionContextSummary #method:setOptionContextSummary#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetOptionContextSummaryMethodInfo,
#endif
    applicationSetOptionContextSummary      ,


-- ** setResourceBasePath #method:setResourceBasePath#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetResourceBasePathMethodInfo,
#endif
    applicationSetResourceBasePath          ,


-- ** setVersion #method:setVersion#

#if defined(ENABLE_OVERLOADING)
    ApplicationSetVersionMethodInfo         ,
#endif
    applicationSetVersion                   ,


-- ** unbindBusyProperty #method:unbindBusyProperty#

#if defined(ENABLE_OVERLOADING)
    ApplicationUnbindBusyPropertyMethodInfo ,
#endif
    applicationUnbindBusyProperty           ,


-- ** unmarkBusy #method:unmarkBusy#

#if defined(ENABLE_OVERLOADING)
    ApplicationUnmarkBusyMethodInfo         ,
#endif
    applicationUnmarkBusy                   ,


-- ** withdrawNotification #method:withdrawNotification#

#if defined(ENABLE_OVERLOADING)
    ApplicationWithdrawNotificationMethodInfo,
#endif
    applicationWithdrawNotification         ,




 -- * Properties


-- ** actionGroup #attr:actionGroup#
-- | The group of actions that the application exports.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationActionGroupPropertyInfo      ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationActionGroup                  ,
#endif
    clearApplicationActionGroup             ,
    constructApplicationActionGroup         ,
    setApplicationActionGroup               ,


-- ** applicationId #attr:applicationId#
-- | The unique identifier for the application.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationApplicationIdPropertyInfo    ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationApplicationId                ,
#endif
    clearApplicationApplicationId           ,
    constructApplicationApplicationId       ,
    getApplicationApplicationId             ,
    setApplicationApplicationId             ,


-- ** flags #attr:flags#
-- | Flags specifying the behaviour of the application.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationFlagsPropertyInfo            ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationFlags                        ,
#endif
    constructApplicationFlags               ,
    getApplicationFlags                     ,
    setApplicationFlags                     ,


-- ** inactivityTimeout #attr:inactivityTimeout#
-- | Time (in milliseconds) to stay alive after becoming idle.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationInactivityTimeoutPropertyInfo,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationInactivityTimeout            ,
#endif
    constructApplicationInactivityTimeout   ,
    getApplicationInactivityTimeout         ,
    setApplicationInactivityTimeout         ,


-- ** isBusy #attr:isBusy#
-- | Whether the application is currently marked as busy through
-- 'GI.Gio.Objects.Application.applicationMarkBusy' or 'GI.Gio.Objects.Application.applicationBindBusyProperty'.
-- 
-- /Since: 2.44/

#if defined(ENABLE_OVERLOADING)
    ApplicationIsBusyPropertyInfo           ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationIsBusy                       ,
#endif
    getApplicationIsBusy                    ,


-- ** isRegistered #attr:isRegistered#
-- | Whether 'GI.Gio.Objects.Application.applicationRegister' has been called.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationIsRegisteredPropertyInfo     ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationIsRegistered                 ,
#endif
    getApplicationIsRegistered              ,


-- ** isRemote #attr:isRemote#
-- | Whether this application instance is remote.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationIsRemotePropertyInfo         ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationIsRemote                     ,
#endif
    getApplicationIsRemote                  ,


-- ** resourceBasePath #attr:resourceBasePath#
-- | The base resource path for the application.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ApplicationResourceBasePathPropertyInfo ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationResourceBasePath             ,
#endif
    clearApplicationResourceBasePath        ,
    constructApplicationResourceBasePath    ,
    getApplicationResourceBasePath          ,
    setApplicationResourceBasePath          ,


-- ** version #attr:version#
-- | The human-readable version number of the application.
-- 
-- /Since: 2.80/

#if defined(ENABLE_OVERLOADING)
    ApplicationVersionPropertyInfo          ,
#endif
#if defined(ENABLE_OVERLOADING)
    applicationVersion                      ,
#endif
    constructApplicationVersion             ,
    getApplicationVersion                   ,
    setApplicationVersion                   ,




 -- * Signals


-- ** activate #signal:activate#

    ApplicationActivateCallback             ,
#if defined(ENABLE_OVERLOADING)
    ApplicationActivateSignalInfo           ,
#endif
    afterApplicationActivate                ,
    onApplicationActivate                   ,


-- ** commandLine #signal:commandLine#

    ApplicationCommandLineCallback          ,
#if defined(ENABLE_OVERLOADING)
    ApplicationCommandLineSignalInfo        ,
#endif
    afterApplicationCommandLine             ,
    onApplicationCommandLine                ,


-- ** handleLocalOptions #signal:handleLocalOptions#

    ApplicationHandleLocalOptionsCallback   ,
#if defined(ENABLE_OVERLOADING)
    ApplicationHandleLocalOptionsSignalInfo ,
#endif
    afterApplicationHandleLocalOptions      ,
    onApplicationHandleLocalOptions         ,


-- ** nameLost #signal:nameLost#

    ApplicationNameLostCallback             ,
#if defined(ENABLE_OVERLOADING)
    ApplicationNameLostSignalInfo           ,
#endif
    afterApplicationNameLost                ,
    onApplicationNameLost                   ,


-- ** open #signal:open#

    ApplicationOpenCallback                 ,
#if defined(ENABLE_OVERLOADING)
    ApplicationOpenSignalInfo               ,
#endif
    afterApplicationOpen                    ,
    onApplicationOpen                       ,


-- ** shutdown #signal:shutdown#

    ApplicationShutdownCallback             ,
#if defined(ENABLE_OVERLOADING)
    ApplicationShutdownSignalInfo           ,
#endif
    afterApplicationShutdown                ,
    onApplicationShutdown                   ,


-- ** startup #signal:startup#

    ApplicationStartupCallback              ,
#if defined(ENABLE_OVERLOADING)
    ApplicationStartupSignalInfo            ,
#endif
    afterApplicationStartup                 ,
    onApplicationStartup                    ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Enums as GLib.Enums
import qualified GI.GLib.Flags as GLib.Flags
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.DateTime as GLib.DateTime
import qualified GI.GLib.Structs.OptionEntry as GLib.OptionEntry
import qualified GI.GLib.Structs.OptionGroup as GLib.OptionGroup
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import qualified GI.GLib.Structs.VariantDict as GLib.VariantDict
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Action as Gio.Action
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionMap as Gio.ActionMap
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AppInfo as Gio.AppInfo
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncInitable as Gio.AsyncInitable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Drive as Gio.Drive
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Icon as Gio.Icon
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Mount as Gio.Mount
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Seekable as Gio.Seekable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Volume as Gio.Volume
import {-# SOURCE #-} qualified GI.Gio.Objects.AppLaunchContext as Gio.AppLaunchContext
import {-# SOURCE #-} qualified GI.Gio.Objects.ApplicationCommandLine as Gio.ApplicationCommandLine
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusAuthObserver as Gio.DBusAuthObserver
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusMessage as Gio.DBusMessage
import {-# SOURCE #-} qualified GI.Gio.Objects.FileEnumerator as Gio.FileEnumerator
import {-# SOURCE #-} qualified GI.Gio.Objects.FileIOStream as Gio.FileIOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInfo as Gio.FileInfo
import {-# SOURCE #-} qualified GI.Gio.Objects.FileInputStream as Gio.FileInputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.FileMonitor as Gio.FileMonitor
import {-# SOURCE #-} qualified GI.Gio.Objects.FileOutputStream as Gio.FileOutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuAttributeIter as Gio.MenuAttributeIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuLinkIter as Gio.MenuLinkIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import {-# SOURCE #-} qualified GI.Gio.Objects.MountOperation as Gio.MountOperation
import {-# SOURCE #-} qualified GI.Gio.Objects.Notification as Gio.Notification
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.UnixFDList as Gio.UnixFDList
import {-# SOURCE #-} qualified GI.Gio.Structs.ActionEntry as Gio.ActionEntry
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusAnnotationInfo as Gio.DBusAnnotationInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusArgInfo as Gio.DBusArgInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusInterfaceInfo as Gio.DBusInterfaceInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusMethodInfo as Gio.DBusMethodInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusPropertyInfo as Gio.DBusPropertyInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSignalInfo as Gio.DBusSignalInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSubtreeVTable as Gio.DBusSubtreeVTable
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfo as Gio.FileAttributeInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeInfoList as Gio.FileAttributeInfoList
import {-# SOURCE #-} qualified GI.Gio.Structs.FileAttributeMatcher as Gio.FileAttributeMatcher
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GLib.Enums as GLib.Enums
import qualified GI.GLib.Flags as GLib.Flags
import qualified GI.GLib.Structs.OptionEntry as GLib.OptionEntry
import qualified GI.GLib.Structs.OptionGroup as GLib.OptionGroup
import qualified GI.GLib.Structs.VariantDict as GLib.VariantDict
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionMap as Gio.ActionMap
import {-# SOURCE #-} qualified GI.Gio.Interfaces.File as Gio.File
import {-# SOURCE #-} qualified GI.Gio.Objects.ApplicationCommandLine as Gio.ApplicationCommandLine
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.Notification as Gio.Notification

#endif

-- | Memory-managed wrapper type.
newtype Application = Application (SP.ManagedPtr Application)
    deriving (Eq)

instance SP.ManagedPtrNewtype Application where
    toManagedPtr (Application p) = p

foreign import ccall "g_application_get_type"
    c_g_application_get_type :: IO B.Types.GType

instance B.Types.TypedObject Application where
    glibType = c_g_application_get_type

instance B.Types.GObject Application

-- | Type class for types which can be safely cast to t'Application', for instance with `toApplication`.
class (SP.GObject o, O.IsDescendantOf Application o) => IsApplication o
instance (SP.GObject o, O.IsDescendantOf Application o) => IsApplication o

instance O.HasParentTypes Application
type instance O.ParentTypes Application = '[GObject.Object.Object, Gio.ActionGroup.ActionGroup, Gio.ActionMap.ActionMap]

-- | Cast to t'Application', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toApplication :: (MIO.MonadIO m, IsApplication o) => o -> m Application
toApplication = MIO.liftIO . B.ManagedPtr.unsafeCastTo Application

-- | Convert t'Application' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Application) where
    gvalueGType_ = c_g_application_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Application)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Application)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Application ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveApplicationMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveApplicationMethod "actionAdded" o = Gio.ActionGroup.ActionGroupActionAddedMethodInfo
    ResolveApplicationMethod "actionEnabledChanged" o = Gio.ActionGroup.ActionGroupActionEnabledChangedMethodInfo
    ResolveApplicationMethod "actionRemoved" o = Gio.ActionGroup.ActionGroupActionRemovedMethodInfo
    ResolveApplicationMethod "actionStateChanged" o = Gio.ActionGroup.ActionGroupActionStateChangedMethodInfo
    ResolveApplicationMethod "activate" o = ApplicationActivateMethodInfo
    ResolveApplicationMethod "activateAction" o = Gio.ActionGroup.ActionGroupActivateActionMethodInfo
    ResolveApplicationMethod "addAction" o = Gio.ActionMap.ActionMapAddActionMethodInfo
    ResolveApplicationMethod "addActionEntries" o = Gio.ActionMap.ActionMapAddActionEntriesMethodInfo
    ResolveApplicationMethod "addMainOption" o = ApplicationAddMainOptionMethodInfo
    ResolveApplicationMethod "addMainOptionEntries" o = ApplicationAddMainOptionEntriesMethodInfo
    ResolveApplicationMethod "addOptionGroup" o = ApplicationAddOptionGroupMethodInfo
    ResolveApplicationMethod "bindBusyProperty" o = ApplicationBindBusyPropertyMethodInfo
    ResolveApplicationMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveApplicationMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveApplicationMethod "changeActionState" o = Gio.ActionGroup.ActionGroupChangeActionStateMethodInfo
    ResolveApplicationMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveApplicationMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveApplicationMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveApplicationMethod "hasAction" o = Gio.ActionGroup.ActionGroupHasActionMethodInfo
    ResolveApplicationMethod "hold" o = ApplicationHoldMethodInfo
    ResolveApplicationMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveApplicationMethod "listActions" o = Gio.ActionGroup.ActionGroupListActionsMethodInfo
    ResolveApplicationMethod "lookupAction" o = Gio.ActionMap.ActionMapLookupActionMethodInfo
    ResolveApplicationMethod "markBusy" o = ApplicationMarkBusyMethodInfo
    ResolveApplicationMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveApplicationMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveApplicationMethod "open" o = ApplicationOpenMethodInfo
    ResolveApplicationMethod "queryAction" o = Gio.ActionGroup.ActionGroupQueryActionMethodInfo
    ResolveApplicationMethod "quit" o = ApplicationQuitMethodInfo
    ResolveApplicationMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveApplicationMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveApplicationMethod "register" o = ApplicationRegisterMethodInfo
    ResolveApplicationMethod "release" o = ApplicationReleaseMethodInfo
    ResolveApplicationMethod "removeAction" o = Gio.ActionMap.ActionMapRemoveActionMethodInfo
    ResolveApplicationMethod "removeActionEntries" o = Gio.ActionMap.ActionMapRemoveActionEntriesMethodInfo
    ResolveApplicationMethod "run" o = ApplicationRunMethodInfo
    ResolveApplicationMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveApplicationMethod "sendNotification" o = ApplicationSendNotificationMethodInfo
    ResolveApplicationMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveApplicationMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveApplicationMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveApplicationMethod "unbindBusyProperty" o = ApplicationUnbindBusyPropertyMethodInfo
    ResolveApplicationMethod "unmarkBusy" o = ApplicationUnmarkBusyMethodInfo
    ResolveApplicationMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveApplicationMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveApplicationMethod "withdrawNotification" o = ApplicationWithdrawNotificationMethodInfo
    ResolveApplicationMethod "getActionEnabled" o = Gio.ActionGroup.ActionGroupGetActionEnabledMethodInfo
    ResolveApplicationMethod "getActionParameterType" o = Gio.ActionGroup.ActionGroupGetActionParameterTypeMethodInfo
    ResolveApplicationMethod "getActionState" o = Gio.ActionGroup.ActionGroupGetActionStateMethodInfo
    ResolveApplicationMethod "getActionStateHint" o = Gio.ActionGroup.ActionGroupGetActionStateHintMethodInfo
    ResolveApplicationMethod "getActionStateType" o = Gio.ActionGroup.ActionGroupGetActionStateTypeMethodInfo
    ResolveApplicationMethod "getApplicationId" o = ApplicationGetApplicationIdMethodInfo
    ResolveApplicationMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveApplicationMethod "getDbusConnection" o = ApplicationGetDbusConnectionMethodInfo
    ResolveApplicationMethod "getDbusObjectPath" o = ApplicationGetDbusObjectPathMethodInfo
    ResolveApplicationMethod "getFlags" o = ApplicationGetFlagsMethodInfo
    ResolveApplicationMethod "getInactivityTimeout" o = ApplicationGetInactivityTimeoutMethodInfo
    ResolveApplicationMethod "getIsBusy" o = ApplicationGetIsBusyMethodInfo
    ResolveApplicationMethod "getIsRegistered" o = ApplicationGetIsRegisteredMethodInfo
    ResolveApplicationMethod "getIsRemote" o = ApplicationGetIsRemoteMethodInfo
    ResolveApplicationMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveApplicationMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveApplicationMethod "getResourceBasePath" o = ApplicationGetResourceBasePathMethodInfo
    ResolveApplicationMethod "getVersion" o = ApplicationGetVersionMethodInfo
    ResolveApplicationMethod "setActionGroup" o = ApplicationSetActionGroupMethodInfo
    ResolveApplicationMethod "setApplicationId" o = ApplicationSetApplicationIdMethodInfo
    ResolveApplicationMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveApplicationMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveApplicationMethod "setDefault" o = ApplicationSetDefaultMethodInfo
    ResolveApplicationMethod "setFlags" o = ApplicationSetFlagsMethodInfo
    ResolveApplicationMethod "setInactivityTimeout" o = ApplicationSetInactivityTimeoutMethodInfo
    ResolveApplicationMethod "setOptionContextDescription" o = ApplicationSetOptionContextDescriptionMethodInfo
    ResolveApplicationMethod "setOptionContextParameterString" o = ApplicationSetOptionContextParameterStringMethodInfo
    ResolveApplicationMethod "setOptionContextSummary" o = ApplicationSetOptionContextSummaryMethodInfo
    ResolveApplicationMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveApplicationMethod "setResourceBasePath" o = ApplicationSetResourceBasePathMethodInfo
    ResolveApplicationMethod "setVersion" o = ApplicationSetVersionMethodInfo
    ResolveApplicationMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveApplicationMethod t Application, O.OverloadedMethod info Application p) => OL.IsLabel t (Application -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveApplicationMethod t Application, O.OverloadedMethod info Application p, R.HasField t Application p) => R.HasField t Application p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveApplicationMethod t Application, O.OverloadedMethodInfo info Application) => OL.IsLabel t (O.MethodProxy info Application) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Application::activate
-- | The [activate](#g:signal:activate) signal is emitted on the primary instance when an
-- activation occurs. See 'GI.Gio.Objects.Application.applicationActivate'.
type ApplicationActivateCallback =
    IO ()

type C_ApplicationActivateCallback =
    Ptr Application ->                      -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ApplicationActivateCallback`.
foreign import ccall "wrapper"
    mk_ApplicationActivateCallback :: C_ApplicationActivateCallback -> IO (FunPtr C_ApplicationActivateCallback)

wrap_ApplicationActivateCallback :: 
    GObject a => (a -> ApplicationActivateCallback) ->
    C_ApplicationActivateCallback
wrap_ApplicationActivateCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [activate](#signal:activate) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #activate callback
-- @
-- 
-- 
onApplicationActivate :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationActivateCallback) -> m SignalHandlerId
onApplicationActivate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationActivateCallback wrapped
    wrapped'' <- mk_ApplicationActivateCallback wrapped'
    connectSignalFunPtr obj "activate" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [activate](#signal:activate) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #activate callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationActivate :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationActivateCallback) -> m SignalHandlerId
afterApplicationActivate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationActivateCallback wrapped
    wrapped'' <- mk_ApplicationActivateCallback wrapped'
    connectSignalFunPtr obj "activate" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationActivateSignalInfo
instance SignalInfo ApplicationActivateSignalInfo where
    type HaskellCallbackType ApplicationActivateSignalInfo = ApplicationActivateCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationActivateCallback cb
        cb'' <- mk_ApplicationActivateCallback cb'
        connectSignalFunPtr obj "activate" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::activate"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:activate"})

#endif

-- signal Application::command-line
-- | The [commandLine](#g:signal:commandLine) signal is emitted on the primary instance when
-- a commandline is not handled locally. See 'GI.Gio.Objects.Application.applicationRun' and
-- the t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine' documentation for more information.
type ApplicationCommandLineCallback =
    Gio.ApplicationCommandLine.ApplicationCommandLine
    -- ^ /@commandLine@/: a t'GI.Gio.Objects.ApplicationCommandLine.ApplicationCommandLine' representing the
    --     passed commandline
    -> IO Int32
    -- ^ __Returns:__ An integer that is set as the exit status for the calling
    --   process. See 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineSetExitStatus'.

type C_ApplicationCommandLineCallback =
    Ptr Application ->                      -- object
    Ptr Gio.ApplicationCommandLine.ApplicationCommandLine ->
    Ptr () ->                               -- user_data
    IO Int32

-- | Generate a function pointer callable from C code, from a `C_ApplicationCommandLineCallback`.
foreign import ccall "wrapper"
    mk_ApplicationCommandLineCallback :: C_ApplicationCommandLineCallback -> IO (FunPtr C_ApplicationCommandLineCallback)

wrap_ApplicationCommandLineCallback :: 
    GObject a => (a -> ApplicationCommandLineCallback) ->
    C_ApplicationCommandLineCallback
wrap_ApplicationCommandLineCallback gi'cb gi'selfPtr commandLine _ = do
    commandLine' <- (newObject Gio.ApplicationCommandLine.ApplicationCommandLine) commandLine
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  commandLine'
    return result


-- | Connect a signal handler for the [commandLine](#signal:commandLine) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #commandLine callback
-- @
-- 
-- 
onApplicationCommandLine :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationCommandLineCallback) -> m SignalHandlerId
onApplicationCommandLine obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationCommandLineCallback wrapped
    wrapped'' <- mk_ApplicationCommandLineCallback wrapped'
    connectSignalFunPtr obj "command-line" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [commandLine](#signal:commandLine) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #commandLine callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationCommandLine :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationCommandLineCallback) -> m SignalHandlerId
afterApplicationCommandLine obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationCommandLineCallback wrapped
    wrapped'' <- mk_ApplicationCommandLineCallback wrapped'
    connectSignalFunPtr obj "command-line" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationCommandLineSignalInfo
instance SignalInfo ApplicationCommandLineSignalInfo where
    type HaskellCallbackType ApplicationCommandLineSignalInfo = ApplicationCommandLineCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationCommandLineCallback cb
        cb'' <- mk_ApplicationCommandLineCallback cb'
        connectSignalFunPtr obj "command-line" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::command-line"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:commandLine"})

#endif

-- signal Application::handle-local-options
-- | The [handleLocalOptions](#g:signal:handleLocalOptions) signal is emitted on the local instance
-- after the parsing of the commandline options has occurred.
-- 
-- You can add options to be recognised during commandline option
-- parsing using 'GI.Gio.Objects.Application.applicationAddMainOptionEntries' and
-- 'GI.Gio.Objects.Application.applicationAddOptionGroup'.
-- 
-- Signal handlers can inspect /@options@/ (along with values pointed to
-- from the /@argData@/ of an installed @/GOptionEntrys/@) in order to
-- decide to perform certain actions, including direct local handling
-- (which may be useful for options like --version).
-- 
-- In the event that the application is marked
-- 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' the \"normal processing\" will
-- send the /@options@/ dictionary to the primary instance where it can be
-- read with 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetOptionsDict'.  The signal
-- handler can modify the dictionary before returning, and the
-- modified dictionary will be sent.
-- 
-- In the event that 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' is not set,
-- \"normal processing\" will treat the remaining uncollected command
-- line arguments as filenames or URIs.  If there are no arguments,
-- the application is activated by 'GI.Gio.Objects.Application.applicationActivate'.  One or
-- more arguments results in a call to 'GI.Gio.Objects.Application.applicationOpen'.
-- 
-- If you want to handle the local commandline arguments for yourself
-- by converting them to calls to 'GI.Gio.Objects.Application.applicationOpen' or
-- 'GI.Gio.Interfaces.ActionGroup.actionGroupActivateAction' then you must be sure to register
-- the application first.  You should probably not call
-- 'GI.Gio.Objects.Application.applicationActivate' for yourself, however: just return -1 and
-- allow the default handler to do it for you.  This will ensure that
-- the @--gapplication-service@ switch works properly (i.e. no activation
-- in that case).
-- 
-- Note that this signal is emitted from the default implementation of
-- @/local_command_line()/@.  If you override that function and don\'t
-- chain up then this signal will never be emitted.
-- 
-- You can override @/local_command_line()/@ if you need more powerful
-- capabilities than what is provided here, but this should not
-- normally be required.
-- 
-- /Since: 2.40/
type ApplicationHandleLocalOptionsCallback =
    GLib.VariantDict.VariantDict
    -- ^ /@options@/: the options dictionary
    -> IO Int32
    -- ^ __Returns:__ an exit code. If you have handled your options and want
    -- to exit the process, return a non-negative option, 0 for success,
    -- and a positive value for failure. To continue, return -1 to let
    -- the default option processing continue.

type C_ApplicationHandleLocalOptionsCallback =
    Ptr Application ->                      -- object
    Ptr GLib.VariantDict.VariantDict ->
    Ptr () ->                               -- user_data
    IO Int32

-- | Generate a function pointer callable from C code, from a `C_ApplicationHandleLocalOptionsCallback`.
foreign import ccall "wrapper"
    mk_ApplicationHandleLocalOptionsCallback :: C_ApplicationHandleLocalOptionsCallback -> IO (FunPtr C_ApplicationHandleLocalOptionsCallback)

wrap_ApplicationHandleLocalOptionsCallback :: 
    GObject a => (a -> ApplicationHandleLocalOptionsCallback) ->
    C_ApplicationHandleLocalOptionsCallback
wrap_ApplicationHandleLocalOptionsCallback gi'cb gi'selfPtr options _ = do
    B.ManagedPtr.withTransient  options $ \options' -> do
        result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  options'
        return result


-- | Connect a signal handler for the [handleLocalOptions](#signal:handleLocalOptions) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #handleLocalOptions callback
-- @
-- 
-- 
onApplicationHandleLocalOptions :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationHandleLocalOptionsCallback) -> m SignalHandlerId
onApplicationHandleLocalOptions obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationHandleLocalOptionsCallback wrapped
    wrapped'' <- mk_ApplicationHandleLocalOptionsCallback wrapped'
    connectSignalFunPtr obj "handle-local-options" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [handleLocalOptions](#signal:handleLocalOptions) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #handleLocalOptions callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationHandleLocalOptions :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationHandleLocalOptionsCallback) -> m SignalHandlerId
afterApplicationHandleLocalOptions obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationHandleLocalOptionsCallback wrapped
    wrapped'' <- mk_ApplicationHandleLocalOptionsCallback wrapped'
    connectSignalFunPtr obj "handle-local-options" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationHandleLocalOptionsSignalInfo
instance SignalInfo ApplicationHandleLocalOptionsSignalInfo where
    type HaskellCallbackType ApplicationHandleLocalOptionsSignalInfo = ApplicationHandleLocalOptionsCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationHandleLocalOptionsCallback cb
        cb'' <- mk_ApplicationHandleLocalOptionsCallback cb'
        connectSignalFunPtr obj "handle-local-options" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::handle-local-options"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:handleLocalOptions"})

#endif

-- signal Application::name-lost
-- | The [nameLost](#g:signal:nameLost) signal is emitted only on the registered primary instance
-- when a new instance has taken over. This can only happen if the application
-- is using the 'GI.Gio.Flags.ApplicationFlagsAllowReplacement' flag.
-- 
-- The default handler for this signal calls 'GI.Gio.Objects.Application.applicationQuit'.
-- 
-- /Since: 2.60/
type ApplicationNameLostCallback =
    IO Bool
    -- ^ __Returns:__ 'P.True' if the signal has been handled

type C_ApplicationNameLostCallback =
    Ptr Application ->                      -- object
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_ApplicationNameLostCallback`.
foreign import ccall "wrapper"
    mk_ApplicationNameLostCallback :: C_ApplicationNameLostCallback -> IO (FunPtr C_ApplicationNameLostCallback)

wrap_ApplicationNameLostCallback :: 
    GObject a => (a -> ApplicationNameLostCallback) ->
    C_ApplicationNameLostCallback
wrap_ApplicationNameLostCallback gi'cb gi'selfPtr _ = do
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [nameLost](#signal:nameLost) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #nameLost callback
-- @
-- 
-- 
onApplicationNameLost :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationNameLostCallback) -> m SignalHandlerId
onApplicationNameLost obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationNameLostCallback wrapped
    wrapped'' <- mk_ApplicationNameLostCallback wrapped'
    connectSignalFunPtr obj "name-lost" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [nameLost](#signal:nameLost) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #nameLost callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationNameLost :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationNameLostCallback) -> m SignalHandlerId
afterApplicationNameLost obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationNameLostCallback wrapped
    wrapped'' <- mk_ApplicationNameLostCallback wrapped'
    connectSignalFunPtr obj "name-lost" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationNameLostSignalInfo
instance SignalInfo ApplicationNameLostSignalInfo where
    type HaskellCallbackType ApplicationNameLostSignalInfo = ApplicationNameLostCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationNameLostCallback cb
        cb'' <- mk_ApplicationNameLostCallback cb'
        connectSignalFunPtr obj "name-lost" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::name-lost"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:nameLost"})

#endif

-- signal Application::open
-- | The [open](#g:signal:open) signal is emitted on the primary instance when there are
-- files to open. See 'GI.Gio.Objects.Application.applicationOpen' for more information.
type ApplicationOpenCallback =
    [Gio.File.File]
    -- ^ /@files@/: an array of @/GFiles/@
    -> T.Text
    -- ^ /@hint@/: a hint provided by the calling instance
    -> IO ()

type C_ApplicationOpenCallback =
    Ptr Application ->                      -- object
    Ptr (Ptr Gio.File.File) ->
    Int32 ->
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ApplicationOpenCallback`.
foreign import ccall "wrapper"
    mk_ApplicationOpenCallback :: C_ApplicationOpenCallback -> IO (FunPtr C_ApplicationOpenCallback)

wrap_ApplicationOpenCallback :: 
    GObject a => (a -> ApplicationOpenCallback) ->
    C_ApplicationOpenCallback
wrap_ApplicationOpenCallback gi'cb gi'selfPtr files nFiles hint _ = do
    files' <- (unpackPtrArrayWithLength nFiles) files
    files'' <- mapM (newObject Gio.File.File) files'
    hint' <- cstringToText hint
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  files'' hint'


-- | Connect a signal handler for the [open](#signal:open) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #open callback
-- @
-- 
-- 
onApplicationOpen :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationOpenCallback) -> m SignalHandlerId
onApplicationOpen obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationOpenCallback wrapped
    wrapped'' <- mk_ApplicationOpenCallback wrapped'
    connectSignalFunPtr obj "open" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [open](#signal:open) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #open callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationOpen :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationOpenCallback) -> m SignalHandlerId
afterApplicationOpen obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationOpenCallback wrapped
    wrapped'' <- mk_ApplicationOpenCallback wrapped'
    connectSignalFunPtr obj "open" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationOpenSignalInfo
instance SignalInfo ApplicationOpenSignalInfo where
    type HaskellCallbackType ApplicationOpenSignalInfo = ApplicationOpenCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationOpenCallback cb
        cb'' <- mk_ApplicationOpenCallback cb'
        connectSignalFunPtr obj "open" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::open"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:open"})

#endif

-- signal Application::shutdown
-- | The [shutdown](#g:signal:shutdown) signal is emitted only on the registered primary instance
-- immediately after the main loop terminates.
type ApplicationShutdownCallback =
    IO ()

type C_ApplicationShutdownCallback =
    Ptr Application ->                      -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ApplicationShutdownCallback`.
foreign import ccall "wrapper"
    mk_ApplicationShutdownCallback :: C_ApplicationShutdownCallback -> IO (FunPtr C_ApplicationShutdownCallback)

wrap_ApplicationShutdownCallback :: 
    GObject a => (a -> ApplicationShutdownCallback) ->
    C_ApplicationShutdownCallback
wrap_ApplicationShutdownCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [shutdown](#signal:shutdown) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #shutdown callback
-- @
-- 
-- 
onApplicationShutdown :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationShutdownCallback) -> m SignalHandlerId
onApplicationShutdown obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationShutdownCallback wrapped
    wrapped'' <- mk_ApplicationShutdownCallback wrapped'
    connectSignalFunPtr obj "shutdown" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [shutdown](#signal:shutdown) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #shutdown callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationShutdown :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationShutdownCallback) -> m SignalHandlerId
afterApplicationShutdown obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationShutdownCallback wrapped
    wrapped'' <- mk_ApplicationShutdownCallback wrapped'
    connectSignalFunPtr obj "shutdown" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationShutdownSignalInfo
instance SignalInfo ApplicationShutdownSignalInfo where
    type HaskellCallbackType ApplicationShutdownSignalInfo = ApplicationShutdownCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationShutdownCallback cb
        cb'' <- mk_ApplicationShutdownCallback cb'
        connectSignalFunPtr obj "shutdown" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::shutdown"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:shutdown"})

#endif

-- signal Application::startup
-- | The [startup](#g:signal:startup) signal is emitted on the primary instance immediately
-- after registration. See 'GI.Gio.Objects.Application.applicationRegister'.
type ApplicationStartupCallback =
    IO ()

type C_ApplicationStartupCallback =
    Ptr Application ->                      -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ApplicationStartupCallback`.
foreign import ccall "wrapper"
    mk_ApplicationStartupCallback :: C_ApplicationStartupCallback -> IO (FunPtr C_ApplicationStartupCallback)

wrap_ApplicationStartupCallback :: 
    GObject a => (a -> ApplicationStartupCallback) ->
    C_ApplicationStartupCallback
wrap_ApplicationStartupCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [startup](#signal:startup) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' application #startup callback
-- @
-- 
-- 
onApplicationStartup :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationStartupCallback) -> m SignalHandlerId
onApplicationStartup obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationStartupCallback wrapped
    wrapped'' <- mk_ApplicationStartupCallback wrapped'
    connectSignalFunPtr obj "startup" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [startup](#signal:startup) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' application #startup callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterApplicationStartup :: (IsApplication a, MonadIO m) => a -> ((?self :: a) => ApplicationStartupCallback) -> m SignalHandlerId
afterApplicationStartup obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ApplicationStartupCallback wrapped
    wrapped'' <- mk_ApplicationStartupCallback wrapped'
    connectSignalFunPtr obj "startup" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ApplicationStartupSignalInfo
instance SignalInfo ApplicationStartupSignalInfo where
    type HaskellCallbackType ApplicationStartupSignalInfo = ApplicationStartupCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ApplicationStartupCallback cb
        cb'' <- mk_ApplicationStartupCallback cb'
        connectSignalFunPtr obj "startup" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application::startup"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:signal:startup"})

#endif

-- VVV Prop "action-group"
   -- Type: TInterface (Name {namespace = "Gio", name = "ActionGroup"})
   -- Flags: [PropertyWritable]
   -- Nullable: (Nothing,Just True)

-- | Set the value of the “@action-group@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' application [ #actionGroup 'Data.GI.Base.Attributes.:=' value ]
-- @
setApplicationActionGroup :: (MonadIO m, IsApplication o, Gio.ActionGroup.IsActionGroup a) => o -> a -> m ()
setApplicationActionGroup obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "action-group" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@action-group@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationActionGroup :: (IsApplication o, MIO.MonadIO m, Gio.ActionGroup.IsActionGroup a) => a -> m (GValueConstruct o)
constructApplicationActionGroup val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "action-group" (P.Just val)

-- | Set the value of the “@action-group@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #actionGroup
-- @
clearApplicationActionGroup :: (MonadIO m, IsApplication o) => o -> m ()
clearApplicationActionGroup obj = liftIO $ B.Properties.setObjectPropertyObject obj "action-group" (Nothing :: Maybe Gio.ActionGroup.ActionGroup)

#if defined(ENABLE_OVERLOADING)
data ApplicationActionGroupPropertyInfo
instance AttrInfo ApplicationActionGroupPropertyInfo where
    type AttrAllowedOps ApplicationActionGroupPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint ApplicationActionGroupPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationActionGroupPropertyInfo = Gio.ActionGroup.IsActionGroup
    type AttrTransferTypeConstraint ApplicationActionGroupPropertyInfo = Gio.ActionGroup.IsActionGroup
    type AttrTransferType ApplicationActionGroupPropertyInfo = Gio.ActionGroup.ActionGroup
    type AttrGetType ApplicationActionGroupPropertyInfo = ()
    type AttrLabel ApplicationActionGroupPropertyInfo = "action-group"
    type AttrOrigin ApplicationActionGroupPropertyInfo = Application
    attrGet = undefined
    attrSet = setApplicationActionGroup
    attrPut = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.ActionGroup.ActionGroup v
    attrConstruct = constructApplicationActionGroup
    attrClear = clearApplicationActionGroup
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.actionGroup"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:actionGroup"
        })
#endif

-- VVV Prop "application-id"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstruct]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@application-id@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #applicationId
-- @
getApplicationApplicationId :: (MonadIO m, IsApplication o) => o -> m (Maybe T.Text)
getApplicationApplicationId obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "application-id"

-- | Set the value of the “@application-id@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' application [ #applicationId 'Data.GI.Base.Attributes.:=' value ]
-- @
setApplicationApplicationId :: (MonadIO m, IsApplication o) => o -> T.Text -> m ()
setApplicationApplicationId obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "application-id" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@application-id@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationApplicationId :: (IsApplication o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructApplicationApplicationId val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "application-id" (P.Just val)

-- | Set the value of the “@application-id@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #applicationId
-- @
clearApplicationApplicationId :: (MonadIO m, IsApplication o) => o -> m ()
clearApplicationApplicationId obj = liftIO $ B.Properties.setObjectPropertyString obj "application-id" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data ApplicationApplicationIdPropertyInfo
instance AttrInfo ApplicationApplicationIdPropertyInfo where
    type AttrAllowedOps ApplicationApplicationIdPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint ApplicationApplicationIdPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationApplicationIdPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint ApplicationApplicationIdPropertyInfo = (~) T.Text
    type AttrTransferType ApplicationApplicationIdPropertyInfo = T.Text
    type AttrGetType ApplicationApplicationIdPropertyInfo = (Maybe T.Text)
    type AttrLabel ApplicationApplicationIdPropertyInfo = "application-id"
    type AttrOrigin ApplicationApplicationIdPropertyInfo = Application
    attrGet = getApplicationApplicationId
    attrSet = setApplicationApplicationId
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationApplicationId
    attrClear = clearApplicationApplicationId
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationId"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:applicationId"
        })
#endif

-- VVV Prop "flags"
   -- Type: TInterface (Name {namespace = "Gio", name = "ApplicationFlags"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@flags@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #flags
-- @
getApplicationFlags :: (MonadIO m, IsApplication o) => o -> m [Gio.Flags.ApplicationFlags]
getApplicationFlags obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "flags"

-- | Set the value of the “@flags@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' application [ #flags 'Data.GI.Base.Attributes.:=' value ]
-- @
setApplicationFlags :: (MonadIO m, IsApplication o) => o -> [Gio.Flags.ApplicationFlags] -> m ()
setApplicationFlags obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFlags obj "flags" val

-- | Construct a t'GValueConstruct' with valid value for the “@flags@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationFlags :: (IsApplication o, MIO.MonadIO m) => [Gio.Flags.ApplicationFlags] -> m (GValueConstruct o)
constructApplicationFlags val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "flags" val

#if defined(ENABLE_OVERLOADING)
data ApplicationFlagsPropertyInfo
instance AttrInfo ApplicationFlagsPropertyInfo where
    type AttrAllowedOps ApplicationFlagsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint ApplicationFlagsPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationFlagsPropertyInfo = (~) [Gio.Flags.ApplicationFlags]
    type AttrTransferTypeConstraint ApplicationFlagsPropertyInfo = (~) [Gio.Flags.ApplicationFlags]
    type AttrTransferType ApplicationFlagsPropertyInfo = [Gio.Flags.ApplicationFlags]
    type AttrGetType ApplicationFlagsPropertyInfo = [Gio.Flags.ApplicationFlags]
    type AttrLabel ApplicationFlagsPropertyInfo = "flags"
    type AttrOrigin ApplicationFlagsPropertyInfo = Application
    attrGet = getApplicationFlags
    attrSet = setApplicationFlags
    attrPut = setApplicationFlags
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationFlags
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.flags"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:flags"
        })
#endif

-- VVV Prop "inactivity-timeout"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@inactivity-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #inactivityTimeout
-- @
getApplicationInactivityTimeout :: (MonadIO m, IsApplication o) => o -> m Word32
getApplicationInactivityTimeout obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "inactivity-timeout"

-- | Set the value of the “@inactivity-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' application [ #inactivityTimeout 'Data.GI.Base.Attributes.:=' value ]
-- @
setApplicationInactivityTimeout :: (MonadIO m, IsApplication o) => o -> Word32 -> m ()
setApplicationInactivityTimeout obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "inactivity-timeout" val

-- | Construct a t'GValueConstruct' with valid value for the “@inactivity-timeout@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationInactivityTimeout :: (IsApplication o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructApplicationInactivityTimeout val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "inactivity-timeout" val

#if defined(ENABLE_OVERLOADING)
data ApplicationInactivityTimeoutPropertyInfo
instance AttrInfo ApplicationInactivityTimeoutPropertyInfo where
    type AttrAllowedOps ApplicationInactivityTimeoutPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint ApplicationInactivityTimeoutPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationInactivityTimeoutPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint ApplicationInactivityTimeoutPropertyInfo = (~) Word32
    type AttrTransferType ApplicationInactivityTimeoutPropertyInfo = Word32
    type AttrGetType ApplicationInactivityTimeoutPropertyInfo = Word32
    type AttrLabel ApplicationInactivityTimeoutPropertyInfo = "inactivity-timeout"
    type AttrOrigin ApplicationInactivityTimeoutPropertyInfo = Application
    attrGet = getApplicationInactivityTimeout
    attrSet = setApplicationInactivityTimeout
    attrPut = setApplicationInactivityTimeout
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationInactivityTimeout
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.inactivityTimeout"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:inactivityTimeout"
        })
#endif

-- VVV Prop "is-busy"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@is-busy@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #isBusy
-- @
getApplicationIsBusy :: (MonadIO m, IsApplication o) => o -> m Bool
getApplicationIsBusy obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-busy"

#if defined(ENABLE_OVERLOADING)
data ApplicationIsBusyPropertyInfo
instance AttrInfo ApplicationIsBusyPropertyInfo where
    type AttrAllowedOps ApplicationIsBusyPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ApplicationIsBusyPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationIsBusyPropertyInfo = (~) ()
    type AttrTransferTypeConstraint ApplicationIsBusyPropertyInfo = (~) ()
    type AttrTransferType ApplicationIsBusyPropertyInfo = ()
    type AttrGetType ApplicationIsBusyPropertyInfo = Bool
    type AttrLabel ApplicationIsBusyPropertyInfo = "is-busy"
    type AttrOrigin ApplicationIsBusyPropertyInfo = Application
    attrGet = getApplicationIsBusy
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.isBusy"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:isBusy"
        })
#endif

-- VVV Prop "is-registered"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@is-registered@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #isRegistered
-- @
getApplicationIsRegistered :: (MonadIO m, IsApplication o) => o -> m Bool
getApplicationIsRegistered obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-registered"

#if defined(ENABLE_OVERLOADING)
data ApplicationIsRegisteredPropertyInfo
instance AttrInfo ApplicationIsRegisteredPropertyInfo where
    type AttrAllowedOps ApplicationIsRegisteredPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ApplicationIsRegisteredPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationIsRegisteredPropertyInfo = (~) ()
    type AttrTransferTypeConstraint ApplicationIsRegisteredPropertyInfo = (~) ()
    type AttrTransferType ApplicationIsRegisteredPropertyInfo = ()
    type AttrGetType ApplicationIsRegisteredPropertyInfo = Bool
    type AttrLabel ApplicationIsRegisteredPropertyInfo = "is-registered"
    type AttrOrigin ApplicationIsRegisteredPropertyInfo = Application
    attrGet = getApplicationIsRegistered
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.isRegistered"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:isRegistered"
        })
#endif

-- VVV Prop "is-remote"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@is-remote@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #isRemote
-- @
getApplicationIsRemote :: (MonadIO m, IsApplication o) => o -> m Bool
getApplicationIsRemote obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-remote"

#if defined(ENABLE_OVERLOADING)
data ApplicationIsRemotePropertyInfo
instance AttrInfo ApplicationIsRemotePropertyInfo where
    type AttrAllowedOps ApplicationIsRemotePropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ApplicationIsRemotePropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationIsRemotePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ApplicationIsRemotePropertyInfo = (~) ()
    type AttrTransferType ApplicationIsRemotePropertyInfo = ()
    type AttrGetType ApplicationIsRemotePropertyInfo = Bool
    type AttrLabel ApplicationIsRemotePropertyInfo = "is-remote"
    type AttrOrigin ApplicationIsRemotePropertyInfo = Application
    attrGet = getApplicationIsRemote
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.isRemote"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:isRemote"
        })
#endif

-- VVV Prop "resource-base-path"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@resource-base-path@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #resourceBasePath
-- @
getApplicationResourceBasePath :: (MonadIO m, IsApplication o) => o -> m (Maybe T.Text)
getApplicationResourceBasePath obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "resource-base-path"

-- | Set the value of the “@resource-base-path@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' application [ #resourceBasePath 'Data.GI.Base.Attributes.:=' value ]
-- @
setApplicationResourceBasePath :: (MonadIO m, IsApplication o) => o -> T.Text -> m ()
setApplicationResourceBasePath obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "resource-base-path" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@resource-base-path@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationResourceBasePath :: (IsApplication o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructApplicationResourceBasePath val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "resource-base-path" (P.Just val)

-- | Set the value of the “@resource-base-path@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #resourceBasePath
-- @
clearApplicationResourceBasePath :: (MonadIO m, IsApplication o) => o -> m ()
clearApplicationResourceBasePath obj = liftIO $ B.Properties.setObjectPropertyString obj "resource-base-path" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data ApplicationResourceBasePathPropertyInfo
instance AttrInfo ApplicationResourceBasePathPropertyInfo where
    type AttrAllowedOps ApplicationResourceBasePathPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint ApplicationResourceBasePathPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationResourceBasePathPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint ApplicationResourceBasePathPropertyInfo = (~) T.Text
    type AttrTransferType ApplicationResourceBasePathPropertyInfo = T.Text
    type AttrGetType ApplicationResourceBasePathPropertyInfo = (Maybe T.Text)
    type AttrLabel ApplicationResourceBasePathPropertyInfo = "resource-base-path"
    type AttrOrigin ApplicationResourceBasePathPropertyInfo = Application
    attrGet = getApplicationResourceBasePath
    attrSet = setApplicationResourceBasePath
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationResourceBasePath
    attrClear = clearApplicationResourceBasePath
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.resourceBasePath"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:resourceBasePath"
        })
#endif

-- VVV Prop "version"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just False)

-- | Get the value of the “@version@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' application #version
-- @
getApplicationVersion :: (MonadIO m, IsApplication o) => o -> m (Maybe T.Text)
getApplicationVersion obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "version"

-- | Set the value of the “@version@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' application [ #version 'Data.GI.Base.Attributes.:=' value ]
-- @
setApplicationVersion :: (MonadIO m, IsApplication o) => o -> T.Text -> m ()
setApplicationVersion obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "version" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@version@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructApplicationVersion :: (IsApplication o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructApplicationVersion val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "version" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data ApplicationVersionPropertyInfo
instance AttrInfo ApplicationVersionPropertyInfo where
    type AttrAllowedOps ApplicationVersionPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint ApplicationVersionPropertyInfo = IsApplication
    type AttrSetTypeConstraint ApplicationVersionPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint ApplicationVersionPropertyInfo = (~) T.Text
    type AttrTransferType ApplicationVersionPropertyInfo = T.Text
    type AttrGetType ApplicationVersionPropertyInfo = (Maybe T.Text)
    type AttrLabel ApplicationVersionPropertyInfo = "version"
    type AttrOrigin ApplicationVersionPropertyInfo = Application
    attrGet = getApplicationVersion
    attrSet = setApplicationVersion
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructApplicationVersion
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.version"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#g:attr:version"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Application
type instance O.AttributeList Application = ApplicationAttributeList
type ApplicationAttributeList = ('[ '("actionGroup", ApplicationActionGroupPropertyInfo), '("applicationId", ApplicationApplicationIdPropertyInfo), '("flags", ApplicationFlagsPropertyInfo), '("inactivityTimeout", ApplicationInactivityTimeoutPropertyInfo), '("isBusy", ApplicationIsBusyPropertyInfo), '("isRegistered", ApplicationIsRegisteredPropertyInfo), '("isRemote", ApplicationIsRemotePropertyInfo), '("resourceBasePath", ApplicationResourceBasePathPropertyInfo), '("version", ApplicationVersionPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
applicationActionGroup :: AttrLabelProxy "actionGroup"
applicationActionGroup = AttrLabelProxy

applicationApplicationId :: AttrLabelProxy "applicationId"
applicationApplicationId = AttrLabelProxy

applicationFlags :: AttrLabelProxy "flags"
applicationFlags = AttrLabelProxy

applicationInactivityTimeout :: AttrLabelProxy "inactivityTimeout"
applicationInactivityTimeout = AttrLabelProxy

applicationIsBusy :: AttrLabelProxy "isBusy"
applicationIsBusy = AttrLabelProxy

applicationIsRegistered :: AttrLabelProxy "isRegistered"
applicationIsRegistered = AttrLabelProxy

applicationIsRemote :: AttrLabelProxy "isRemote"
applicationIsRemote = AttrLabelProxy

applicationResourceBasePath :: AttrLabelProxy "resourceBasePath"
applicationResourceBasePath = AttrLabelProxy

applicationVersion :: AttrLabelProxy "version"
applicationVersion = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Application = ApplicationSignalList
type ApplicationSignalList = ('[ '("actionAdded", Gio.ActionGroup.ActionGroupActionAddedSignalInfo), '("actionEnabledChanged", Gio.ActionGroup.ActionGroupActionEnabledChangedSignalInfo), '("actionRemoved", Gio.ActionGroup.ActionGroupActionRemovedSignalInfo), '("actionStateChanged", Gio.ActionGroup.ActionGroupActionStateChangedSignalInfo), '("activate", ApplicationActivateSignalInfo), '("commandLine", ApplicationCommandLineSignalInfo), '("handleLocalOptions", ApplicationHandleLocalOptionsSignalInfo), '("nameLost", ApplicationNameLostSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("open", ApplicationOpenSignalInfo), '("shutdown", ApplicationShutdownSignalInfo), '("startup", ApplicationStartupSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Application::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "application_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the application id" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ApplicationFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the application flags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Application" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_new" g_application_new :: 
    CString ->                              -- application_id : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "ApplicationFlags"})
    IO (Ptr Application)

-- | Creates a new t'GI.Gio.Objects.Application.Application' instance.
-- 
-- If non-'P.Nothing', the application id must be valid.  See
-- 'GI.Gio.Objects.Application.applicationIdIsValid'.
-- 
-- If no application ID is given then some features of t'GI.Gio.Objects.Application.Application'
-- (most notably application uniqueness) will be disabled.
applicationNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@applicationId@/: the application id
    -> [Gio.Flags.ApplicationFlags]
    -- ^ /@flags@/: the application flags
    -> m (Maybe Application)
    -- ^ __Returns:__ a new t'GI.Gio.Objects.Application.Application' instance
applicationNew applicationId flags = liftIO $ do
    maybeApplicationId <- case applicationId of
        Nothing -> return FP.nullPtr
        Just jApplicationId -> do
            jApplicationId' <- textToCString jApplicationId
            return jApplicationId'
    let flags' = gflagsToWord flags
    result <- g_application_new maybeApplicationId flags'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Application) result'
        return result''
    freeMem maybeApplicationId
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Application::activate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_activate" g_application_activate :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Activates the application.
-- 
-- In essence, this results in the [Application::activate]("GI.Gio.Objects.Application#g:signal:activate") signal being
-- emitted in the primary instance.
-- 
-- The application must be registered before calling this function.
-- 
-- /Since: 2.28/
applicationActivate ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m ()
applicationActivate application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_activate application'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationActivateMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationActivateMethodInfo a signature where
    overloadedMethod = applicationActivate

instance O.OverloadedMethodInfo ApplicationActivateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationActivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationActivate"
        })


#endif

-- method Application::add_main_option
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "long_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the long name of an option used to specify it in a commandline"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "short_name"
--           , argType = TBasicType TInt8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the short name of an option"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "OptionFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags from #GOptionFlags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "arg"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "OptionArg" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the type of the option, as a #GOptionArg"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "description"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the description for the option in `--help` output"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "arg_description"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the placeholder to use for the extra argument\n   parsed by the option in `--help` output"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_add_main_option" g_application_add_main_option :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- long_name : TBasicType TUTF8
    Int8 ->                                 -- short_name : TBasicType TInt8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "OptionFlags"})
    CUInt ->                                -- arg : TInterface (Name {namespace = "GLib", name = "OptionArg"})
    CString ->                              -- description : TBasicType TUTF8
    CString ->                              -- arg_description : TBasicType TUTF8
    IO ()

-- | Add an option to be handled by /@application@/.
-- 
-- Calling this function is the equivalent of calling
-- 'GI.Gio.Objects.Application.applicationAddMainOptionEntries' with a single t'GI.GLib.Structs.OptionEntry.OptionEntry'
-- that has its arg_data member set to 'P.Nothing'.
-- 
-- The parsed arguments will be packed into a t'GI.GLib.Structs.VariantDict.VariantDict' which
-- is passed to [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions"). If
-- 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' is set, then it will also
-- be sent to the primary instance. See
-- 'GI.Gio.Objects.Application.applicationAddMainOptionEntries' for more details.
-- 
-- See t'GI.GLib.Structs.OptionEntry.OptionEntry' for more documentation of the arguments.
-- 
-- /Since: 2.42/
applicationAddMainOption ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: the t'GI.Gio.Objects.Application.Application'
    -> T.Text
    -- ^ /@longName@/: the long name of an option used to specify it in a commandline
    -> Int8
    -- ^ /@shortName@/: the short name of an option
    -> [GLib.Flags.OptionFlags]
    -- ^ /@flags@/: flags from t'GI.GLib.Flags.OptionFlags'
    -> GLib.Enums.OptionArg
    -- ^ /@arg@/: the type of the option, as a t'GI.GLib.Enums.OptionArg'
    -> T.Text
    -- ^ /@description@/: the description for the option in @--help@ output
    -> Maybe (T.Text)
    -- ^ /@argDescription@/: the placeholder to use for the extra argument
    --    parsed by the option in @--help@ output
    -> m ()
applicationAddMainOption application longName shortName flags arg description argDescription = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    longName' <- textToCString longName
    let flags' = gflagsToWord flags
    let arg' = (fromIntegral . fromEnum) arg
    description' <- textToCString description
    maybeArgDescription <- case argDescription of
        Nothing -> return FP.nullPtr
        Just jArgDescription -> do
            jArgDescription' <- textToCString jArgDescription
            return jArgDescription'
    g_application_add_main_option application' longName' shortName flags' arg' description' maybeArgDescription
    touchManagedPtr application
    freeMem longName'
    freeMem description'
    freeMem maybeArgDescription
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationAddMainOptionMethodInfo
instance (signature ~ (T.Text -> Int8 -> [GLib.Flags.OptionFlags] -> GLib.Enums.OptionArg -> T.Text -> Maybe (T.Text) -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationAddMainOptionMethodInfo a signature where
    overloadedMethod = applicationAddMainOption

instance O.OverloadedMethodInfo ApplicationAddMainOptionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationAddMainOption",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationAddMainOption"
        })


#endif

-- method Application::add_main_option_entries
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "entries"
--           , argType =
--               TCArray
--                 True
--                 (-1)
--                 (-1)
--                 (TInterface Name { namespace = "GLib" , name = "OptionEntry" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the\n  main options for the application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_add_main_option_entries" g_application_add_main_option_entries :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr (Ptr GLib.OptionEntry.OptionEntry) -> -- entries : TCArray True (-1) (-1) (TInterface (Name {namespace = "GLib", name = "OptionEntry"}))
    IO ()

-- | Adds main option entries to be handled by /@application@/.
-- 
-- This function is comparable to 'GI.GLib.Structs.OptionContext.optionContextAddMainEntries'.
-- 
-- After the commandline arguments are parsed, the
-- [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions") signal will be emitted.  At this
-- point, the application can inspect the values pointed to by /@argData@/
-- in the given @/GOptionEntrys/@.
-- 
-- Unlike t'GI.GLib.Structs.OptionContext.OptionContext', t'GI.Gio.Objects.Application.Application' supports giving a 'P.Nothing'
-- /@argData@/ for a non-callback t'GI.GLib.Structs.OptionEntry.OptionEntry'.  This results in the
-- argument in question being packed into a t'GI.GLib.Structs.VariantDict.VariantDict' which is also
-- passed to [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions"), where it can be
-- inspected and modified.  If 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' is
-- set, then the resulting dictionary is sent to the primary instance,
-- where 'GI.Gio.Objects.ApplicationCommandLine.applicationCommandLineGetOptionsDict' will return it.
-- As it has been passed outside the process at this point, the types of all
-- values in the options dict must be checked before being used.
-- This \"packing\" is done according to the type of the argument --
-- booleans for normal flags, strings for strings, bytestrings for
-- filenames, etc.  The packing only occurs if the flag is given (ie: we
-- do not pack a \"false\" t'GVariant' in the case that a flag is missing).
-- 
-- In general, it is recommended that all commandline arguments are
-- parsed locally.  The options dictionary should then be used to
-- transmit the result of the parsing to the primary instance, where
-- @/g_variant_dict_lookup()/@ can be used.  For local options, it is
-- possible to either use /@argData@/ in the usual way, or to consult (and
-- potentially remove) the option from the options dictionary.
-- 
-- This function is new in GLib 2.40.  Before then, the only real choice
-- was to send all of the commandline arguments (options and all) to the
-- primary instance for handling.  t'GI.Gio.Objects.Application.Application' ignored them completely
-- on the local side.  Calling this function \"opts in\" to the new
-- behaviour, and in particular, means that unrecognized options will be
-- treated as errors.  Unrecognized options have never been ignored when
-- 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' is unset.
-- 
-- If [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions") needs to see the list of
-- filenames, then the use of 'GI.GLib.Constants.OPTION_REMAINING' is recommended.  If
-- /@argData@/ is 'P.Nothing' then 'GI.GLib.Constants.OPTION_REMAINING' can be used as a key into
-- the options dictionary.  If you do use 'GI.GLib.Constants.OPTION_REMAINING' then you
-- need to handle these arguments for yourself because once they are
-- consumed, they will no longer be visible to the default handling
-- (which treats them as filenames to be opened).
-- 
-- It is important to use the proper GVariant format when retrieving
-- the options with @/g_variant_dict_lookup()/@:
-- * for 'GI.GLib.Enums.OptionArgNone', use @b@
-- * for 'GI.GLib.Enums.OptionArgString', use @&s@
-- * for 'GI.GLib.Enums.OptionArgInt', use @i@
-- * for 'GI.GLib.Enums.OptionArgInt64', use @x@
-- * for 'GI.GLib.Enums.OptionArgDouble', use @d@
-- * for 'GI.GLib.Enums.OptionArgFilename', use @^&ay@
-- * for 'GI.GLib.Enums.OptionArgStringArray', use @^a&s@
-- * for 'GI.GLib.Enums.OptionArgFilenameArray', use @^a&ay@
-- 
-- 
-- /Since: 2.40/
applicationAddMainOptionEntries ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> [GLib.OptionEntry.OptionEntry]
    -- ^ /@entries@/: the
    --   main options for the application
    -> m ()
applicationAddMainOptionEntries application entries = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    entries' <- mapM unsafeManagedPtrGetPtr entries
    entries'' <- packZeroTerminatedPtrArray entries'
    g_application_add_main_option_entries application' entries''
    touchManagedPtr application
    mapM_ touchManagedPtr entries
    freeMem entries''
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationAddMainOptionEntriesMethodInfo
instance (signature ~ ([GLib.OptionEntry.OptionEntry] -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationAddMainOptionEntriesMethodInfo a signature where
    overloadedMethod = applicationAddMainOptionEntries

instance O.OverloadedMethodInfo ApplicationAddMainOptionEntriesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationAddMainOptionEntries",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationAddMainOptionEntries"
        })


#endif

-- method Application::add_option_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "OptionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GOptionGroup" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_add_option_group" g_application_add_option_group :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr GLib.OptionGroup.OptionGroup ->     -- group : TInterface (Name {namespace = "GLib", name = "OptionGroup"})
    IO ()

-- | Adds a t'GI.GLib.Structs.OptionGroup.OptionGroup' to the commandline handling of /@application@/.
-- 
-- This function is comparable to 'GI.GLib.Structs.OptionContext.optionContextAddGroup'.
-- 
-- Unlike 'GI.Gio.Objects.Application.applicationAddMainOptionEntries', this function does
-- not deal with 'P.Nothing' /@argData@/ and never transmits options to the
-- primary instance.
-- 
-- The reason for that is because, by the time the options arrive at the
-- primary instance, it is typically too late to do anything with them.
-- Taking the GTK option group as an example: GTK will already have been
-- initialised by the time the [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") handler runs.
-- In the case that this is not the first-running instance of the
-- application, the existing instance may already have been running for
-- a very long time.
-- 
-- This means that the options from t'GI.GLib.Structs.OptionGroup.OptionGroup' are only really usable
-- in the case that the instance of the application being run is the
-- first instance.  Passing options like @--display=@ or @--gdk-debug=@
-- on future runs will have no effect on the existing primary instance.
-- 
-- Calling this function will cause the options in the supplied option
-- group to be parsed, but it does not cause you to be \"opted in\" to the
-- new functionality whereby unrecognized options are rejected even if
-- 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' was given.
-- 
-- /Since: 2.40/
applicationAddOptionGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: the t'GI.Gio.Objects.Application.Application'
    -> GLib.OptionGroup.OptionGroup
    -- ^ /@group@/: a t'GI.GLib.Structs.OptionGroup.OptionGroup'
    -> m ()
applicationAddOptionGroup application group = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    group' <- B.ManagedPtr.disownBoxed group
    g_application_add_option_group application' group'
    touchManagedPtr application
    touchManagedPtr group
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationAddOptionGroupMethodInfo
instance (signature ~ (GLib.OptionGroup.OptionGroup -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationAddOptionGroupMethodInfo a signature where
    overloadedMethod = applicationAddOptionGroup

instance O.OverloadedMethodInfo ApplicationAddOptionGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationAddOptionGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationAddOptionGroup"
        })


#endif

-- method Application::bind_busy_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of a boolean property of @object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_bind_busy_property" g_application_bind_busy_property :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property : TBasicType TUTF8
    IO ()

-- | Marks /@application@/ as busy (see 'GI.Gio.Objects.Application.applicationMarkBusy') while
-- /@property@/ on /@object@/ is 'P.True'.
-- 
-- The binding holds a reference to /@application@/ while it is active, but
-- not to /@object@/. Instead, the binding is destroyed when /@object@/ is
-- finalized.
-- 
-- /Since: 2.44/
applicationBindBusyProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a, GObject.Object.IsObject b) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> b
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@property@/: the name of a boolean property of /@object@/
    -> m ()
applicationBindBusyProperty application object property = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    object' <- unsafeManagedPtrCastPtr object
    property' <- textToCString property
    g_application_bind_busy_property application' object' property'
    touchManagedPtr application
    touchManagedPtr object
    freeMem property'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationBindBusyPropertyMethodInfo
instance (signature ~ (b -> T.Text -> m ()), MonadIO m, IsApplication a, GObject.Object.IsObject b) => O.OverloadedMethod ApplicationBindBusyPropertyMethodInfo a signature where
    overloadedMethod = applicationBindBusyProperty

instance O.OverloadedMethodInfo ApplicationBindBusyPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationBindBusyProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationBindBusyProperty"
        })


#endif

-- method Application::get_application_id
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_application_id" g_application_get_application_id :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CString

-- | Gets the unique identifier for /@application@/.
-- 
-- /Since: 2.28/
applicationGetApplicationId ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the identifier for /@application@/, owned by /@application@/
applicationGetApplicationId application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_application_id application'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr application
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationGetApplicationIdMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetApplicationIdMethodInfo a signature where
    overloadedMethod = applicationGetApplicationId

instance O.OverloadedMethodInfo ApplicationGetApplicationIdMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetApplicationId",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetApplicationId"
        })


#endif

-- method Application::get_dbus_connection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusConnection" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_dbus_connection" g_application_get_dbus_connection :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO (Ptr Gio.DBusConnection.DBusConnection)

-- | Gets the t'GI.Gio.Objects.DBusConnection.DBusConnection' being used by the application, or 'P.Nothing'.
-- 
-- If t'GI.Gio.Objects.Application.Application' is using its D-Bus backend then this function will
-- return the t'GI.Gio.Objects.DBusConnection.DBusConnection' being used for uniqueness and
-- communication with the desktop environment and other instances of the
-- application.
-- 
-- If t'GI.Gio.Objects.Application.Application' is not using D-Bus then this function will return
-- 'P.Nothing'.  This includes the situation where the D-Bus backend would
-- normally be in use but we were unable to connect to the bus.
-- 
-- This function must not be called before the application has been
-- registered.  See 'GI.Gio.Objects.Application.applicationGetIsRegistered'.
-- 
-- /Since: 2.34/
applicationGetDbusConnection ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m (Maybe Gio.DBusConnection.DBusConnection)
    -- ^ __Returns:__ a t'GI.Gio.Objects.DBusConnection.DBusConnection', or 'P.Nothing'
applicationGetDbusConnection application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_dbus_connection application'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gio.DBusConnection.DBusConnection) result'
        return result''
    touchManagedPtr application
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationGetDbusConnectionMethodInfo
instance (signature ~ (m (Maybe Gio.DBusConnection.DBusConnection)), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetDbusConnectionMethodInfo a signature where
    overloadedMethod = applicationGetDbusConnection

instance O.OverloadedMethodInfo ApplicationGetDbusConnectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetDbusConnection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetDbusConnection"
        })


#endif

-- method Application::get_dbus_object_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_dbus_object_path" g_application_get_dbus_object_path :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CString

-- | Gets the D-Bus object path being used by the application, or 'P.Nothing'.
-- 
-- If t'GI.Gio.Objects.Application.Application' is using its D-Bus backend then this function will
-- return the D-Bus object path that t'GI.Gio.Objects.Application.Application' is using.  If the
-- application is the primary instance then there is an object published
-- at this path.  If the application is not the primary instance then
-- the result of this function is undefined.
-- 
-- If t'GI.Gio.Objects.Application.Application' is not using D-Bus then this function will return
-- 'P.Nothing'.  This includes the situation where the D-Bus backend would
-- normally be in use but we were unable to connect to the bus.
-- 
-- This function must not be called before the application has been
-- registered.  See 'GI.Gio.Objects.Application.applicationGetIsRegistered'.
-- 
-- /Since: 2.34/
applicationGetDbusObjectPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the object path, or 'P.Nothing'
applicationGetDbusObjectPath application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_dbus_object_path application'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr application
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationGetDbusObjectPathMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetDbusObjectPathMethodInfo a signature where
    overloadedMethod = applicationGetDbusObjectPath

instance O.OverloadedMethodInfo ApplicationGetDbusObjectPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetDbusObjectPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetDbusObjectPath"
        })


#endif

-- method Application::get_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "ApplicationFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_flags" g_application_get_flags :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CUInt

-- | Gets the flags for /@application@/.
-- 
-- See t'GI.Gio.Flags.ApplicationFlags'.
-- 
-- /Since: 2.28/
applicationGetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m [Gio.Flags.ApplicationFlags]
    -- ^ __Returns:__ the flags for /@application@/
applicationGetFlags application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_flags application'
    let result' = wordToGFlags result
    touchManagedPtr application
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationGetFlagsMethodInfo
instance (signature ~ (m [Gio.Flags.ApplicationFlags]), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetFlagsMethodInfo a signature where
    overloadedMethod = applicationGetFlags

instance O.OverloadedMethodInfo ApplicationGetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetFlags"
        })


#endif

-- method Application::get_inactivity_timeout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_inactivity_timeout" g_application_get_inactivity_timeout :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO Word32

-- | Gets the current inactivity timeout for the application.
-- 
-- This is the amount of time (in milliseconds) after the last call to
-- 'GI.Gio.Objects.Application.applicationRelease' before the application stops running.
-- 
-- /Since: 2.28/
applicationGetInactivityTimeout ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m Word32
    -- ^ __Returns:__ the timeout, in milliseconds
applicationGetInactivityTimeout application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_inactivity_timeout application'
    touchManagedPtr application
    return result

#if defined(ENABLE_OVERLOADING)
data ApplicationGetInactivityTimeoutMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetInactivityTimeoutMethodInfo a signature where
    overloadedMethod = applicationGetInactivityTimeout

instance O.OverloadedMethodInfo ApplicationGetInactivityTimeoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetInactivityTimeout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetInactivityTimeout"
        })


#endif

-- method Application::get_is_busy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_is_busy" g_application_get_is_busy :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CInt

-- | Gets the application\'s current busy state, as set through
-- 'GI.Gio.Objects.Application.applicationMarkBusy' or 'GI.Gio.Objects.Application.applicationBindBusyProperty'.
-- 
-- /Since: 2.44/
applicationGetIsBusy ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@application@/ is currently marked as busy
applicationGetIsBusy application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_is_busy application'
    let result' = (/= 0) result
    touchManagedPtr application
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationGetIsBusyMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetIsBusyMethodInfo a signature where
    overloadedMethod = applicationGetIsBusy

instance O.OverloadedMethodInfo ApplicationGetIsBusyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetIsBusy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetIsBusy"
        })


#endif

-- method Application::get_is_registered
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_is_registered" g_application_get_is_registered :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CInt

-- | Checks if /@application@/ is registered.
-- 
-- An application is registered if 'GI.Gio.Objects.Application.applicationRegister' has been
-- successfully called.
-- 
-- /Since: 2.28/
applicationGetIsRegistered ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@application@/ is registered
applicationGetIsRegistered application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_is_registered application'
    let result' = (/= 0) result
    touchManagedPtr application
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationGetIsRegisteredMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetIsRegisteredMethodInfo a signature where
    overloadedMethod = applicationGetIsRegistered

instance O.OverloadedMethodInfo ApplicationGetIsRegisteredMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetIsRegistered",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetIsRegistered"
        })


#endif

-- method Application::get_is_remote
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_is_remote" g_application_get_is_remote :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CInt

-- | Checks if /@application@/ is remote.
-- 
-- If /@application@/ is remote then it means that another instance of
-- application already exists (the \'primary\' instance).  Calls to
-- perform actions on /@application@/ will result in the actions being
-- performed by the primary instance.
-- 
-- The value of this property cannot be accessed before
-- 'GI.Gio.Objects.Application.applicationRegister' has been called.  See
-- 'GI.Gio.Objects.Application.applicationGetIsRegistered'.
-- 
-- /Since: 2.28/
applicationGetIsRemote ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@application@/ is remote
applicationGetIsRemote application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_is_remote application'
    let result' = (/= 0) result
    touchManagedPtr application
    return result'

#if defined(ENABLE_OVERLOADING)
data ApplicationGetIsRemoteMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetIsRemoteMethodInfo a signature where
    overloadedMethod = applicationGetIsRemote

instance O.OverloadedMethodInfo ApplicationGetIsRemoteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetIsRemote",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetIsRemote"
        })


#endif

-- method Application::get_resource_base_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_resource_base_path" g_application_get_resource_base_path :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CString

-- | Gets the resource base path of /@application@/.
-- 
-- See 'GI.Gio.Objects.Application.applicationSetResourceBasePath' for more information.
-- 
-- /Since: 2.42/
applicationGetResourceBasePath ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the base resource path, if one is set
applicationGetResourceBasePath application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_resource_base_path application'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr application
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationGetResourceBasePathMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetResourceBasePathMethodInfo a signature where
    overloadedMethod = applicationGetResourceBasePath

instance O.OverloadedMethodInfo ApplicationGetResourceBasePathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetResourceBasePath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetResourceBasePath"
        })


#endif

-- method Application::get_version
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_version" g_application_get_version :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO CString

-- | Gets the version of /@application@/.
-- 
-- /Since: 2.80/
applicationGetVersion ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the version of /@application@/
applicationGetVersion application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    result <- g_application_get_version application'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr application
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ApplicationGetVersionMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationGetVersionMethodInfo a signature where
    overloadedMethod = applicationGetVersion

instance O.OverloadedMethodInfo ApplicationGetVersionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationGetVersion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationGetVersion"
        })


#endif

-- method Application::hold
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_hold" g_application_hold :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Increases the use count of /@application@/.
-- 
-- Use this function to indicate that the application has a reason to
-- continue to run.  For example, 'GI.Gio.Objects.Application.applicationHold' is called by GTK
-- when a toplevel window is on the screen.
-- 
-- To cancel the hold, call 'GI.Gio.Objects.Application.applicationRelease'.
applicationHold ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m ()
applicationHold application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_hold application'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationHoldMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationHoldMethodInfo a signature where
    overloadedMethod = applicationHold

instance O.OverloadedMethodInfo ApplicationHoldMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationHold",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationHold"
        })


#endif

-- method Application::mark_busy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_mark_busy" g_application_mark_busy :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Increases the busy count of /@application@/.
-- 
-- Use this function to indicate that the application is busy, for instance
-- while a long running operation is pending.
-- 
-- The busy state will be exposed to other processes, so a session shell will
-- use that information to indicate the state to the user (e.g. with a
-- spinner).
-- 
-- To cancel the busy indication, use 'GI.Gio.Objects.Application.applicationUnmarkBusy'.
-- 
-- The application must be registered before calling this function.
-- 
-- /Since: 2.38/
applicationMarkBusy ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m ()
applicationMarkBusy application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_mark_busy application'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationMarkBusyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationMarkBusyMethodInfo a signature where
    overloadedMethod = applicationMarkBusy

instance O.OverloadedMethodInfo ApplicationMarkBusyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationMarkBusy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationMarkBusy"
        })


#endif

-- method Application::open
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "files"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 2
--                 (TInterface Name { namespace = "Gio" , name = "File" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of #GFiles to open"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_files"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of the @files array"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hint"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a hint (or \"\"), but never %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_files"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of the @files array"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_open" g_application_open :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr (Ptr Gio.File.File) ->              -- files : TCArray False (-1) 2 (TInterface (Name {namespace = "Gio", name = "File"}))
    Int32 ->                                -- n_files : TBasicType TInt
    CString ->                              -- hint : TBasicType TUTF8
    IO ()

-- | Opens the given files.
-- 
-- In essence, this results in the [Application::open]("GI.Gio.Objects.Application#g:signal:open") signal being emitted
-- in the primary instance.
-- 
-- /@nFiles@/ must be greater than zero.
-- 
-- /@hint@/ is simply passed through to the [open](#g:signal:open) signal.  It is
-- intended to be used by applications that have multiple modes for
-- opening files (eg: \"view\" vs \"edit\", etc).  Unless you have a need
-- for this functionality, you should use \"\".
-- 
-- The application must be registered before calling this function
-- and it must have the 'GI.Gio.Flags.ApplicationFlagsHandlesOpen' flag set.
-- 
-- /Since: 2.28/
applicationOpen ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> [Gio.File.File]
    -- ^ /@files@/: an array of @/GFiles/@ to open
    -> T.Text
    -- ^ /@hint@/: a hint (or \"\"), but never 'P.Nothing'
    -> m ()
applicationOpen application files hint = liftIO $ do
    let nFiles = fromIntegral $ P.length files
    application' <- unsafeManagedPtrCastPtr application
    files' <- mapM unsafeManagedPtrCastPtr files
    files'' <- packPtrArray files'
    hint' <- textToCString hint
    g_application_open application' files'' nFiles hint'
    touchManagedPtr application
    mapM_ touchManagedPtr files
    freeMem files''
    freeMem hint'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationOpenMethodInfo
instance (signature ~ ([Gio.File.File] -> T.Text -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationOpenMethodInfo a signature where
    overloadedMethod = applicationOpen

instance O.OverloadedMethodInfo ApplicationOpenMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationOpen",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationOpen"
        })


#endif

-- method Application::quit
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_quit" g_application_quit :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Immediately quits the application.
-- 
-- Upon return to the mainloop, 'GI.Gio.Objects.Application.applicationRun' will return,
-- calling only the \'shutdown\' function before doing so.
-- 
-- The hold count is ignored.
-- Take care if your code has called 'GI.Gio.Objects.Application.applicationHold' on the application and
-- is therefore still expecting it to exist.
-- (Note that you may have called 'GI.Gio.Objects.Application.applicationHold' indirectly, for example
-- through @/gtk_application_add_window()/@.)
-- 
-- The result of calling 'GI.Gio.Objects.Application.applicationRun' again after it returns is
-- unspecified.
-- 
-- /Since: 2.32/
applicationQuit ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m ()
applicationQuit application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_quit application'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationQuitMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationQuitMethodInfo a signature where
    overloadedMethod = applicationQuit

instance O.OverloadedMethodInfo ApplicationQuitMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationQuit",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationQuit"
        })


#endif

-- method Application::register
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_application_register" g_application_register :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Attempts registration of the application.
-- 
-- This is the point at which the application discovers if it is the
-- primary instance or merely acting as a remote for an already-existing
-- primary instance.  This is implemented by attempting to acquire the
-- application identifier as a unique bus name on the session bus using
-- GDBus.
-- 
-- If there is no application ID or if 'GI.Gio.Flags.ApplicationFlagsNonUnique' was
-- given, then this process will always become the primary instance.
-- 
-- Due to the internal architecture of GDBus, method calls can be
-- dispatched at any time (even if a main loop is not running).  For
-- this reason, you must ensure that any object paths that you wish to
-- register are registered before calling this function.
-- 
-- If the application has already been registered then 'P.True' is
-- returned with no work performed.
-- 
-- The [Application::startup]("GI.Gio.Objects.Application#g:signal:startup") signal is emitted if registration succeeds
-- and /@application@/ is the primary instance (including the non-unique
-- case).
-- 
-- In the event of an error (such as /@cancellable@/ being cancelled, or a
-- failure to connect to the session bus), 'P.False' is returned and /@error@/
-- is set appropriately.
-- 
-- Note: the return value of this function is not an indicator that this
-- instance is or is not the primary instance of the application.  See
-- 'GI.Gio.Objects.Application.applicationGetIsRemote' for that.
-- 
-- /Since: 2.28/
applicationRegister ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
applicationRegister application cancellable = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_application_register application' maybeCancellable
        touchManagedPtr application
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ApplicationRegisterMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsApplication a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ApplicationRegisterMethodInfo a signature where
    overloadedMethod = applicationRegister

instance O.OverloadedMethodInfo ApplicationRegisterMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationRegister",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationRegister"
        })


#endif

-- method Application::release
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_release" g_application_release :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Decrease the use count of /@application@/.
-- 
-- When the use count reaches zero, the application will stop running.
-- 
-- Never call this function except to cancel the effect of a previous
-- call to 'GI.Gio.Objects.Application.applicationHold'.
applicationRelease ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m ()
applicationRelease application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_release application'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationReleaseMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationReleaseMethodInfo a signature where
    overloadedMethod = applicationRelease

instance O.OverloadedMethodInfo ApplicationReleaseMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationRelease",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationRelease"
        })


#endif

-- method Application::run
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "argc"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the argc from main() (or 0 if @argv is %NULL)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "argv"
--           , argType = TCArray False (-1) 1 (TBasicType TFileName)
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "\n    the argv from main(), or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "argc"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the argc from main() (or 0 if @argv is %NULL)"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_run" g_application_run :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Int32 ->                                -- argc : TBasicType TInt
    Ptr CString ->                          -- argv : TCArray False (-1) 1 (TBasicType TFileName)
    IO Int32

-- | Runs the application.
-- 
-- This function is intended to be run from @/main()/@ and its return value
-- is intended to be returned by @/main()/@. Although you are expected to pass
-- the /@argc@/, /@argv@/ parameters from @/main()/@ to this function, it is possible
-- to pass 'P.Nothing' if /@argv@/ is not available or commandline handling is not
-- required.  Note that on Windows, /@argc@/ and /@argv@/ are ignored, and
-- @/g_win32_get_command_line()/@ is called internally (for proper support
-- of Unicode commandline arguments).
-- 
-- t'GI.Gio.Objects.Application.Application' will attempt to parse the commandline arguments.  You
-- can add commandline flags to the list of recognised options by way of
-- 'GI.Gio.Objects.Application.applicationAddMainOptionEntries'.  After this, the
-- [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions") signal is emitted, from which the
-- application can inspect the values of its @/GOptionEntrys/@.
-- 
-- [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions") is a good place to handle options
-- such as @--version@, where an immediate reply from the local process is
-- desired (instead of communicating with an already-running instance).
-- A [Application::handleLocalOptions]("GI.Gio.Objects.Application#g:signal:handleLocalOptions") handler can stop further processing
-- by returning a non-negative value, which then becomes the exit status of
-- the process.
-- 
-- What happens next depends on the flags: if
-- 'GI.Gio.Flags.ApplicationFlagsHandlesCommandLine' was specified then the remaining
-- commandline arguments are sent to the primary instance, where a
-- [Application::commandLine]("GI.Gio.Objects.Application#g:signal:commandLine") signal is emitted.  Otherwise, the
-- remaining commandline arguments are assumed to be a list of files.
-- If there are no files listed, the application is activated via the
-- [Application::activate]("GI.Gio.Objects.Application#g:signal:activate") signal.  If there are one or more files, and
-- 'GI.Gio.Flags.ApplicationFlagsHandlesOpen' was specified then the files are opened
-- via the [Application::open]("GI.Gio.Objects.Application#g:signal:open") signal.
-- 
-- If you are interested in doing more complicated local handling of the
-- commandline then you should implement your own t'GI.Gio.Objects.Application.Application' subclass
-- and override @/local_command_line()/@. In this case, you most likely want
-- to return 'P.True' from your @/local_command_line()/@ implementation to
-- suppress the default handling. See
-- [gapplication-example-cmdline2.c][https:\/\/gitlab.gnome.org\/GNOME\/glib\/-\/blob\/HEAD\/gio\/tests\/gapplication-example-cmdline2.c]
-- for an example.
-- 
-- If, after the above is done, the use count of the application is zero
-- then the exit status is returned immediately.  If the use count is
-- non-zero then the default main context is iterated until the use count
-- falls to zero, at which point 0 is returned.
-- 
-- If the 'GI.Gio.Flags.ApplicationFlagsIsService' flag is set, then the service will
-- run for as much as 10 seconds with a use count of zero while waiting
-- for the message that caused the activation to arrive.  After that,
-- if the use count falls to zero the application will exit immediately,
-- except in the case that 'GI.Gio.Objects.Application.applicationSetInactivityTimeout' is in
-- use.
-- 
-- This function sets the prgname ('GI.GLib.Functions.setPrgname'), if not already set,
-- to the basename of argv[0].
-- 
-- Much like 'GI.GLib.Structs.MainLoop.mainLoopRun', this function will acquire the main context
-- for the duration that the application is running.
-- 
-- Since 2.40, applications that are not explicitly flagged as services
-- or launchers (ie: neither 'GI.Gio.Flags.ApplicationFlagsIsService' or
-- 'GI.Gio.Flags.ApplicationFlagsIsLauncher' are given as flags) will check (from the
-- default handler for local_command_line) if \"--gapplication-service\"
-- was given in the command line.  If this flag is present then normal
-- commandline processing is interrupted and the
-- 'GI.Gio.Flags.ApplicationFlagsIsService' flag is set.  This provides a \"compromise\"
-- solution whereby running an application directly from the commandline
-- will invoke it in the normal way (which can be useful for debugging)
-- while still allowing applications to be D-Bus activated in service
-- mode.  The D-Bus service file should invoke the executable with
-- \"--gapplication-service\" as the sole commandline argument.  This
-- approach is suitable for use by most graphical applications but
-- should not be used from applications like editors that need precise
-- control over when processes invoked via the commandline will exit and
-- what their exit status will be.
-- 
-- /Since: 2.28/
applicationRun ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Maybe ([[Char]])
    -- ^ /@argv@/: 
    --     the argv from @/main()/@, or 'P.Nothing'
    -> m Int32
    -- ^ __Returns:__ the exit status
applicationRun application argv = liftIO $ do
    let argc = case argv of
            Nothing -> 0
            Just jArgv -> fromIntegral $ P.length jArgv
    application' <- unsafeManagedPtrCastPtr application
    maybeArgv <- case argv of
        Nothing -> return FP.nullPtr
        Just jArgv -> do
            jArgv' <- packFileNameArray jArgv
            return jArgv'
    result <- g_application_run application' argc maybeArgv
    touchManagedPtr application
    (mapCArrayWithLength argc) freeMem maybeArgv
    freeMem maybeArgv
    return result

#if defined(ENABLE_OVERLOADING)
data ApplicationRunMethodInfo
instance (signature ~ (Maybe ([[Char]]) -> m Int32), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationRunMethodInfo a signature where
    overloadedMethod = applicationRun

instance O.OverloadedMethodInfo ApplicationRunMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationRun",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationRun"
        })


#endif

-- method Application::send_notification
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "id of the notification, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "notification"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Notification" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GNotification to send"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_send_notification" g_application_send_notification :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- id : TBasicType TUTF8
    Ptr Gio.Notification.Notification ->    -- notification : TInterface (Name {namespace = "Gio", name = "Notification"})
    IO ()

-- | Sends a notification on behalf of /@application@/ to the desktop shell.
-- There is no guarantee that the notification is displayed immediately,
-- or even at all.
-- 
-- Notifications may persist after the application exits. It will be
-- D-Bus-activated when the notification or one of its actions is
-- activated.
-- 
-- Modifying /@notification@/ after this call has no effect. However, the
-- object can be reused for a later call to this function.
-- 
-- /@id@/ may be any string that uniquely identifies the event for the
-- application. It does not need to be in any special format. For
-- example, \"new-message\" might be appropriate for a notification about
-- new messages.
-- 
-- If a previous notification was sent with the same /@id@/, it will be
-- replaced with /@notification@/ and shown again as if it was a new
-- notification. This works even for notifications sent from a previous
-- execution of the application, as long as /@id@/ is the same string.
-- 
-- /@id@/ may be @NULL@, but it is impossible to replace or withdraw
-- notifications without an id.
-- 
-- If /@notification@/ is no longer relevant, it can be withdrawn with
-- 'GI.Gio.Objects.Application.applicationWithdrawNotification'.
-- 
-- It is an error to call this function if /@application@/ has no
-- application ID.
-- 
-- /Since: 2.40/
applicationSendNotification ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a, Gio.Notification.IsNotification b) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Maybe (T.Text)
    -- ^ /@id@/: id of the notification, or 'P.Nothing'
    -> b
    -- ^ /@notification@/: the t'GI.Gio.Objects.Notification.Notification' to send
    -> m ()
applicationSendNotification application id notification = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeId <- case id of
        Nothing -> return FP.nullPtr
        Just jId -> do
            jId' <- textToCString jId
            return jId'
    notification' <- unsafeManagedPtrCastPtr notification
    g_application_send_notification application' maybeId notification'
    touchManagedPtr application
    touchManagedPtr notification
    freeMem maybeId
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSendNotificationMethodInfo
instance (signature ~ (Maybe (T.Text) -> b -> m ()), MonadIO m, IsApplication a, Gio.Notification.IsNotification b) => O.OverloadedMethod ApplicationSendNotificationMethodInfo a signature where
    overloadedMethod = applicationSendNotification

instance O.OverloadedMethodInfo ApplicationSendNotificationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSendNotification",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSendNotification"
        })


#endif

-- method Application::set_action_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GActionGroup, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_action_group" g_application_set_action_group :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr Gio.ActionGroup.ActionGroup ->      -- action_group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    IO ()

{-# DEPRECATED applicationSetActionGroup ["(Since version 2.32)","Use the t'GI.Gio.Interfaces.ActionMap.ActionMap' interface instead.  Never ever","mix use of this API with use of t'GI.Gio.Interfaces.ActionMap.ActionMap' on the same /@application@/","or things will go very badly wrong.  This function is known to","introduce buggy behaviour (ie: signals not emitted on changes to the","action group), so you should really use t'GI.Gio.Interfaces.ActionMap.ActionMap' instead."] #-}
-- | This used to be how actions were associated with a t'GI.Gio.Objects.Application.Application'.
-- Now there is t'GI.Gio.Interfaces.ActionMap.ActionMap' for that.
-- 
-- /Since: 2.28/
applicationSetActionGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a, Gio.ActionGroup.IsActionGroup b) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Maybe (b)
    -- ^ /@actionGroup@/: a t'GI.Gio.Interfaces.ActionGroup.ActionGroup', or 'P.Nothing'
    -> m ()
applicationSetActionGroup application actionGroup = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeActionGroup <- case actionGroup of
        Nothing -> return FP.nullPtr
        Just jActionGroup -> do
            jActionGroup' <- unsafeManagedPtrCastPtr jActionGroup
            return jActionGroup'
    g_application_set_action_group application' maybeActionGroup
    touchManagedPtr application
    whenJust actionGroup touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetActionGroupMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsApplication a, Gio.ActionGroup.IsActionGroup b) => O.OverloadedMethod ApplicationSetActionGroupMethodInfo a signature where
    overloadedMethod = applicationSetActionGroup

instance O.OverloadedMethodInfo ApplicationSetActionGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetActionGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetActionGroup"
        })


#endif

-- method Application::set_application_id
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "application_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the identifier for @application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_application_id" g_application_set_application_id :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- application_id : TBasicType TUTF8
    IO ()

-- | Sets the unique identifier for /@application@/.
-- 
-- The application id can only be modified if /@application@/ has not yet
-- been registered.
-- 
-- If non-'P.Nothing', the application id must be valid.  See
-- 'GI.Gio.Objects.Application.applicationIdIsValid'.
-- 
-- /Since: 2.28/
applicationSetApplicationId ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Maybe (T.Text)
    -- ^ /@applicationId@/: the identifier for /@application@/
    -> m ()
applicationSetApplicationId application applicationId = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeApplicationId <- case applicationId of
        Nothing -> return FP.nullPtr
        Just jApplicationId -> do
            jApplicationId' <- textToCString jApplicationId
            return jApplicationId'
    g_application_set_application_id application' maybeApplicationId
    touchManagedPtr application
    freeMem maybeApplicationId
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetApplicationIdMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetApplicationIdMethodInfo a signature where
    overloadedMethod = applicationSetApplicationId

instance O.OverloadedMethodInfo ApplicationSetApplicationIdMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetApplicationId",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetApplicationId"
        })


#endif

-- method Application::set_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the application to set as default, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_default" g_application_set_default :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Sets or unsets the default application for the process, as returned
-- by 'GI.Gio.Objects.Application.applicationGetDefault'.
-- 
-- This function does not take its own reference on /@application@/.  If
-- /@application@/ is destroyed then the default application will revert
-- back to 'P.Nothing'.
-- 
-- /Since: 2.32/
applicationSetDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    Maybe (a)
    -- ^ /@application@/: the application to set as default, or 'P.Nothing'
    -> m ()
applicationSetDefault application = liftIO $ do
    maybeApplication <- case application of
        Nothing -> return FP.nullPtr
        Just jApplication -> do
            jApplication' <- unsafeManagedPtrCastPtr jApplication
            return jApplication'
    g_application_set_default maybeApplication
    whenJust application touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetDefaultMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetDefaultMethodInfo a signature where
    overloadedMethod i = applicationSetDefault (Just i)

instance O.OverloadedMethodInfo ApplicationSetDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetDefault"
        })


#endif

-- method Application::set_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ApplicationFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the flags for @application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_flags" g_application_set_flags :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "ApplicationFlags"})
    IO ()

-- | Sets the flags for /@application@/.
-- 
-- The flags can only be modified if /@application@/ has not yet been
-- registered.
-- 
-- See t'GI.Gio.Flags.ApplicationFlags'.
-- 
-- /Since: 2.28/
applicationSetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> [Gio.Flags.ApplicationFlags]
    -- ^ /@flags@/: the flags for /@application@/
    -> m ()
applicationSetFlags application flags = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    let flags' = gflagsToWord flags
    g_application_set_flags application' flags'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetFlagsMethodInfo
instance (signature ~ ([Gio.Flags.ApplicationFlags] -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetFlagsMethodInfo a signature where
    overloadedMethod = applicationSetFlags

instance O.OverloadedMethodInfo ApplicationSetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetFlags"
        })


#endif

-- method Application::set_inactivity_timeout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "inactivity_timeout"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the timeout, in milliseconds"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_inactivity_timeout" g_application_set_inactivity_timeout :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Word32 ->                               -- inactivity_timeout : TBasicType TUInt
    IO ()

-- | Sets the current inactivity timeout for the application.
-- 
-- This is the amount of time (in milliseconds) after the last call to
-- 'GI.Gio.Objects.Application.applicationRelease' before the application stops running.
-- 
-- This call has no side effects of its own.  The value set here is only
-- used for next time 'GI.Gio.Objects.Application.applicationRelease' drops the use count to
-- zero.  Any timeouts currently in progress are not impacted.
-- 
-- /Since: 2.28/
applicationSetInactivityTimeout ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Word32
    -- ^ /@inactivityTimeout@/: the timeout, in milliseconds
    -> m ()
applicationSetInactivityTimeout application inactivityTimeout = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_set_inactivity_timeout application' inactivityTimeout
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetInactivityTimeoutMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetInactivityTimeoutMethodInfo a signature where
    overloadedMethod = applicationSetInactivityTimeout

instance O.OverloadedMethodInfo ApplicationSetInactivityTimeoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetInactivityTimeout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetInactivityTimeout"
        })


#endif

-- method Application::set_option_context_description
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "description"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string to be shown in `--help` output\n after the list of options, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_option_context_description" g_application_set_option_context_description :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- description : TBasicType TUTF8
    IO ()

-- | Adds a description to the /@application@/ option context.
-- 
-- See 'GI.GLib.Structs.OptionContext.optionContextSetDescription' for more information.
-- 
-- /Since: 2.56/
applicationSetOptionContextDescription ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: the t'GI.Gio.Objects.Application.Application'
    -> Maybe (T.Text)
    -- ^ /@description@/: a string to be shown in @--help@ output
    --  after the list of options, or 'P.Nothing'
    -> m ()
applicationSetOptionContextDescription application description = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeDescription <- case description of
        Nothing -> return FP.nullPtr
        Just jDescription -> do
            jDescription' <- textToCString jDescription
            return jDescription'
    g_application_set_option_context_description application' maybeDescription
    touchManagedPtr application
    freeMem maybeDescription
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetOptionContextDescriptionMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetOptionContextDescriptionMethodInfo a signature where
    overloadedMethod = applicationSetOptionContextDescription

instance O.OverloadedMethodInfo ApplicationSetOptionContextDescriptionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetOptionContextDescription",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetOptionContextDescription"
        })


#endif

-- method Application::set_option_context_parameter_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameter_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string which is displayed\n  in the first line of `--help` output, after the usage summary `programname [OPTION...]`."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_option_context_parameter_string" g_application_set_option_context_parameter_string :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- parameter_string : TBasicType TUTF8
    IO ()

-- | Sets the parameter string to be used by the commandline handling of /@application@/.
-- 
-- This function registers the argument to be passed to @/g_option_context_new()/@
-- when the internal t'GI.GLib.Structs.OptionContext.OptionContext' of /@application@/ is created.
-- 
-- See @/g_option_context_new()/@ for more information about /@parameterString@/.
-- 
-- /Since: 2.56/
applicationSetOptionContextParameterString ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: the t'GI.Gio.Objects.Application.Application'
    -> Maybe (T.Text)
    -- ^ /@parameterString@/: a string which is displayed
    --   in the first line of @--help@ output, after the usage summary @programname [OPTION...]@.
    -> m ()
applicationSetOptionContextParameterString application parameterString = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeParameterString <- case parameterString of
        Nothing -> return FP.nullPtr
        Just jParameterString -> do
            jParameterString' <- textToCString jParameterString
            return jParameterString'
    g_application_set_option_context_parameter_string application' maybeParameterString
    touchManagedPtr application
    freeMem maybeParameterString
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetOptionContextParameterStringMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetOptionContextParameterStringMethodInfo a signature where
    overloadedMethod = applicationSetOptionContextParameterString

instance O.OverloadedMethodInfo ApplicationSetOptionContextParameterStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetOptionContextParameterString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetOptionContextParameterString"
        })


#endif

-- method Application::set_option_context_summary
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "summary"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string to be shown in `--help` output\n before the list of options, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_option_context_summary" g_application_set_option_context_summary :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- summary : TBasicType TUTF8
    IO ()

-- | Adds a summary to the /@application@/ option context.
-- 
-- See 'GI.GLib.Structs.OptionContext.optionContextSetSummary' for more information.
-- 
-- /Since: 2.56/
applicationSetOptionContextSummary ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: the t'GI.Gio.Objects.Application.Application'
    -> Maybe (T.Text)
    -- ^ /@summary@/: a string to be shown in @--help@ output
    --  before the list of options, or 'P.Nothing'
    -> m ()
applicationSetOptionContextSummary application summary = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeSummary <- case summary of
        Nothing -> return FP.nullPtr
        Just jSummary -> do
            jSummary' <- textToCString jSummary
            return jSummary'
    g_application_set_option_context_summary application' maybeSummary
    touchManagedPtr application
    freeMem maybeSummary
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetOptionContextSummaryMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetOptionContextSummaryMethodInfo a signature where
    overloadedMethod = applicationSetOptionContextSummary

instance O.OverloadedMethodInfo ApplicationSetOptionContextSummaryMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetOptionContextSummary",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetOptionContextSummary"
        })


#endif

-- method Application::set_resource_base_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "resource_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the resource path to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_resource_base_path" g_application_set_resource_base_path :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- resource_path : TBasicType TUTF8
    IO ()

-- | Sets (or unsets) the base resource path of /@application@/.
-- 
-- The path is used to automatically load various
-- [application resources]t'GI.Gio.Structs.Resource.Resource' such as menu layouts and
-- action descriptions. The various types of resources will be found at
-- fixed names relative to the given base path.
-- 
-- By default, the resource base path is determined from the application
-- ID by prefixing \'\/\' and replacing each \'.\' with \'\/\'.  This is done at
-- the time that the t'GI.Gio.Objects.Application.Application' object is constructed.  Changes to
-- the application ID after that point will not have an impact on the
-- resource base path.
-- 
-- As an example, if the application has an ID of \"org.example.app\" then
-- the default resource base path will be \"\/org\/example\/app\".  If this
-- is a @/GtkApplication/@ (and you have not manually changed the path)
-- then Gtk will then search for the menus of the application at
-- \"\/org\/example\/app\/gtk\/menus.ui\".
-- 
-- See t'GI.Gio.Structs.Resource.Resource' for more information about adding resources to your
-- application.
-- 
-- You can disable automatic resource loading functionality by setting
-- the path to 'P.Nothing'.
-- 
-- Changing the resource base path once the application is running is
-- not recommended.  The point at which the resource path is consulted
-- for forming paths for various purposes is unspecified.  When writing
-- a sub-class of t'GI.Gio.Objects.Application.Application' you should either set the
-- [Application:resourceBasePath]("GI.Gio.Objects.Application#g:attr:resourceBasePath") property at construction time, or call
-- this function during the instance initialization. Alternatively, you
-- can call this function in the t'GI.Gio.Structs.ApplicationClass.ApplicationClass'.@/startup/@ virtual function,
-- before chaining up to the parent implementation.
-- 
-- /Since: 2.42/
applicationSetResourceBasePath ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> Maybe (T.Text)
    -- ^ /@resourcePath@/: the resource path to use
    -> m ()
applicationSetResourceBasePath application resourcePath = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    maybeResourcePath <- case resourcePath of
        Nothing -> return FP.nullPtr
        Just jResourcePath -> do
            jResourcePath' <- textToCString jResourcePath
            return jResourcePath'
    g_application_set_resource_base_path application' maybeResourcePath
    touchManagedPtr application
    freeMem maybeResourcePath
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetResourceBasePathMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetResourceBasePathMethodInfo a signature where
    overloadedMethod = applicationSetResourceBasePath

instance O.OverloadedMethodInfo ApplicationSetResourceBasePathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetResourceBasePath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetResourceBasePath"
        })


#endif

-- method Application::set_version
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "version"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the version of @application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_set_version" g_application_set_version :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- version : TBasicType TUTF8
    IO ()

-- | Sets the version number of /@application@/. This will be used to implement
-- a @--version@ command line argument
-- 
-- The application version can only be modified if /@application@/ has not yet
-- been registered.
-- 
-- /Since: 2.80/
applicationSetVersion ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> T.Text
    -- ^ /@version@/: the version of /@application@/
    -> m ()
applicationSetVersion application version = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    version' <- textToCString version
    g_application_set_version application' version'
    touchManagedPtr application
    freeMem version'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationSetVersionMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationSetVersionMethodInfo a signature where
    overloadedMethod = applicationSetVersion

instance O.OverloadedMethodInfo ApplicationSetVersionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationSetVersion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationSetVersion"
        })


#endif

-- method Application::unbind_busy_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of a boolean property of @object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_unbind_busy_property" g_application_unbind_busy_property :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property : TBasicType TUTF8
    IO ()

-- | Destroys a binding between /@property@/ and the busy state of
-- /@application@/ that was previously created with
-- 'GI.Gio.Objects.Application.applicationBindBusyProperty'.
-- 
-- /Since: 2.44/
applicationUnbindBusyProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a, GObject.Object.IsObject b) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> b
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@property@/: the name of a boolean property of /@object@/
    -> m ()
applicationUnbindBusyProperty application object property = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    object' <- unsafeManagedPtrCastPtr object
    property' <- textToCString property
    g_application_unbind_busy_property application' object' property'
    touchManagedPtr application
    touchManagedPtr object
    freeMem property'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationUnbindBusyPropertyMethodInfo
instance (signature ~ (b -> T.Text -> m ()), MonadIO m, IsApplication a, GObject.Object.IsObject b) => O.OverloadedMethod ApplicationUnbindBusyPropertyMethodInfo a signature where
    overloadedMethod = applicationUnbindBusyProperty

instance O.OverloadedMethodInfo ApplicationUnbindBusyPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationUnbindBusyProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationUnbindBusyProperty"
        })


#endif

-- method Application::unmark_busy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_unmark_busy" g_application_unmark_busy :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    IO ()

-- | Decreases the busy count of /@application@/.
-- 
-- When the busy count reaches zero, the new state will be propagated
-- to other processes.
-- 
-- This function must only be called to cancel the effect of a previous
-- call to 'GI.Gio.Objects.Application.applicationMarkBusy'.
-- 
-- /Since: 2.38/
applicationUnmarkBusy ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> m ()
applicationUnmarkBusy application = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    g_application_unmark_busy application'
    touchManagedPtr application
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationUnmarkBusyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationUnmarkBusyMethodInfo a signature where
    overloadedMethod = applicationUnmarkBusy

instance O.OverloadedMethodInfo ApplicationUnmarkBusyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationUnmarkBusy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationUnmarkBusy"
        })


#endif

-- method Application::withdraw_notification
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "application"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Application" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GApplication" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "id of a previously sent notification"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_application_withdraw_notification" g_application_withdraw_notification :: 
    Ptr Application ->                      -- application : TInterface (Name {namespace = "Gio", name = "Application"})
    CString ->                              -- id : TBasicType TUTF8
    IO ()

-- | Withdraws a notification that was sent with
-- 'GI.Gio.Objects.Application.applicationSendNotification'.
-- 
-- This call does nothing if a notification with /@id@/ doesn\'t exist or
-- the notification was never sent.
-- 
-- This function works even for notifications sent in previous
-- executions of this application, as long /@id@/ is the same as it was for
-- the sent notification.
-- 
-- Note that notifications are dismissed when the user clicks on one
-- of the buttons in a notification or triggers its default action, so
-- there is no need to explicitly withdraw the notification in that case.
-- 
-- /Since: 2.40/
applicationWithdrawNotification ::
    (B.CallStack.HasCallStack, MonadIO m, IsApplication a) =>
    a
    -- ^ /@application@/: a t'GI.Gio.Objects.Application.Application'
    -> T.Text
    -- ^ /@id@/: id of a previously sent notification
    -> m ()
applicationWithdrawNotification application id = liftIO $ do
    application' <- unsafeManagedPtrCastPtr application
    id' <- textToCString id
    g_application_withdraw_notification application' id'
    touchManagedPtr application
    freeMem id'
    return ()

#if defined(ENABLE_OVERLOADING)
data ApplicationWithdrawNotificationMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsApplication a) => O.OverloadedMethod ApplicationWithdrawNotificationMethodInfo a signature where
    overloadedMethod = applicationWithdrawNotification

instance O.OverloadedMethodInfo ApplicationWithdrawNotificationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Application.applicationWithdrawNotification",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Application.html#v:applicationWithdrawNotification"
        })


#endif

-- method Application::get_default
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Application" })
-- throws : False
-- Skip return : False

foreign import ccall "g_application_get_default" g_application_get_default :: 
    IO (Ptr Application)

-- | Returns the default t'GI.Gio.Objects.Application.Application' instance for this process.
-- 
-- Normally there is only one t'GI.Gio.Objects.Application.Application' per process and it becomes
-- the default when it is created.  You can exercise more control over
-- this by using 'GI.Gio.Objects.Application.applicationSetDefault'.
-- 
-- If there is no default application then 'P.Nothing' is returned.
-- 
-- /Since: 2.32/
applicationGetDefault ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m (Maybe Application)
    -- ^ __Returns:__ the default application for this process, or 'P.Nothing'
applicationGetDefault  = liftIO $ do
    result <- g_application_get_default
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Application) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Application::id_is_valid
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "application_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a potential application identifier"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_application_id_is_valid" g_application_id_is_valid :: 
    CString ->                              -- application_id : TBasicType TUTF8
    IO CInt

-- | Checks if /@applicationId@/ is a valid application identifier.
-- 
-- A valid ID is required for calls to 'GI.Gio.Objects.Application.applicationNew' and
-- 'GI.Gio.Objects.Application.applicationSetApplicationId'.
-- 
-- Application identifiers follow the same format as
-- <https://dbus.freedesktop.org/doc/dbus-specification.html#message-protocol-names-bus D-Bus well-known bus names>.
-- For convenience, the restrictions on application identifiers are
-- reproduced here:
-- 
-- * Application identifiers are composed of 1 or more elements separated by a
--   period (@.@) character. All elements must contain at least one character.
-- * Each element must only contain the ASCII characters @[A-Z][a-z][0-9]_-@,
--   with @-@ discouraged in new application identifiers. Each element must not
--   begin with a digit.
-- * Application identifiers must contain at least one @.@ (period) character
--   (and thus at least two elements).
-- * Application identifiers must not begin with a @.@ (period) character.
-- * Application identifiers must not exceed 255 characters.
-- 
-- 
-- Note that the hyphen (@-@) character is allowed in application identifiers,
-- but is problematic or not allowed in various specifications and APIs that
-- refer to D-Bus, such as
-- <http://docs.flatpak.org/en/latest/introduction.html#identifiers Flatpak application IDs>,
-- the
-- <https://specifications.freedesktop.org/desktop-entry-spec/desktop-entry-spec-latest.html#dbus `DBusActivatable` interface in the Desktop Entry Specification>,
-- and the convention that an application\'s \"main\" interface and object path
-- resemble its application identifier and bus name. To avoid situations that
-- require special-case handling, it is recommended that new application
-- identifiers consistently replace hyphens with underscores.
-- 
-- Like D-Bus interface names, application identifiers should start with the
-- reversed DNS domain name of the author of the interface (in lower-case), and
-- it is conventional for the rest of the application identifier to consist of
-- words run together, with initial capital letters.
-- 
-- As with D-Bus interface names, if the author\'s DNS domain name contains
-- hyphen\/minus characters they should be replaced by underscores, and if it
-- contains leading digits they should be escaped by prepending an underscore.
-- For example, if the owner of 7-zip.org used an application identifier for an
-- archiving application, it might be named @org._7_zip.Archiver@.
applicationIdIsValid ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@applicationId@/: a potential application identifier
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@applicationId@/ is valid
applicationIdIsValid applicationId = liftIO $ do
    applicationId' <- textToCString applicationId
    result <- g_application_id_is_valid applicationId'
    let result' = (/= 0) result
    freeMem applicationId'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif


