{-# LANGUAGE PatternSynonyms, ScopedTypeVariables, ViewPatterns #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Constants
    ( 
    pattern VOLUME_MONITOR_EXTENSION_POINT_NAME,
    pattern VOLUME_IDENTIFIER_KIND_UUID     ,
    pattern VOLUME_IDENTIFIER_KIND_UNIX_DEVICE,
    pattern VOLUME_IDENTIFIER_KIND_NFS_MOUNT,
    pattern VOLUME_IDENTIFIER_KIND_LABEL    ,
    pattern VOLUME_IDENTIFIER_KIND_HAL_UDI  ,
    pattern VOLUME_IDENTIFIER_KIND_CLASS    ,
    pattern VFS_EXTENSION_POINT_NAME        ,
    pattern TLS_DATABASE_PURPOSE_AUTHENTICATE_SERVER,
    pattern TLS_DATABASE_PURPOSE_AUTHENTICATE_CLIENT,
    pattern TLS_BACKEND_EXTENSION_POINT_NAME,
    pattern SETTINGS_BACKEND_EXTENSION_POINT_NAME,
    pattern PROXY_RESOLVER_EXTENSION_POINT_NAME,
    pattern PROXY_EXTENSION_POINT_NAME      ,
    pattern POWER_PROFILE_MONITOR_EXTENSION_POINT_NAME,
    pattern NETWORK_MONITOR_EXTENSION_POINT_NAME,
    pattern NATIVE_VOLUME_MONITOR_EXTENSION_POINT_NAME,
    pattern MENU_LINK_SUBMENU               ,
    pattern MENU_LINK_SECTION               ,
    pattern MENU_EXPORTER_MAX_SECTION_SIZE  ,
    pattern MENU_ATTRIBUTE_TARGET           ,
    pattern MENU_ATTRIBUTE_LABEL            ,
    pattern MENU_ATTRIBUTE_ICON             ,
    pattern MENU_ATTRIBUTE_ACTION_NAMESPACE ,
    pattern MENU_ATTRIBUTE_ACTION           ,
    pattern MEMORY_MONITOR_EXTENSION_POINT_NAME,
    pattern FILE_ATTRIBUTE_UNIX_UID         ,
    pattern FILE_ATTRIBUTE_UNIX_RDEV        ,
    pattern FILE_ATTRIBUTE_UNIX_NLINK       ,
    pattern FILE_ATTRIBUTE_UNIX_MODE        ,
    pattern FILE_ATTRIBUTE_UNIX_IS_MOUNTPOINT,
    pattern FILE_ATTRIBUTE_UNIX_INODE       ,
    pattern FILE_ATTRIBUTE_UNIX_GID         ,
    pattern FILE_ATTRIBUTE_UNIX_DEVICE      ,
    pattern FILE_ATTRIBUTE_UNIX_BLOCK_SIZE  ,
    pattern FILE_ATTRIBUTE_UNIX_BLOCKS      ,
    pattern FILE_ATTRIBUTE_TRASH_ORIG_PATH  ,
    pattern FILE_ATTRIBUTE_TRASH_ITEM_COUNT ,
    pattern FILE_ATTRIBUTE_TRASH_DELETION_DATE,
    pattern FILE_ATTRIBUTE_TIME_MODIFIED_USEC,
    pattern FILE_ATTRIBUTE_TIME_MODIFIED_NSEC,
    pattern FILE_ATTRIBUTE_TIME_MODIFIED    ,
    pattern FILE_ATTRIBUTE_TIME_CREATED_USEC,
    pattern FILE_ATTRIBUTE_TIME_CREATED_NSEC,
    pattern FILE_ATTRIBUTE_TIME_CREATED     ,
    pattern FILE_ATTRIBUTE_TIME_CHANGED_USEC,
    pattern FILE_ATTRIBUTE_TIME_CHANGED_NSEC,
    pattern FILE_ATTRIBUTE_TIME_CHANGED     ,
    pattern FILE_ATTRIBUTE_TIME_ACCESS_USEC ,
    pattern FILE_ATTRIBUTE_TIME_ACCESS_NSEC ,
    pattern FILE_ATTRIBUTE_TIME_ACCESS      ,
    pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_XXLARGE,
    pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_XLARGE,
    pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_NORMAL,
    pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_LARGE,
    pattern FILE_ATTRIBUTE_THUMBNAIL_PATH   ,
    pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_XXLARGE,
    pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_XLARGE,
    pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_NORMAL,
    pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_LARGE,
    pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID,
    pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_XXLARGE,
    pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_XLARGE,
    pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_NORMAL,
    pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_LARGE,
    pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED,
    pattern FILE_ATTRIBUTE_STANDARD_TYPE    ,
    pattern FILE_ATTRIBUTE_STANDARD_TARGET_URI,
    pattern FILE_ATTRIBUTE_STANDARD_SYMLINK_TARGET,
    pattern FILE_ATTRIBUTE_STANDARD_SYMBOLIC_ICON,
    pattern FILE_ATTRIBUTE_STANDARD_SORT_ORDER,
    pattern FILE_ATTRIBUTE_STANDARD_SIZE    ,
    pattern FILE_ATTRIBUTE_STANDARD_NAME    ,
    pattern FILE_ATTRIBUTE_STANDARD_IS_VOLATILE,
    pattern FILE_ATTRIBUTE_STANDARD_IS_VIRTUAL,
    pattern FILE_ATTRIBUTE_STANDARD_IS_SYMLINK,
    pattern FILE_ATTRIBUTE_STANDARD_IS_HIDDEN,
    pattern FILE_ATTRIBUTE_STANDARD_IS_BACKUP,
    pattern FILE_ATTRIBUTE_STANDARD_ICON    ,
    pattern FILE_ATTRIBUTE_STANDARD_FAST_CONTENT_TYPE,
    pattern FILE_ATTRIBUTE_STANDARD_EDIT_NAME,
    pattern FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME,
    pattern FILE_ATTRIBUTE_STANDARD_DESCRIPTION,
    pattern FILE_ATTRIBUTE_STANDARD_COPY_NAME,
    pattern FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE,
    pattern FILE_ATTRIBUTE_STANDARD_ALLOCATED_SIZE,
    pattern FILE_ATTRIBUTE_SELINUX_CONTEXT  ,
    pattern FILE_ATTRIBUTE_RECENT_MODIFIED  ,
    pattern FILE_ATTRIBUTE_PREVIEW_ICON     ,
    pattern FILE_ATTRIBUTE_OWNER_USER_REAL  ,
    pattern FILE_ATTRIBUTE_OWNER_USER       ,
    pattern FILE_ATTRIBUTE_OWNER_GROUP      ,
    pattern FILE_ATTRIBUTE_MOUNTABLE_UNIX_DEVICE_FILE,
    pattern FILE_ATTRIBUTE_MOUNTABLE_UNIX_DEVICE,
    pattern FILE_ATTRIBUTE_MOUNTABLE_START_STOP_TYPE,
    pattern FILE_ATTRIBUTE_MOUNTABLE_IS_MEDIA_CHECK_AUTOMATIC,
    pattern FILE_ATTRIBUTE_MOUNTABLE_HAL_UDI,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_UNMOUNT,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_STOP,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_START_DEGRADED,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_START,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_POLL,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_MOUNT,
    pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_EJECT,
    pattern FILE_ATTRIBUTE_ID_FILESYSTEM    ,
    pattern FILE_ATTRIBUTE_ID_FILE          ,
    pattern FILE_ATTRIBUTE_GVFS_BACKEND     ,
    pattern FILE_ATTRIBUTE_FILESYSTEM_USE_PREVIEW,
    pattern FILE_ATTRIBUTE_FILESYSTEM_USED  ,
    pattern FILE_ATTRIBUTE_FILESYSTEM_TYPE  ,
    pattern FILE_ATTRIBUTE_FILESYSTEM_SIZE  ,
    pattern FILE_ATTRIBUTE_FILESYSTEM_REMOTE,
    pattern FILE_ATTRIBUTE_FILESYSTEM_READONLY,
    pattern FILE_ATTRIBUTE_FILESYSTEM_FREE  ,
    pattern FILE_ATTRIBUTE_ETAG_VALUE       ,
    pattern FILE_ATTRIBUTE_DOS_REPARSE_POINT_TAG,
    pattern FILE_ATTRIBUTE_DOS_IS_SYSTEM    ,
    pattern FILE_ATTRIBUTE_DOS_IS_MOUNTPOINT,
    pattern FILE_ATTRIBUTE_DOS_IS_ARCHIVE   ,
    pattern FILE_ATTRIBUTE_ACCESS_CAN_WRITE ,
    pattern FILE_ATTRIBUTE_ACCESS_CAN_TRASH ,
    pattern FILE_ATTRIBUTE_ACCESS_CAN_RENAME,
    pattern FILE_ATTRIBUTE_ACCESS_CAN_READ  ,
    pattern FILE_ATTRIBUTE_ACCESS_CAN_EXECUTE,
    pattern FILE_ATTRIBUTE_ACCESS_CAN_DELETE,
    pattern DRIVE_IDENTIFIER_KIND_UNIX_DEVICE,
    pattern DEBUG_CONTROLLER_EXTENSION_POINT_NAME,
    pattern DBUS_METHOD_INVOCATION_UNHANDLED,
    pattern DBUS_METHOD_INVOCATION_HANDLED  ,

    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Extension point for volume monitor functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
pattern VOLUME_MONITOR_EXTENSION_POINT_NAME = "gio-volume-monitor" :: T.Text

-- | The string used to obtain a UUID with 'GI.Gio.Interfaces.Volume.volumeGetIdentifier'.
pattern VOLUME_IDENTIFIER_KIND_UUID = "uuid" :: T.Text

-- | The string used to obtain a Unix device path with 'GI.Gio.Interfaces.Volume.volumeGetIdentifier'.
pattern VOLUME_IDENTIFIER_KIND_UNIX_DEVICE = "unix-device" :: T.Text

-- | The string used to obtain a NFS mount with 'GI.Gio.Interfaces.Volume.volumeGetIdentifier'.
pattern VOLUME_IDENTIFIER_KIND_NFS_MOUNT = "nfs-mount" :: T.Text

-- | The string used to obtain a filesystem label with 'GI.Gio.Interfaces.Volume.volumeGetIdentifier'.
pattern VOLUME_IDENTIFIER_KIND_LABEL = "label" :: T.Text

{-# DEPRECATED VOLUME_IDENTIFIER_KIND_HAL_UDI ["(Since version 2.58)","Do not use, HAL is deprecated."] #-}
-- | The string used to obtain a Hal UDI with 'GI.Gio.Interfaces.Volume.volumeGetIdentifier'.
pattern VOLUME_IDENTIFIER_KIND_HAL_UDI = "hal-udi" :: T.Text

-- | The string used to obtain the volume class with 'GI.Gio.Interfaces.Volume.volumeGetIdentifier'.
-- 
-- Known volume classes include @device@, @network@, and @loop@. Other
-- classes may be added in the future.
-- 
-- This is intended to be used by applications to classify t'GI.Gio.Interfaces.Volume.Volume'
-- instances into different sections - for example a file manager or
-- file chooser can use this information to show @network@ volumes under
-- a \"Network\" heading and @device@ volumes under a \"Devices\" heading.
pattern VOLUME_IDENTIFIER_KIND_CLASS = "class" :: T.Text

-- | Extension point for t'GI.Gio.Objects.Vfs.Vfs' functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
pattern VFS_EXTENSION_POINT_NAME = "gio-vfs" :: T.Text

-- | The purpose used to verify the server certificate in a TLS connection. This
-- is the most common purpose in use. Used by TLS clients.
pattern TLS_DATABASE_PURPOSE_AUTHENTICATE_SERVER = "1.3.6.1.5.5.7.3.1" :: T.Text

-- | The purpose used to verify the client certificate in a TLS connection.
-- Used by TLS servers.
pattern TLS_DATABASE_PURPOSE_AUTHENTICATE_CLIENT = "1.3.6.1.5.5.7.3.2" :: T.Text

-- | Extension point for TLS functionality via t'GI.Gio.Interfaces.TlsBackend.TlsBackend'.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
pattern TLS_BACKEND_EXTENSION_POINT_NAME = "gio-tls-backend" :: T.Text

-- | Extension point for t'GI.Gio.Objects.SettingsBackend.SettingsBackend' functionality.
pattern SETTINGS_BACKEND_EXTENSION_POINT_NAME = "gsettings-backend" :: T.Text

-- | Extension point for proxy resolving functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
pattern PROXY_RESOLVER_EXTENSION_POINT_NAME = "gio-proxy-resolver" :: T.Text

-- | Extension point for proxy functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
-- 
-- /Since: 2.26/
pattern PROXY_EXTENSION_POINT_NAME = "gio-proxy" :: T.Text

-- | Extension point for power profile usage monitoring functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
-- 
-- /Since: 2.70/
pattern POWER_PROFILE_MONITOR_EXTENSION_POINT_NAME = "gio-power-profile-monitor" :: T.Text

-- | Extension point for network status monitoring functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
-- 
-- /Since: 2.30/
pattern NETWORK_MONITOR_EXTENSION_POINT_NAME = "gio-network-monitor" :: T.Text

-- | /No description available in the introspection data./
pattern NATIVE_VOLUME_MONITOR_EXTENSION_POINT_NAME = "gio-native-volume-monitor" :: T.Text

-- | The name of the link that associates a menu item with a submenu.
-- 
-- See also 'GI.Gio.Objects.MenuItem.menuItemSetLink'.
-- 
-- /Since: 2.32/
pattern MENU_LINK_SUBMENU = "submenu" :: T.Text

-- | The name of the link that associates a menu item with a section.  The linked
-- menu will usually be shown in place of the menu item, using the item\'s label
-- as a header.
-- 
-- See also 'GI.Gio.Objects.MenuItem.menuItemSetLink'.
-- 
-- /Since: 2.32/
pattern MENU_LINK_SECTION = "section" :: T.Text

-- | The maximum number of entries in a menu section supported by
-- 'GI.Gio.Objects.DBusConnection.dBusConnectionExportMenuModel'.
-- 
-- The exact value of the limit may change in future GLib versions.
-- 
-- /Since: 2.76/
pattern MENU_EXPORTER_MAX_SECTION_SIZE = 1000 :: Int32

-- | The menu item attribute which holds the target with which the item\'s action
-- will be activated.
-- 
-- See also @/g_menu_item_set_action_and_target()/@
-- 
-- /Since: 2.32/
pattern MENU_ATTRIBUTE_TARGET = "target" :: T.Text

-- | The menu item attribute which holds the label of the item.
-- 
-- /Since: 2.32/
pattern MENU_ATTRIBUTE_LABEL = "label" :: T.Text

-- | The menu item attribute which holds the icon of the item.
-- 
-- The icon is stored in the format returned by 'GI.Gio.Interfaces.Icon.iconSerialize'.
-- 
-- This attribute is intended only to represent \'noun\' icons such as
-- favicons for a webpage, or application icons.  It should not be used
-- for \'verbs\' (ie: stock icons).
-- 
-- /Since: 2.38/
pattern MENU_ATTRIBUTE_ICON = "icon" :: T.Text

-- | The menu item attribute that holds the namespace for all action names in
-- menus that are linked from this item.
-- 
-- /Since: 2.36/
pattern MENU_ATTRIBUTE_ACTION_NAMESPACE = "action-namespace" :: T.Text

-- | The menu item attribute which holds the action name of the item.  Action
-- names are namespaced with an identifier for the action group in which the
-- action resides. For example, \"win.\" for window-specific actions and \"app.\"
-- for application-wide actions.
-- 
-- See also @/g_menu_model_get_item_attribute()/@ and @/g_menu_item_set_attribute()/@.
-- 
-- /Since: 2.32/
pattern MENU_ATTRIBUTE_ACTION = "action" :: T.Text

-- | Extension point for memory usage monitoring functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
-- 
-- /Since: 2.64/
pattern MEMORY_MONITOR_EXTENSION_POINT_NAME = "gio-memory-monitor" :: T.Text

-- | A key in the \"unix\" namespace for getting the user ID for the file.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_UID = "unix::uid" :: T.Text

-- | A key in the \"unix\" namespace for getting the device ID for the file
-- (if it is a special file).
-- 
-- See the documentation for @lstat()@.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_RDEV = "unix::rdev" :: T.Text

-- | A key in the \"unix\" namespace for getting the number of hard links
-- for a file.
-- 
-- See the documentation for @lstat()@.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_NLINK = "unix::nlink" :: T.Text

-- | A key in the \"unix\" namespace for getting the mode of the file
-- (e.g. whether the file is a regular file, symlink, etc).
-- 
-- See the documentation for @lstat()@: this attribute is equivalent to
-- the @st_mode@ member of @struct stat@, and includes both the file type
-- and permissions.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_MODE = "unix::mode" :: T.Text

-- | A key in the \"unix\" namespace for checking if the file represents a
-- UNIX mount point.
-- 
-- This attribute is 'P.True' if the file is a UNIX mount point.
-- 
-- Since 2.58, @\/@ is considered to be a mount point.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_UNIX_IS_MOUNTPOINT = "unix::is-mountpoint" :: T.Text

-- | A key in the \"unix\" namespace for getting the inode of the file.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
pattern FILE_ATTRIBUTE_UNIX_INODE = "unix::inode" :: T.Text

-- | A key in the \"unix\" namespace for getting the group ID for the file.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_GID = "unix::gid" :: T.Text

-- | A key in the \"unix\" namespace for getting the device id of the device the
-- file is located on (see @/stat()/@ documentation).
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_DEVICE = "unix::device" :: T.Text

-- | A key in the \"unix\" namespace for getting the block size for the file
-- system.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_UNIX_BLOCK_SIZE = "unix::block-size" :: T.Text

-- | A key in the \"unix\" namespace for getting the number of blocks allocated
-- for the file.
-- 
-- This attribute is only available for UNIX file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
pattern FILE_ATTRIBUTE_UNIX_BLOCKS = "unix::blocks" :: T.Text

-- | A key in the \"trash\" namespace for getting the original path of a file
-- inside the @trash:\/\/\/@ folder before it was trashed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
-- 
-- /Since: 2.24/
pattern FILE_ATTRIBUTE_TRASH_ORIG_PATH = "trash::orig-path" :: T.Text

-- | A key in the \"trash\" namespace for getting the number of (toplevel) items
-- that are present in the @trash:\/\/\/@ folder.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_TRASH_ITEM_COUNT = "trash::item-count" :: T.Text

-- | A key in the \"trash\" namespace for getting the deletion date and time
-- of a file inside the @trash:\/\/\/@ folder.
-- 
-- The format of the returned string is @YYYY-MM-DDThh:mm:ss@.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
-- 
-- /Since: 2.24/
pattern FILE_ATTRIBUTE_TRASH_DELETION_DATE = "trash::deletion-date" :: T.Text

-- | A key in the \"time\" namespace for getting the microseconds of the time
-- the file was last modified.
-- 
-- This should be used in conjunction with 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_MODIFIED'.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_TIME_MODIFIED_USEC = "time::modified-usec" :: T.Text

-- | A key in the \"time\" namespace for getting the nanoseconds of the time
-- the file was last modified. This should be used in conjunction with
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_MODIFIED'. Corresponding t'GI.Gio.Enums.FileAttributeType' is
-- 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- /Since: 2.74/
pattern FILE_ATTRIBUTE_TIME_MODIFIED_NSEC = "time::modified-nsec" :: T.Text

-- | A key in the \"time\" namespace for getting the time the file was last
-- modified.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64', and
-- contains the time since the file was modified, in seconds since the UNIX
-- epoch.
pattern FILE_ATTRIBUTE_TIME_MODIFIED = "time::modified" :: T.Text

-- | A key in the \"time\" namespace for getting the microseconds of the time
-- the file was created.
-- 
-- This should be used in conjunction with 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_CREATED'.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_TIME_CREATED_USEC = "time::created-usec" :: T.Text

-- | A key in the \"time\" namespace for getting the nanoseconds of the time
-- the file was created. This should be used in conjunction with
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_CREATED'. Corresponding t'GI.Gio.Enums.FileAttributeType' is
-- 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- /Since: 2.74/
pattern FILE_ATTRIBUTE_TIME_CREATED_NSEC = "time::created-nsec" :: T.Text

-- | A key in the \"time\" namespace for getting the time the file was created.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64',
-- and contains the time since the file was created, in seconds since the UNIX
-- epoch.
-- 
-- This may correspond to Linux @stx_btime@, FreeBSD @st_birthtim@, NetBSD
-- @st_birthtime@ or NTFS @ctime@.
pattern FILE_ATTRIBUTE_TIME_CREATED = "time::created" :: T.Text

-- | A key in the \"time\" namespace for getting the microseconds of the time
-- the file was last changed.
-- 
-- This should be used in conjunction with 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_CHANGED'.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_TIME_CHANGED_USEC = "time::changed-usec" :: T.Text

-- | A key in the \"time\" namespace for getting the nanoseconds of the time
-- the file was last changed. This should be used in conjunction with
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_CHANGED'. Corresponding t'GI.Gio.Enums.FileAttributeType' is
-- 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- /Since: 2.74/
pattern FILE_ATTRIBUTE_TIME_CHANGED_NSEC = "time::changed-nsec" :: T.Text

-- | A key in the \"time\" namespace for getting the time the file was last
-- changed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64',
-- and contains the time since the file was last changed, in seconds since
-- the UNIX epoch.
-- 
-- This corresponds to the traditional UNIX ctime.
pattern FILE_ATTRIBUTE_TIME_CHANGED = "time::changed" :: T.Text

-- | A key in the \"time\" namespace for getting the microseconds of the time
-- the file was last accessed.
-- 
-- This should be used in conjunction with 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_ACCESS'.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_TIME_ACCESS_USEC = "time::access-usec" :: T.Text

-- | A key in the \"time\" namespace for getting the nanoseconds of the time
-- the file was last accessed. This should be used in conjunction with
-- 'GI.Gio.Constants.FILE_ATTRIBUTE_TIME_ACCESS'. Corresponding t'GI.Gio.Enums.FileAttributeType' is
-- 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- /Since: 2.74/
pattern FILE_ATTRIBUTE_TIME_ACCESS_NSEC = "time::access-nsec" :: T.Text

-- | A key in the \"time\" namespace for getting the time the file was last
-- accessed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64', and
-- contains the time since the file was last accessed, in seconds since the
-- UNIX epoch.
pattern FILE_ATTRIBUTE_TIME_ACCESS = "time::access" :: T.Text

-- | A key in the \"thumbnail\" namespace for getting the path to the xx-large
-- thumbnail image.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_XXLARGE = "thumbnail::path-xxlarge" :: T.Text

-- | A key in the \"thumbnail\" namespace for getting the path to the x-large
-- thumbnail image.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_XLARGE = "thumbnail::path-xlarge" :: T.Text

-- | A key in the \"thumbnail\" namespace for getting the path to the normal
-- thumbnail image.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_NORMAL = "thumbnail::path-normal" :: T.Text

-- | A key in the \"thumbnail\" namespace for getting the path to the large
-- thumbnail image.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_PATH_LARGE = "thumbnail::path-large" :: T.Text

-- | A key in the \"thumbnail\" namespace for getting the path to the thumbnail
-- image with the biggest size available.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
pattern FILE_ATTRIBUTE_THUMBNAIL_PATH = "thumbnail::path" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking whether the xx-large
-- thumbnail is outdated.
-- 
-- This attribute is 'P.True' if the x-large thumbnail is up-to-date with the file
-- it represents, and 'P.False' if the file has been modified since the thumbnail
-- was generated.
-- 
-- If 'GI.Gio.Constants.FILE_ATTRIBUTE_THUMBNAILING_FAILED_XXLARGE' is 'P.True' and this attribute
-- is 'P.False', it indicates that thumbnailing may be attempted again and may
-- succeed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_XXLARGE = "thumbnail::is-valid-xxlarge" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking whether the x-large
-- thumbnail is outdated.
-- 
-- This attribute is 'P.True' if the x-large thumbnail is up-to-date with the file
-- it represents, and 'P.False' if the file has been modified since the thumbnail
-- was generated.
-- 
-- If 'GI.Gio.Constants.FILE_ATTRIBUTE_THUMBNAILING_FAILED_XLARGE' is 'P.True' and this attribute
-- is 'P.False', it indicates that thumbnailing may be attempted again and may
-- succeed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_XLARGE = "thumbnail::is-valid-xlarge" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking whether the normal
-- thumbnail is outdated.
-- 
-- This attribute is 'P.True' if the normal thumbnail is up-to-date with the file
-- it represents, and 'P.False' if the file has been modified since the thumbnail
-- was generated.
-- 
-- If 'GI.Gio.Constants.FILE_ATTRIBUTE_THUMBNAILING_FAILED_NORMAL' is 'P.True' and this attribute
-- is 'P.False', it indicates that thumbnailing may be attempted again and may
-- succeed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_NORMAL = "thumbnail::is-valid-normal" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking whether the large
-- thumbnail is outdated.
-- 
-- This attribute is 'P.True' if the large thumbnail is up-to-date with the file
-- it represents, and 'P.False' if the file has been modified since the thumbnail
-- was generated.
-- 
-- If 'GI.Gio.Constants.FILE_ATTRIBUTE_THUMBNAILING_FAILED_LARGE' is 'P.True' and this attribute
-- is 'P.False', it indicates that thumbnailing may be attempted again and may
-- succeed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID_LARGE = "thumbnail::is-valid-large" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking whether the thumbnail is outdated.
-- 
-- This attribute is 'P.True' if the thumbnail is up-to-date with the file it represents,
-- and 'P.False' if the file has been modified since the thumbnail was generated.
-- 
-- If 'GI.Gio.Constants.FILE_ATTRIBUTE_THUMBNAILING_FAILED' is 'P.True' and this attribute is 'P.False',
-- it indicates that thumbnailing may be attempted again and may succeed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.40/
pattern FILE_ATTRIBUTE_THUMBNAIL_IS_VALID = "thumbnail::is-valid" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking if thumbnailing failed
-- for the xx-large image.
-- 
-- This attribute is 'P.True' if thumbnailing failed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_XXLARGE = "thumbnail::failed-xxlarge" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking if thumbnailing failed
-- for the x-large image.
-- 
-- This attribute is 'P.True' if thumbnailing failed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_XLARGE = "thumbnail::failed-xlarge" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking if thumbnailing failed
-- for the normal image.
-- 
-- This attribute is 'P.True' if thumbnailing failed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_NORMAL = "thumbnail::failed-normal" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking if thumbnailing failed
-- for the large image.
-- 
-- This attribute is 'P.True' if thumbnailing failed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.76/
pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED_LARGE = "thumbnail::failed-large" :: T.Text

-- | A key in the \"thumbnail\" namespace for checking if thumbnailing failed.
-- 
-- This attribute is 'P.True' if thumbnailing failed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_THUMBNAILING_FAILED = "thumbnail::failed" :: T.Text

-- | A key in the \"standard\" namespace for storing file types.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- The value for this key should contain a t'GI.Gio.Enums.FileType'.
pattern FILE_ATTRIBUTE_STANDARD_TYPE = "standard::type" :: T.Text

-- | A key in the \"standard\" namespace for getting the target URI for the file, in
-- the case of 'GI.Gio.Enums.FileTypeShortcut' or 'GI.Gio.Enums.FileTypeMountable' files.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_STANDARD_TARGET_URI = "standard::target-uri" :: T.Text

-- | A key in the \"standard\" namespace for getting the symlink target, if the file
-- is a symlink.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
pattern FILE_ATTRIBUTE_STANDARD_SYMLINK_TARGET = "standard::symlink-target" :: T.Text

-- | A key in the \"standard\" namespace for getting the symbolic icon for the file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeObject'.
-- 
-- The value for this key should contain a t'GI.Gio.Interfaces.Icon.Icon'.
-- 
-- /Since: 2.34/
pattern FILE_ATTRIBUTE_STANDARD_SYMBOLIC_ICON = "standard::symbolic-icon" :: T.Text

-- | A key in the \"standard\" namespace for setting the sort order of a file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeInt32'.
-- 
-- An example use would be in file managers, which would use this key
-- to set the order files are displayed. Files with smaller sort order
-- should be sorted first, and files without sort order as if sort order
-- was zero.
pattern FILE_ATTRIBUTE_STANDARD_SORT_ORDER = "standard::sort-order" :: T.Text

-- | A key in the \"standard\" namespace for getting the file\'s size (in bytes).
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
pattern FILE_ATTRIBUTE_STANDARD_SIZE = "standard::size" :: T.Text

-- | A key in the \"standard\" namespace for getting the name of the file.
-- 
-- The name is the on-disk filename which may not be in any known encoding,
-- and can thus not be generally displayed as is. It is guaranteed to be set on
-- every file.
-- 
-- Use 'GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME' if you need to display the
-- name in a user interface.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeByteString'.
pattern FILE_ATTRIBUTE_STANDARD_NAME = "standard::name" :: T.Text

-- | A key in the \"standard\" namespace for checking if a file is
-- volatile. This is meant for opaque, non-POSIX-like backends to
-- indicate that the URI is not persistent. Applications should look
-- at 'GI.Gio.Constants.FILE_ATTRIBUTE_STANDARD_SYMLINK_TARGET' for the persistent URI.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.46/
pattern FILE_ATTRIBUTE_STANDARD_IS_VOLATILE = "standard::is-volatile" :: T.Text

-- | A key in the \"standard\" namespace for checking if a file is virtual.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_STANDARD_IS_VIRTUAL = "standard::is-virtual" :: T.Text

-- | A key in the \"standard\" namespace for checking if the file is a symlink.
-- Typically the actual type is something else, if we followed the symlink
-- to get the type.
-- 
-- On Windows NTFS mountpoints are considered to be symlinks as well.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_STANDARD_IS_SYMLINK = "standard::is-symlink" :: T.Text

-- | A key in the \"standard\" namespace for checking if a file is hidden.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_STANDARD_IS_HIDDEN = "standard::is-hidden" :: T.Text

-- | A key in the \"standard\" namespace for checking if a file is a backup file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_STANDARD_IS_BACKUP = "standard::is-backup" :: T.Text

-- | A key in the \"standard\" namespace for getting the icon for the file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeObject'.
-- 
-- The value for this key should contain a t'GI.Gio.Interfaces.Icon.Icon'.
pattern FILE_ATTRIBUTE_STANDARD_ICON = "standard::icon" :: T.Text

-- | A key in the \"standard\" namespace for getting the fast content type.
-- 
-- The fast content type isn\'t as reliable as the regular one, as it
-- only uses the filename to guess it, but it is faster to calculate than the
-- regular content type.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_STANDARD_FAST_CONTENT_TYPE = "standard::fast-content-type" :: T.Text

-- | A key in the \"standard\" namespace for edit name of the file.
-- 
-- An edit name is similar to the display name, but it is meant to be
-- used when you want to rename the file in the UI. The display name
-- might contain information you don\'t want in the new filename (such as
-- \"(invalid unicode)\" if the filename was in an invalid encoding).
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_STANDARD_EDIT_NAME = "standard::edit-name" :: T.Text

-- | A key in the \"standard\" namespace for getting the display name of the file.
-- 
-- A display name is guaranteed to be in UTF-8 and can thus be displayed in
-- the UI. It is guaranteed to be set on every file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME = "standard::display-name" :: T.Text

-- | A key in the \"standard\" namespace for getting the description of the file.
-- 
-- The description is a utf8 string that describes the file, generally containing
-- the filename, but can also contain further information. Example descriptions
-- could be \"filename (on hostname)\" for a remote file or \"filename (in trash)\"
-- for a file in the trash. This is useful for instance as the window title
-- when displaying a directory or for a bookmarks menu.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_STANDARD_DESCRIPTION = "standard::description" :: T.Text

-- | A key in the \"standard\" namespace for getting the copy name of the file.
-- 
-- The copy name is an optional version of the name. If available it\'s always
-- in UTF8, and corresponds directly to the original filename (only transcoded to
-- UTF8). This is useful if you want to copy the file to another filesystem that
-- might have a different encoding. If the filename is not a valid string in the
-- encoding selected for the filesystem it is in then the copy name will not be set.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_STANDARD_COPY_NAME = "standard::copy-name" :: T.Text

-- | A key in the \"standard\" namespace for getting the content type of the file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
-- 
-- The value for this key should contain a valid content type.
pattern FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE = "standard::content-type" :: T.Text

-- | A key in the \"standard\" namespace for getting the amount of disk space
-- that is consumed by the file (in bytes).
-- 
-- This will generally be larger than the file size (due to block size
-- overhead) but can occasionally be smaller (for example, for sparse files).
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
-- 
-- /Since: 2.20/
pattern FILE_ATTRIBUTE_STANDARD_ALLOCATED_SIZE = "standard::allocated-size" :: T.Text

-- | A key in the \"selinux\" namespace for getting the file\'s SELinux
-- context.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
-- 
-- Note that this attribute is only available if GLib has been built
-- with SELinux support.
pattern FILE_ATTRIBUTE_SELINUX_CONTEXT = "selinux::context" :: T.Text

-- | A key in the \"recent\" namespace for getting time, when the metadata for the
-- file in @recent:\/\/\/@ was last changed.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeInt64'.
-- 
-- /Since: 2.52/
pattern FILE_ATTRIBUTE_RECENT_MODIFIED = "recent::modified" :: T.Text

-- | A key in the \"preview\" namespace for getting a t'GI.Gio.Interfaces.Icon.Icon' that can be
-- used to get preview of the file.
-- 
-- For example, it may be a low resolution thumbnail without metadata.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeObject'.
-- 
-- The value for this key should contain a t'GI.Gio.Interfaces.Icon.Icon'.
-- 
-- /Since: 2.20/
pattern FILE_ATTRIBUTE_PREVIEW_ICON = "preview::icon" :: T.Text

-- | A key in the \"owner\" namespace for getting the real name of the
-- user that owns the file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_OWNER_USER_REAL = "owner::user-real" :: T.Text

-- | A key in the \"owner\" namespace for getting the user name of the
-- file\'s owner.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_OWNER_USER = "owner::user" :: T.Text

-- | A key in the \"owner\" namespace for getting the file owner\'s group.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_OWNER_GROUP = "owner::group" :: T.Text

-- | A key in the \"mountable\" namespace for getting the unix device file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_UNIX_DEVICE_FILE = "mountable::unix-device-file" :: T.Text

-- | A key in the \"mountable\" namespace for getting the unix device.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
pattern FILE_ATTRIBUTE_MOUNTABLE_UNIX_DEVICE = "mountable::unix-device" :: T.Text

-- | A key in the \"mountable\" namespace for getting the t'GI.Gio.Enums.DriveStartStopType'.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_START_STOP_TYPE = "mountable::start-stop-type" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) is automatically polled for media.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_IS_MEDIA_CHECK_AUTOMATIC = "mountable::is-media-check-automatic" :: T.Text

-- | A key in the \"mountable\" namespace for getting the HAL UDI for the mountable
-- file.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_MOUNTABLE_HAL_UDI = "mountable::hal-udi" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE)  is unmountable.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_UNMOUNT = "mountable::can-unmount" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) can be stopped.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_STOP = "mountable::can-stop" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) can be started degraded.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_START_DEGRADED = "mountable::can-start-degraded" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) can be started.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_START = "mountable::can-start" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) can be polled.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.22/
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_POLL = "mountable::can-poll" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) is mountable.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_MOUNT = "mountable::can-mount" :: T.Text

-- | A key in the \"mountable\" namespace for checking if a file (of
-- type G_FILE_TYPE_MOUNTABLE) can be ejected.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_MOUNTABLE_CAN_EJECT = "mountable::can-eject" :: T.Text

-- | A key in the \"id\" namespace for getting the file system identifier.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
-- 
-- An example use would be during drag and drop to see if the source
-- and target are on the same filesystem (default to move) or not (default
-- to copy).
pattern FILE_ATTRIBUTE_ID_FILESYSTEM = "id::filesystem" :: T.Text

-- | A key in the \"id\" namespace for getting a file identifier.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
-- 
-- An example use would be during listing files, to avoid recursive
-- directory scanning.
pattern FILE_ATTRIBUTE_ID_FILE = "id::file" :: T.Text

-- | A key in the \"gvfs\" namespace that gets the name of the current
-- GVFS backend in use.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_GVFS_BACKEND = "gvfs::backend" :: T.Text

-- | A key in the \"filesystem\" namespace for hinting a file manager
-- application whether it should preview (e.g. thumbnail) files on the
-- file system.
-- 
-- The value for this key contain a t'GI.Gio.Enums.FilesystemPreviewType'.
pattern FILE_ATTRIBUTE_FILESYSTEM_USE_PREVIEW = "filesystem::use-preview" :: T.Text

-- | A key in the \"filesystem\" namespace for getting the number of bytes
-- used by data on the file system.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
-- 
-- /Since: 2.32/
pattern FILE_ATTRIBUTE_FILESYSTEM_USED = "filesystem::used" :: T.Text

-- | A key in the \"filesystem\" namespace for getting the file system\'s type.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_FILESYSTEM_TYPE = "filesystem::type" :: T.Text

-- | A key in the \"filesystem\" namespace for getting the total size (in
-- bytes) of the file system, used in 'GI.Gio.Interfaces.File.fileQueryFilesystemInfo'.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
pattern FILE_ATTRIBUTE_FILESYSTEM_SIZE = "filesystem::size" :: T.Text

-- | A key in the \"filesystem\" namespace for checking if the file system
-- is remote.
-- 
-- Is set to 'P.True' if the file system is remote.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_FILESYSTEM_REMOTE = "filesystem::remote" :: T.Text

-- | A key in the \"filesystem\" namespace for checking if the file system
-- is read only.
-- 
-- Is set to 'P.True' if the file system is read only.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_FILESYSTEM_READONLY = "filesystem::readonly" :: T.Text

-- | A key in the \"filesystem\" namespace for getting the number of bytes
-- of free space left on the file system.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint64'.
pattern FILE_ATTRIBUTE_FILESYSTEM_FREE = "filesystem::free" :: T.Text

-- | A key in the \"etag\" namespace for getting the value of the file\'s
-- entity tag.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeString'.
pattern FILE_ATTRIBUTE_ETAG_VALUE = "etag::value" :: T.Text

-- | A key in the \"dos\" namespace for getting the file NTFS reparse tag.
-- 
-- This value is 0 for files that are not reparse points.
-- 
-- See the <https://msdn.microsoft.com/en-us/library/dd541667.aspx Reparse Tags>
-- page for possible reparse tag values.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeUint32'.
-- 
-- /Since: 2.60/
pattern FILE_ATTRIBUTE_DOS_REPARSE_POINT_TAG = "dos::reparse-point-tag" :: T.Text

-- | A key in the \"dos\" namespace for checking if the file\'s backup flag
-- is set.
-- 
-- This attribute is 'P.True' if the backup flag is set.
-- 
-- This attribute is only available for DOS file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_DOS_IS_SYSTEM = "dos::is-system" :: T.Text

-- | A key in the \"dos\" namespace for checking if the file is a NTFS mount point
-- (a volume mount or a junction point).
-- 
-- This attribute is 'P.True' if file is a reparse point of type
-- <https://msdn.microsoft.com/en-us/library/dd541667.aspx IO_REPARSE_TAG_MOUNT_POINT>.
-- 
-- This attribute is only available for DOS file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- /Since: 2.60/
pattern FILE_ATTRIBUTE_DOS_IS_MOUNTPOINT = "dos::is-mountpoint" :: T.Text

-- | A key in the \"dos\" namespace for checking if the file\'s archive flag
-- is set.
-- 
-- This attribute is 'P.True' if the archive flag is set.
-- 
-- This attribute is only available for DOS file systems.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
pattern FILE_ATTRIBUTE_DOS_IS_ARCHIVE = "dos::is-archive" :: T.Text

-- | A key in the \"access\" namespace for getting write privileges.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- This attribute will be 'P.True' if the user is able to write to the file.
pattern FILE_ATTRIBUTE_ACCESS_CAN_WRITE = "access::can-write" :: T.Text

-- | A key in the \"access\" namespace for checking trashing privileges.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- This attribute will be 'P.True' if the user is able to move the file to
-- the trash.
pattern FILE_ATTRIBUTE_ACCESS_CAN_TRASH = "access::can-trash" :: T.Text

-- | A key in the \"access\" namespace for checking renaming privileges.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- This attribute will be 'P.True' if the user is able to rename the file.
pattern FILE_ATTRIBUTE_ACCESS_CAN_RENAME = "access::can-rename" :: T.Text

-- | A key in the \"access\" namespace for getting read privileges.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- This attribute will be 'P.True' if the user is able to read the file.
pattern FILE_ATTRIBUTE_ACCESS_CAN_READ = "access::can-read" :: T.Text

-- | A key in the \"access\" namespace for getting execution privileges.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- This attribute will be 'P.True' if the user is able to execute the file.
pattern FILE_ATTRIBUTE_ACCESS_CAN_EXECUTE = "access::can-execute" :: T.Text

-- | A key in the \"access\" namespace for checking deletion privileges.
-- 
-- Corresponding t'GI.Gio.Enums.FileAttributeType' is 'GI.Gio.Enums.FileAttributeTypeBoolean'.
-- 
-- This attribute will be 'P.True' if the user is able to delete the file.
pattern FILE_ATTRIBUTE_ACCESS_CAN_DELETE = "access::can-delete" :: T.Text

-- | The string used to obtain a Unix device path with 'GI.Gio.Interfaces.Drive.driveGetIdentifier'.
-- 
-- /Since: 2.58/
pattern DRIVE_IDENTIFIER_KIND_UNIX_DEVICE = "unix-device" :: T.Text

-- | Extension point for debug control functionality.
-- See <http://developer.gnome.org/gio/stable/overview.html#extending-gio Extending GIO>.
-- 
-- /Since: 2.72/
pattern DEBUG_CONTROLLER_EXTENSION_POINT_NAME = "gio-debug-controller" :: T.Text

-- | The value returned by handlers of the signals generated by
-- the @gdbus-codegen@ tool to indicate that a method call has not been
-- handled by an implementation. It is equal to 'P.False', but using
-- this macro is sometimes more readable.
-- 
-- In code that needs to be backwards-compatible with older GLib,
-- use 'P.False' instead.
-- 
-- /Since: 2.68/
pattern DBUS_METHOD_INVOCATION_UNHANDLED = P.False :: Bool

-- | The value returned by handlers of the signals generated by
-- the @gdbus-codegen@ tool to indicate that a method call has been
-- handled by an implementation. It is equal to 'P.True', but using
-- this macro is sometimes more readable.
-- 
-- In code that needs to be backwards-compatible with older GLib,
-- use 'P.True' instead, often written like this:
-- 
-- 
-- 	
-- >
-- >  g_dbus_method_invocation_return_error (invocation, ...);
-- >  return TRUE;    // handled
-- 
-- 
-- /Since: 2.68/
pattern DBUS_METHOD_INVOCATION_HANDLED = P.True :: Bool


