{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The ATK interface implemented by components with text content.
-- 
-- t'GI.Atk.Interfaces.Text.Text' should be implemented by @/AtkObjects/@ on behalf of widgets
-- that have text content which is either attributed or otherwise
-- non-trivial.  @/AtkObjects/@ whose text content is simple,
-- unattributed, and very brief may expose that content via
-- @/atk_object_get_name/@ instead; however if the text is editable,
-- multi-line, typically longer than three or four words, attributed,
-- selectable, or if the object already uses the \'name\' ATK property
-- for other information, the t'GI.Atk.Interfaces.Text.Text' interface should be used to
-- expose the text content.  In the case of editable text content,
-- t'GI.Atk.Interfaces.EditableText.EditableText' (a subtype of the t'GI.Atk.Interfaces.Text.Text' interface) should be
-- implemented instead.
-- 
--  t'GI.Atk.Interfaces.Text.Text' provides not only traversal facilities and change
-- notification for text content, but also caret tracking and glyph
-- bounding box calculations.  Note that the text strings are exposed
-- as UTF-8, and are therefore potentially multi-byte, and
-- caret-to-byte offset mapping makes no assumptions about the
-- character length; also bounding box glyph-to-offset mapping may be
-- complex for languages which use ligatures.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Atk.Interfaces.Text
    ( 

-- * Exported types
    Text(..)                                ,
    IsText                                  ,
    toText                                  ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [addSelection]("GI.Atk.Interfaces.Text#g:method:addSelection"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeSelection]("GI.Atk.Interfaces.Text#g:method:removeSelection"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [scrollSubstringTo]("GI.Atk.Interfaces.Text#g:method:scrollSubstringTo"), [scrollSubstringToPoint]("GI.Atk.Interfaces.Text#g:method:scrollSubstringToPoint"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBoundedRanges]("GI.Atk.Interfaces.Text#g:method:getBoundedRanges"), [getCaretOffset]("GI.Atk.Interfaces.Text#g:method:getCaretOffset"), [getCharacterAtOffset]("GI.Atk.Interfaces.Text#g:method:getCharacterAtOffset"), [getCharacterCount]("GI.Atk.Interfaces.Text#g:method:getCharacterCount"), [getCharacterExtents]("GI.Atk.Interfaces.Text#g:method:getCharacterExtents"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDefaultAttributes]("GI.Atk.Interfaces.Text#g:method:getDefaultAttributes"), [getNSelections]("GI.Atk.Interfaces.Text#g:method:getNSelections"), [getOffsetAtPoint]("GI.Atk.Interfaces.Text#g:method:getOffsetAtPoint"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRangeExtents]("GI.Atk.Interfaces.Text#g:method:getRangeExtents"), [getRunAttributes]("GI.Atk.Interfaces.Text#g:method:getRunAttributes"), [getSelection]("GI.Atk.Interfaces.Text#g:method:getSelection"), [getStringAtOffset]("GI.Atk.Interfaces.Text#g:method:getStringAtOffset"), [getText]("GI.Atk.Interfaces.Text#g:method:getText"), [getTextAfterOffset]("GI.Atk.Interfaces.Text#g:method:getTextAfterOffset"), [getTextAtOffset]("GI.Atk.Interfaces.Text#g:method:getTextAtOffset"), [getTextBeforeOffset]("GI.Atk.Interfaces.Text#g:method:getTextBeforeOffset").
-- 
-- ==== Setters
-- [setCaretOffset]("GI.Atk.Interfaces.Text#g:method:setCaretOffset"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setSelection]("GI.Atk.Interfaces.Text#g:method:setSelection").

#if defined(ENABLE_OVERLOADING)
    ResolveTextMethod                       ,
#endif

-- ** addSelection #method:addSelection#

#if defined(ENABLE_OVERLOADING)
    TextAddSelectionMethodInfo              ,
#endif
    textAddSelection                        ,


-- ** freeRanges #method:freeRanges#

    textFreeRanges                          ,


-- ** getBoundedRanges #method:getBoundedRanges#

#if defined(ENABLE_OVERLOADING)
    TextGetBoundedRangesMethodInfo          ,
#endif
    textGetBoundedRanges                    ,


-- ** getCaretOffset #method:getCaretOffset#

#if defined(ENABLE_OVERLOADING)
    TextGetCaretOffsetMethodInfo            ,
#endif
    textGetCaretOffset                      ,


-- ** getCharacterAtOffset #method:getCharacterAtOffset#

#if defined(ENABLE_OVERLOADING)
    TextGetCharacterAtOffsetMethodInfo      ,
#endif
    textGetCharacterAtOffset                ,


-- ** getCharacterCount #method:getCharacterCount#

#if defined(ENABLE_OVERLOADING)
    TextGetCharacterCountMethodInfo         ,
#endif
    textGetCharacterCount                   ,


-- ** getCharacterExtents #method:getCharacterExtents#

#if defined(ENABLE_OVERLOADING)
    TextGetCharacterExtentsMethodInfo       ,
#endif
    textGetCharacterExtents                 ,


-- ** getDefaultAttributes #method:getDefaultAttributes#

#if defined(ENABLE_OVERLOADING)
    TextGetDefaultAttributesMethodInfo      ,
#endif
    textGetDefaultAttributes                ,


-- ** getNSelections #method:getNSelections#

#if defined(ENABLE_OVERLOADING)
    TextGetNSelectionsMethodInfo            ,
#endif
    textGetNSelections                      ,


-- ** getOffsetAtPoint #method:getOffsetAtPoint#

#if defined(ENABLE_OVERLOADING)
    TextGetOffsetAtPointMethodInfo          ,
#endif
    textGetOffsetAtPoint                    ,


-- ** getRangeExtents #method:getRangeExtents#

#if defined(ENABLE_OVERLOADING)
    TextGetRangeExtentsMethodInfo           ,
#endif
    textGetRangeExtents                     ,


-- ** getRunAttributes #method:getRunAttributes#

#if defined(ENABLE_OVERLOADING)
    TextGetRunAttributesMethodInfo          ,
#endif
    textGetRunAttributes                    ,


-- ** getSelection #method:getSelection#

#if defined(ENABLE_OVERLOADING)
    TextGetSelectionMethodInfo              ,
#endif
    textGetSelection                        ,


-- ** getStringAtOffset #method:getStringAtOffset#

#if defined(ENABLE_OVERLOADING)
    TextGetStringAtOffsetMethodInfo         ,
#endif
    textGetStringAtOffset                   ,


-- ** getText #method:getText#

#if defined(ENABLE_OVERLOADING)
    TextGetTextMethodInfo                   ,
#endif
    textGetText                             ,


-- ** getTextAfterOffset #method:getTextAfterOffset#

#if defined(ENABLE_OVERLOADING)
    TextGetTextAfterOffsetMethodInfo        ,
#endif
    textGetTextAfterOffset                  ,


-- ** getTextAtOffset #method:getTextAtOffset#

#if defined(ENABLE_OVERLOADING)
    TextGetTextAtOffsetMethodInfo           ,
#endif
    textGetTextAtOffset                     ,


-- ** getTextBeforeOffset #method:getTextBeforeOffset#

#if defined(ENABLE_OVERLOADING)
    TextGetTextBeforeOffsetMethodInfo       ,
#endif
    textGetTextBeforeOffset                 ,


-- ** removeSelection #method:removeSelection#

#if defined(ENABLE_OVERLOADING)
    TextRemoveSelectionMethodInfo           ,
#endif
    textRemoveSelection                     ,


-- ** scrollSubstringTo #method:scrollSubstringTo#

#if defined(ENABLE_OVERLOADING)
    TextScrollSubstringToMethodInfo         ,
#endif
    textScrollSubstringTo                   ,


-- ** scrollSubstringToPoint #method:scrollSubstringToPoint#

#if defined(ENABLE_OVERLOADING)
    TextScrollSubstringToPointMethodInfo    ,
#endif
    textScrollSubstringToPoint              ,


-- ** setCaretOffset #method:setCaretOffset#

#if defined(ENABLE_OVERLOADING)
    TextSetCaretOffsetMethodInfo            ,
#endif
    textSetCaretOffset                      ,


-- ** setSelection #method:setSelection#

#if defined(ENABLE_OVERLOADING)
    TextSetSelectionMethodInfo              ,
#endif
    textSetSelection                        ,




 -- * Signals


-- ** textAttributesChanged #signal:textAttributesChanged#

    TextTextAttributesChangedCallback       ,
#if defined(ENABLE_OVERLOADING)
    TextTextAttributesChangedSignalInfo     ,
#endif
    afterTextTextAttributesChanged          ,
    onTextTextAttributesChanged             ,


-- ** textCaretMoved #signal:textCaretMoved#

    TextTextCaretMovedCallback              ,
#if defined(ENABLE_OVERLOADING)
    TextTextCaretMovedSignalInfo            ,
#endif
    afterTextTextCaretMoved                 ,
    onTextTextCaretMoved                    ,


-- ** textChanged #signal:textChanged#

    TextTextChangedCallback                 ,
#if defined(ENABLE_OVERLOADING)
    TextTextChangedSignalInfo               ,
#endif
    afterTextTextChanged                    ,
    onTextTextChanged                       ,


-- ** textInsert #signal:textInsert#

    TextTextInsertCallback                  ,
#if defined(ENABLE_OVERLOADING)
    TextTextInsertSignalInfo                ,
#endif
    afterTextTextInsert                     ,
    onTextTextInsert                        ,


-- ** textRemove #signal:textRemove#

    TextTextRemoveCallback                  ,
#if defined(ENABLE_OVERLOADING)
    TextTextRemoveSignalInfo                ,
#endif
    afterTextTextRemove                     ,
    onTextTextRemove                        ,


-- ** textSelectionChanged #signal:textSelectionChanged#

    TextTextSelectionChangedCallback        ,
#if defined(ENABLE_OVERLOADING)
    TextTextSelectionChangedSignalInfo      ,
#endif
    afterTextTextSelectionChanged           ,
    onTextTextSelectionChanged              ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.Atk.Enums as Atk.Enums
import {-# SOURCE #-} qualified GI.Atk.Structs.TextRange as Atk.TextRange
import {-# SOURCE #-} qualified GI.Atk.Structs.TextRectangle as Atk.TextRectangle
import qualified GI.GObject.Objects.Object as GObject.Object

#else
import {-# SOURCE #-} qualified GI.Atk.Enums as Atk.Enums
import {-# SOURCE #-} qualified GI.Atk.Structs.TextRange as Atk.TextRange
import {-# SOURCE #-} qualified GI.Atk.Structs.TextRectangle as Atk.TextRectangle
import qualified GI.GObject.Objects.Object as GObject.Object

#endif

-- interface Text 
-- | Memory-managed wrapper type.
newtype Text = Text (SP.ManagedPtr Text)
    deriving (Eq)

instance SP.ManagedPtrNewtype Text where
    toManagedPtr (Text p) = p

foreign import ccall "atk_text_get_type"
    c_atk_text_get_type :: IO B.Types.GType

instance B.Types.TypedObject Text where
    glibType = c_atk_text_get_type

instance B.Types.GObject Text

-- | Type class for types which can be safely cast to t'Text', for instance with `toText`.
class (SP.GObject o, O.IsDescendantOf Text o) => IsText o
instance (SP.GObject o, O.IsDescendantOf Text o) => IsText o

instance O.HasParentTypes Text
type instance O.ParentTypes Text = '[GObject.Object.Object]

-- | Cast to t'Text', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toText :: (MIO.MonadIO m, IsText o) => o -> m Text
toText = MIO.liftIO . B.ManagedPtr.unsafeCastTo Text

-- | Convert t'Text' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Text) where
    gvalueGType_ = c_atk_text_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Text)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Text)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Text ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Text
type instance O.AttributeList Text = TextAttributeList
type TextAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveTextMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTextMethod "addSelection" o = TextAddSelectionMethodInfo
    ResolveTextMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveTextMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveTextMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveTextMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveTextMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveTextMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveTextMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveTextMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveTextMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveTextMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveTextMethod "removeSelection" o = TextRemoveSelectionMethodInfo
    ResolveTextMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveTextMethod "scrollSubstringTo" o = TextScrollSubstringToMethodInfo
    ResolveTextMethod "scrollSubstringToPoint" o = TextScrollSubstringToPointMethodInfo
    ResolveTextMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveTextMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveTextMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveTextMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveTextMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveTextMethod "getBoundedRanges" o = TextGetBoundedRangesMethodInfo
    ResolveTextMethod "getCaretOffset" o = TextGetCaretOffsetMethodInfo
    ResolveTextMethod "getCharacterAtOffset" o = TextGetCharacterAtOffsetMethodInfo
    ResolveTextMethod "getCharacterCount" o = TextGetCharacterCountMethodInfo
    ResolveTextMethod "getCharacterExtents" o = TextGetCharacterExtentsMethodInfo
    ResolveTextMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveTextMethod "getDefaultAttributes" o = TextGetDefaultAttributesMethodInfo
    ResolveTextMethod "getNSelections" o = TextGetNSelectionsMethodInfo
    ResolveTextMethod "getOffsetAtPoint" o = TextGetOffsetAtPointMethodInfo
    ResolveTextMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveTextMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveTextMethod "getRangeExtents" o = TextGetRangeExtentsMethodInfo
    ResolveTextMethod "getRunAttributes" o = TextGetRunAttributesMethodInfo
    ResolveTextMethod "getSelection" o = TextGetSelectionMethodInfo
    ResolveTextMethod "getStringAtOffset" o = TextGetStringAtOffsetMethodInfo
    ResolveTextMethod "getText" o = TextGetTextMethodInfo
    ResolveTextMethod "getTextAfterOffset" o = TextGetTextAfterOffsetMethodInfo
    ResolveTextMethod "getTextAtOffset" o = TextGetTextAtOffsetMethodInfo
    ResolveTextMethod "getTextBeforeOffset" o = TextGetTextBeforeOffsetMethodInfo
    ResolveTextMethod "setCaretOffset" o = TextSetCaretOffsetMethodInfo
    ResolveTextMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveTextMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveTextMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveTextMethod "setSelection" o = TextSetSelectionMethodInfo
    ResolveTextMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTextMethod t Text, O.OverloadedMethod info Text p) => OL.IsLabel t (Text -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTextMethod t Text, O.OverloadedMethod info Text p, R.HasField t Text p) => R.HasField t Text p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTextMethod t Text, O.OverloadedMethodInfo info Text) => OL.IsLabel t (O.MethodProxy info Text) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Text::add_selection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the starting character offset of the selected region"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the offset of the first character after the selected region."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_add_selection" atk_text_add_selection :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    IO CInt

-- | Adds a selection bounded by the specified offsets.
textAddSelection ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@startOffset@/: the starting character offset of the selected region
    -> Int32
    -- ^ /@endOffset@/: the offset of the first character after the selected region.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if successful, 'P.False' otherwise
textAddSelection text startOffset endOffset = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_add_selection text' startOffset endOffset
    let result' = (/= 0) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextAddSelectionMethodInfo
instance (signature ~ (Int32 -> Int32 -> m Bool), MonadIO m, IsText a) => O.OverloadedMethod TextAddSelectionMethodInfo a signature where
    overloadedMethod = textAddSelection

instance O.OverloadedMethodInfo TextAddSelectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textAddSelection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textAddSelection"
        })


#endif

-- method Text::get_bounded_ranges
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rect"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextRectangle" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "An AtkTextRectangle giving the dimensions of the bounding box."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Specify whether coordinates are relative to the screen or widget window."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x_clip_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextClipType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Specify the horizontal clip type."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y_clip_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextClipType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Specify the vertical clip type."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TCArray
--                  True
--                  (-1)
--                  (-1)
--                  (TInterface Name { namespace = "Atk" , name = "TextRange" }))
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_bounded_ranges" atk_text_get_bounded_ranges :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Ptr Atk.TextRectangle.TextRectangle ->  -- rect : TInterface (Name {namespace = "Atk", name = "TextRectangle"})
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    CUInt ->                                -- x_clip_type : TInterface (Name {namespace = "Atk", name = "TextClipType"})
    CUInt ->                                -- y_clip_type : TInterface (Name {namespace = "Atk", name = "TextClipType"})
    IO (Ptr (Ptr Atk.TextRange.TextRange))

-- | Get the ranges of text in the specified bounding box.
-- 
-- /Since: 1.3/
textGetBoundedRanges ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Atk.TextRectangle.TextRectangle
    -- ^ /@rect@/: An AtkTextRectangle giving the dimensions of the bounding box.
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: Specify whether coordinates are relative to the screen or widget window.
    -> Atk.Enums.TextClipType
    -- ^ /@xClipType@/: Specify the horizontal clip type.
    -> Atk.Enums.TextClipType
    -- ^ /@yClipType@/: Specify the vertical clip type.
    -> m [Atk.TextRange.TextRange]
    -- ^ __Returns:__ Array of AtkTextRange. The last
    --          element of the array returned by this function will be NULL.
textGetBoundedRanges text rect coordType xClipType yClipType = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    rect' <- unsafeManagedPtrGetPtr rect
    let coordType' = (fromIntegral . fromEnum) coordType
    let xClipType' = (fromIntegral . fromEnum) xClipType
    let yClipType' = (fromIntegral . fromEnum) yClipType
    result <- atk_text_get_bounded_ranges text' rect' coordType' xClipType' yClipType'
    checkUnexpectedReturnNULL "textGetBoundedRanges" result
    result' <- unpackZeroTerminatedPtrArray result
    result'' <- mapM (wrapBoxed Atk.TextRange.TextRange) result'
    freeMem result
    touchManagedPtr text
    touchManagedPtr rect
    return result''

#if defined(ENABLE_OVERLOADING)
data TextGetBoundedRangesMethodInfo
instance (signature ~ (Atk.TextRectangle.TextRectangle -> Atk.Enums.CoordType -> Atk.Enums.TextClipType -> Atk.Enums.TextClipType -> m [Atk.TextRange.TextRange]), MonadIO m, IsText a) => O.OverloadedMethod TextGetBoundedRangesMethodInfo a signature where
    overloadedMethod = textGetBoundedRanges

instance O.OverloadedMethodInfo TextGetBoundedRangesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetBoundedRanges",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetBoundedRanges"
        })


#endif

-- method Text::get_caret_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_caret_offset" atk_text_get_caret_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    IO Int32

-- | Gets the offset of the position of the caret (cursor).
textGetCaretOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> m Int32
    -- ^ __Returns:__ the character offset of the position of the caret or -1 if
    --          the caret is not located inside the element or in the case of
    --          any other failure.
textGetCaretOffset text = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_get_caret_offset text'
    touchManagedPtr text
    return result

#if defined(ENABLE_OVERLOADING)
data TextGetCaretOffsetMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsText a) => O.OverloadedMethod TextGetCaretOffsetMethodInfo a signature where
    overloadedMethod = textGetCaretOffset

instance O.OverloadedMethodInfo TextGetCaretOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetCaretOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetCaretOffset"
        })


#endif

-- method Text::get_character_at_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a character offset within @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUniChar)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_character_at_offset" atk_text_get_character_at_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    IO CInt

-- | Gets the specified text.
textGetCharacterAtOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: a character offset within /@text@/
    -> m Char
    -- ^ __Returns:__ the character at /@offset@/ or 0 in the case of failure.
textGetCharacterAtOffset text offset = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_get_character_at_offset text' offset
    let result' = (chr . fromIntegral) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextGetCharacterAtOffsetMethodInfo
instance (signature ~ (Int32 -> m Char), MonadIO m, IsText a) => O.OverloadedMethod TextGetCharacterAtOffsetMethodInfo a signature where
    overloadedMethod = textGetCharacterAtOffset

instance O.OverloadedMethodInfo TextGetCharacterAtOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetCharacterAtOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetCharacterAtOffset"
        })


#endif

-- method Text::get_character_count
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_character_count" atk_text_get_character_count :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    IO Int32

-- | Gets the character count.
textGetCharacterCount ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> m Int32
    -- ^ __Returns:__ the number of characters or -1 in case of failure.
textGetCharacterCount text = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_get_character_count text'
    touchManagedPtr text
    return result

#if defined(ENABLE_OVERLOADING)
data TextGetCharacterCountMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsText a) => O.OverloadedMethod TextGetCharacterCountMethodInfo a signature where
    overloadedMethod = textGetCharacterCount

instance O.OverloadedMethodInfo TextGetCharacterCountMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetCharacterCount",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetCharacterCount"
        })


#endif

-- method Text::get_character_extents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The offset of the text character for which bounding information is required."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Pointer for the x coordinate of the bounding box"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Pointer for the y coordinate of the bounding box"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Pointer for the width of the bounding box"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Pointer for the height of the bounding box"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "coords"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specify whether coordinates are relative to the screen or widget window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_character_extents" atk_text_get_character_extents :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    Ptr Int32 ->                            -- x : TBasicType TInt
    Ptr Int32 ->                            -- y : TBasicType TInt
    Ptr Int32 ->                            -- width : TBasicType TInt
    Ptr Int32 ->                            -- height : TBasicType TInt
    CUInt ->                                -- coords : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO ()

-- | If the extent can not be obtained (e.g. missing support), all of x, y, width,
-- height are set to -1.
-- 
-- Get the bounding box containing the glyph representing the character at
--     a particular text offset.
textGetCharacterExtents ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: The offset of the text character for which bounding information is required.
    -> Atk.Enums.CoordType
    -- ^ /@coords@/: specify whether coordinates are relative to the screen or widget window
    -> m ((Int32, Int32, Int32, Int32))
textGetCharacterExtents text offset coords = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    x <- allocMem :: IO (Ptr Int32)
    y <- allocMem :: IO (Ptr Int32)
    width <- allocMem :: IO (Ptr Int32)
    height <- allocMem :: IO (Ptr Int32)
    let coords' = (fromIntegral . fromEnum) coords
    atk_text_get_character_extents text' offset x y width height coords'
    x' <- peek x
    y' <- peek y
    width' <- peek width
    height' <- peek height
    touchManagedPtr text
    freeMem x
    freeMem y
    freeMem width
    freeMem height
    return (x', y', width', height')

#if defined(ENABLE_OVERLOADING)
data TextGetCharacterExtentsMethodInfo
instance (signature ~ (Int32 -> Atk.Enums.CoordType -> m ((Int32, Int32, Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetCharacterExtentsMethodInfo a signature where
    overloadedMethod = textGetCharacterExtents

instance O.OverloadedMethodInfo TextGetCharacterExtentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetCharacterExtents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetCharacterExtents"
        })


#endif

-- method Text::get_default_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TGSList (TBasicType TPtr))
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_default_attributes" atk_text_get_default_attributes :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    IO (Ptr (GSList (Ptr ())))

-- | Creates an @/AtkAttributeSet/@ which consists of the default values of
-- attributes for the text. See the enum AtkTextAttribute for types of text
-- attributes that can be returned. Note that other attributes may also be
-- returned.
textGetDefaultAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> m ([Ptr ()])
    -- ^ __Returns:__ an @/AtkAttributeSet/@ which contains the default text
    --          attributes for this t'GI.Atk.Interfaces.Text.Text'. This @/AtkAttributeSet/@ should be freed by
    --          a call to 'GI.Atk.Functions.attributeSetFree'.
textGetDefaultAttributes text = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_get_default_attributes text'
    result' <- unpackGSList result
    g_slist_free result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextGetDefaultAttributesMethodInfo
instance (signature ~ (m ([Ptr ()])), MonadIO m, IsText a) => O.OverloadedMethod TextGetDefaultAttributesMethodInfo a signature where
    overloadedMethod = textGetDefaultAttributes

instance O.OverloadedMethodInfo TextGetDefaultAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetDefaultAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetDefaultAttributes"
        })


#endif

-- method Text::get_n_selections
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_n_selections" atk_text_get_n_selections :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    IO Int32

-- | Gets the number of selected regions.
textGetNSelections ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> m Int32
    -- ^ __Returns:__ The number of selected regions, or -1 in the case of failure.
textGetNSelections text = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_get_n_selections text'
    touchManagedPtr text
    return result

#if defined(ENABLE_OVERLOADING)
data TextGetNSelectionsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsText a) => O.OverloadedMethod TextGetNSelectionsMethodInfo a signature where
    overloadedMethod = textGetNSelections

instance O.OverloadedMethodInfo TextGetNSelectionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetNSelections",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetNSelections"
        })


#endif

-- method Text::get_offset_at_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "screen x-position of character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "screen y-position of character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coords"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specify whether coordinates are relative to the screen or\nwidget window"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_offset_at_point" atk_text_get_offset_at_point :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    CUInt ->                                -- coords : TInterface (Name {namespace = "Atk", name = "CoordType"})
    IO Int32

-- | Gets the offset of the character located at coordinates /@x@/ and /@y@/. /@x@/ and /@y@/
-- are interpreted as being relative to the screen or this widget\'s window
-- depending on /@coords@/.
textGetOffsetAtPoint ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@x@/: screen x-position of character
    -> Int32
    -- ^ /@y@/: screen y-position of character
    -> Atk.Enums.CoordType
    -- ^ /@coords@/: specify whether coordinates are relative to the screen or
    -- widget window
    -> m Int32
    -- ^ __Returns:__ the offset to the character which is located at  the specified
    --          /@x@/ and /@y@/ coordinates of -1 in case of failure.
textGetOffsetAtPoint text x y coords = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let coords' = (fromIntegral . fromEnum) coords
    result <- atk_text_get_offset_at_point text' x y coords'
    touchManagedPtr text
    return result

#if defined(ENABLE_OVERLOADING)
data TextGetOffsetAtPointMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.CoordType -> m Int32), MonadIO m, IsText a) => O.OverloadedMethod TextGetOffsetAtPointMethodInfo a signature where
    overloadedMethod = textGetOffsetAtPoint

instance O.OverloadedMethodInfo TextGetOffsetAtPointMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetOffsetAtPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetOffsetAtPoint"
        })


#endif

-- method Text::get_range_extents
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The offset of the first text character for which boundary\n       information is required."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The offset of the text character after the last character\n       for which boundary information is required."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coord_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Specify whether coordinates are relative to the screen or widget window."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rect"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextRectangle" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A pointer to a AtkTextRectangle which is filled in by this function."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_range_extents" atk_text_get_range_extents :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    CUInt ->                                -- coord_type : TInterface (Name {namespace = "Atk", name = "CoordType"})
    Ptr Atk.TextRectangle.TextRectangle ->  -- rect : TInterface (Name {namespace = "Atk", name = "TextRectangle"})
    IO ()

-- | Get the bounding box for text within the specified range.
-- 
-- If the extents can not be obtained (e.g. or missing support), the rectangle
-- fields are set to -1.
-- 
-- /Since: 1.3/
textGetRangeExtents ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@startOffset@/: The offset of the first text character for which boundary
    --        information is required.
    -> Int32
    -- ^ /@endOffset@/: The offset of the text character after the last character
    --        for which boundary information is required.
    -> Atk.Enums.CoordType
    -- ^ /@coordType@/: Specify whether coordinates are relative to the screen or widget window.
    -> m (Atk.TextRectangle.TextRectangle)
textGetRangeExtents text startOffset endOffset coordType = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let coordType' = (fromIntegral . fromEnum) coordType
    rect <- SP.callocBytes 16 :: IO (Ptr Atk.TextRectangle.TextRectangle)
    atk_text_get_range_extents text' startOffset endOffset coordType' rect
    rect' <- (wrapPtr Atk.TextRectangle.TextRectangle) rect
    touchManagedPtr text
    return rect'

#if defined(ENABLE_OVERLOADING)
data TextGetRangeExtentsMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.CoordType -> m (Atk.TextRectangle.TextRectangle)), MonadIO m, IsText a) => O.OverloadedMethod TextGetRangeExtentsMethodInfo a signature where
    overloadedMethod = textGetRangeExtents

instance O.OverloadedMethodInfo TextGetRangeExtentsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetRangeExtents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetRangeExtents"
        })


#endif

-- method Text::get_run_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the character offset at which to get the attributes, -1 means the offset of\nthe character to be inserted at the caret location."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the address to put the start offset of the range"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the address to put the end offset of the range"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TGSList (TBasicType TPtr))
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_run_attributes" atk_text_get_run_attributes :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    Ptr Int32 ->                            -- start_offset : TBasicType TInt
    Ptr Int32 ->                            -- end_offset : TBasicType TInt
    IO (Ptr (GSList (Ptr ())))

-- | Creates an @/AtkAttributeSet/@ which consists of the attributes explicitly
-- set at the position /@offset@/ in the text. /@startOffset@/ and /@endOffset@/ are
-- set to the start and end of the range around /@offset@/ where the attributes are
-- invariant. Note that /@endOffset@/ is the offset of the first character
-- after the range.  See the enum AtkTextAttribute for types of text
-- attributes that can be returned. Note that other attributes may also be
-- returned.
textGetRunAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: the character offset at which to get the attributes, -1 means the offset of
    -- the character to be inserted at the caret location.
    -> m (([Ptr ()], Int32, Int32))
    -- ^ __Returns:__ an @/AtkAttributeSet/@ which contains the attributes
    --         explicitly set at /@offset@/. This @/AtkAttributeSet/@ should be freed by
    --         a call to 'GI.Atk.Functions.attributeSetFree'.
textGetRunAttributes text offset = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    startOffset <- allocMem :: IO (Ptr Int32)
    endOffset <- allocMem :: IO (Ptr Int32)
    result <- atk_text_get_run_attributes text' offset startOffset endOffset
    result' <- unpackGSList result
    g_slist_free result
    startOffset' <- peek startOffset
    endOffset' <- peek endOffset
    touchManagedPtr text
    freeMem startOffset
    freeMem endOffset
    return (result', startOffset', endOffset')

#if defined(ENABLE_OVERLOADING)
data TextGetRunAttributesMethodInfo
instance (signature ~ (Int32 -> m (([Ptr ()], Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetRunAttributesMethodInfo a signature where
    overloadedMethod = textGetRunAttributes

instance O.OverloadedMethodInfo TextGetRunAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetRunAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetRunAttributes"
        })


#endif

-- method Text::get_selection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "selection_num"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The selection number.  The selected regions are\nassigned numbers that correspond to how far the region is from the\nstart of the text.  The selected region closest to the beginning\nof the text region is assigned the number 0, etc.  Note that adding,\nmoving or deleting a selected region can change the numbering."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "passes back the starting character offset of the selected region"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "passes back the ending character offset (offset immediately past)\nof the selected region"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_selection" atk_text_get_selection :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- selection_num : TBasicType TInt
    Ptr Int32 ->                            -- start_offset : TBasicType TInt
    Ptr Int32 ->                            -- end_offset : TBasicType TInt
    IO CString

-- | Gets the text from the specified selection.
textGetSelection ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@selectionNum@/: The selection number.  The selected regions are
    -- assigned numbers that correspond to how far the region is from the
    -- start of the text.  The selected region closest to the beginning
    -- of the text region is assigned the number 0, etc.  Note that adding,
    -- moving or deleting a selected region can change the numbering.
    -> m ((T.Text, Int32, Int32))
    -- ^ __Returns:__ a newly allocated string containing the selected text. Use 'GI.GLib.Functions.free'
    --          to free the returned string.
textGetSelection text selectionNum = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    startOffset <- allocMem :: IO (Ptr Int32)
    endOffset <- allocMem :: IO (Ptr Int32)
    result <- atk_text_get_selection text' selectionNum startOffset endOffset
    checkUnexpectedReturnNULL "textGetSelection" result
    result' <- cstringToText result
    freeMem result
    startOffset' <- peek startOffset
    endOffset' <- peek endOffset
    touchManagedPtr text
    freeMem startOffset
    freeMem endOffset
    return (result', startOffset', endOffset')

#if defined(ENABLE_OVERLOADING)
data TextGetSelectionMethodInfo
instance (signature ~ (Int32 -> m ((T.Text, Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetSelectionMethodInfo a signature where
    overloadedMethod = textGetSelection

instance O.OverloadedMethodInfo TextGetSelectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetSelection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetSelection"
        })


#endif

-- method Text::get_string_at_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "position" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "granularity"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextGranularity" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An #AtkTextGranularity"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the starting character offset of the returned string, or -1\n               in the case of error (e.g. invalid offset, not implemented)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the offset of the first character after the returned string,\n             or -1 in the case of error (e.g. invalid offset, not implemented)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_string_at_offset" atk_text_get_string_at_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    CUInt ->                                -- granularity : TInterface (Name {namespace = "Atk", name = "TextGranularity"})
    Ptr Int32 ->                            -- start_offset : TBasicType TInt
    Ptr Int32 ->                            -- end_offset : TBasicType TInt
    IO CString

-- | Gets a portion of the text exposed through an t'GI.Atk.Interfaces.Text.Text' according to a given /@offset@/
-- and a specific /@granularity@/, along with the start and end offsets defining the
-- boundaries of such a portion of text.
-- 
-- If /@granularity@/ is ATK_TEXT_GRANULARITY_CHAR the character at the
-- offset is returned.
-- 
-- If /@granularity@/ is ATK_TEXT_GRANULARITY_WORD the returned string
-- is from the word start at or before the offset to the word start after
-- the offset.
-- 
-- The returned string will contain the word at the offset if the offset
-- is inside a word and will contain the word before the offset if the
-- offset is not inside a word.
-- 
-- If /@granularity@/ is ATK_TEXT_GRANULARITY_SENTENCE the returned string
-- is from the sentence start at or before the offset to the sentence
-- start after the offset.
-- 
-- The returned string will contain the sentence at the offset if the offset
-- is inside a sentence and will contain the sentence before the offset
-- if the offset is not inside a sentence.
-- 
-- If /@granularity@/ is ATK_TEXT_GRANULARITY_LINE the returned string
-- is from the line start at or before the offset to the line
-- start after the offset.
-- 
-- If /@granularity@/ is ATK_TEXT_GRANULARITY_PARAGRAPH the returned string
-- is from the start of the paragraph at or before the offset to the start
-- of the following paragraph after the offset.
-- 
-- /Since: 2.10/
textGetStringAtOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: position
    -> Atk.Enums.TextGranularity
    -- ^ /@granularity@/: An t'GI.Atk.Enums.TextGranularity'
    -> m ((Maybe T.Text, Int32, Int32))
    -- ^ __Returns:__ a newly allocated string containing the text at
    --          the /@offset@/ bounded by the specified /@granularity@/. Use 'GI.GLib.Functions.free'
    --          to free the returned string.  Returns 'P.Nothing' if the offset is invalid
    --          or no implementation is available.
textGetStringAtOffset text offset granularity = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let granularity' = (fromIntegral . fromEnum) granularity
    startOffset <- allocMem :: IO (Ptr Int32)
    endOffset <- allocMem :: IO (Ptr Int32)
    result <- atk_text_get_string_at_offset text' offset granularity' startOffset endOffset
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    startOffset' <- peek startOffset
    endOffset' <- peek endOffset
    touchManagedPtr text
    freeMem startOffset
    freeMem endOffset
    return (maybeResult, startOffset', endOffset')

#if defined(ENABLE_OVERLOADING)
data TextGetStringAtOffsetMethodInfo
instance (signature ~ (Int32 -> Atk.Enums.TextGranularity -> m ((Maybe T.Text, Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetStringAtOffsetMethodInfo a signature where
    overloadedMethod = textGetStringAtOffset

instance O.OverloadedMethodInfo TextGetStringAtOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetStringAtOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetStringAtOffset"
        })


#endif

-- method Text::get_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a starting character offset within @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an ending character offset within @text, or -1 for the end of the string."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_text" atk_text_get_text :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    IO CString

-- | Gets the specified text.
textGetText ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@startOffset@/: a starting character offset within /@text@/
    -> Int32
    -- ^ /@endOffset@/: an ending character offset within /@text@/, or -1 for the end of the string.
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string containing the text from /@startOffset@/ up
    --          to, but not including /@endOffset@/. Use 'GI.GLib.Functions.free' to free the returned
    --          string.
textGetText text startOffset endOffset = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_get_text text' startOffset endOffset
    checkUnexpectedReturnNULL "textGetText" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextGetTextMethodInfo
instance (signature ~ (Int32 -> Int32 -> m T.Text), MonadIO m, IsText a) => O.OverloadedMethod TextGetTextMethodInfo a signature where
    overloadedMethod = textGetText

instance O.OverloadedMethodInfo TextGetTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetText"
        })


#endif

-- method Text::get_text_after_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "position" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "boundary_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextBoundary" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An #AtkTextBoundary"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the starting character offset of the returned string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the offset of the first character after the\n             returned substring"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_text_after_offset" atk_text_get_text_after_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    CUInt ->                                -- boundary_type : TInterface (Name {namespace = "Atk", name = "TextBoundary"})
    Ptr Int32 ->                            -- start_offset : TBasicType TInt
    Ptr Int32 ->                            -- end_offset : TBasicType TInt
    IO CString

{-# DEPRECATED textGetTextAfterOffset ["(Since version 2.9.3)","Please use 'GI.Atk.Interfaces.Text.textGetStringAtOffset' instead."] #-}
-- | Gets the specified text.
textGetTextAfterOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: position
    -> Atk.Enums.TextBoundary
    -- ^ /@boundaryType@/: An t'GI.Atk.Enums.TextBoundary'
    -> m ((T.Text, Int32, Int32))
    -- ^ __Returns:__ a newly allocated string containing the text after /@offset@/ bounded
    --          by the specified /@boundaryType@/. Use 'GI.GLib.Functions.free' to free the returned
    --          string.
textGetTextAfterOffset text offset boundaryType = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let boundaryType' = (fromIntegral . fromEnum) boundaryType
    startOffset <- allocMem :: IO (Ptr Int32)
    endOffset <- allocMem :: IO (Ptr Int32)
    result <- atk_text_get_text_after_offset text' offset boundaryType' startOffset endOffset
    checkUnexpectedReturnNULL "textGetTextAfterOffset" result
    result' <- cstringToText result
    freeMem result
    startOffset' <- peek startOffset
    endOffset' <- peek endOffset
    touchManagedPtr text
    freeMem startOffset
    freeMem endOffset
    return (result', startOffset', endOffset')

#if defined(ENABLE_OVERLOADING)
data TextGetTextAfterOffsetMethodInfo
instance (signature ~ (Int32 -> Atk.Enums.TextBoundary -> m ((T.Text, Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetTextAfterOffsetMethodInfo a signature where
    overloadedMethod = textGetTextAfterOffset

instance O.OverloadedMethodInfo TextGetTextAfterOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetTextAfterOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetTextAfterOffset"
        })


#endif

-- method Text::get_text_at_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "position" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "boundary_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextBoundary" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An #AtkTextBoundary"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the starting character offset of the returned string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the offset of the first character after the\n             returned substring"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_text_at_offset" atk_text_get_text_at_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    CUInt ->                                -- boundary_type : TInterface (Name {namespace = "Atk", name = "TextBoundary"})
    Ptr Int32 ->                            -- start_offset : TBasicType TInt
    Ptr Int32 ->                            -- end_offset : TBasicType TInt
    IO CString

{-# DEPRECATED textGetTextAtOffset ["This method is deprecated since ATK version","2.9.4. Please use 'GI.Atk.Interfaces.Text.textGetStringAtOffset' instead."] #-}
-- | Gets the specified text.
-- 
-- If the boundary_type if ATK_TEXT_BOUNDARY_CHAR the character at the
-- offset is returned.
-- 
-- If the boundary_type is ATK_TEXT_BOUNDARY_WORD_START the returned string
-- is from the word start at or before the offset to the word start after
-- the offset.
-- 
-- The returned string will contain the word at the offset if the offset
-- is inside a word and will contain the word before the offset if the
-- offset is not inside a word.
-- 
-- If the boundary type is ATK_TEXT_BOUNDARY_SENTENCE_START the returned
-- string is from the sentence start at or before the offset to the sentence
-- start after the offset.
-- 
-- The returned string will contain the sentence at the offset if the offset
-- is inside a sentence and will contain the sentence before the offset
-- if the offset is not inside a sentence.
-- 
-- If the boundary type is ATK_TEXT_BOUNDARY_LINE_START the returned
-- string is from the line start at or before the offset to the line
-- start after the offset.
textGetTextAtOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: position
    -> Atk.Enums.TextBoundary
    -- ^ /@boundaryType@/: An t'GI.Atk.Enums.TextBoundary'
    -> m ((T.Text, Int32, Int32))
    -- ^ __Returns:__ a newly allocated string containing the text at /@offset@/ bounded
    --          by the specified /@boundaryType@/. Use 'GI.GLib.Functions.free' to free the returned
    --          string.
textGetTextAtOffset text offset boundaryType = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let boundaryType' = (fromIntegral . fromEnum) boundaryType
    startOffset <- allocMem :: IO (Ptr Int32)
    endOffset <- allocMem :: IO (Ptr Int32)
    result <- atk_text_get_text_at_offset text' offset boundaryType' startOffset endOffset
    checkUnexpectedReturnNULL "textGetTextAtOffset" result
    result' <- cstringToText result
    freeMem result
    startOffset' <- peek startOffset
    endOffset' <- peek endOffset
    touchManagedPtr text
    freeMem startOffset
    freeMem endOffset
    return (result', startOffset', endOffset')

#if defined(ENABLE_OVERLOADING)
data TextGetTextAtOffsetMethodInfo
instance (signature ~ (Int32 -> Atk.Enums.TextBoundary -> m ((T.Text, Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetTextAtOffsetMethodInfo a signature where
    overloadedMethod = textGetTextAtOffset

instance O.OverloadedMethodInfo TextGetTextAtOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetTextAtOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetTextAtOffset"
        })


#endif

-- method Text::get_text_before_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "position" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "boundary_type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "TextBoundary" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An #AtkTextBoundary"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the starting character offset of the returned string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the offset of the first character after the\n             returned substring"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_get_text_before_offset" atk_text_get_text_before_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    CUInt ->                                -- boundary_type : TInterface (Name {namespace = "Atk", name = "TextBoundary"})
    Ptr Int32 ->                            -- start_offset : TBasicType TInt
    Ptr Int32 ->                            -- end_offset : TBasicType TInt
    IO CString

{-# DEPRECATED textGetTextBeforeOffset ["(Since version 2.9.3)","Please use 'GI.Atk.Interfaces.Text.textGetStringAtOffset' instead."] #-}
-- | Gets the specified text.
textGetTextBeforeOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: position
    -> Atk.Enums.TextBoundary
    -- ^ /@boundaryType@/: An t'GI.Atk.Enums.TextBoundary'
    -> m ((T.Text, Int32, Int32))
    -- ^ __Returns:__ a newly allocated string containing the text before /@offset@/ bounded
    --          by the specified /@boundaryType@/. Use 'GI.GLib.Functions.free' to free the returned
    --          string.
textGetTextBeforeOffset text offset boundaryType = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let boundaryType' = (fromIntegral . fromEnum) boundaryType
    startOffset <- allocMem :: IO (Ptr Int32)
    endOffset <- allocMem :: IO (Ptr Int32)
    result <- atk_text_get_text_before_offset text' offset boundaryType' startOffset endOffset
    checkUnexpectedReturnNULL "textGetTextBeforeOffset" result
    result' <- cstringToText result
    freeMem result
    startOffset' <- peek startOffset
    endOffset' <- peek endOffset
    touchManagedPtr text
    freeMem startOffset
    freeMem endOffset
    return (result', startOffset', endOffset')

#if defined(ENABLE_OVERLOADING)
data TextGetTextBeforeOffsetMethodInfo
instance (signature ~ (Int32 -> Atk.Enums.TextBoundary -> m ((T.Text, Int32, Int32))), MonadIO m, IsText a) => O.OverloadedMethod TextGetTextBeforeOffsetMethodInfo a signature where
    overloadedMethod = textGetTextBeforeOffset

instance O.OverloadedMethodInfo TextGetTextBeforeOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textGetTextBeforeOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textGetTextBeforeOffset"
        })


#endif

-- method Text::remove_selection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "selection_num"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The selection number.  The selected regions are\nassigned numbers that correspond to how far the region is from the\nstart of the text.  The selected region closest to the beginning\nof the text region is assigned the number 0, etc.  Note that adding,\nmoving or deleting a selected region can change the numbering."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_remove_selection" atk_text_remove_selection :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- selection_num : TBasicType TInt
    IO CInt

-- | Removes the specified selection.
textRemoveSelection ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@selectionNum@/: The selection number.  The selected regions are
    -- assigned numbers that correspond to how far the region is from the
    -- start of the text.  The selected region closest to the beginning
    -- of the text region is assigned the number 0, etc.  Note that adding,
    -- moving or deleting a selected region can change the numbering.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if successful, 'P.False' otherwise
textRemoveSelection text selectionNum = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_remove_selection text' selectionNum
    let result' = (/= 0) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextRemoveSelectionMethodInfo
instance (signature ~ (Int32 -> m Bool), MonadIO m, IsText a) => O.OverloadedMethod TextRemoveSelectionMethodInfo a signature where
    overloadedMethod = textRemoveSelection

instance O.OverloadedMethodInfo TextRemoveSelectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textRemoveSelection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textRemoveSelection"
        })


#endif

-- method Text::scroll_substring_to
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "start offset in the @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "end offset in the @text, or -1 for the end of the text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "ScrollType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "specify where the object should be made visible."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_scroll_substring_to" atk_text_scroll_substring_to :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    CUInt ->                                -- type : TInterface (Name {namespace = "Atk", name = "ScrollType"})
    IO CInt

-- | Makes a substring of /@text@/ visible on the screen by scrolling all necessary parents.
-- 
-- /Since: 2.32/
textScrollSubstringTo ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@startOffset@/: start offset in the /@text@/
    -> Int32
    -- ^ /@endOffset@/: end offset in the /@text@/, or -1 for the end of the text.
    -> Atk.Enums.ScrollType
    -- ^ /@type@/: specify where the object should be made visible.
    -> m Bool
    -- ^ __Returns:__ whether scrolling was successful.
textScrollSubstringTo text startOffset endOffset type_ = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let type_' = (fromIntegral . fromEnum) type_
    result <- atk_text_scroll_substring_to text' startOffset endOffset type_'
    let result' = (/= 0) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextScrollSubstringToMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.ScrollType -> m Bool), MonadIO m, IsText a) => O.OverloadedMethod TextScrollSubstringToMethodInfo a signature where
    overloadedMethod = textScrollSubstringTo

instance O.OverloadedMethodInfo TextScrollSubstringToMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textScrollSubstringTo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textScrollSubstringTo"
        })


#endif

-- method Text::scroll_substring_to_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "start offset in the @text"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "end offset in the @text, or -1 for the end of the text."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "coords"
--           , argType =
--               TInterface Name { namespace = "Atk" , name = "CoordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "specify whether coordinates are relative to the screen or to the\nparent object."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "x-position where to scroll to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "y-position where to scroll to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_scroll_substring_to_point" atk_text_scroll_substring_to_point :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    CUInt ->                                -- coords : TInterface (Name {namespace = "Atk", name = "CoordType"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    IO CInt

-- | Move the top-left of a substring of /@text@/ to a given position of the screen
-- by scrolling all necessary parents.
-- 
-- /Since: 2.32/
textScrollSubstringToPoint ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@startOffset@/: start offset in the /@text@/
    -> Int32
    -- ^ /@endOffset@/: end offset in the /@text@/, or -1 for the end of the text.
    -> Atk.Enums.CoordType
    -- ^ /@coords@/: specify whether coordinates are relative to the screen or to the
    -- parent object.
    -> Int32
    -- ^ /@x@/: x-position where to scroll to
    -> Int32
    -- ^ /@y@/: y-position where to scroll to
    -> m Bool
    -- ^ __Returns:__ whether scrolling was successful.
textScrollSubstringToPoint text startOffset endOffset coords x y = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    let coords' = (fromIntegral . fromEnum) coords
    result <- atk_text_scroll_substring_to_point text' startOffset endOffset coords' x y
    let result' = (/= 0) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextScrollSubstringToPointMethodInfo
instance (signature ~ (Int32 -> Int32 -> Atk.Enums.CoordType -> Int32 -> Int32 -> m Bool), MonadIO m, IsText a) => O.OverloadedMethod TextScrollSubstringToPointMethodInfo a signature where
    overloadedMethod = textScrollSubstringToPoint

instance O.OverloadedMethodInfo TextScrollSubstringToPointMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textScrollSubstringToPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textScrollSubstringToPoint"
        })


#endif

-- method Text::set_caret_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the character offset of the new caret position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_set_caret_offset" atk_text_set_caret_offset :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- offset : TBasicType TInt
    IO CInt

-- | Sets the caret (cursor) position to the specified /@offset@/.
-- 
-- In the case of rich-text content, this method should either grab focus
-- or move the sequential focus navigation starting point (if the application
-- supports this concept) as if the user had clicked on the new caret position.
-- Typically, this means that the target of this operation is the node containing
-- the new caret position or one of its ancestors. In other words, after this
-- method is called, if the user advances focus, it should move to the first
-- focusable node following the new caret position.
-- 
-- Calling this method should also scroll the application viewport in a way
-- that matches the behavior of the application\'s typical caret motion or tab
-- navigation as closely as possible. This also means that if the application\'s
-- caret motion or focus navigation does not trigger a scroll operation, this
-- method should not trigger one either. If the application does not have a caret
-- motion or focus navigation operation, this method should try to scroll the new
-- caret position into view while minimizing unnecessary scroll motion.
textSetCaretOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@offset@/: the character offset of the new caret position
    -> m Bool
    -- ^ __Returns:__ 'P.True' if successful, 'P.False' otherwise.
textSetCaretOffset text offset = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_set_caret_offset text' offset
    let result' = (/= 0) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextSetCaretOffsetMethodInfo
instance (signature ~ (Int32 -> m Bool), MonadIO m, IsText a) => O.OverloadedMethod TextSetCaretOffsetMethodInfo a signature where
    overloadedMethod = textSetCaretOffset

instance O.OverloadedMethodInfo TextSetCaretOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textSetCaretOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textSetCaretOffset"
        })


#endif

-- method Text::set_selection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TInterface Name { namespace = "Atk" , name = "Text" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an #AtkText" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "selection_num"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The selection number.  The selected regions are\nassigned numbers that correspond to how far the region is from the\nstart of the text.  The selected region closest to the beginning\nof the text region is assigned the number 0, etc.  Note that adding,\nmoving or deleting a selected region can change the numbering."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the new starting character offset of the selection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the new end position of (e.g. offset immediately past)\nthe selection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_set_selection" atk_text_set_selection :: 
    Ptr Text ->                             -- text : TInterface (Name {namespace = "Atk", name = "Text"})
    Int32 ->                                -- selection_num : TBasicType TInt
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    IO CInt

-- | Changes the start and end offset of the specified selection.
textSetSelection ::
    (B.CallStack.HasCallStack, MonadIO m, IsText a) =>
    a
    -- ^ /@text@/: an t'GI.Atk.Interfaces.Text.Text'
    -> Int32
    -- ^ /@selectionNum@/: The selection number.  The selected regions are
    -- assigned numbers that correspond to how far the region is from the
    -- start of the text.  The selected region closest to the beginning
    -- of the text region is assigned the number 0, etc.  Note that adding,
    -- moving or deleting a selected region can change the numbering.
    -> Int32
    -- ^ /@startOffset@/: the new starting character offset of the selection
    -> Int32
    -- ^ /@endOffset@/: the new end position of (e.g. offset immediately past)
    -- the selection
    -> m Bool
    -- ^ __Returns:__ 'P.True' if successful, 'P.False' otherwise
textSetSelection text selectionNum startOffset endOffset = liftIO $ do
    text' <- unsafeManagedPtrCastPtr text
    result <- atk_text_set_selection text' selectionNum startOffset endOffset
    let result' = (/= 0) result
    touchManagedPtr text
    return result'

#if defined(ENABLE_OVERLOADING)
data TextSetSelectionMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> m Bool), MonadIO m, IsText a) => O.OverloadedMethod TextSetSelectionMethodInfo a signature where
    overloadedMethod = textSetSelection

instance O.OverloadedMethodInfo TextSetSelectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text.textSetSelection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#v:textSetSelection"
        })


#endif

-- method Text::free_ranges
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "ranges"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 (-1)
--                 (TInterface Name { namespace = "Atk" , name = "TextRange" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A pointer to an array of #AtkTextRange which is\n  to be freed."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "atk_text_free_ranges" atk_text_free_ranges :: 
    Ptr Atk.TextRange.TextRange ->          -- ranges : TCArray False (-1) (-1) (TInterface (Name {namespace = "Atk", name = "TextRange"}))
    IO ()

-- | Frees the memory associated with an array of AtkTextRange. It is assumed
-- that the array was returned by the function atk_text_get_bounded_ranges
-- and is NULL terminated.
-- 
-- /Since: 1.3/
textFreeRanges ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Atk.TextRange.TextRange]
    -- ^ /@ranges@/: A pointer to an array of t'GI.Atk.Structs.TextRange.TextRange' which is
    --   to be freed.
    -> m ()
textFreeRanges ranges = liftIO $ do
    ranges' <- mapM unsafeManagedPtrGetPtr ranges
    ranges'' <- packBlockArray 32 ranges'
    atk_text_free_ranges ranges''
    mapM_ touchManagedPtr ranges
    freeMem ranges''
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- signal Text::text-attributes-changed
-- | The \"text-attributes-changed\" signal is emitted when the text
-- attributes of the text of an object which implements AtkText
-- changes.
type TextTextAttributesChangedCallback =
    IO ()

type C_TextTextAttributesChangedCallback =
    Ptr Text ->                             -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_TextTextAttributesChangedCallback`.
foreign import ccall "wrapper"
    mk_TextTextAttributesChangedCallback :: C_TextTextAttributesChangedCallback -> IO (FunPtr C_TextTextAttributesChangedCallback)

wrap_TextTextAttributesChangedCallback :: 
    GObject a => (a -> TextTextAttributesChangedCallback) ->
    C_TextTextAttributesChangedCallback
wrap_TextTextAttributesChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [textAttributesChanged](#signal:textAttributesChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' text #textAttributesChanged callback
-- @
-- 
-- 
onTextTextAttributesChanged :: (IsText a, MonadIO m) => a -> ((?self :: a) => TextTextAttributesChangedCallback) -> m SignalHandlerId
onTextTextAttributesChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextAttributesChangedCallback wrapped
    wrapped'' <- mk_TextTextAttributesChangedCallback wrapped'
    connectSignalFunPtr obj "text-attributes-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [textAttributesChanged](#signal:textAttributesChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' text #textAttributesChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterTextTextAttributesChanged :: (IsText a, MonadIO m) => a -> ((?self :: a) => TextTextAttributesChangedCallback) -> m SignalHandlerId
afterTextTextAttributesChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextAttributesChangedCallback wrapped
    wrapped'' <- mk_TextTextAttributesChangedCallback wrapped'
    connectSignalFunPtr obj "text-attributes-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data TextTextAttributesChangedSignalInfo
instance SignalInfo TextTextAttributesChangedSignalInfo where
    type HaskellCallbackType TextTextAttributesChangedSignalInfo = TextTextAttributesChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_TextTextAttributesChangedCallback cb
        cb'' <- mk_TextTextAttributesChangedCallback cb'
        connectSignalFunPtr obj "text-attributes-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text::text-attributes-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#g:signal:textAttributesChanged"})

#endif

-- signal Text::text-caret-moved
-- | The \"text-caret-moved\" signal is emitted when the caret
-- position of the text of an object which implements AtkText
-- changes.
type TextTextCaretMovedCallback =
    Int32
    -- ^ /@arg1@/: The new position of the text caret.
    -> IO ()

type C_TextTextCaretMovedCallback =
    Ptr Text ->                             -- object
    Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_TextTextCaretMovedCallback`.
foreign import ccall "wrapper"
    mk_TextTextCaretMovedCallback :: C_TextTextCaretMovedCallback -> IO (FunPtr C_TextTextCaretMovedCallback)

wrap_TextTextCaretMovedCallback :: 
    GObject a => (a -> TextTextCaretMovedCallback) ->
    C_TextTextCaretMovedCallback
wrap_TextTextCaretMovedCallback gi'cb gi'selfPtr arg1 _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  arg1


-- | Connect a signal handler for the [textCaretMoved](#signal:textCaretMoved) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' text #textCaretMoved callback
-- @
-- 
-- 
onTextTextCaretMoved :: (IsText a, MonadIO m) => a -> ((?self :: a) => TextTextCaretMovedCallback) -> m SignalHandlerId
onTextTextCaretMoved obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextCaretMovedCallback wrapped
    wrapped'' <- mk_TextTextCaretMovedCallback wrapped'
    connectSignalFunPtr obj "text-caret-moved" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [textCaretMoved](#signal:textCaretMoved) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' text #textCaretMoved callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterTextTextCaretMoved :: (IsText a, MonadIO m) => a -> ((?self :: a) => TextTextCaretMovedCallback) -> m SignalHandlerId
afterTextTextCaretMoved obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextCaretMovedCallback wrapped
    wrapped'' <- mk_TextTextCaretMovedCallback wrapped'
    connectSignalFunPtr obj "text-caret-moved" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data TextTextCaretMovedSignalInfo
instance SignalInfo TextTextCaretMovedSignalInfo where
    type HaskellCallbackType TextTextCaretMovedSignalInfo = TextTextCaretMovedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_TextTextCaretMovedCallback cb
        cb'' <- mk_TextTextCaretMovedCallback cb'
        connectSignalFunPtr obj "text-caret-moved" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text::text-caret-moved"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#g:signal:textCaretMoved"})

#endif

-- signal Text::text-changed
{-# DEPRECATED TextTextChangedCallback ["(Since version 2.9.4)","Use t'GI.Atk.Objects.Object.Object'::@/text-insert/@ or","t'GI.Atk.Objects.Object.Object'::@/text-remove/@ instead."] #-}
-- | The \"text-changed\" signal is emitted when the text of the
-- object which implements the AtkText interface changes, This
-- signal will have a detail which is either \"insert\" or
-- \"delete\" which identifies whether the text change was an
-- insertion or a deletion.
type TextTextChangedCallback =
    Int32
    -- ^ /@arg1@/: The position (character offset) of the insertion or deletion.
    -> Int32
    -- ^ /@arg2@/: The length (in characters) of text inserted or deleted.
    -> IO ()

type C_TextTextChangedCallback =
    Ptr Text ->                             -- object
    Int32 ->
    Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_TextTextChangedCallback`.
foreign import ccall "wrapper"
    mk_TextTextChangedCallback :: C_TextTextChangedCallback -> IO (FunPtr C_TextTextChangedCallback)

wrap_TextTextChangedCallback :: 
    GObject a => (a -> TextTextChangedCallback) ->
    C_TextTextChangedCallback
wrap_TextTextChangedCallback gi'cb gi'selfPtr arg1 arg2 _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  arg1 arg2


-- | Connect a signal handler for the [textChanged](#signal:textChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' text #textChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@text-changed::detail@” instead.
-- 
onTextTextChanged :: (IsText a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => TextTextChangedCallback) -> m SignalHandlerId
onTextTextChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextChangedCallback wrapped
    wrapped'' <- mk_TextTextChangedCallback wrapped'
    connectSignalFunPtr obj "text-changed" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [textChanged](#signal:textChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' text #textChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@text-changed::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterTextTextChanged :: (IsText a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => TextTextChangedCallback) -> m SignalHandlerId
afterTextTextChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextChangedCallback wrapped
    wrapped'' <- mk_TextTextChangedCallback wrapped'
    connectSignalFunPtr obj "text-changed" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data TextTextChangedSignalInfo
instance SignalInfo TextTextChangedSignalInfo where
    type HaskellCallbackType TextTextChangedSignalInfo = TextTextChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_TextTextChangedCallback cb
        cb'' <- mk_TextTextChangedCallback cb'
        connectSignalFunPtr obj "text-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text::text-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#g:signal:textChanged"})

#endif

-- signal Text::text-insert
-- | The \"text-insert\" signal is emitted when a new text is
-- inserted. If the signal was not triggered by the user
-- (e.g. typing or pasting text), the \"system\" detail should be
-- included.
type TextTextInsertCallback =
    Int32
    -- ^ /@arg1@/: The position (character offset) of the insertion.
    -> Int32
    -- ^ /@arg2@/: The length (in characters) of text inserted.
    -> T.Text
    -- ^ /@arg3@/: The new text inserted
    -> IO ()

type C_TextTextInsertCallback =
    Ptr Text ->                             -- object
    Int32 ->
    Int32 ->
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_TextTextInsertCallback`.
foreign import ccall "wrapper"
    mk_TextTextInsertCallback :: C_TextTextInsertCallback -> IO (FunPtr C_TextTextInsertCallback)

wrap_TextTextInsertCallback :: 
    GObject a => (a -> TextTextInsertCallback) ->
    C_TextTextInsertCallback
wrap_TextTextInsertCallback gi'cb gi'selfPtr arg1 arg2 arg3 _ = do
    arg3' <- cstringToText arg3
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  arg1 arg2 arg3'


-- | Connect a signal handler for the [textInsert](#signal:textInsert) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' text #textInsert callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@text-insert::detail@” instead.
-- 
onTextTextInsert :: (IsText a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => TextTextInsertCallback) -> m SignalHandlerId
onTextTextInsert obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextInsertCallback wrapped
    wrapped'' <- mk_TextTextInsertCallback wrapped'
    connectSignalFunPtr obj "text-insert" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [textInsert](#signal:textInsert) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' text #textInsert callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@text-insert::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterTextTextInsert :: (IsText a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => TextTextInsertCallback) -> m SignalHandlerId
afterTextTextInsert obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextInsertCallback wrapped
    wrapped'' <- mk_TextTextInsertCallback wrapped'
    connectSignalFunPtr obj "text-insert" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data TextTextInsertSignalInfo
instance SignalInfo TextTextInsertSignalInfo where
    type HaskellCallbackType TextTextInsertSignalInfo = TextTextInsertCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_TextTextInsertCallback cb
        cb'' <- mk_TextTextInsertCallback cb'
        connectSignalFunPtr obj "text-insert" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text::text-insert"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#g:signal:textInsert"})

#endif

-- signal Text::text-remove
-- | The \"text-remove\" signal is emitted when a new text is
-- removed. If the signal was not triggered by the user
-- (e.g. typing or pasting text), the \"system\" detail should be
-- included.
type TextTextRemoveCallback =
    Int32
    -- ^ /@arg1@/: The position (character offset) of the removal.
    -> Int32
    -- ^ /@arg2@/: The length (in characters) of text removed.
    -> T.Text
    -- ^ /@arg3@/: The old text removed
    -> IO ()

type C_TextTextRemoveCallback =
    Ptr Text ->                             -- object
    Int32 ->
    Int32 ->
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_TextTextRemoveCallback`.
foreign import ccall "wrapper"
    mk_TextTextRemoveCallback :: C_TextTextRemoveCallback -> IO (FunPtr C_TextTextRemoveCallback)

wrap_TextTextRemoveCallback :: 
    GObject a => (a -> TextTextRemoveCallback) ->
    C_TextTextRemoveCallback
wrap_TextTextRemoveCallback gi'cb gi'selfPtr arg1 arg2 arg3 _ = do
    arg3' <- cstringToText arg3
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  arg1 arg2 arg3'


-- | Connect a signal handler for the [textRemove](#signal:textRemove) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' text #textRemove callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@text-remove::detail@” instead.
-- 
onTextTextRemove :: (IsText a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => TextTextRemoveCallback) -> m SignalHandlerId
onTextTextRemove obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextRemoveCallback wrapped
    wrapped'' <- mk_TextTextRemoveCallback wrapped'
    connectSignalFunPtr obj "text-remove" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [textRemove](#signal:textRemove) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' text #textRemove callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@text-remove::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterTextTextRemove :: (IsText a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => TextTextRemoveCallback) -> m SignalHandlerId
afterTextTextRemove obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextRemoveCallback wrapped
    wrapped'' <- mk_TextTextRemoveCallback wrapped'
    connectSignalFunPtr obj "text-remove" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data TextTextRemoveSignalInfo
instance SignalInfo TextTextRemoveSignalInfo where
    type HaskellCallbackType TextTextRemoveSignalInfo = TextTextRemoveCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_TextTextRemoveCallback cb
        cb'' <- mk_TextTextRemoveCallback cb'
        connectSignalFunPtr obj "text-remove" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text::text-remove"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#g:signal:textRemove"})

#endif

-- signal Text::text-selection-changed
-- | The \"text-selection-changed\" signal is emitted when the
-- selected text of an object which implements AtkText changes.
type TextTextSelectionChangedCallback =
    IO ()

type C_TextTextSelectionChangedCallback =
    Ptr Text ->                             -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_TextTextSelectionChangedCallback`.
foreign import ccall "wrapper"
    mk_TextTextSelectionChangedCallback :: C_TextTextSelectionChangedCallback -> IO (FunPtr C_TextTextSelectionChangedCallback)

wrap_TextTextSelectionChangedCallback :: 
    GObject a => (a -> TextTextSelectionChangedCallback) ->
    C_TextTextSelectionChangedCallback
wrap_TextTextSelectionChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [textSelectionChanged](#signal:textSelectionChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' text #textSelectionChanged callback
-- @
-- 
-- 
onTextTextSelectionChanged :: (IsText a, MonadIO m) => a -> ((?self :: a) => TextTextSelectionChangedCallback) -> m SignalHandlerId
onTextTextSelectionChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextSelectionChangedCallback wrapped
    wrapped'' <- mk_TextTextSelectionChangedCallback wrapped'
    connectSignalFunPtr obj "text-selection-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [textSelectionChanged](#signal:textSelectionChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' text #textSelectionChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterTextTextSelectionChanged :: (IsText a, MonadIO m) => a -> ((?self :: a) => TextTextSelectionChangedCallback) -> m SignalHandlerId
afterTextTextSelectionChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_TextTextSelectionChangedCallback wrapped
    wrapped'' <- mk_TextTextSelectionChangedCallback wrapped'
    connectSignalFunPtr obj "text-selection-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data TextTextSelectionChangedSignalInfo
instance SignalInfo TextTextSelectionChangedSignalInfo where
    type HaskellCallbackType TextTextSelectionChangedSignalInfo = TextTextSelectionChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_TextTextSelectionChangedCallback cb
        cb'' <- mk_TextTextSelectionChangedCallback cb'
        connectSignalFunPtr obj "text-selection-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Atk.Interfaces.Text::text-selection-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-atk-2.0.28/docs/GI-Atk-Interfaces-Text.html#g:signal:textSelectionChanged"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Text = TextSignalList
type TextSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo), '("textAttributesChanged", TextTextAttributesChangedSignalInfo), '("textCaretMoved", TextTextCaretMovedSignalInfo), '("textChanged", TextTextChangedSignalInfo), '("textInsert", TextTextInsertSignalInfo), '("textRemove", TextTextRemoveSignalInfo), '("textSelectionChanged", TextTextSelectionChangedSignalInfo)] :: [(Symbol, DK.Type)])

#endif


