// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package core

import "context"

type (
	// Perm represents an individuals repository
	// permission.
	Perm struct {
		UserID  int64  `db:"perm_user_id"  json:"-"`
		RepoUID string `db:"perm_repo_uid" json:"-"`
		Read    bool   `db:"perm_read"     json:"read"`
		Write   bool   `db:"perm_write"    json:"write"`
		Admin   bool   `db:"perm_admin"    json:"admin"`
		Synced  int64  `db:"perm_synced"   json:"-"`
		Created int64  `db:"perm_created"  json:"-"`
		Updated int64  `db:"perm_updated"  json:"-"`
	}

	// Collaborator represents a project collaborator,
	// and provides the account and repository permissions
	// details.
	Collaborator struct {
		UserID  int64  `db:"perm_user_id"  json:"user_id"`
		RepoUID string `db:"perm_repo_uid" json:"repo_id"`
		Login   string `db:"user_login"    json:"login"`
		Avatar  string `db:"user_avatar"   json:"avatar"`
		Read    bool   `db:"perm_read"     json:"read"`
		Write   bool   `db:"perm_write"    json:"write"`
		Admin   bool   `db:"perm_admin"    json:"admin"`
		Synced  int64  `db:"perm_synced"   json:"synced"`
		Created int64  `db:"perm_created"  json:"created"`
		Updated int64  `db:"perm_updated"  json:"updated"`
	}

	// PermStore defines operations for working with
	// repository permissions.
	PermStore interface {
		// Find returns a project member from the
		// datastore.
		Find(ctx context.Context, repoUID string, userID int64) (*Perm, error)

		// List returns a list of project members from the
		// datastore.
		List(ctx context.Context, repoUID string) ([]*Collaborator, error)

		// Update persists an updated project member
		// to the datastore.
		Update(context.Context, *Perm) error

		// Delete deletes a project member from the
		// datastore.
		Delete(context.Context, *Perm) error
	}
)
