/*
 * Decompiled with CFR 0.152.
 */
package org.apache.skywalking.oap.server.core.profiling.trace.analyze;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.skywalking.oap.server.core.profiling.trace.ProfileLanguageType;
import org.apache.skywalking.oap.server.core.profiling.trace.ProfileThreadSnapshotRecord;
import org.apache.skywalking.oap.server.core.profiling.trace.analyze.GoProfileAnalyzer;
import org.apache.skywalking.oap.server.core.profiling.trace.analyze.ProfileAnalyzeCollector;
import org.apache.skywalking.oap.server.core.profiling.trace.analyze.ProfileStack;
import org.apache.skywalking.oap.server.core.query.input.SegmentProfileAnalyzeQuery;
import org.apache.skywalking.oap.server.core.query.type.ProfileAnalyzation;
import org.apache.skywalking.oap.server.core.query.type.ProfileStackTree;
import org.apache.skywalking.oap.server.core.storage.profiling.trace.IProfileThreadSnapshotQueryDAO;
import org.apache.skywalking.oap.server.library.module.ModuleManager;
import org.apache.skywalking.oap.server.library.util.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProfileAnalyzer {
    private static final Logger LOGGER = LoggerFactory.getLogger(ProfileAnalyzer.class);
    private static final ProfileAnalyzeCollector ANALYZE_COLLECTOR = new ProfileAnalyzeCollector();
    private final int threadSnapshotAnalyzeBatchSize;
    private final int analyzeSnapshotMaxSize;
    private final ModuleManager moduleManager;
    protected IProfileThreadSnapshotQueryDAO profileThreadSnapshotQueryDAO;

    public ProfileAnalyzer(ModuleManager moduleManager, int snapshotAnalyzeBatchSize, int analyzeSnapshotMaxSize) {
        this.moduleManager = moduleManager;
        this.threadSnapshotAnalyzeBatchSize = snapshotAnalyzeBatchSize;
        this.analyzeSnapshotMaxSize = analyzeSnapshotMaxSize;
    }

    public ProfileAnalyzation analyze(List<SegmentProfileAnalyzeQuery> queries) throws IOException {
        ProfileAnalyzation analyzation = new ProfileAnalyzation();
        SequenceSearch sequenceSearch = this.getAllSequenceRange(queries);
        List<Object> javaRecords = new ArrayList();
        if (sequenceSearch != null) {
            if (sequenceSearch.getTotalSequenceCount() > this.analyzeSnapshotMaxSize) {
                analyzation.setTip("Out of snapshot analyze limit, " + sequenceSearch.getTotalSequenceCount() + " snapshots found, but analysis first " + this.analyzeSnapshotMaxSize + " snapshots only.");
            }
            List records = sequenceSearch.getRanges().parallelStream().map(r -> {
                try {
                    return this.getProfileThreadSnapshotQueryDAO().queryRecords(r.getSegmentId(), r.getMinSequence(), r.getMaxSequence());
                }
                catch (IOException e) {
                    LOGGER.warn(e.getMessage(), (Throwable)e);
                    return Collections.emptyList();
                }
            }).flatMap(Collection::stream).collect(Collectors.toList());
            if (LOGGER.isDebugEnabled()) {
                int totalRanges = sequenceSearch.getRanges().size();
                LOGGER.debug("Profile analyze fetched records, segmentId(s)={}, ranges={}, recordsCount={}", new Object[]{sequenceSearch.getRanges().stream().map(SequenceRange::getSegmentId).distinct().collect(Collectors.toList()), totalRanges, records.size()});
            }
            javaRecords = records.stream().filter(rec -> rec.getLanguage() == ProfileLanguageType.JAVA).collect(Collectors.toList());
        } else if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Profile analyze: no Java records found in time window, will try Go fallback");
        }
        if (!javaRecords.isEmpty()) {
            LOGGER.info("Analyzing {} Java profile records", (Object)javaRecords.size());
            List<ProfileStack> stacks = javaRecords.stream().map(rec -> {
                try {
                    return ProfileStack.deserialize(rec);
                }
                catch (Exception ex) {
                    LOGGER.warn("Deserialize stack failed, segmentId={}, sequence={}, dumpTime={}", new Object[]{rec.getSegmentId(), rec.getSequence(), rec.getDumpTime(), ex});
                    return null;
                }
            }).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            List<ProfileStackTree> trees = this.analyzeByStack(stacks);
            if (trees != null && !trees.isEmpty()) {
                analyzation.getTrees().addAll(trees);
                return analyzation;
            }
        }
        ArrayList<ProfileThreadSnapshotRecord> goRecords = new ArrayList<ProfileThreadSnapshotRecord>();
        for (SegmentProfileAnalyzeQuery q : queries) {
            String segId = q.getSegmentId();
            try {
                int minSeq = this.getProfileThreadSnapshotQueryDAO().queryMinSequence(segId, 0L, Long.MAX_VALUE);
                int maxSeqExclusive = this.getProfileThreadSnapshotQueryDAO().queryMaxSequence(segId, 0L, Long.MAX_VALUE) + 1;
                if (maxSeqExclusive <= minSeq) continue;
                List<ProfileThreadSnapshotRecord> full = this.getProfileThreadSnapshotQueryDAO().queryRecords(segId, minSeq, maxSeqExclusive);
                for (ProfileThreadSnapshotRecord r2 : full) {
                    if (r2.getLanguage() != ProfileLanguageType.GO) continue;
                    goRecords.add(r2);
                }
            }
            catch (IOException e) {
                LOGGER.warn("Go fallback: full-range fetch failed for segmentId={}", (Object)segId, (Object)e);
            }
        }
        if (!goRecords.isEmpty()) {
            LOGGER.info("Java analysis found no data, fallback to Go: analyzing {} Go profile records", (Object)goRecords.size());
            GoProfileAnalyzer goAnalyzer = new GoProfileAnalyzer();
            ProfileAnalyzation goAnalyzation = goAnalyzer.analyzeRecords(goRecords, queries);
            if (goAnalyzation != null && !goAnalyzation.getTrees().isEmpty()) {
                analyzation.getTrees().addAll(goAnalyzation.getTrees());
            }
        } else if (sequenceSearch == null && javaRecords.isEmpty()) {
            analyzation.setTip("Data not found");
        }
        return analyzation;
    }

    protected SequenceSearch getAllSequenceRange(List<SegmentProfileAnalyzeQuery> queries) {
        List searches = queries.parallelStream().map(r -> {
            try {
                return this.getAllSequenceRange(r.getSegmentId(), r.getTimeRange().getStart(), r.getTimeRange().getEnd());
            }
            catch (IOException e) {
                LOGGER.warn(e.getMessage(), (Throwable)e);
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
        return searches.stream().reduce(new SequenceSearch(0), SequenceSearch::combine);
    }

    protected SequenceSearch getAllSequenceRange(String segmentId, long start, long end) throws IOException {
        int batchMax;
        int minSequence = this.getProfileThreadSnapshotQueryDAO().queryMinSequence(segmentId, start, end);
        int maxSequence = this.getProfileThreadSnapshotQueryDAO().queryMaxSequence(segmentId, start, end) + 1;
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Profile analyze sequence window: segmentId={}, start={}, end={}, minSeq={}, maxSeq(exclusive)={}", new Object[]{segmentId, start, end, minSequence, maxSequence});
        }
        if (maxSequence <= 0) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Profile analyze not found any sequence in window: segmentId={}, start={}, end={}", new Object[]{segmentId, start, end});
            }
            return null;
        }
        SequenceSearch sequenceSearch = new SequenceSearch(maxSequence - minSequence);
        maxSequence = Math.min(maxSequence, minSequence + this.analyzeSnapshotMaxSize);
        do {
            batchMax = Math.min(minSequence + this.threadSnapshotAnalyzeBatchSize, maxSequence);
            sequenceSearch.getRanges().add(new SequenceRange(segmentId, minSequence, batchMax));
        } while ((minSequence = batchMax) < maxSequence);
        return sequenceSearch;
    }

    protected List<ProfileStackTree> analyzeByStack(List<ProfileStack> stacks) {
        if (CollectionUtils.isEmpty(stacks)) {
            return null;
        }
        Map<String, ProfileStackTree> stackTrees = stacks.parallelStream().filter(s -> CollectionUtils.isNotEmpty(s.getStack())).collect(Collectors.groupingBy(s -> s.getStack().get(0), ANALYZE_COLLECTOR));
        return new ArrayList<ProfileStackTree>(stackTrees.values());
    }

    protected IProfileThreadSnapshotQueryDAO getProfileThreadSnapshotQueryDAO() {
        if (this.profileThreadSnapshotQueryDAO == null) {
            this.profileThreadSnapshotQueryDAO = (IProfileThreadSnapshotQueryDAO)this.moduleManager.find("storage").provider().getService(IProfileThreadSnapshotQueryDAO.class);
        }
        return this.profileThreadSnapshotQueryDAO;
    }

    private static class SequenceSearch {
        private LinkedList<SequenceRange> ranges = new LinkedList();
        private int totalSequenceCount;

        public SequenceSearch(int totalSequenceCount) {
            this.totalSequenceCount = totalSequenceCount;
        }

        public LinkedList<SequenceRange> getRanges() {
            return this.ranges;
        }

        public int getTotalSequenceCount() {
            return this.totalSequenceCount;
        }

        public SequenceSearch combine(SequenceSearch search) {
            this.ranges.addAll(search.ranges);
            this.totalSequenceCount += search.totalSequenceCount;
            return this;
        }
    }

    private static class SequenceRange {
        private String segmentId;
        private int minSequence;
        private int maxSequence;

        public SequenceRange(String segmentId, int minSequence, int maxSequence) {
            this.segmentId = segmentId;
            this.minSequence = minSequence;
            this.maxSequence = maxSequence;
        }

        public String getSegmentId() {
            return this.segmentId;
        }

        public int getMinSequence() {
            return this.minSequence;
        }

        public int getMaxSequence() {
            return this.maxSequence;
        }
    }
}

