/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.file;

import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.rocketmq.common.message.MessageQueue;
import org.apache.rocketmq.tieredstore.MessageStoreConfig;
import org.apache.rocketmq.tieredstore.MessageStoreExecutor;
import org.apache.rocketmq.tieredstore.file.FlatFileFactory;
import org.apache.rocketmq.tieredstore.file.FlatMessageFile;
import org.apache.rocketmq.tieredstore.metadata.MetadataStore;
import org.apache.rocketmq.tieredstore.metadata.entity.TopicMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlatFileStore {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    private final MetadataStore metadataStore;
    private final MessageStoreConfig storeConfig;
    private final MessageStoreExecutor executor;
    private final FlatFileFactory flatFileFactory;
    private final ConcurrentMap<MessageQueue, FlatMessageFile> flatFileConcurrentMap;

    public FlatFileStore(MessageStoreConfig storeConfig, MetadataStore metadataStore, MessageStoreExecutor executor) {
        this.storeConfig = storeConfig;
        this.metadataStore = metadataStore;
        this.executor = executor;
        this.flatFileFactory = new FlatFileFactory(metadataStore, storeConfig, executor);
        this.flatFileConcurrentMap = new ConcurrentHashMap<MessageQueue, FlatMessageFile>();
    }

    public boolean load() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            this.flatFileConcurrentMap.clear();
            this.recover();
            log.info("FlatFileStore recover finished, total cost={}ms", (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
        }
        catch (Exception e) {
            long costTime = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            log.info("FlatFileStore recover error, total cost={}ms", (Object)costTime);
            LoggerFactory.getLogger((String)"RocketmqBroker").error("FlatFileStore recover error, total cost={}ms", (Object)costTime, (Object)e);
            return false;
        }
        return true;
    }

    public void recover() {
        Semaphore semaphore = new Semaphore(this.storeConfig.getTieredStoreMaxPendingLimit() / 4);
        ArrayList futures = new ArrayList();
        this.metadataStore.iterateTopic(topicMetadata -> {
            semaphore.acquireUninterruptibly();
            futures.add(this.recoverAsync((TopicMetadata)topicMetadata).whenComplete((unused, throwable) -> {
                if (throwable != null) {
                    log.error("FlatFileStore recover file error, topic={}", (Object)topicMetadata.getTopic(), throwable);
                }
                semaphore.release();
            }));
        });
        CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();
    }

    public CompletableFuture<Void> recoverAsync(TopicMetadata topicMetadata) {
        return CompletableFuture.runAsync(() -> {
            Stopwatch stopwatch = Stopwatch.createStarted();
            AtomicLong queueCount = new AtomicLong();
            this.metadataStore.iterateQueue(topicMetadata.getTopic(), queueMetadata -> {
                FlatMessageFile flatFile = this.computeIfAbsent(new MessageQueue(topicMetadata.getTopic(), this.storeConfig.getBrokerName(), queueMetadata.getQueue().getQueueId()));
                queueCount.incrementAndGet();
                log.debug("FlatFileStore recover file, topicId={}, topic={}, queueId={}, cost={}ms", new Object[]{flatFile.getTopicId(), flatFile.getMessageQueue().getTopic(), flatFile.getMessageQueue().getQueueId(), stopwatch.elapsed(TimeUnit.MILLISECONDS)});
            });
            log.info("FlatFileStore recover file, topic={}, total={}, cost={}ms", new Object[]{topicMetadata.getTopic(), queueCount.get(), stopwatch.elapsed(TimeUnit.MILLISECONDS)});
        }, this.executor.bufferCommitExecutor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void scheduleDeleteExpireFile() {
        if (!this.storeConfig.isTieredStoreDeleteFileEnable()) {
            return;
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        ImmutableList<FlatMessageFile> fileList = this.deepCopyFlatFileToList();
        for (FlatMessageFile flatFile : fileList) {
            flatFile.getFileLock().lock();
            try {
                flatFile.destroyExpiredFile(System.currentTimeMillis() - TimeUnit.HOURS.toMillis(flatFile.getFileReservedHours()));
            }
            catch (Exception e) {
                log.error("FlatFileStore delete expire file error", (Throwable)e);
            }
            finally {
                flatFile.getFileLock().unlock();
            }
        }
        log.info("FlatFileStore schedule delete expired file, count={}, cost={}ms", (Object)fileList.size(), (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    public MetadataStore getMetadataStore() {
        return this.metadataStore;
    }

    public MessageStoreConfig getStoreConfig() {
        return this.storeConfig;
    }

    public FlatFileFactory getFlatFileFactory() {
        return this.flatFileFactory;
    }

    public FlatMessageFile computeIfAbsent(MessageQueue messageQueue) {
        return this.flatFileConcurrentMap.computeIfAbsent(messageQueue, mq -> new FlatMessageFile(this.flatFileFactory, mq.getTopic(), mq.getQueueId()));
    }

    public FlatMessageFile getFlatFile(MessageQueue messageQueue) {
        return (FlatMessageFile)this.flatFileConcurrentMap.get(messageQueue);
    }

    public ImmutableList<FlatMessageFile> deepCopyFlatFileToList() {
        return ImmutableList.copyOf(this.flatFileConcurrentMap.values());
    }

    public void shutdown() {
        this.flatFileConcurrentMap.values().forEach(FlatMessageFile::shutdown);
    }

    public void destroyFile(MessageQueue mq) {
        if (mq == null) {
            return;
        }
        FlatMessageFile flatFile = (FlatMessageFile)this.flatFileConcurrentMap.remove(mq);
        if (flatFile != null) {
            flatFile.shutdown();
            flatFile.destroy();
        }
        log.info("FlatFileStore destroy file, topic={}, queueId={}", (Object)mq.getTopic(), (Object)mq.getQueueId());
    }

    public void destroy() {
        this.shutdown();
        this.flatFileConcurrentMap.values().forEach(FlatMessageFile::destroy);
        this.flatFileConcurrentMap.clear();
    }
}

