/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.ratis.shell.cli.sh;

import org.apache.ratis.RaftTestUtil;
import org.apache.ratis.client.RaftClient;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.MiniRaftCluster;
import org.apache.ratis.server.impl.RaftServerTestUtil;
import org.apache.ratis.server.raftlog.RaftLog;
import org.apache.ratis.statemachine.impl.SimpleStateMachine4Testing;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Slf4jUtils;
import org.apache.ratis.util.SizeInBytes;
import org.apache.ratis.util.TimeDuration;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.event.Level;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

public abstract class PeerCommandIntegrationTest <CLUSTER extends MiniRaftCluster>
    extends AbstractCommandIntegrationTestWithGrpc implements MiniRaftCluster.Factory.Get<CLUSTER> {

  {
    Slf4jUtils.setLogLevel(RaftServer.Division.LOG, Level.WARN);
    Slf4jUtils.setLogLevel(RaftLog.LOG, Level.WARN);
    Slf4jUtils.setLogLevel(RaftClient.LOG, Level.WARN);
  }

  {
    final RaftProperties prop = getProperties();
    prop.setClass(MiniRaftCluster.STATEMACHINE_CLASS_KEY,
        SimpleStateMachine4Testing.class, StateMachine.class);
    RaftServerConfigKeys.Log.setSegmentSizeMax(prop, SizeInBytes.valueOf("8KB"));
  }

  @Test
  public void testPeerAddRemoveCommand() throws Exception {
    runWithNewCluster(NUM_SERVERS, this::runTestPeerRemoveCommand);
    runWithNewCluster(NUM_SERVERS, this::runTestPeerAddCommand);
  }

  void runTestPeerRemoveCommand(MiniRaftCluster cluster) throws Exception {
    final RaftServer.Division leader = RaftTestUtil.waitForLeader(cluster);
    final String address = getClusterAddress(cluster);
    RaftServer.Division toRemove = cluster.getFollowers().get(0);
    final List<RaftPeer> peers = Arrays.asList(cluster.getFollowers().get(1).getPeer(), leader.getPeer());
    final StringPrintStream out = new StringPrintStream();
    RatisShell shell = new RatisShell(out.getPrintStream());
    Assertions.assertTrue(cluster.getFollowers().contains(toRemove));
    int ret = shell.run("peer", "remove", "-peers", address, "-peerId",
        toRemove.getPeer().getId().toString());

    Assertions.assertEquals(0, ret);
    RaftServerTestUtil.waitAndCheckNewConf(cluster, peers,1, null);
  }

  void runTestPeerAddCommand(MiniRaftCluster cluster) throws Exception {
    LOG.info("Start testMultiGroup" + cluster.printServers());

    RaftTestUtil.waitForLeader(cluster);
    final List<RaftPeer> peers = cluster.getPeers();
    final List<RaftPeer> newPeers = cluster.addNewPeers(1, true).getAddedPeers();

    RaftServerTestUtil.waitAndCheckNewConf(cluster, peers, 0, null);
    StringBuilder sb = new StringBuilder();
    for (RaftPeer peer : peers) {
      sb.append(peer.getAdminAddress());
      sb.append(",");
    }
    final StringPrintStream out = new StringPrintStream();
    RatisShell shell = new RatisShell(out.getPrintStream());

    final RaftPeer newPeer0 = newPeers.get(0);
    int ret = shell.run("peer", "add", "-peers", sb.toString(), "-address",
        newPeer0.getAdminAddress(), "-peerId", newPeer0.getId().toString());

    Assertions.assertEquals(0, ret);
    peers.add(newPeer0);
    RaftServerTestUtil.waitAndCheckNewConf(cluster, peers, 0, null);

  }

  @Test
  public void testPeerSetPriorityCommand() throws Exception {
    runWithNewCluster(NUM_SERVERS, this::runTestPeerSetPriorityCommand);
  }

  void runTestPeerSetPriorityCommand(MiniRaftCluster cluster) throws Exception {
    final RaftServer.Division leader = RaftTestUtil.waitForLeader(cluster);
    final String address = getClusterAddress(cluster);

    RaftServer.Division newLeader = cluster.getFollowers().get(0);
    final StringPrintStream out = new StringPrintStream();
    RatisShell shell = new RatisShell(out.getPrintStream());
    Assertions.assertTrue(cluster.getFollowers().contains(newLeader));
    int ret = shell.run("peer", "setPriority", "-peers", address, "-addressPriority",
        newLeader.getPeer().getAddress()+ "|" + 2);
    Assertions.assertEquals(0, ret);
    JavaUtils.attempt(() -> {
      Assertions.assertEquals(cluster.getLeader().getId(), newLeader.getId());
    }, 10, TimeDuration.valueOf(1, TimeUnit.SECONDS), "testPeerSetPriorityCommand", LOG);
  }

}
