/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentFailureReason;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.ChecksumException;
import com.sleepycat.je.log.ChecksumValidator;
import com.sleepycat.je.log.FileHandle;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.log.LogEntryHeader;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.LoggerUtils;
import java.io.FileNotFoundException;
import java.nio.ByteBuffer;
import java.util.logging.Logger;

public abstract class FileReader {
    protected final EnvironmentImpl envImpl;
    protected final FileManager fileManager;
    protected final ReadWindow window;
    private ByteBuffer saveBuffer;
    private final boolean singleFile;
    protected boolean eof;
    protected final boolean forward;
    private int nRead;
    protected LogEntryHeader currentEntryHeader;
    protected long currentEntryPrevOffset;
    protected long currentEntryOffset;
    protected long nextEntryOffset;
    protected long startLsn;
    private final long finishLsn;
    protected ChecksumValidator cksumValidator;
    private boolean doChecksumOnRead;
    private boolean alwaysValidateChecksum;
    protected final Logger logger;

    public FileReader(EnvironmentImpl envImpl, int readBufferSize, boolean forward, long startLsn, Long singleFileNumber, long endOfFileLsn, long finishLsn) throws DatabaseException {
        this.envImpl = envImpl;
        this.fileManager = envImpl.getFileManager();
        this.singleFile = singleFileNumber != null;
        this.forward = forward;
        this.doChecksumOnRead = envImpl.getLogManager().getChecksumOnRead();
        if (this.doChecksumOnRead) {
            this.cksumValidator = new ChecksumValidator();
        }
        this.window = this.makeWindow(readBufferSize);
        this.saveBuffer = ByteBuffer.allocate(readBufferSize);
        this.nRead = 0;
        this.startLsn = startLsn;
        this.finishLsn = finishLsn;
        this.logger = envImpl.getLogger();
        this.initStartingPosition(endOfFileLsn, singleFileNumber);
    }

    protected ReadWindow makeWindow(int readBufferSize) throws DatabaseException {
        return new ReadWindow(readBufferSize, this.envImpl);
    }

    protected void initStartingPosition(long endOfFileLsn, Long ignoreSingleFileNumber) {
        this.eof = false;
        if (this.forward) {
            if (this.startLsn != -1L) {
                this.window.initAtFileStart(this.startLsn);
            } else {
                Long firstNum = this.fileManager.getFirstFileNum();
                if (firstNum == null) {
                    this.eof = true;
                } else {
                    this.window.initAtFileStart(DbLsn.makeLsn((long)firstNum, 0));
                }
            }
            this.nextEntryOffset = this.window.getEndOffset();
        } else {
            assert (this.startLsn != -1L);
            this.window.initAtFileStart(endOfFileLsn);
            this.currentEntryPrevOffset = DbLsn.getFileNumber(this.startLsn) == DbLsn.getFileNumber(endOfFileLsn) ? DbLsn.getFileOffset(this.startLsn) : 0L;
            this.currentEntryOffset = DbLsn.getFileOffset(endOfFileLsn);
        }
    }

    public void setAlwaysValidateChecksum(boolean validate) {
        this.alwaysValidateChecksum = validate;
    }

    public int getNumRead() {
        return this.nRead;
    }

    public long getNRepeatIteratorReads() {
        return this.window.getNRepeatIteratorReads();
    }

    public long getLastLsn() {
        return DbLsn.makeLsn(this.window.currentFileNum(), this.currentEntryOffset);
    }

    public int getLastEntrySize() {
        return this.currentEntryHeader.getEntrySize();
    }

    public boolean readNextEntry() {
        try {
            return this.readNextEntryAllowExceptions();
        }
        catch (FileNotFoundException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_FILE_NOT_FOUND, (Throwable)e);
        }
        catch (ChecksumException e) {
            throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_CHECKSUM, (Throwable)e);
        }
    }

    public final boolean readNextEntryAllowExceptions() throws FileNotFoundException, ChecksumException {
        boolean foundEntry = false;
        long savedCurrentEntryOffset = this.currentEntryOffset;
        long savedNextEntryOffset = this.nextEntryOffset;
        try {
            while (!this.eof && !foundEntry) {
                boolean collectData;
                boolean isChecksumTarget;
                boolean isTarget;
                this.getLogEntryInReadBuffer();
                ByteBuffer dataBuffer = this.readData(14, true);
                this.readBasicHeader(dataBuffer);
                if (this.currentEntryHeader.isVariableLength()) {
                    this.startChecksum(dataBuffer);
                    int optionalPortionLen = this.currentEntryHeader.getVariablePortionSize();
                    dataBuffer = this.readData(optionalPortionLen, true);
                    this.addToChecksum(dataBuffer, optionalPortionLen);
                    this.currentEntryHeader.readVariablePortion(dataBuffer);
                }
                if (this.forward) {
                    this.currentEntryOffset = this.nextEntryOffset;
                    this.nextEntryOffset += (long)(this.currentEntryHeader.getSize() + this.currentEntryHeader.getItemSize());
                }
                try {
                    isTarget = this.isTargetEntry();
                    boolean bl = isChecksumTarget = isTarget || this.alwaysValidateChecksum;
                    if (!this.currentEntryHeader.isVariableLength()) {
                        this.startChecksum(dataBuffer, isChecksumTarget);
                    }
                    collectData = isChecksumTarget && this.doChecksumOnRead || isTarget;
                    dataBuffer = this.readData(this.currentEntryHeader.getItemSize(), collectData);
                }
                catch (Throwable e) {
                    if (this.forward) {
                        this.currentEntryOffset = savedCurrentEntryOffset;
                        this.nextEntryOffset = savedNextEntryOffset;
                    }
                    throw e;
                }
                this.validateChecksum(dataBuffer, isChecksumTarget);
                if (isTarget) {
                    if (!this.processEntry(dataBuffer)) continue;
                    foundEntry = true;
                    ++this.nRead;
                    continue;
                }
                if (!collectData) continue;
                this.skipEntry(dataBuffer);
            }
        }
        catch (EOFException e) {
            this.eof = true;
        }
        catch (DatabaseException e) {
            this.eof = true;
            this.reportProblem(e);
            throw e;
        }
        return foundEntry;
    }

    protected void skipEntry(ByteBuffer entryBuffer) {
        entryBuffer.position(entryBuffer.position() + this.currentEntryHeader.getItemSize());
    }

    private void reportProblem(Exception e) {
        StringBuilder sb = new StringBuilder();
        sb.append("Halted log file reading at file 0x").append(Long.toHexString(this.window.currentFileNum())).append(" offset 0x").append(Long.toHexString(this.nextEntryOffset)).append(" offset(decimal)=").append(this.nextEntryOffset).append(" prev=0x").append(Long.toHexString(this.currentEntryPrevOffset));
        if (this.currentEntryHeader != null) {
            LogEntryType problemType = LogEntryType.findType(this.currentEntryHeader.getType());
            sb.append(":\nentry=").append(problemType).append("type=").append(this.currentEntryHeader.getType()).append(",version=").append(this.currentEntryHeader.getVersion()).append(")\nprev=0x").append(Long.toHexString(this.currentEntryPrevOffset)).append("\nsize=").append(this.currentEntryHeader.getItemSize()).append("\nNext entry should be at 0x").append(Long.toHexString(this.nextEntryOffset + (long)this.currentEntryHeader.getSize() + (long)this.currentEntryHeader.getItemSize()));
        }
        LoggerUtils.traceAndLogException(this.envImpl, "FileReader", "readNextEntry", sb.toString(), e);
    }

    private void getLogEntryInReadBuffer() throws ChecksumException, EOFException, FileNotFoundException, DatabaseException {
        if (this.forward) {
            this.setForwardPosition();
        } else {
            this.setBackwardPosition();
        }
    }

    protected void setForwardPosition() throws EOFException, DatabaseException, ChecksumException, FileNotFoundException {
        long nextLsn;
        if (this.finishLsn != -1L && DbLsn.compareTo(nextLsn = DbLsn.makeLsn(this.window.currentFileNum(), this.nextEntryOffset), this.finishLsn) >= 0) {
            throw new EOFException();
        }
    }

    protected void setBackwardPosition() throws ChecksumException, FileNotFoundException, EOFException, DatabaseException {
        if (this.currentEntryPrevOffset != 0L && this.window.containsOffset(this.currentEntryPrevOffset)) {
            long nextLsn = DbLsn.makeLsn(this.window.currentFileNum(), this.currentEntryPrevOffset);
            if (this.finishLsn != -1L && DbLsn.compareTo(nextLsn, this.finishLsn) == -1) {
                throw new EOFException("finish=" + DbLsn.getNoFormatString(this.finishLsn) + "next=" + DbLsn.getNoFormatString(nextLsn));
            }
            this.window.positionBuffer(this.currentEntryPrevOffset);
        } else {
            long nextTarget;
            long nextWindowStart;
            long nextFile;
            if (this.currentEntryPrevOffset == 0L) {
                this.currentEntryPrevOffset = this.fileManager.getFileHeaderPrevOffset(this.window.currentFileNum());
                Long prevFileNum = this.fileManager.getFollowingFileNum(this.window.currentFileNum(), false);
                if (prevFileNum == null) {
                    throw new EOFException("No file following " + this.window.currentFileNum());
                }
                if (this.finishLsn != -1L && prevFileNum < DbLsn.getFileNumber(this.finishLsn)) {
                    throw new EOFException("finish=" + DbLsn.getNoFormatString(this.finishLsn) + " nextFile=0x" + Long.toHexString(prevFileNum));
                }
                if (this.window.currentFileNum() - prevFileNum != 1L) {
                    this.handleGapInBackwardsScan(prevFileNum);
                }
                nextFile = prevFileNum;
                nextWindowStart = this.currentEntryPrevOffset;
                nextTarget = this.currentEntryPrevOffset;
            } else if (this.currentEntryOffset - this.currentEntryPrevOffset > (long)this.window.capacity()) {
                nextFile = this.window.currentFileNum();
                nextWindowStart = this.currentEntryPrevOffset;
                nextTarget = this.currentEntryPrevOffset;
            } else {
                nextFile = this.window.currentFileNum();
                long newPosition = this.currentEntryOffset - (long)this.window.capacity();
                nextWindowStart = newPosition < 0L ? 0L : newPosition;
                nextTarget = this.currentEntryPrevOffset;
            }
            long nextLsn = DbLsn.makeLsn(nextFile, this.currentEntryPrevOffset);
            if (this.finishLsn != -1L && DbLsn.compareTo(nextLsn, this.finishLsn) == -1) {
                throw new EOFException("finish=" + DbLsn.getNoFormatString(this.finishLsn) + " next=" + DbLsn.getNoFormatString(nextLsn));
            }
            this.window.slideAndFill(nextFile, nextWindowStart, nextTarget, this.forward);
        }
        this.currentEntryOffset = this.currentEntryPrevOffset;
    }

    private void readBasicHeader(ByteBuffer dataBuffer) throws ChecksumException, DatabaseException {
        this.currentEntryHeader = new LogEntryHeader(dataBuffer, this.window.logVersion, this.window.getCurrentLsn());
        this.currentEntryPrevOffset = this.currentEntryHeader.getPrevOffset();
    }

    private void startChecksum(ByteBuffer dataBuffer) throws ChecksumException {
        this.startChecksum(dataBuffer, true);
    }

    private void startChecksum(ByteBuffer dataBuffer, boolean isChecksumTarget) throws ChecksumException {
        if (!this.doChecksumOnRead) {
            return;
        }
        if (!isChecksumTarget) {
            return;
        }
        this.cksumValidator.reset();
        int originalPosition = dataBuffer.position();
        if (this.currentEntryHeader.isInvisible()) {
            LogEntryHeader.turnOffInvisible(dataBuffer, originalPosition - 14);
        }
        int headerSizeMinusChecksum = this.currentEntryHeader.getInvariantSizeMinusChecksum();
        int entryTypeStart = originalPosition - headerSizeMinusChecksum;
        dataBuffer.position(entryTypeStart);
        this.cksumValidator.update(dataBuffer, headerSizeMinusChecksum);
        dataBuffer.position(originalPosition);
    }

    private void addToChecksum(ByteBuffer dataBuffer, int length) throws ChecksumException {
        if (!this.doChecksumOnRead) {
            return;
        }
        this.cksumValidator.update(dataBuffer, length);
    }

    private void validateChecksum(ByteBuffer dataBuffer, boolean isChecksumTarget) throws ChecksumException {
        if (!this.doChecksumOnRead) {
            return;
        }
        if (!isChecksumTarget) {
            return;
        }
        this.cksumValidator.update(dataBuffer, this.currentEntryHeader.getItemSize());
        this.cksumValidator.validate(this.currentEntryHeader.getChecksum(), this.window.currentFileNum(), this.currentEntryOffset);
    }

    private ByteBuffer readData(int amountToRead, boolean collectData) throws ChecksumException, EOFException, FileNotFoundException, DatabaseException {
        int alreadyRead = 0;
        ByteBuffer completeBuffer = null;
        this.saveBuffer.clear();
        while (alreadyRead < amountToRead && !this.eof) {
            int bytesNeeded = amountToRead - alreadyRead;
            if (this.window.hasRemaining()) {
                if (collectData) {
                    if (alreadyRead > 0 || this.window.remaining() < bytesNeeded) {
                        this.copyToSaveBuffer(bytesNeeded);
                        alreadyRead = this.saveBuffer.position();
                        completeBuffer = this.saveBuffer;
                        continue;
                    }
                    completeBuffer = this.window.getBuffer();
                    alreadyRead = amountToRead;
                    continue;
                }
                int positionIncrement = this.window.remaining() > bytesNeeded ? bytesNeeded : this.window.remaining();
                alreadyRead += positionIncrement;
                this.window.incrementBufferPosition(positionIncrement);
                completeBuffer = this.window.getBuffer();
                continue;
            }
            if (!this.window.fillNext(this.singleFile, bytesNeeded)) continue;
            this.nextEntryOffset = 0L;
        }
        this.saveBuffer.flip();
        return completeBuffer;
    }

    public void skipData(int amountToSkip) throws ChecksumException, EOFException, FileNotFoundException, DatabaseException {
        try {
            this.readData(amountToSkip, false);
        }
        catch (DatabaseException e) {
            this.reportProblem(e);
            throw e;
        }
    }

    private void copyToSaveBuffer(int bytesNeeded) {
        ByteBuffer temp;
        int bytesFromThisBuffer = bytesNeeded <= this.window.remaining() ? bytesNeeded : this.window.remaining();
        if (this.saveBuffer.capacity() - this.saveBuffer.position() < bytesFromThisBuffer) {
            temp = ByteBuffer.allocate(this.saveBuffer.capacity() + bytesFromThisBuffer);
            this.saveBuffer.flip();
            temp.put(this.saveBuffer);
            this.saveBuffer = temp;
        }
        temp = this.window.getBuffer().slice();
        temp.limit(bytesFromThisBuffer);
        this.saveBuffer.put(temp);
        this.window.incrementBufferPosition(bytesFromThisBuffer);
    }

    public int getAndResetNReads() {
        return this.window.getAndResetNReads();
    }

    protected boolean isTargetEntry() throws DatabaseException {
        return true;
    }

    protected abstract boolean processEntry(ByteBuffer var1) throws DatabaseException;

    public boolean entryIsReplicated() {
        if (this.currentEntryHeader == null) {
            throw EnvironmentFailureException.unexpectedState("entryIsReplicated should not be used before reader is initialized");
        }
        return this.currentEntryHeader.getReplicated();
    }

    protected void handleGapInBackwardsScan(long prevFileNum) {
        throw new EnvironmentFailureException(this.envImpl, EnvironmentFailureReason.LOG_INTEGRITY, "Cannot read backward over cleaned file from " + this.window.currentFileNum() + " to " + prevFileNum);
    }

    protected static class ReadWindow {
        private long fileNum;
        private int logVersion;
        protected long startOffset;
        protected long endOffset;
        protected ByteBuffer readBuffer;
        private final int maxReadBufferSize;
        protected final EnvironmentImpl envImpl;
        protected final FileManager fileManager;
        private long nRepeatIteratorReads;
        private int nReadOperations;

        protected ReadWindow(int readBufferSize, EnvironmentImpl envImpl) {
            DbConfigManager configManager = envImpl.getConfigManager();
            this.maxReadBufferSize = configManager.getInt(EnvironmentParams.LOG_ITERATOR_MAX_SIZE);
            this.envImpl = envImpl;
            this.fileManager = envImpl.getFileManager();
            this.readBuffer = ByteBuffer.allocate(readBufferSize);
            this.readBuffer.flip();
        }

        public void initAtFileStart(long startLsn) {
            this.setFileNum(DbLsn.getFileNumber(startLsn), -1);
            this.endOffset = this.startOffset = DbLsn.getFileOffset(startLsn);
        }

        public long getEndOffset() {
            return this.endOffset;
        }

        protected void setFileNum(long fileNum, int logVersion) {
            this.fileNum = fileNum;
            this.logVersion = logVersion;
        }

        public long currentFileNum() {
            return this.fileNum;
        }

        boolean containsOffset(long targetOffset) {
            return targetOffset >= this.startOffset && targetOffset < this.endOffset;
        }

        public boolean containsLsn(long targetFileNumber, long targetOffset) {
            return this.fileNum == targetFileNumber && this.containsOffset(targetOffset);
        }

        public void positionBuffer(long targetOffset) {
            assert (this.containsOffset(targetOffset)) : this + " doesn't contain " + DbLsn.getNoFormatString(targetOffset);
            this.readBuffer.position((int)(targetOffset - this.startOffset));
        }

        void incrementBufferPosition(int increment) {
            int currentPosition = this.readBuffer.position();
            this.readBuffer.position(currentPosition + increment);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void slideAndFill(long windowfileNum, long windowStartOffset, long targetOffset, boolean forward) throws ChecksumException, FileNotFoundException, DatabaseException {
            FileHandle fileHandle = this.fileManager.getFileHandle(windowfileNum);
            try {
                this.startOffset = windowStartOffset;
                this.setFileNum(windowfileNum, fileHandle.getLogVersion());
                boolean foundData = this.fillFromFile(fileHandle, targetOffset);
                if (!foundData && !forward) {
                    throw EnvironmentFailureException.unexpectedState("Detected a log file gap when reading backwards. Target position = " + DbLsn.getNoFormatString(DbLsn.makeLsn(windowfileNum, targetOffset)) + " starting position = " + DbLsn.getNoFormatString(DbLsn.makeLsn(windowfileNum, windowStartOffset)) + " end position = " + DbLsn.getNoFormatString(DbLsn.makeLsn(windowfileNum, this.endOffset)));
                }
            }
            finally {
                fileHandle.release();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected boolean fillNext(boolean singleFile, int bytesNeeded) throws ChecksumException, FileNotFoundException, EOFException, DatabaseException {
            this.adjustReadBufferSize(bytesNeeded);
            FileHandle fileHandle = null;
            try {
                fileHandle = this.fileManager.getFileHandle(this.fileNum);
                this.startOffset = this.endOffset;
                if (this.fillFromFile(fileHandle, this.startOffset)) {
                    boolean bl = false;
                    return bl;
                }
                if (singleFile) {
                    throw new EOFException("Single file only");
                }
                Long nextFile = this.fileManager.getFollowingFileNum(this.fileNum, true);
                if (nextFile == null) {
                    throw new EOFException();
                }
                fileHandle.release();
                fileHandle = null;
                fileHandle = this.fileManager.getFileHandle(nextFile);
                this.setFileNum(nextFile, fileHandle.getLogVersion());
                this.startOffset = 0L;
                this.fillFromFile(fileHandle, 0L);
                boolean bl = true;
                return bl;
            }
            finally {
                if (fileHandle != null) {
                    fileHandle.release();
                }
            }
        }

        protected boolean fillFromFile(FileHandle fileHandle, long targetOffset) throws DatabaseException {
            boolean foundData = false;
            this.readBuffer.clear();
            if (this.fileManager.readFromFile(fileHandle.getFile(), this.readBuffer, this.startOffset, fileHandle.getFileNum(), false)) {
                foundData = true;
                ++this.nReadOperations;
                this.logVersion = fileHandle.getLogVersion();
            }
            this.endOffset = this.startOffset + (long)this.readBuffer.position();
            this.readBuffer.flip();
            this.readBuffer.position((int)(targetOffset - this.startOffset));
            return foundData;
        }

        protected void adjustReadBufferSize(int amountToRead) {
            int readBufferSize = this.readBuffer.capacity();
            if (amountToRead > readBufferSize) {
                if (readBufferSize < this.maxReadBufferSize) {
                    if (amountToRead < this.maxReadBufferSize) {
                        readBufferSize = amountToRead;
                        int remainder = readBufferSize % 1024;
                        readBufferSize += 1024 - remainder;
                        readBufferSize = Math.min(readBufferSize, this.maxReadBufferSize);
                    } else {
                        readBufferSize = this.maxReadBufferSize;
                    }
                    this.readBuffer = ByteBuffer.allocate(readBufferSize);
                }
                if (amountToRead > this.readBuffer.capacity()) {
                    ++this.nRepeatIteratorReads;
                }
            }
        }

        int capacity() {
            return this.readBuffer.capacity();
        }

        int remaining() {
            return this.readBuffer.remaining();
        }

        boolean hasRemaining() {
            return this.readBuffer.hasRemaining();
        }

        ByteBuffer getBuffer() {
            return this.readBuffer;
        }

        int getAndResetNReads() {
            int tmp = this.nReadOperations;
            this.nReadOperations = 0;
            return tmp;
        }

        long getNRepeatIteratorReads() {
            return this.nRepeatIteratorReads;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            long start = DbLsn.makeLsn(this.fileNum, this.startOffset);
            long end = DbLsn.makeLsn(this.fileNum, this.endOffset);
            sb.append("window covers ");
            sb.append(DbLsn.getNoFormatString(start)).append(" to ");
            sb.append(DbLsn.getNoFormatString(end));
            sb.append(" positioned at ");
            long target = this.getCurrentLsn();
            sb.append(DbLsn.getNoFormatString(target));
            return sb.toString();
        }

        long getCurrentLsn() {
            return DbLsn.makeLsn(this.fileNum, this.startOffset + (long)this.readBuffer.position());
        }
    }

    public static class EOFException
    extends Exception {
        public EOFException() {
        }

        public EOFException(String message) {
            super(message);
        }
    }
}

