/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

"use client";

import { createContext, useContext, useEffect, useState } from "react";
import { ThemeProvider as MuiThemeProvider, createTheme } from "@mui/material/styles";
import CssBaseline from "@mui/material/CssBaseline";

type ThemeContextType = {
    isDarkMode: boolean;
    toggleTheme: () => void;
};

const ThemeContext = createContext<ThemeContextType>({
    isDarkMode: false,
    toggleTheme: () => {},
});

export const useTheme = () => useContext(ThemeContext);

export function ThemeProvider({ children }: { children: React.ReactNode }) {
    const [isDarkMode, setIsDarkMode] = useState(false);
    const [mounted, setMounted] = useState(false);

    useEffect(() => {
        // Check if user has already set a preference
        const storedTheme = localStorage.getItem("theme");
        const prefersDark = window.matchMedia("(prefers-color-scheme: dark)").matches;

        const shouldBeDark = storedTheme === "dark" || (!storedTheme && prefersDark);

        // Apply dark mode styles
        if (shouldBeDark) {
            document.documentElement.classList.add("dark");
            // Also add the special class to navbar if it exists
            document.getElementById("navbar")?.classList.add("navbar-dark-mode");
        } else {
            document.documentElement.classList.remove("dark");
            document.getElementById("navbar")?.classList.remove("navbar-dark-mode");
        }

        setIsDarkMode(shouldBeDark);
        setMounted(true);
    }, []);

    const toggleTheme = () => {
        setIsDarkMode((prev) => {
            const newMode = !prev;
            if (newMode) {
                document.documentElement.classList.add("dark");
                document.getElementById("navbar")?.classList.add("navbar-dark-mode");
                localStorage.setItem("theme", "dark");
            } else {
                document.documentElement.classList.remove("dark");
                document.getElementById("navbar")?.classList.remove("navbar-dark-mode");
                localStorage.setItem("theme", "light");
            }
            return newMode;
        });
    };

    // Create MUI theme based on current mode
    const theme = createTheme({
        palette: {
            mode: isDarkMode ? "dark" : "light",
            primary: {
                main: "#1976d2",
                light: "#42a5f5",
                dark: "#1565c0",
                contrastText: "#fff",
            },
            secondary: {
                main: "#9c27b0",
                light: "#ba68c8",
                dark: "#7b1fa2",
                contrastText: "#fff",
            },
            background: {
                default: isDarkMode ? "#121212" : "#fafafa",
                paper: isDarkMode ? "#1e1e1e" : "#ffffff",
            },
        },
        components: {
            MuiPaper: {
                styleOverrides: {
                    root: {
                        transition: "background-color 0.3s ease",
                    },
                },
            },
        },
    });

    return (
        <ThemeContext.Provider value={{ isDarkMode, toggleTheme }}>
            <MuiThemeProvider theme={theme}>
                <CssBaseline />
                {children}
            </MuiThemeProvider>
        </ThemeContext.Provider>
    );
}
