/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools;

import com.fasterxml.jackson.databind.ObjectReader;
import java.io.IOException;
import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import joptsimple.OptionParser;
import org.apache.kafka.clients.admin.AbstractOptions;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.DescribeTopicsOptions;
import org.apache.kafka.clients.admin.ListOffsetsOptions;
import org.apache.kafka.clients.admin.ListOffsetsResult;
import org.apache.kafka.clients.admin.OffsetSpec;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.LeaderNotAvailableException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.util.CommandLineUtils;
import org.apache.kafka.tools.consumer.group.CsvUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OffsetsUtils {
    public static final Logger LOGGER = LoggerFactory.getLogger(OffsetsUtils.class);
    private static final String TOPIC_PARTITION_SEPARATOR = ":";
    private final Admin adminClient;
    private final OffsetsUtilsOptions opts;
    private final OptionParser parser;

    public OffsetsUtils(Admin adminClient, OptionParser parser, OffsetsUtilsOptions opts) {
        this.adminClient = adminClient;
        this.opts = opts;
        this.parser = parser;
    }

    public static void printOffsetsToReset(Map<String, Map<TopicPartition, OffsetAndMetadata>> groupAssignmentsToReset) {
        int maxGroupLen = Math.max(15, groupAssignmentsToReset.keySet().stream().mapToInt(String::length).max().orElse(0));
        int maxTopicLen = Math.max(15, groupAssignmentsToReset.values().stream().flatMap(assignments -> assignments.keySet().stream()).mapToInt(tp -> tp.topic().length()).max().orElse(0));
        String format = "%n%" + -maxGroupLen + "s %" + -maxTopicLen + "s %-10s %s";
        if (!groupAssignmentsToReset.isEmpty()) {
            System.out.printf(format, "GROUP", "TOPIC", "PARTITION", "NEW-OFFSET");
        }
        groupAssignmentsToReset.forEach((groupId, assignment) -> assignment.forEach((consumerAssignment, offsetAndMetadata) -> System.out.printf(format, groupId, consumerAssignment.topic(), consumerAssignment.partition(), offsetAndMetadata.offset())));
        System.out.println();
    }

    public Optional<Map<String, Map<TopicPartition, OffsetAndMetadata>>> resetPlanFromFile() {
        if (this.opts.resetFromFileOpt != null && !this.opts.resetFromFileOpt.isEmpty()) {
            try {
                String resetPlanPath = this.opts.resetFromFileOpt.get(0);
                String resetPlanCsv = Utils.readFileAsString((String)resetPlanPath);
                Map<String, Map<TopicPartition, OffsetAndMetadata>> resetPlan = this.parseResetPlan(resetPlanCsv);
                return Optional.of(resetPlan);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return Optional.empty();
    }

    private Map<String, Map<TopicPartition, OffsetAndMetadata>> parseResetPlan(String resetPlanCsv) {
        ObjectReader csvReader = CsvUtils.readerFor(CsvUtils.CsvRecordNoGroup.class);
        String[] lines = resetPlanCsv.split("\n");
        boolean isSingleGroupQuery = this.opts.groupOpt.size() == 1;
        boolean isOldCsvFormat = false;
        try {
            if (lines.length > 0) {
                csvReader.readValue(lines[0], CsvUtils.CsvRecordNoGroup.class);
                isOldCsvFormat = true;
            }
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        HashMap<String, Map<TopicPartition, OffsetAndMetadata>> dataMap = new HashMap<String, Map<TopicPartition, OffsetAndMetadata>>();
        try {
            if (isSingleGroupQuery && isOldCsvFormat) {
                String group = this.opts.groupOpt.get(0);
                for (String line : lines) {
                    CsvUtils.CsvRecordNoGroup rec = (CsvUtils.CsvRecordNoGroup)csvReader.readValue(line, CsvUtils.CsvRecordNoGroup.class);
                    dataMap.computeIfAbsent(group, k -> new HashMap()).put(new TopicPartition(rec.getTopic(), rec.getPartition()), new OffsetAndMetadata(rec.getOffset()));
                }
            } else {
                csvReader = CsvUtils.readerFor(CsvUtils.CsvRecordWithGroup.class);
                for (String line : lines) {
                    CsvUtils.CsvRecordWithGroup rec = (CsvUtils.CsvRecordWithGroup)csvReader.readValue(line, CsvUtils.CsvRecordWithGroup.class);
                    dataMap.computeIfAbsent(rec.getGroup(), k -> new HashMap()).put(new TopicPartition(rec.getTopic(), rec.getPartition()), new OffsetAndMetadata(rec.getOffset()));
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return dataMap;
    }

    private Map<TopicPartition, Long> checkOffsetsRange(Map<TopicPartition, Long> requestedOffsets) {
        Map<TopicPartition, LogOffsetResult> logStartOffsets = this.getLogStartOffsets(requestedOffsets.keySet());
        Map<TopicPartition, LogOffsetResult> logEndOffsets = this.getLogEndOffsets(requestedOffsets.keySet());
        HashMap<TopicPartition, Long> res = new HashMap<TopicPartition, Long>();
        requestedOffsets.forEach((topicPartition, offset) -> {
            LogOffsetResult logEndOffset = (LogOffsetResult)logEndOffsets.get(topicPartition);
            if (logEndOffset != null) {
                if (logEndOffset instanceof LogOffset && offset > ((LogOffset)logEndOffset).value) {
                    long endOffset = ((LogOffset)logEndOffset).value;
                    LOGGER.warn("New offset (" + offset + ") is higher than latest offset for topic partition " + String.valueOf(topicPartition) + ". Value will be set to " + endOffset);
                    res.put((TopicPartition)topicPartition, endOffset);
                } else {
                    LogOffsetResult logStartOffset = (LogOffsetResult)logStartOffsets.get(topicPartition);
                    if (logStartOffset instanceof LogOffset && offset < ((LogOffset)logStartOffset).value) {
                        long startOffset = ((LogOffset)logStartOffset).value;
                        LOGGER.warn("New offset (" + offset + ") is lower than earliest offset for topic partition " + String.valueOf(topicPartition) + ". Value will be set to " + startOffset);
                        res.put((TopicPartition)topicPartition, startOffset);
                    } else {
                        res.put((TopicPartition)topicPartition, (Long)offset);
                    }
                }
            } else {
                throw new IllegalStateException("Unexpected non-existing offset value for topic partition " + String.valueOf(topicPartition));
            }
        });
        return res;
    }

    private Map<TopicPartition, LogOffsetResult> getLogTimestampOffsets(Collection<TopicPartition> topicPartitions, long timestamp) {
        try {
            Map timestampOffsets = topicPartitions.stream().collect(Collectors.toMap(Function.identity(), tp -> OffsetSpec.forTimestamp((long)timestamp)));
            Map offsets = (Map)this.adminClient.listOffsets(timestampOffsets, this.withTimeoutMs(new ListOffsetsOptions())).all().get();
            HashMap successfulOffsetsForTimes = new HashMap();
            HashMap<TopicPartition, ListOffsetsResult.ListOffsetsResultInfo> unsuccessfulOffsetsForTimes = new HashMap<TopicPartition, ListOffsetsResult.ListOffsetsResultInfo>();
            offsets.forEach((tp, offsetsResultInfo) -> {
                if (offsetsResultInfo.offset() != -1L) {
                    successfulOffsetsForTimes.put(tp, offsetsResultInfo);
                } else {
                    unsuccessfulOffsetsForTimes.put((TopicPartition)tp, (ListOffsetsResult.ListOffsetsResultInfo)offsetsResultInfo);
                }
            });
            Map<TopicPartition, LogOffsetResult> successfulLogTimestampOffsets = successfulOffsetsForTimes.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> new LogOffset(((ListOffsetsResult.ListOffsetsResultInfo)e.getValue()).offset())));
            unsuccessfulOffsetsForTimes.forEach((tp, offsetResultInfo) -> System.out.println("\nWarn: Partition " + tp.partition() + " from topic " + tp.topic() + " is empty. Falling back to latest known offset."));
            successfulLogTimestampOffsets.putAll(this.getLogEndOffsets(unsuccessfulOffsetsForTimes.keySet()));
            return successfulLogTimestampOffsets;
        }
        catch (InterruptedException | ExecutionException e2) {
            throw new RuntimeException(e2);
        }
    }

    private Map<TopicPartition, LogOffsetResult> getLogStartOffsets(Collection<TopicPartition> topicPartitions) {
        return this.getLogOffsets(topicPartitions, OffsetSpec.earliest());
    }

    public Map<TopicPartition, LogOffsetResult> getLogEndOffsets(Collection<TopicPartition> topicPartitions) {
        return this.getLogOffsets(topicPartitions, OffsetSpec.latest());
    }

    public Map<TopicPartition, LogOffsetResult> getLogOffsets(Collection<TopicPartition> topicPartitions, OffsetSpec offsetSpec) {
        try {
            Map startOffsets = topicPartitions.stream().collect(Collectors.toMap(Function.identity(), tp -> offsetSpec));
            Map offsets = (Map)this.adminClient.listOffsets(startOffsets, this.withTimeoutMs(new ListOffsetsOptions())).all().get();
            return topicPartitions.stream().collect(Collectors.toMap(Function.identity(), tp -> offsets.containsKey(tp) ? new LogOffset(((ListOffsetsResult.ListOffsetsResultInfo)offsets.get(tp)).offset()) : new Unknown()));
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public List<TopicPartition> parseTopicPartitionsToReset(List<String> topicArgs) throws ExecutionException, InterruptedException {
        ArrayList topicsWithPartitions = new ArrayList();
        ArrayList topics = new ArrayList();
        topicArgs.forEach(topicArg -> {
            if (topicArg.contains(TOPIC_PARTITION_SEPARATOR)) {
                topicsWithPartitions.add(topicArg);
            } else {
                topics.add(topicArg);
            }
        });
        List<TopicPartition> specifiedPartitions = topicsWithPartitions.stream().flatMap(this::parseTopicsWithPartitions).collect(Collectors.toList());
        ArrayList unspecifiedPartitions = new ArrayList();
        if (!topics.isEmpty()) {
            Map descriptionMap = (Map)this.adminClient.describeTopics(topics, this.withTimeoutMs(new DescribeTopicsOptions())).allTopicNames().get();
            descriptionMap.forEach((topic, description) -> description.partitions().forEach(tpInfo -> unspecifiedPartitions.add(new TopicPartition(topic, tpInfo.partition()))));
        }
        specifiedPartitions.addAll(unspecifiedPartitions);
        return specifiedPartitions;
    }

    public Stream<TopicPartition> parseTopicsWithPartitions(String topicArg) {
        ToIntFunction<String> partitionNum = partition -> {
            try {
                return Integer.parseInt(partition);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid partition '" + partition + "' specified in topic arg '" + topicArg + "''");
            }
        };
        String[] arr = topicArg.split(TOPIC_PARTITION_SEPARATOR);
        if (arr.length != 2) {
            throw new IllegalArgumentException("Invalid topic arg '" + topicArg + "', expected topic name and partitions");
        }
        String topic = arr[0];
        String partitions = arr[1];
        return Arrays.stream(partitions.split(",")).map(partition -> new TopicPartition(topic, partitionNum.applyAsInt((String)partition)));
    }

    public Map<TopicPartition, OffsetAndMetadata> resetToOffset(Collection<TopicPartition> partitionsToReset) {
        long offset = this.opts.resetToOffsetOpt != null && !this.opts.resetToOffsetOpt.isEmpty() ? this.opts.resetToOffsetOpt.get(0) : 0L;
        return this.checkOffsetsRange(partitionsToReset.stream().collect(Collectors.toMap(Function.identity(), tp -> offset))).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> new OffsetAndMetadata(((Long)e.getValue()).longValue())));
    }

    public Map<TopicPartition, OffsetAndMetadata> resetToEarliest(Collection<TopicPartition> partitionsToReset) {
        Map<TopicPartition, LogOffsetResult> logStartOffsets = this.getLogStartOffsets(partitionsToReset);
        return partitionsToReset.stream().collect(Collectors.toMap(Function.identity(), topicPartition -> {
            LogOffsetResult logOffsetResult = (LogOffsetResult)logStartOffsets.get(topicPartition);
            if (!(logOffsetResult instanceof LogOffset)) {
                CommandLineUtils.printUsageAndExit((OptionParser)this.parser, (String)("Error getting starting offset of topic partition: " + String.valueOf(topicPartition)));
            }
            return new OffsetAndMetadata(((LogOffset)logOffsetResult).value);
        }));
    }

    public Map<TopicPartition, OffsetAndMetadata> resetToLatest(Collection<TopicPartition> partitionsToReset) {
        Map<TopicPartition, LogOffsetResult> logEndOffsets = this.getLogEndOffsets(partitionsToReset);
        return partitionsToReset.stream().collect(Collectors.toMap(Function.identity(), topicPartition -> {
            LogOffsetResult logOffsetResult = (LogOffsetResult)logEndOffsets.get(topicPartition);
            if (!(logOffsetResult instanceof LogOffset)) {
                CommandLineUtils.printUsageAndExit((OptionParser)this.parser, (String)("Error getting ending offset of topic partition: " + String.valueOf(topicPartition)));
            }
            return new OffsetAndMetadata(((LogOffset)logOffsetResult).value);
        }));
    }

    public Map<TopicPartition, OffsetAndMetadata> resetByShiftBy(Collection<TopicPartition> partitionsToReset, Map<TopicPartition, OffsetAndMetadata> currentCommittedOffsets) {
        Map<TopicPartition, Long> requestedOffsets = partitionsToReset.stream().collect(Collectors.toMap(Function.identity(), topicPartition -> {
            long shiftBy = this.opts.resetShiftByOpt;
            OffsetAndMetadata currentOffset = (OffsetAndMetadata)currentCommittedOffsets.get(topicPartition);
            if (currentOffset == null) {
                throw new IllegalArgumentException("Cannot shift offset for partition " + String.valueOf(topicPartition) + " since there is no current committed offset");
            }
            return currentOffset.offset() + shiftBy;
        }));
        return this.checkOffsetsRange(requestedOffsets).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> new OffsetAndMetadata(((Long)e.getValue()).longValue())));
    }

    public Map<TopicPartition, OffsetAndMetadata> resetToDateTime(Collection<TopicPartition> partitionsToReset) {
        try {
            long timestamp = Utils.getDateTime((String)this.opts.resetToDatetimeOpt.get(0));
            Map<TopicPartition, LogOffsetResult> logTimestampOffsets = this.getLogTimestampOffsets(partitionsToReset, timestamp);
            return partitionsToReset.stream().collect(Collectors.toMap(Function.identity(), topicPartition -> {
                LogOffsetResult logTimestampOffset = (LogOffsetResult)logTimestampOffsets.get(topicPartition);
                if (!(logTimestampOffset instanceof LogOffset)) {
                    CommandLineUtils.printUsageAndExit((OptionParser)this.parser, (String)("Error getting offset by timestamp of topic partition: " + String.valueOf(topicPartition)));
                }
                return new OffsetAndMetadata(((LogOffset)logTimestampOffset).value);
            }));
        }
        catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    public Map<TopicPartition, OffsetAndMetadata> resetByDuration(Collection<TopicPartition> partitionsToReset) {
        String duration = this.opts.resetByDurationOpt;
        Duration durationParsed = Duration.parse(duration);
        Instant now = Instant.now();
        durationParsed.negated().addTo(now);
        long timestamp = now.minus(durationParsed).toEpochMilli();
        Map<TopicPartition, LogOffsetResult> logTimestampOffsets = this.getLogTimestampOffsets(partitionsToReset, timestamp);
        return partitionsToReset.stream().collect(Collectors.toMap(Function.identity(), topicPartition -> {
            LogOffsetResult logTimestampOffset = (LogOffsetResult)logTimestampOffsets.get(topicPartition);
            if (!(logTimestampOffset instanceof LogOffset)) {
                CommandLineUtils.printUsageAndExit((OptionParser)this.parser, (String)("Error getting offset by timestamp of topic partition: " + String.valueOf(topicPartition)));
            }
            return new OffsetAndMetadata(((LogOffset)logTimestampOffset).value);
        }));
    }

    public Map<TopicPartition, OffsetAndMetadata> resetFromFile(String groupId) {
        return this.resetPlanFromFile().map(resetPlan -> {
            Map resetPlanForGroup = (Map)resetPlan.get(groupId);
            if (resetPlanForGroup == null) {
                OffsetsUtils.printError("No reset plan for group " + groupId + " found", Optional.empty());
                return Map.of();
            }
            Map<TopicPartition, Long> requestedOffsets = resetPlanForGroup.keySet().stream().collect(Collectors.toMap(Function.identity(), topicPartition -> ((OffsetAndMetadata)resetPlanForGroup.get(topicPartition)).offset()));
            return this.checkOffsetsRange(requestedOffsets).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> new OffsetAndMetadata(((Long)e.getValue()).longValue())));
        }).orElseGet(Map::of);
    }

    public Map<TopicPartition, OffsetAndMetadata> resetToCurrent(Collection<TopicPartition> partitionsToReset, Map<TopicPartition, OffsetAndMetadata> currentCommittedOffsets) {
        ArrayList<TopicPartition> partitionsToResetWithCommittedOffset = new ArrayList<TopicPartition>();
        ArrayList<TopicPartition> partitionsToResetWithoutCommittedOffset = new ArrayList<TopicPartition>();
        for (TopicPartition topicPartition2 : partitionsToReset) {
            if (currentCommittedOffsets.containsKey(topicPartition2)) {
                partitionsToResetWithCommittedOffset.add(topicPartition2);
                continue;
            }
            partitionsToResetWithoutCommittedOffset.add(topicPartition2);
        }
        Map<TopicPartition, OffsetAndMetadata> preparedOffsetsForPartitionsWithCommittedOffset = partitionsToResetWithCommittedOffset.stream().collect(Collectors.toMap(Function.identity(), topicPartition -> {
            OffsetAndMetadata committedOffset = (OffsetAndMetadata)currentCommittedOffsets.get(topicPartition);
            if (committedOffset == null) {
                throw new IllegalStateException("Expected a valid current offset for topic partition: " + String.valueOf(topicPartition));
            }
            return new OffsetAndMetadata(committedOffset.offset());
        }));
        Map<TopicPartition, OffsetAndMetadata> preparedOffsetsForPartitionsWithoutCommittedOffset = this.getLogEndOffsets(partitionsToResetWithoutCommittedOffset).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> {
            if (!(e.getValue() instanceof LogOffset)) {
                CommandLineUtils.printUsageAndExit((OptionParser)this.parser, (String)("Error getting ending offset of topic partition: " + String.valueOf(e.getKey())));
            }
            return new OffsetAndMetadata(((LogOffset)e.getValue()).value);
        }));
        preparedOffsetsForPartitionsWithCommittedOffset.putAll(preparedOffsetsForPartitionsWithoutCommittedOffset);
        return preparedOffsetsForPartitionsWithCommittedOffset;
    }

    public void checkAllTopicPartitionsValid(Collection<TopicPartition> partitionsToReset) {
        List<TopicPartition> partitionsNotExistList = this.filterNonExistentPartitions(partitionsToReset);
        if (!partitionsNotExistList.isEmpty()) {
            String partitionStr = partitionsNotExistList.stream().map(TopicPartition::toString).collect(Collectors.joining(","));
            throw new UnknownTopicOrPartitionException("The partitions \"" + partitionStr + "\" do not exist");
        }
        List<TopicPartition> partitionsWithoutLeader = this.filterNoneLeaderPartitions(partitionsToReset);
        if (!partitionsWithoutLeader.isEmpty()) {
            String partitionStr = partitionsWithoutLeader.stream().map(TopicPartition::toString).collect(Collectors.joining(","));
            throw new LeaderNotAvailableException("The partitions \"" + partitionStr + "\" have no leader");
        }
    }

    public List<TopicPartition> filterNoneLeaderPartitions(Collection<TopicPartition> topicPartitions) {
        Set topics = topicPartitions.stream().map(TopicPartition::topic).collect(Collectors.toSet());
        try {
            return ((Map)this.adminClient.describeTopics(topics, this.withTimeoutMs(new DescribeTopicsOptions())).allTopicNames().get()).entrySet().stream().flatMap(entry -> ((TopicDescription)entry.getValue()).partitions().stream().filter(partitionInfo -> partitionInfo.leader() == null).map(partitionInfo -> new TopicPartition((String)entry.getKey(), partitionInfo.partition()))).filter(topicPartitions::contains).toList();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public List<TopicPartition> filterNonExistentPartitions(Collection<TopicPartition> topicPartitions) {
        Set topics = topicPartitions.stream().map(TopicPartition::topic).collect(Collectors.toSet());
        try {
            List existPartitions = ((Map)this.adminClient.describeTopics(topics, this.withTimeoutMs(new DescribeTopicsOptions())).allTopicNames().get()).entrySet().stream().flatMap(entry -> ((TopicDescription)entry.getValue()).partitions().stream().map(partitionInfo -> new TopicPartition((String)entry.getKey(), partitionInfo.partition()))).toList();
            return topicPartitions.stream().filter(tp -> !existPartitions.contains(tp)).toList();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private <T extends AbstractOptions<T>> T withTimeoutMs(T options) {
        int t = (int)this.opts.timeoutMsOpt;
        return (T)options.timeoutMs(Integer.valueOf(t));
    }

    private static void printError(String msg, Optional<Throwable> e) {
        System.out.println("\nError: " + msg);
        e.ifPresent(Throwable::printStackTrace);
    }

    public static class OffsetsUtilsOptions {
        List<String> groupOpt;
        List<Long> resetToOffsetOpt;
        List<String> resetFromFileOpt;
        List<String> resetToDatetimeOpt;
        String resetByDurationOpt;
        Long resetShiftByOpt;
        long timeoutMsOpt;

        public OffsetsUtilsOptions(List<String> groupOpt, List<Long> resetToOffsetOpt, List<String> resetFromFileOpt, List<String> resetToDatetimeOpt, String resetByDurationOpt, Long resetShiftByOpt, long timeoutMsOpt) {
            this.groupOpt = groupOpt;
            this.resetToOffsetOpt = resetToOffsetOpt;
            this.resetFromFileOpt = resetFromFileOpt;
            this.resetToDatetimeOpt = resetToDatetimeOpt;
            this.resetByDurationOpt = resetByDurationOpt;
            this.resetShiftByOpt = resetShiftByOpt;
            this.timeoutMsOpt = timeoutMsOpt;
        }

        public OffsetsUtilsOptions(List<String> groupOpt, List<String> resetToDatetimeOpt, long timeoutMsOpt) {
            this.groupOpt = groupOpt;
            this.resetToDatetimeOpt = resetToDatetimeOpt;
            this.timeoutMsOpt = timeoutMsOpt;
        }
    }

    public record LogOffset(long value) implements LogOffsetResult
    {
    }

    public static interface LogOffsetResult {
    }

    public static class Unknown
    implements LogOffsetResult {
    }

    public static class Ignore
    implements LogOffsetResult {
    }
}

