/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.inlong.manager.client.api.service;

import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;

import java.io.IOException;
import java.util.function.Supplier;
/**
 * AuthInterceptor
 * Before okhttp call a request, uniformly encapsulate the relevant parameters of authentication
 */
public class AuthInterceptor implements Interceptor {

    private final String username;
    private final String password;
    private final Supplier<String> tenantGetter;

    public AuthInterceptor(String username, String password, Supplier<String> tenantGetter) {
        this.username = username;
        this.password = password;
        this.tenantGetter = tenantGetter;
    }

    @Override
    public Response intercept(Chain chain) throws IOException {
        Request oldRequest = chain.request();
        HttpUrl.Builder builder = oldRequest.url()
                .newBuilder()
                .addEncodedQueryParameter("username", username)
                .addEncodedQueryParameter("password", password);

        Request newRequest = oldRequest.newBuilder()
                .method(oldRequest.method(), oldRequest.body())
                .url(builder.build())
                .addHeader("tenant", tenantGetter.get())
                .build();
        return chain.proceed(newRequest);
    }
}
